<?php
/**
 * Authentication Handler
 * Manages user authentication, sessions, and access control
 */

class Auth
{
    private PDO $db;
    private static ?Auth $instance = null;

    public function __construct()
    {
        $this->db = getDB();
    }

    public static function getInstance(): Auth
    {
        if (self::$instance === null) {
            self::$instance = new Auth();
        }
        return self::$instance;
    }

    /**
     * Start secure session
     */
    public static function startSession(): void
    {
        if (session_status() === PHP_SESSION_NONE) {
            // Secure session settings
            ini_set('session.use_strict_mode', 1);
            ini_set('session.cookie_httponly', 1);
            ini_set('session.use_only_cookies', 1);

            session_start();
        }
    }

    /**
     * Check if user is logged in
     */
    public static function isLoggedIn(): bool
    {
        self::startSession();

        if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
            return false;
        }

        if (!isset($_SESSION['user_id']) || !isset($_SESSION['login_time'])) {
            return false;
        }

        // Check session timeout (8 hours)
        $timeout = 8 * 60 * 60;
        if (time() - $_SESSION['login_time'] > $timeout) {
            self::logout();
            return false;
        }

        // IP check disabled - can cause issues with proxies/load balancers
        // if (isset($_SESSION['ip']) && $_SESSION['ip'] !== $_SERVER['REMOTE_ADDR']) {
        //     self::logout();
        //     return false;
        // }

        return true;
    }

    /**
     * Require login - redirect if not authenticated
     */
    public static function requireLogin(): void
    {
        if (!self::isLoggedIn()) {
            header('Location: login.php');
            exit;
        }
    }

    /**
     * Require admin role
     */
    public static function requireAdmin(): void
    {
        self::requireLogin();
        if ($_SESSION['role'] !== 'admin') {
            http_response_code(403);
            die('Access denied');
        }
    }

    /**
     * Get current user
     */
    public static function getCurrentUser(): ?array
    {
        if (!self::isLoggedIn()) {
            return null;
        }

        return [
            'id' => $_SESSION['user_id'],
            'username' => $_SESSION['username'],
            'role' => $_SESSION['role']
        ];
    }

    /**
     * Logout user
     */
    public static function logout(): void
    {
        self::startSession();

        $_SESSION = [];

        if (ini_get("session.use_cookies")) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000,
                $params["path"], $params["domain"],
                $params["secure"], $params["httponly"]
            );
        }

        session_destroy();
    }

    /**
     * Create password hash
     */
    public static function hashPassword(string $password): string
    {
        return password_hash($password, PASSWORD_BCRYPT, ['cost' => 12]);
    }

    /**
     * Create new user
     */
    public function createUser(string $username, string $password, string $role = 'user', string $email = ''): ?int
    {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO users (username, password, email, role, is_active, created_at)
                VALUES (?, ?, ?, ?, 1, NOW())
            ");

            $stmt->execute([
                $username,
                self::hashPassword($password),
                $email,
                $role
            ]);

            return (int) $this->db->lastInsertId();
        } catch (PDOException $e) {
            return null;
        }
    }

    /**
     * Update user password
     */
    public function updatePassword(int $userId, string $newPassword): bool
    {
        try {
            $stmt = $this->db->prepare("UPDATE users SET password = ?, updated_at = NOW() WHERE id = ?");
            return $stmt->execute([self::hashPassword($newPassword), $userId]);
        } catch (PDOException $e) {
            return false;
        }
    }

    /**
     * Get all users
     */
    public function getAllUsers(): array
    {
        $stmt = $this->db->query("SELECT id, username, email, role, is_active, last_login, login_ip, created_at FROM users ORDER BY created_at DESC");
        return $stmt->fetchAll();
    }

    /**
     * Delete user
     */
    public function deleteUser(int $userId): bool
    {
        // Don't delete the last admin
        $stmt = $this->db->query("SELECT COUNT(*) FROM users WHERE role = 'admin' AND is_active = 1");
        $adminCount = $stmt->fetchColumn();

        $stmt = $this->db->prepare("SELECT role FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();

        if ($user && $user['role'] === 'admin' && $adminCount <= 1) {
            return false; // Can't delete the last admin
        }

        $stmt = $this->db->prepare("DELETE FROM users WHERE id = ?");
        return $stmt->execute([$userId]);
    }

    /**
     * Toggle user active status
     */
    public function toggleUser(int $userId): bool
    {
        $stmt = $this->db->prepare("UPDATE users SET is_active = NOT is_active WHERE id = ?");
        return $stmt->execute([$userId]);
    }

    /**
     * Generate API token
     */
    public function generateApiToken(int $userId): ?string
    {
        $token = bin2hex(random_bytes(32));
        $hashedToken = hash('sha256', $token);

        try {
            $stmt = $this->db->prepare("UPDATE users SET api_token = ?, updated_at = NOW() WHERE id = ?");
            $stmt->execute([$hashedToken, $userId]);
            return $token;
        } catch (PDOException $e) {
            return null;
        }
    }

    /**
     * Validate API token
     */
    public function validateApiToken(string $token): ?array
    {
        $hashedToken = hash('sha256', $token);

        $stmt = $this->db->prepare("SELECT id, username, role FROM users WHERE api_token = ? AND is_active = 1");
        $stmt->execute([$hashedToken]);

        return $stmt->fetch() ?: null;
    }
}
