<?php
/**
 * VOD Import System - Cron Worker
 * Run via cron: * * * * * php /var/www/html/VOD/workers/cron.php
 */

// Prevent web access
if (php_sapi_name() !== 'cli') {
    die('CLI only');
}

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../classes/ServerManager.php';
require_once __DIR__ . '/../classes/MovieImporter.php';
require_once __DIR__ . '/../classes/FolderWatch.php';
require_once __DIR__ . '/../classes/Logger.php';

$logger = new Logger('cron');
$logger->info('Cron job started');

// Get command line arguments
$action = $argv[1] ?? 'all';

try {
    switch ($action) {
        case 'check-servers':
            checkServers($logger);
            break;

        case 'smart-sync':
            smartSyncAll($logger);
            break;

        case 'check-links':
            checkBrokenLinks($logger);
            break;

        case 'process-watch':
            processWatchFolder($logger);
            break;

        case 'cleanup':
            runCleanup($logger);
            break;

        case 'all':
        default:
            // Run all tasks based on intervals
            runScheduledTasks($logger);
            break;
    }
} catch (Exception $e) {
    $logger->error('Cron job failed', ['error' => $e->getMessage()]);
}

$logger->info('Cron job completed');

/**
 * Check all active servers
 */
function checkServers(Logger $logger): void
{
    $sm = new ServerManager();
    $servers = $sm->getAllServers(true);

    $logger->info('Checking servers', ['count' => count($servers)]);

    foreach ($servers as $server) {
        $result = $sm->checkServer($server['id']);
        $logger->info('Server checked', [
            'server' => $server['name'],
            'status' => $result['status'] ?? 'unknown'
        ]);
    }
}

/**
 * Smart sync all active servers
 */
function smartSyncAll(Logger $logger): void
{
    $sm = new ServerManager();
    $importer = new MovieImporter();
    $servers = $sm->getAllServers(true);

    $logger->info('Starting smart sync for all servers');

    $totalNew = 0;

    foreach ($servers as $server) {
        if ($server['type'] !== 'xtream') continue;

        $result = $importer->smartSync($server['id']);

        if ($result['success']) {
            $totalNew += $result['imported'];
            $logger->info('Smart sync completed', [
                'server' => $server['name'],
                'new' => $result['new_found'],
                'imported' => $result['imported']
            ]);
        } else {
            $logger->error('Smart sync failed', [
                'server' => $server['name'],
                'error' => $result['error'] ?? 'Unknown'
            ]);
        }

        // Delay between servers
        sleep(5);
    }

    $logger->info('Smart sync completed for all servers', ['total_imported' => $totalNew]);
}

/**
 * Check and update broken links
 */
function checkBrokenLinks(Logger $logger): void
{
    $db = getDB();
    $importer = new MovieImporter();

    // Get movies with links that haven't been checked recently
    $stmt = $db->query("
        SELECT DISTINCT m.id
        FROM movies m
        JOIN movie_links ml ON m.id = ml.movie_id
        WHERE ml.last_check IS NULL
           OR ml.last_check < DATE_SUB(NOW(), INTERVAL 24 HOUR)
        ORDER BY ml.last_check ASC
        LIMIT 100
    ");

    $movies = $stmt->fetchAll(PDO::FETCH_COLUMN);

    $logger->info('Checking movie links', ['movies' => count($movies)]);

    $broken = 0;
    $working = 0;

    foreach ($movies as $movieId) {
        $results = $importer->checkMovieLinks($movieId);

        foreach ($results as $result) {
            if ($result['is_working']) {
                $working++;
            } else {
                $broken++;
            }
        }

        // Delay between checks
        usleep(500000); // 500ms
    }

    $logger->info('Link check completed', ['working' => $working, 'broken' => $broken]);
}

/**
 * Process watch folder
 */
function processWatchFolder(Logger $logger): void
{
    $fw = new FolderWatch();

    $scan = $fw->scanWatchFolder();
    $logger->info('Watch folder scanned', [
        'valid' => count($scan['valid']),
        'invalid' => count($scan['invalid'])
    ]);

    if (!empty($scan['valid'])) {
        $result = $fw->processWatchFolder();
        $logger->info('Watch folder processed', [
            'processed' => count($result['processed']),
            'errors' => count($result['errors'])
        ]);
    }

    // Auto cleanup orphans if enabled
    $fw->autoCleanupOrphans();
}

/**
 * Run cleanup tasks
 */
function runCleanup(Logger $logger): void
{
    $db = getDB();

    // Clear old logs
    $loggerInstance = new Logger('cleanup');
    $deleted = $loggerInstance->clearOldLogs(30);
    $logger->info('Old logs cleared', ['deleted' => $deleted]);

    // Clear old cache files
    $cacheDir = CACHE_PATH;
    $files = glob($cacheDir . '/*');
    $now = time();
    $cleared = 0;

    foreach ($files as $file) {
        if (is_file($file) && ($now - filemtime($file)) > 3600) {
            unlink($file);
            $cleared++;
        }
    }

    $logger->info('Cache cleared', ['files' => $cleared]);

    // Clear completed queue items older than 7 days
    $stmt = $db->prepare("
        DELETE FROM import_queue
        WHERE status IN ('completed', 'cancelled')
        AND created_at < DATE_SUB(NOW(), INTERVAL 7 DAY)
    ");
    $stmt->execute();
    $logger->info('Old queue items cleared', ['deleted' => $stmt->rowCount()]);
}

/**
 * Run scheduled tasks based on intervals
 */
function runScheduledTasks(Logger $logger): void
{
    $db = getDB();

    // Check if we should run each task
    $autoSyncInterval = getSetting('auto_sync_interval', 3600);
    $linkCheckInterval = getSetting('check_links_interval', 86400);

    // Get last run times from settings or logs
    $lastSync = getLastRunTime('smart_sync');
    $lastLinkCheck = getLastRunTime('link_check');
    $lastWatch = getLastRunTime('watch_process');

    $now = time();

    // Smart sync (default: hourly)
    if ($now - $lastSync > $autoSyncInterval) {
        smartSyncAll($logger);
        setLastRunTime('smart_sync', $now);
    }

    // Link check (default: daily)
    if ($now - $lastLinkCheck > $linkCheckInterval) {
        checkBrokenLinks($logger);
        setLastRunTime('link_check', $now);
    }

    // Watch folder (every 5 minutes)
    if ($now - $lastWatch > 300) {
        processWatchFolder($logger);
        setLastRunTime('watch_process', $now);
    }

    // Cleanup (daily at midnight - check if it's a new day)
    $lastCleanup = getLastRunTime('cleanup');
    if (date('Y-m-d', $lastCleanup) !== date('Y-m-d')) {
        runCleanup($logger);
        setLastRunTime('cleanup', $now);
    }
}

/**
 * Get last run time for a task
 */
function getLastRunTime(string $task): int
{
    $file = CACHE_PATH . "/cron_{$task}.time";

    if (file_exists($file)) {
        return (int) file_get_contents($file);
    }

    return 0;
}

/**
 * Set last run time for a task
 */
function setLastRunTime(string $task, int $time): void
{
    $file = CACHE_PATH . "/cron_{$task}.time";
    file_put_contents($file, $time);
}
