<?php
/**
 * Logger Class
 * Handles logging to files and database
 */

class Logger
{
    private string $channel;
    private string $logPath;
    private ?PDO $db = null;
    private bool $debugMode;

    private const LEVELS = [
        'DEBUG' => 0,
        'INFO' => 1,
        'WARNING' => 2,
        'ERROR' => 3,
        'CRITICAL' => 4
    ];

    public function __construct(string $channel = 'app', ?PDO $db = null)
    {
        $this->channel = $channel;
        $this->logPath = defined('LOGS_PATH') ? LOGS_PATH : __DIR__ . '/../logs';
        $this->debugMode = defined('DEBUG_MODE') ? DEBUG_MODE : false;
        $this->db = $db;

        if (!is_dir($this->logPath)) {
            mkdir($this->logPath, 0755, true);
        }
    }

    /**
     * Log a message
     */
    public function log(string $level, string $message, array $context = []): void
    {
        $level = strtoupper($level);

        // Skip debug messages if debug mode is off
        if ($level === 'DEBUG' && !$this->debugMode) {
            return;
        }

        $timestamp = date('Y-m-d H:i:s');
        $contextStr = !empty($context) ? ' ' . json_encode($context, JSON_UNESCAPED_UNICODE) : '';

        // File log
        $logLine = "[{$timestamp}] [{$this->channel}] {$level}: {$message}{$contextStr}\n";
        $logFile = $this->logPath . '/' . date('Y-m-d') . '.log';

        file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);

        // Also log to channel-specific file
        $channelFile = $this->logPath . '/' . $this->channel . '.log';
        file_put_contents($channelFile, $logLine, FILE_APPEND | LOCK_EX);

        // Database log for important events
        if ($this->db && in_array($level, ['INFO', 'WARNING', 'ERROR', 'CRITICAL'])) {
            $this->logToDatabase($level, $message, $context);
        }
    }

    /**
     * Log to database
     */
    private function logToDatabase(string $level, string $message, array $context): void
    {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO import_logs (action, type, status, message, details, ip_address)
                VALUES (:action, :type, :status, :message, :details, :ip)
            ");

            $status = match ($level) {
                'ERROR', 'CRITICAL' => 'error',
                'WARNING' => 'warning',
                default => 'info'
            };

            $stmt->execute([
                'action' => $context['action'] ?? $this->channel,
                'type' => $context['type'] ?? 'system',
                'status' => $status,
                'message' => $message,
                'details' => !empty($context) ? json_encode($context) : null,
                'ip' => $_SERVER['REMOTE_ADDR'] ?? null
            ]);
        } catch (Exception $e) {
            // Silently fail - don't let logging errors break the application
        }
    }

    /**
     * Debug log
     */
    public function debug(string $message, array $context = []): void
    {
        $this->log('DEBUG', $message, $context);
    }

    /**
     * Info log
     */
    public function info(string $message, array $context = []): void
    {
        $this->log('INFO', $message, $context);
    }

    /**
     * Warning log
     */
    public function warning(string $message, array $context = []): void
    {
        $this->log('WARNING', $message, $context);
    }

    /**
     * Error log
     */
    public function error(string $message, array $context = []): void
    {
        $this->log('ERROR', $message, $context);
    }

    /**
     * Critical log
     */
    public function critical(string $message, array $context = []): void
    {
        $this->log('CRITICAL', $message, $context);
    }

    /**
     * Log HTTP request
     */
    public function logRequest(string $url, string $method, int $responseCode, float $responseTime, ?string $error = null): void
    {
        $this->debug("HTTP Request", [
            'url' => $url,
            'method' => $method,
            'response_code' => $responseCode,
            'response_time_ms' => round($responseTime * 1000),
            'error' => $error
        ]);
    }

    /**
     * Get recent logs from file
     */
    public function getRecentLogs(int $lines = 100, ?string $level = null): array
    {
        $logFile = $this->logPath . '/' . $this->channel . '.log';

        if (!file_exists($logFile)) {
            return [];
        }

        $file = new SplFileObject($logFile, 'r');
        $file->seek(PHP_INT_MAX);
        $totalLines = $file->key();

        $startLine = max(0, $totalLines - $lines);
        $logs = [];

        $file->seek($startLine);
        while (!$file->eof()) {
            $line = trim($file->fgets());
            if (empty($line)) continue;

            // Parse log line
            if (preg_match('/^\[([\d\-\s:]+)\]\s+\[(\w+)\]\s+(\w+):\s+(.*)$/', $line, $matches)) {
                $logEntry = [
                    'timestamp' => $matches[1],
                    'channel' => $matches[2],
                    'level' => $matches[3],
                    'message' => $matches[4]
                ];

                // Filter by level if specified
                if ($level === null || strtoupper($level) === $logEntry['level']) {
                    $logs[] = $logEntry;
                }
            }
        }

        return array_reverse($logs);
    }

    /**
     * Get logs from database
     */
    public function getLogsFromDatabase(array $filters = [], int $limit = 100, int $offset = 0): array
    {
        if (!$this->db) {
            $this->db = getDB();
        }

        $sql = "SELECT * FROM import_logs WHERE 1=1";
        $params = [];

        if (!empty($filters['status'])) {
            $sql .= " AND status = :status";
            $params['status'] = $filters['status'];
        }

        if (!empty($filters['type'])) {
            $sql .= " AND type = :type";
            $params['type'] = $filters['type'];
        }

        if (!empty($filters['action'])) {
            $sql .= " AND action = :action";
            $params['action'] = $filters['action'];
        }

        if (!empty($filters['server_id'])) {
            $sql .= " AND server_id = :server_id";
            $params['server_id'] = $filters['server_id'];
        }

        if (!empty($filters['date_from'])) {
            $sql .= " AND created_at >= :date_from";
            $params['date_from'] = $filters['date_from'];
        }

        if (!empty($filters['date_to'])) {
            $sql .= " AND created_at <= :date_to";
            $params['date_to'] = $filters['date_to'];
        }

        $sql .= " ORDER BY created_at DESC LIMIT :limit OFFSET :offset";

        $stmt = $this->db->prepare($sql);

        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue('limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue('offset', $offset, PDO::PARAM_INT);

        $stmt->execute();

        return $stmt->fetchAll();
    }

    /**
     * Clear old logs
     */
    public function clearOldLogs(int $daysToKeep = 30): int
    {
        $deleted = 0;

        // Clear file logs
        $files = glob($this->logPath . '/*.log');
        $cutoffDate = strtotime("-{$daysToKeep} days");

        foreach ($files as $file) {
            if (filemtime($file) < $cutoffDate) {
                unlink($file);
                $deleted++;
            }
        }

        // Clear database logs
        if ($this->db) {
            try {
                $stmt = $this->db->prepare("
                    DELETE FROM import_logs
                    WHERE created_at < DATE_SUB(NOW(), INTERVAL :days DAY)
                ");
                $stmt->execute(['days' => $daysToKeep]);
                $deleted += $stmt->rowCount();
            } catch (Exception $e) {
                // Ignore
            }
        }

        return $deleted;
    }
}
