#!/bin/bash
#############################################
# VOD Import System - Installation Script
# Run as root: bash install.sh
#############################################

set -e

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

echo -e "${BLUE}"
echo "=============================================="
echo "   VOD Import System - Installation Script"
echo "=============================================="
echo -e "${NC}"

# Check if running as root
if [ "$EUID" -ne 0 ]; then
    echo -e "${RED}Please run as root (sudo bash install.sh)${NC}"
    exit 1
fi

# Configuration
DB_NAME="vod_import"
DB_USER="vod_user"
DB_PASS="VodPass@2024"
ADMIN_USER="admin"
ADMIN_PASS="admin123"
INSTALL_DIR="/var/www/html/VOD"

echo -e "${YELLOW}Starting installation...${NC}"

#############################################
# 1. Update System
#############################################
echo -e "${GREEN}[1/10] Updating system packages...${NC}"
apt-get update -y
apt-get upgrade -y

#############################################
# 2. Install Required Packages
#############################################
echo -e "${GREEN}[2/10] Installing required packages...${NC}"
apt-get install -y \
    nginx \
    mysql-server \
    php8.1-fpm \
    php8.1-mysql \
    php8.1-curl \
    php8.1-json \
    php8.1-mbstring \
    php8.1-xml \
    php8.1-zip \
    php8.1-gd \
    php8.1-ssh2 \
    curl \
    wget \
    git \
    unzip \
    sshpass \
    htop \
    net-tools

# If PHP 8.1 not available, try PHP 8.2 or default
if ! command -v php &> /dev/null; then
    apt-get install -y \
        php-fpm \
        php-mysql \
        php-curl \
        php-json \
        php-mbstring \
        php-xml \
        php-zip \
        php-gd
fi

#############################################
# 3. Configure MySQL
#############################################
echo -e "${GREEN}[3/10] Configuring MySQL...${NC}"

# Start MySQL
systemctl start mysql
systemctl enable mysql

# Secure MySQL and create database
mysql -e "CREATE DATABASE IF NOT EXISTS ${DB_NAME} CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
mysql -e "CREATE USER IF NOT EXISTS '${DB_USER}'@'localhost' IDENTIFIED BY '${DB_PASS}';"
mysql -e "GRANT ALL PRIVILEGES ON ${DB_NAME}.* TO '${DB_USER}'@'localhost';"
mysql -e "FLUSH PRIVILEGES;"

echo -e "${GREEN}Database '${DB_NAME}' created successfully${NC}"

#############################################
# 4. Create Directory Structure
#############################################
echo -e "${GREEN}[4/10] Creating directory structure...${NC}"

mkdir -p ${INSTALL_DIR}/{api,assets/{css,js,img},classes,config,cache,logs,media,watch}
chown -R www-data:www-data ${INSTALL_DIR}
chmod -R 755 ${INSTALL_DIR}
chmod -R 777 ${INSTALL_DIR}/{cache,logs,media,watch}

#############################################
# 5. Create Database Schema
#############################################
echo -e "${GREEN}[5/10] Creating database schema...${NC}"

mysql ${DB_NAME} << 'EOSQL'
-- Users table
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100),
    role ENUM('admin', 'user', 'viewer') DEFAULT 'user',
    is_active TINYINT(1) DEFAULT 1,
    api_token VARCHAR(64),
    last_login DATETIME,
    login_ip VARCHAR(45),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Servers table (Source IPTV servers)
CREATE TABLE IF NOT EXISTS servers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    url VARCHAR(255) NOT NULL,
    port INT DEFAULT 80,
    username VARCHAR(100),
    password VARCHAR(100),
    type ENUM('xtream', 'm3u', 'stalker') DEFAULT 'xtream',
    is_active TINYINT(1) DEFAULT 1,
    status ENUM('online', 'offline', 'unknown') DEFAULT 'unknown',
    last_check DATETIME,
    movies_count INT DEFAULT 0,
    series_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Remote servers table (Download destinations)
CREATE TABLE IF NOT EXISTS remote_servers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    type ENUM('ssh', 'ftp', 'sftp', 'local') DEFAULT 'ssh',
    host VARCHAR(255) NOT NULL,
    port INT DEFAULT 22,
    username VARCHAR(100),
    password VARCHAR(255),
    ssh_key TEXT,
    remote_path VARCHAR(255) DEFAULT '/media',
    is_active TINYINT(1) DEFAULT 1,
    status ENUM('online', 'offline', 'unknown') DEFAULT 'unknown',
    last_check DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Categories table
CREATE TABLE IF NOT EXISTS categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    server_id INT,
    external_id INT,
    name VARCHAR(100) NOT NULL,
    type ENUM('movie', 'series', 'live') DEFAULT 'movie',
    parent_id INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (server_id) REFERENCES servers(id) ON DELETE CASCADE,
    UNIQUE KEY unique_category (server_id, external_id, type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Movies table
CREATE TABLE IF NOT EXISTS movies (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tmdb_id INT,
    title VARCHAR(255) NOT NULL,
    clean_name VARCHAR(255),
    original_title VARCHAR(255),
    year INT,
    plot TEXT,
    genre VARCHAR(255),
    director VARCHAR(255),
    cast TEXT,
    rating DECIMAL(3,1),
    duration INT,
    poster VARCHAR(500),
    backdrop VARCHAR(500),
    trailer_url VARCHAR(500),
    is_downloaded TINYINT(1) DEFAULT 0,
    local_path VARCHAR(500),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_title (title),
    INDEX idx_year (year),
    INDEX idx_tmdb (tmdb_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Movie links table
CREATE TABLE IF NOT EXISTS movie_links (
    id INT AUTO_INCREMENT PRIMARY KEY,
    movie_id INT NOT NULL,
    server_id INT NOT NULL,
    stream_id INT,
    category_id INT,
    url VARCHAR(500) NOT NULL,
    container_extension VARCHAR(10) DEFAULT 'mp4',
    quality VARCHAR(20),
    is_working TINYINT(1) DEFAULT 1,
    last_check DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (movie_id) REFERENCES movies(id) ON DELETE CASCADE,
    FOREIGN KEY (server_id) REFERENCES servers(id) ON DELETE CASCADE,
    INDEX idx_movie (movie_id),
    INDEX idx_server (server_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Series table
CREATE TABLE IF NOT EXISTS series (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tmdb_id INT,
    title VARCHAR(255) NOT NULL,
    clean_name VARCHAR(255),
    original_title VARCHAR(255),
    year INT,
    plot TEXT,
    genre VARCHAR(255),
    cast TEXT,
    rating DECIMAL(3,1),
    poster VARCHAR(500),
    backdrop VARCHAR(500),
    category_id INT,
    seasons_count INT DEFAULT 0,
    episodes_count INT DEFAULT 0,
    is_downloaded TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_title (title),
    INDEX idx_tmdb (tmdb_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Episodes table
CREATE TABLE IF NOT EXISTS episodes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    series_id INT NOT NULL,
    server_id INT,
    stream_id INT,
    season_number INT NOT NULL,
    episode_number INT NOT NULL,
    title VARCHAR(255),
    clean_name VARCHAR(255),
    description TEXT,
    duration INT,
    stream_url VARCHAR(500),
    container_extension VARCHAR(10) DEFAULT 'mp4',
    poster VARCHAR(500),
    air_date DATE,
    is_downloaded TINYINT(1) DEFAULT 0,
    local_path VARCHAR(500),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (series_id) REFERENCES series(id) ON DELETE CASCADE,
    INDEX idx_series (series_id),
    INDEX idx_season (season_number)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Download jobs table
CREATE TABLE IF NOT EXISTS download_jobs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    remote_server_id INT NOT NULL,
    movie_id INT,
    episode_id INT,
    source_url VARCHAR(500) NOT NULL,
    filename VARCHAR(255) NOT NULL,
    destination_path VARCHAR(500),
    status ENUM('pending', 'downloading', 'completed', 'failed', 'cancelled') DEFAULT 'pending',
    progress INT DEFAULT 0,
    file_size BIGINT,
    downloaded_size BIGINT DEFAULT 0,
    error_message TEXT,
    started_at DATETIME,
    completed_at DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (remote_server_id) REFERENCES remote_servers(id) ON DELETE CASCADE,
    INDEX idx_status (status),
    INDEX idx_server (remote_server_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Import logs table
CREATE TABLE IF NOT EXISTS import_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    server_id INT,
    action VARCHAR(50),
    type VARCHAR(50),
    target_id INT,
    target_name VARCHAR(255),
    status ENUM('success', 'error', 'warning', 'info') DEFAULT 'info',
    message TEXT,
    details JSON,
    ip_address VARCHAR(45),
    user_agent VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_status (status),
    INDEX idx_action (action),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Settings table
CREATE TABLE IF NOT EXISTS settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    `key` VARCHAR(50) NOT NULL UNIQUE,
    `value` TEXT,
    `type` ENUM('string', 'int', 'bool', 'json') DEFAULT 'string',
    description VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default admin user (password: admin123)
INSERT INTO users (username, password, email, role, is_active) VALUES
('admin', '$2y$12$LQv3c1yqBWVHxkd0LHAkCOYz6TtxMQJqhN8/X4x5gL1GpLqEKRnwq', 'admin@localhost', 'admin', 1)
ON DUPLICATE KEY UPDATE username=username;

-- Insert default settings
INSERT INTO settings (`key`, `value`, `type`, description) VALUES
('site_name', 'VOD Import System', 'string', 'Site name'),
('items_per_page', '24', 'int', 'Items per page'),
('auto_check_links', '1', 'bool', 'Auto check links'),
('tmdb_api_key', '', 'string', 'TMDB API Key'),
('default_quality', '1080p', 'string', 'Default quality'),
('auto_download_movies', '0', 'bool', 'Auto download new movies'),
('auto_download_series', '0', 'bool', 'Auto download new series'),
('default_remote_server', '', 'int', 'Default remote server for downloads'),
('movies_folder', '/movies', 'string', 'Movies folder on remote server'),
('series_folder', '/series', 'string', 'Series folder on remote server')
ON DUPLICATE KEY UPDATE `key`=`key`;

EOSQL

echo -e "${GREEN}Database schema created successfully${NC}"

#############################################
# 6. Configure PHP
#############################################
echo -e "${GREEN}[6/10] Configuring PHP...${NC}"

# Find PHP version
PHP_VERSION=$(php -r "echo PHP_MAJOR_VERSION.'.'.PHP_MINOR_VERSION;")
PHP_FPM_CONF="/etc/php/${PHP_VERSION}/fpm/php.ini"
PHP_FPM_POOL="/etc/php/${PHP_VERSION}/fpm/pool.d/www.conf"

if [ -f "$PHP_FPM_CONF" ]; then
    sed -i 's/upload_max_filesize = .*/upload_max_filesize = 512M/' $PHP_FPM_CONF
    sed -i 's/post_max_size = .*/post_max_size = 512M/' $PHP_FPM_CONF
    sed -i 's/max_execution_time = .*/max_execution_time = 300/' $PHP_FPM_CONF
    sed -i 's/memory_limit = .*/memory_limit = 512M/' $PHP_FPM_CONF
fi

systemctl restart php${PHP_VERSION}-fpm

#############################################
# 7. Configure Nginx
#############################################
echo -e "${GREEN}[7/10] Configuring Nginx...${NC}"

# Get PHP-FPM socket path
PHP_SOCKET="/var/run/php/php${PHP_VERSION}-fpm.sock"

cat > /etc/nginx/sites-available/vod << NGINX
server {
    listen 80;
    server_name _;

    root /var/www/html;
    index index.php index.html;

    client_max_body_size 512M;

    # VOD System
    location /VOD {
        alias ${INSTALL_DIR};
        index index.php index.html;

        location ~ \.php$ {
            fastcgi_pass unix:${PHP_SOCKET};
            fastcgi_param SCRIPT_FILENAME \$request_filename;
            include fastcgi_params;
        }
    }

    # API
    location /VOD/api {
        alias ${INSTALL_DIR}/api;
        try_files \$uri \$uri/ /VOD/api/index.php?\$query_string;

        location ~ \.php$ {
            fastcgi_pass unix:${PHP_SOCKET};
            fastcgi_param SCRIPT_FILENAME \$request_filename;
            include fastcgi_params;
        }
    }

    # Default PHP handling
    location ~ \.php$ {
        include snippets/fastcgi-php.conf;
        fastcgi_pass unix:${PHP_SOCKET};
    }

    location ~ /\.ht {
        deny all;
    }
}
NGINX

# Enable site
ln -sf /etc/nginx/sites-available/vod /etc/nginx/sites-enabled/
rm -f /etc/nginx/sites-enabled/default

# Test and restart nginx
nginx -t && systemctl restart nginx
systemctl enable nginx

#############################################
# 8. Create Configuration File
#############################################
echo -e "${GREEN}[8/10] Creating configuration file...${NC}"

cat > ${INSTALL_DIR}/config/config.php << 'PHPCONFIG'
<?php
/**
 * VOD Import System Configuration
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/../logs/php_errors.log');

date_default_timezone_set('UTC');

// Database Configuration
define('DB_HOST', 'localhost');
define('DB_NAME', 'vod_import');
define('DB_USER', 'vod_user');
define('DB_PASS', 'VodPass@2024');
define('DB_CHARSET', 'utf8mb4');

// Paths
define('BASE_PATH', dirname(__DIR__));
define('CONFIG_PATH', BASE_PATH . '/config');
define('CACHE_PATH', BASE_PATH . '/cache');
define('LOGS_PATH', BASE_PATH . '/logs');
define('MEDIA_PATH', BASE_PATH . '/media');
define('WATCH_PATH', BASE_PATH . '/watch');

// API Settings
define('API_VERSION', '1.0');
define('API_RATE_LIMIT', 100);

// Cache Settings
define('CACHE_ENABLED', true);
define('CACHE_TTL', 300);
define('CACHE_CATEGORIES_TTL', 3600);

// Download Settings
define('MAX_CONCURRENT_DOWNLOADS', 3);
define('DOWNLOAD_TIMEOUT', 7200);
define('DOWNLOAD_CHUNK_SIZE', 8192);

// Link Check Settings
define('LINK_CHECK_TIMEOUT', 10);
define('LINK_CHECK_RETRIES', 3);

// TMDB API (optional)
define('TMDB_API_KEY', '');
define('TMDB_BASE_URL', 'https://api.themoviedb.org/3');
define('TMDB_IMAGE_BASE', 'https://image.tmdb.org/t/p/');

// Debug Mode
define('DEBUG_MODE', true);

/**
 * Get Database Connection
 */
function getDB(): PDO {
    static $pdo = null;

    if ($pdo === null) {
        try {
            $dsn = sprintf(
                'mysql:host=%s;dbname=%s;charset=%s',
                DB_HOST, DB_NAME, DB_CHARSET
            );

            $pdo = new PDO($dsn, DB_USER, DB_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci"
            ]);
        } catch (PDOException $e) {
            if (DEBUG_MODE) {
                die("Database connection failed: " . $e->getMessage());
            }
            die("Database connection failed");
        }
    }

    return $pdo;
}

/**
 * Get Setting Value
 */
function getSetting(string $key, $default = null) {
    static $settings = null;

    if ($settings === null) {
        try {
            $pdo = getDB();
            $stmt = $pdo->query("SELECT `key`, `value`, `type` FROM settings");
            $settings = [];
            while ($row = $stmt->fetch()) {
                $value = $row['value'];
                switch ($row['type']) {
                    case 'int': $value = (int) $value; break;
                    case 'bool': $value = (bool) $value; break;
                    case 'json': $value = json_decode($value, true); break;
                }
                $settings[$row['key']] = $value;
            }
        } catch (Exception $e) {
            return $default;
        }
    }

    return $settings[$key] ?? $default;
}

/**
 * Update Setting Value
 */
function setSetting(string $key, $value, string $type = 'string'): bool {
    try {
        $pdo = getDB();

        if ($type === 'json') {
            $value = json_encode($value);
        } elseif ($type === 'bool') {
            $value = $value ? '1' : '0';
        }

        $stmt = $pdo->prepare("
            INSERT INTO settings (`key`, `value`, `type`)
            VALUES (?, ?, ?)
            ON DUPLICATE KEY UPDATE `value` = VALUES(`value`), `type` = VALUES(`type`)
        ");

        return $stmt->execute([$key, (string) $value, $type]);
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Autoloader
 */
spl_autoload_register(function ($class) {
    $file = BASE_PATH . '/classes/' . $class . '.php';
    if (file_exists($file)) {
        require_once $file;
    }
});

// Create required directories
$dirs = [CACHE_PATH, LOGS_PATH, MEDIA_PATH, WATCH_PATH];
foreach ($dirs as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}
PHPCONFIG

#############################################
# 9. Set Permissions
#############################################
echo -e "${GREEN}[9/10] Setting permissions...${NC}"

chown -R www-data:www-data ${INSTALL_DIR}
chmod -R 755 ${INSTALL_DIR}
chmod -R 777 ${INSTALL_DIR}/{cache,logs,media,watch}

#############################################
# 10. Final Setup
#############################################
echo -e "${GREEN}[10/10] Final setup...${NC}"

# Create test file
cat > ${INSTALL_DIR}/test.php << 'TESTPHP'
<?php
header('Content-Type: application/json');
require_once __DIR__ . '/config/config.php';

try {
    $db = getDB();
    $stmt = $db->query("SELECT COUNT(*) as count FROM users");
    $result = $stmt->fetch();

    echo json_encode([
        'status' => 'ok',
        'php_version' => PHP_VERSION,
        'database' => 'connected',
        'users_count' => $result['count']
    ], JSON_PRETTY_PRINT);
} catch (Exception $e) {
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ]);
}
TESTPHP

# Get server IP
SERVER_IP=$(hostname -I | awk '{print $1}')

echo ""
echo -e "${GREEN}=============================================="
echo "   Installation Complete!"
echo "==============================================${NC}"
echo ""
echo -e "${BLUE}Access URLs:${NC}"
echo "  Main Panel: http://${SERVER_IP}/VOD/"
echo "  API:        http://${SERVER_IP}/VOD/api/"
echo "  Test:       http://${SERVER_IP}/VOD/test.php"
echo ""
echo -e "${BLUE}Database:${NC}"
echo "  Host:     localhost"
echo "  Database: ${DB_NAME}"
echo "  User:     ${DB_USER}"
echo "  Password: ${DB_PASS}"
echo ""
echo -e "${BLUE}Admin Login:${NC}"
echo "  Username: ${ADMIN_USER}"
echo "  Password: ${ADMIN_PASS}"
echo ""
echo -e "${YELLOW}IMPORTANT:${NC}"
echo "  1. Copy your PHP files to: ${INSTALL_DIR}/"
echo "  2. Update config if needed: ${INSTALL_DIR}/config/config.php"
echo "  3. Change admin password after first login!"
echo ""
echo -e "${GREEN}Installation log saved to: /var/log/vod_install.log${NC}"
