<?php
/**
 * Server Manager
 * Manages multiple M3U/XC sources with health checking and smart sync
 */

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/XtreamAPI.php';
require_once __DIR__ . '/M3UParser.php';
require_once __DIR__ . '/Logger.php';

class ServerManager
{
    private PDO $db;
    private Logger $logger;

    public function __construct()
    {
        $this->db = getDB();
        $this->logger = new Logger('server_manager');
    }

    /**
     * Add a new server
     */
    public function addServer(array $data): ?int
    {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO servers (name, type, url, username, password, port, use_https, is_active, priority, config)
                VALUES (:name, :type, :url, :username, :password, :port, :use_https, :is_active, :priority, :config)
            ");

            $stmt->execute([
                'name' => $data['name'],
                'type' => $data['type'] ?? 'xtream',
                'url' => $data['url'],
                'username' => $data['username'] ?? null,
                'password' => $data['password'] ?? null,
                'port' => $data['port'] ?? 80,
                'use_https' => $data['use_https'] ?? 0,
                'is_active' => $data['is_active'] ?? 1,
                'priority' => $data['priority'] ?? 1,
                'config' => isset($data['config']) ? json_encode($data['config']) : null
            ]);

            $serverId = (int) $this->db->lastInsertId();

            $this->logger->info("Server added", ['server_id' => $serverId, 'name' => $data['name']]);

            return $serverId;

        } catch (PDOException $e) {
            $this->logger->error("Failed to add server", ['error' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * Update server
     */
    public function updateServer(int $id, array $data): bool
    {
        try {
            $fields = [];
            $params = ['id' => $id];

            $allowedFields = ['name', 'type', 'url', 'username', 'password', 'port',
                'use_https', 'is_active', 'priority', 'status', 'error_message',
                'movies_count', 'series_count', 'live_count', 'last_check', 'last_sync'];

            foreach ($allowedFields as $field) {
                if (isset($data[$field])) {
                    $fields[] = "{$field} = :{$field}";
                    $params[$field] = $data[$field];
                }
            }

            if (isset($data['config'])) {
                $fields[] = "config = :config";
                $params['config'] = json_encode($data['config']);
            }

            if (empty($fields)) {
                return false;
            }

            $sql = "UPDATE servers SET " . implode(', ', $fields) . " WHERE id = :id";
            $stmt = $this->db->prepare($sql);
            return $stmt->execute($params);

        } catch (PDOException $e) {
            $this->logger->error("Failed to update server", ['error' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Delete server
     */
    public function deleteServer(int $id): bool
    {
        try {
            $stmt = $this->db->prepare("DELETE FROM servers WHERE id = ?");
            $result = $stmt->execute([$id]);

            if ($result) {
                $this->logger->info("Server deleted", ['server_id' => $id]);
            }

            return $result;

        } catch (PDOException $e) {
            $this->logger->error("Failed to delete server", ['error' => $e->getMessage()]);
            return false;
        }
    }

    /**
     * Get server by ID
     */
    public function getServer(int $id): ?array
    {
        $stmt = $this->db->prepare("SELECT * FROM servers WHERE id = ?");
        $stmt->execute([$id]);
        $server = $stmt->fetch();

        if ($server && $server['config']) {
            $server['config'] = json_decode($server['config'], true);
        }

        return $server ?: null;
    }

    /**
     * Get all servers
     */
    public function getAllServers(bool $activeOnly = false): array
    {
        $sql = "SELECT * FROM servers";
        if ($activeOnly) {
            $sql .= " WHERE is_active = 1";
        }
        $sql .= " ORDER BY priority DESC, name ASC";

        $stmt = $this->db->query($sql);
        $servers = $stmt->fetchAll();

        foreach ($servers as &$server) {
            if ($server['config']) {
                $server['config'] = json_decode($server['config'], true);
            }
        }

        return $servers;
    }

    /**
     * Check server health/status
     */
    public function checkServer(int $id): array
    {
        $server = $this->getServer($id);

        if (!$server) {
            return ['success' => false, 'error' => 'Server not found'];
        }

        $result = [
            'server_id' => $id,
            'name' => $server['name'],
            'type' => $server['type'],
            'status' => 'unknown',
            'response_time' => null,
            'movies_count' => 0,
            'series_count' => 0,
            'live_count' => 0,
            'error' => null
        ];

        $startTime = microtime(true);

        try {
            if ($server['type'] === 'xtream') {
                $result = $this->checkXtreamServer($server, $result);
            } else {
                $result = $this->checkM3UServer($server, $result);
            }

            $result['response_time'] = round((microtime(true) - $startTime) * 1000);

            // Update server status in database
            $this->updateServer($id, [
                'status' => $result['status'],
                'movies_count' => $result['movies_count'],
                'series_count' => $result['series_count'],
                'live_count' => $result['live_count'],
                'last_check' => date('Y-m-d H:i:s'),
                'error_message' => $result['error']
            ]);

        } catch (Exception $e) {
            $result['status'] = 'error';
            $result['error'] = $e->getMessage();

            $this->updateServer($id, [
                'status' => 'error',
                'last_check' => date('Y-m-d H:i:s'),
                'error_message' => $e->getMessage()
            ]);
        }

        $this->logger->info("Server check completed", $result);

        return $result;
    }

    /**
     * Check Xtream server
     */
    private function checkXtreamServer(array $server, array $result): array
    {
        $api = new XtreamAPI(
            $server['url'],
            $server['username'],
            $server['password'],
            $server['port'],
            (bool) $server['use_https']
        );

        if (!$api->authenticate()) {
            $result['status'] = 'error';
            $result['error'] = 'Authentication failed';
            return $result;
        }

        $userInfo = $api->getUserInfo();
        $serverInfo = $api->getServerInfo();

        $result['status'] = 'online';
        $result['user_status'] = $userInfo['status'] ?? 'unknown';
        $result['max_connections'] = $userInfo['max_connections'] ?? null;
        $result['expiry'] = $userInfo['exp_date'] ?? null;
        $result['server_timezone'] = $serverInfo['timezone'] ?? null;

        // Get content counts
        $vodCategories = $api->getVODCategories();
        $seriesCategories = $api->getSeriesCategories();
        $liveCategories = $api->getLiveCategories();

        unset($vodCategories['_meta'], $seriesCategories['_meta'], $liveCategories['_meta']);

        $result['vod_categories'] = count($vodCategories);
        $result['series_categories'] = count($seriesCategories);
        $result['live_categories'] = count($liveCategories);

        // Get actual counts
        $vodStreams = $api->getVODStreams();
        $seriesStreams = $api->getSeries();
        $liveStreams = $api->getLiveStreams();

        unset($vodStreams['_meta'], $seriesStreams['_meta'], $liveStreams['_meta']);

        $result['movies_count'] = is_array($vodStreams) ? count($vodStreams) : 0;
        $result['series_count'] = is_array($seriesStreams) ? count($seriesStreams) : 0;
        $result['live_count'] = is_array($liveStreams) ? count($liveStreams) : 0;

        return $result;
    }

    /**
     * Check M3U server
     */
    private function checkM3UServer(array $server, array $result): array
    {
        $parser = new M3UParser();
        $parseResult = $parser->parseUrl($server['url'], 30);

        if (!$parseResult['success']) {
            $result['status'] = 'error';
            $result['error'] = $parseResult['error'] ?? 'Failed to parse M3U';
            return $result;
        }

        $result['status'] = 'online';
        $result['total_entries'] = $parseResult['total'];
        $result['groups'] = $parseResult['groups'];

        // Count by type
        $movies = $parser->getMovies();
        $series = $parser->getSeries();
        $live = $parser->getLive();

        $result['movies_count'] = count($movies);
        $result['series_count'] = count($series);
        $result['live_count'] = count($live);

        return $result;
    }

    /**
     * Check all servers
     */
    public function checkAllServers(bool $activeOnly = true): array
    {
        $servers = $this->getAllServers($activeOnly);
        $results = [];

        foreach ($servers as $server) {
            $results[] = $this->checkServer($server['id']);
        }

        return $results;
    }

    /**
     * Toggle server active status
     */
    public function toggleServer(int $id): bool
    {
        $stmt = $this->db->prepare("UPDATE servers SET is_active = NOT is_active WHERE id = ?");
        return $stmt->execute([$id]);
    }

    /**
     * Get API instance for server
     */
    public function getServerAPI(int $id): ?XtreamAPI
    {
        $server = $this->getServer($id);

        if (!$server || $server['type'] !== 'xtream') {
            return null;
        }

        return new XtreamAPI(
            $server['url'],
            $server['username'],
            $server['password'],
            $server['port'],
            (bool) $server['use_https']
        );
    }

    /**
     * Get M3U parser for server
     */
    public function getServerParser(int $id): ?array
    {
        $server = $this->getServer($id);

        if (!$server || $server['type'] !== 'm3u') {
            return null;
        }

        $parser = new M3UParser();
        return $parser->parseUrl($server['url']);
    }

    /**
     * Sync categories from server
     */
    public function syncCategories(int $serverId): array
    {
        $server = $this->getServer($serverId);

        if (!$server) {
            return ['success' => false, 'error' => 'Server not found'];
        }

        $result = [
            'success' => true,
            'added' => 0,
            'updated' => 0,
            'categories' => []
        ];

        try {
            if ($server['type'] === 'xtream') {
                $api = $this->getServerAPI($serverId);

                if (!$api || !$api->authenticate()) {
                    return ['success' => false, 'error' => 'Authentication failed'];
                }

                // Sync VOD categories
                $vodCats = $api->getVODCategories();
                unset($vodCats['_meta']);

                foreach ($vodCats as $cat) {
                    $this->upsertCategory($serverId, $cat, 'movie');
                    $result['added']++;
                }

                // Sync Series categories
                $seriesCats = $api->getSeriesCategories();
                unset($seriesCats['_meta']);

                foreach ($seriesCats as $cat) {
                    $this->upsertCategory($serverId, $cat, 'series');
                    $result['added']++;
                }

                // Sync Live categories
                $liveCats = $api->getLiveCategories();
                unset($liveCats['_meta']);

                foreach ($liveCats as $cat) {
                    $this->upsertCategory($serverId, $cat, 'live');
                    $result['added']++;
                }

            } else {
                // M3U - use groups as categories
                $parser = new M3UParser();
                $parseResult = $parser->parseUrl($server['url']);

                if (!$parseResult['success']) {
                    return ['success' => false, 'error' => $parseResult['error']];
                }

                foreach ($parseResult['groups'] as $groupName => $count) {
                    $this->upsertCategory($serverId, [
                        'category_id' => md5($groupName),
                        'category_name' => $groupName
                    ], 'movie');
                    $result['added']++;
                }
            }

            $this->logger->info("Categories synced", ['server_id' => $serverId, 'count' => $result['added']]);

        } catch (Exception $e) {
            $result['success'] = false;
            $result['error'] = $e->getMessage();
            $this->logger->error("Category sync failed", ['error' => $e->getMessage()]);
        }

        return $result;
    }

    /**
     * Insert or update category
     */
    private function upsertCategory(int $serverId, array $cat, string $type): void
    {
        $stmt = $this->db->prepare("
            INSERT INTO categories (server_id, external_id, name, type, icon)
            VALUES (:server_id, :external_id, :name, :type, :icon)
            ON DUPLICATE KEY UPDATE name = VALUES(name), icon = VALUES(icon)
        ");

        $stmt->execute([
            'server_id' => $serverId,
            'external_id' => $cat['category_id'] ?? '',
            'name' => $cat['category_name'] ?? 'Unknown',
            'type' => $type,
            'icon' => $cat['icon'] ?? null
        ]);
    }

    /**
     * Get categories for server
     */
    public function getCategories(int $serverId, ?string $type = null): array
    {
        $sql = "SELECT * FROM categories WHERE server_id = ?";
        $params = [$serverId];

        if ($type) {
            $sql .= " AND type = ?";
            $params[] = $type;
        }

        $sql .= " ORDER BY sort_order ASC, name ASC";

        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);

        return $stmt->fetchAll();
    }

    /**
     * Get dashboard statistics
     */
    public function getStats(): array
    {
        $stats = [
            'servers' => [
                'total' => 0,
                'active' => 0,
                'online' => 0,
                'offline' => 0,
                'error' => 0
            ],
            'content' => [
                'movies' => 0,
                'series' => 0,
                'episodes' => 0,
                'links' => 0
            ],
            'downloads' => [
                'pending' => 0,
                'processing' => 0,
                'completed' => 0,
                'failed' => 0
            ]
        ];

        // Server stats
        $stmt = $this->db->query("
            SELECT
                COUNT(*) as total,
                SUM(is_active) as active,
                SUM(CASE WHEN status = 'online' THEN 1 ELSE 0 END) as online,
                SUM(CASE WHEN status = 'offline' THEN 1 ELSE 0 END) as offline,
                SUM(CASE WHEN status = 'error' THEN 1 ELSE 0 END) as error_count
            FROM servers
        ");
        $serverStats = $stmt->fetch();
        $stats['servers'] = [
            'total' => (int) $serverStats['total'],
            'active' => (int) $serverStats['active'],
            'online' => (int) $serverStats['online'],
            'offline' => (int) $serverStats['offline'],
            'error' => (int) $serverStats['error_count']
        ];

        // Content stats
        $stmt = $this->db->query("SELECT COUNT(*) FROM movies");
        $stats['content']['movies'] = (int) $stmt->fetchColumn();

        $stmt = $this->db->query("SELECT COUNT(*) FROM series");
        $stats['content']['series'] = (int) $stmt->fetchColumn();

        $stmt = $this->db->query("SELECT COUNT(*) FROM episodes");
        $stats['content']['episodes'] = (int) $stmt->fetchColumn();

        $stmt = $this->db->query("SELECT COUNT(*) FROM movie_links");
        $stats['content']['links'] = (int) $stmt->fetchColumn();

        // Download queue stats
        $stmt = $this->db->query("
            SELECT status, COUNT(*) as count
            FROM import_queue
            GROUP BY status
        ");
        while ($row = $stmt->fetch()) {
            $stats['downloads'][$row['status']] = (int) $row['count'];
        }

        return $stats;
    }
}
