<?php
/**
 * VOD Import System - Dashboard
 */
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/Auth.php';

// Require authentication
Auth::requireLogin();

$currentUser = Auth::getCurrentUser();
?>
<!DOCTYPE html>
<html lang="en" dir="ltr" data-theme="dark">
<script>
// Apply saved theme immediately to prevent flash
(function() {
    var savedMode = localStorage.getItem('vod-theme-mode') || 'dark';
    var theme = savedMode;
    if (savedMode === 'auto') {
        theme = window.matchMedia('(prefers-color-scheme: dark)').matches ? 'dark' : 'light';
    }
    document.documentElement.setAttribute('data-theme', theme);
})();
</script>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>VOD Import System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        /* Dark Theme (Default) */
        :root, [data-theme="dark"] {
            --primary-color: #6366f1;
            --secondary-color: #4f46e5;
            --success-color: #22c55e;
            --warning-color: #f59e0b;
            --danger-color: #ef4444;
            --dark-bg: #0f172a;
            --card-bg: #1e293b;
            --text-primary: #f8fafc;
            --text-secondary: #94a3b8;
            --border-color: rgba(255,255,255,0.1);
            --input-bg: #1e293b;
            --input-border: rgba(255,255,255,0.2);
            --table-bg: transparent;
            --table-hover: rgba(255,255,255,0.05);
            --dropdown-bg: #1e293b;
            --modal-bg: #1e293b;
        }

        /* Light Theme */
        [data-theme="light"] {
            --primary-color: #4f46e5;
            --secondary-color: #6366f1;
            --success-color: #16a34a;
            --warning-color: #d97706;
            --danger-color: #dc2626;
            --dark-bg: #f1f5f9;
            --card-bg: #ffffff;
            --text-primary: #1e293b;
            --text-secondary: #64748b;
            --border-color: rgba(0,0,0,0.1);
            --input-bg: #ffffff;
            --input-border: #cbd5e1;
            --table-bg: #ffffff;
            --table-hover: rgba(0,0,0,0.02);
            --dropdown-bg: #ffffff;
            --modal-bg: #ffffff;
        }

        [data-theme="light"] body {
            background: #f1f5f9 !important;
            color: #1e293b !important;
        }

        [data-theme="light"] .sidebar {
            background: #ffffff !important;
            border-right: 1px solid #e2e8f0 !important;
        }

        [data-theme="light"] .nav-link {
            color: #64748b !important;
        }

        [data-theme="light"] .nav-link:hover,
        [data-theme="light"] .nav-link.active {
            color: #ffffff !important;
            background: var(--primary-color) !important;
        }

        [data-theme="light"] .card {
            background: #ffffff !important;
            border-color: #e2e8f0 !important;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }

        [data-theme="light"] .card-header {
            background: #f8fafc !important;
            border-bottom-color: #e2e8f0 !important;
            color: #1e293b !important;
        }

        [data-theme="light"] .form-control,
        [data-theme="light"] .form-select {
            background: #ffffff !important;
            border-color: #cbd5e1 !important;
            color: #1e293b !important;
        }

        [data-theme="light"] .form-control:focus,
        [data-theme="light"] .form-select:focus {
            background: #ffffff !important;
            border-color: var(--primary-color) !important;
            color: #1e293b !important;
        }

        [data-theme="light"] .table {
            color: #1e293b !important;
        }

        [data-theme="light"] .table > :not(caption) > * > * {
            background: transparent !important;
            border-bottom-color: #e2e8f0 !important;
            color: #1e293b;
        }

        [data-theme="light"] .modal-content {
            background: #ffffff !important;
            color: #1e293b !important;
        }

        [data-theme="light"] .modal-header {
            border-bottom-color: #e2e8f0 !important;
        }

        [data-theme="light"] .modal-footer {
            border-top-color: #e2e8f0 !important;
        }

        [data-theme="light"] .dropdown-menu,
        [data-theme="light"] .dropdown-menu-dark {
            background: #ffffff !important;
            color: #1e293b !important;
            border-color: #e2e8f0 !important;
        }

        [data-theme="light"] .dropdown-menu .dropdown-item,
        [data-theme="light"] .dropdown-menu-dark .dropdown-item {
            color: #1e293b !important;
        }

        [data-theme="light"] .dropdown-menu .dropdown-item:hover,
        [data-theme="light"] .dropdown-menu-dark .dropdown-item:hover {
            background: #f1f5f9 !important;
        }

        [data-theme="light"] .dropdown-divider {
            border-top-color: #e2e8f0 !important;
        }

        [data-theme="light"] .btn-close {
            filter: none;
        }

        [data-theme="light"] .text-light {
            color: #1e293b !important;
        }

        [data-theme="light"] .text-secondary {
            color: #64748b !important;
        }

        [data-theme="light"] .stat-card {
            background: #ffffff !important;
            border: 1px solid #e2e8f0 !important;
        }

        [data-theme="light"] .stat-card h3 {
            color: var(--primary-color) !important;
        }

        [data-theme="light"] .stat-card p {
            color: #64748b !important;
        }

        [data-theme="light"] .user-info {
            background: #f1f5f9 !important;
        }

        [data-theme="light"] .logo {
            color: #1e293b !important;
        }

        [data-theme="light"] .section-title {
            color: #1e293b !important;
        }

        [data-theme="light"] .btn-outline-secondary {
            color: #64748b !important;
            border-color: #cbd5e1 !important;
        }

        [data-theme="light"] .btn-outline-secondary:hover {
            background: #f1f5f9 !important;
            color: #1e293b !important;
        }

        [data-theme="light"] .progress {
            background: #e2e8f0 !important;
        }

        [data-theme="light"] .badge.bg-secondary {
            background: #e2e8f0 !important;
            color: #64748b !important;
        }

        [data-theme="light"] .movie-card {
            background: #ffffff !important;
            border: 1px solid #e2e8f0 !important;
        }

        [data-theme="light"] .movie-title {
            color: #1e293b !important;
        }

        [data-theme="light"] hr {
            border-color: #e2e8f0 !important;
        }

        [data-theme="light"] .alert-info {
            background: #e0f2fe !important;
            border-color: #7dd3fc !important;
            color: #0369a1 !important;
        }

        [data-theme="light"] .log-entry {
            background: #f8fafc !important;
            border-left-color: var(--primary-color) !important;
        }

        [data-theme="light"] .theme-preview i {
            color: inherit;
        }

        /* Theme Card Styles */
        .theme-card {
            cursor: pointer;
            border-radius: 12px;
            overflow: hidden;
            border: 2px solid var(--border-color);
            transition: all 0.3s ease;
            background: var(--card-bg);
        }
        .theme-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.2);
        }
        .theme-card.active {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(99, 102, 241, 0.3);
        }
        .theme-card-preview {
            height: 80px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 2rem;
        }
        .theme-card-preview.dark-preview {
            background: linear-gradient(135deg, #0f172a 0%, #1e293b 100%);
            color: #f8fafc;
        }
        .theme-card-preview.light-preview {
            background: linear-gradient(135deg, #f1f5f9 0%, #ffffff 100%);
            color: #1e293b;
        }
        .theme-card-preview.auto-preview {
            background: linear-gradient(135deg, #0f172a 0%, #1e293b 50%, #f1f5f9 50%, #ffffff 100%);
            color: #6366f1;
        }
        .theme-card-label {
            padding: 10px;
            text-align: center;
            font-weight: 600;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        .theme-card .theme-check {
            display: none;
            color: var(--primary-color);
        }
        .theme-card.active .theme-check {
            display: inline;
        }

        body {
            background: var(--dark-bg);
            color: var(--text-primary);
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            min-height: 100vh;
        }

        .sidebar {
            background: var(--card-bg);
            min-height: 100vh;
            position: fixed;
            width: 260px;
            left: 0;
            top: 0;
            padding: 1rem;
            border-right: 1px solid rgba(255,255,255,0.1);
        }

        .main-content {
            margin-left: 260px;
            padding: 2rem;
        }

        .nav-link {
            color: var(--text-secondary);
            padding: 0.75rem 1rem;
            border-radius: 8px;
            margin-bottom: 0.25rem;
            transition: all 0.3s;
        }

        .nav-link:hover, .nav-link.active {
            color: var(--text-primary);
            background: var(--primary-color);
        }

        .nav-link i {
            margin-right: 0.5rem;
        }

        .card {
            background: var(--card-bg);
            border: 1px solid rgba(255,255,255,0.1);
            border-radius: 12px;
        }

        .card-header {
            background: transparent;
            border-bottom: 1px solid rgba(255,255,255,0.1);
            color: var(--text-primary);
            font-weight: 600;
            font-size: 1rem;
        }

        /* Settings page styles */
        .form-label {
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
        }

        .form-control, .form-select {
            background: var(--card-bg);
            border: 1px solid rgba(255,255,255,0.15);
            color: var(--text-primary);
            font-weight: 500;
        }

        .form-control:focus, .form-select:focus {
            background: var(--card-bg);
            border-color: var(--primary-color);
            color: var(--text-primary);
            box-shadow: 0 0 0 0.2rem rgba(99, 102, 241, 0.25);
        }

        .form-check-label {
            color: var(--text-primary);
            font-weight: 500;
        }

        .form-check-label strong {
            font-weight: 700;
        }

        [data-theme="light"] .form-control,
        [data-theme="light"] .form-select {
            background: #ffffff;
            border-color: #d1d5db;
            color: #1e293b;
        }

        [data-theme="light"] .form-control:focus,
        [data-theme="light"] .form-select:focus {
            background: #ffffff;
            color: #1e293b;
        }

        .stat-card {
            padding: 1.5rem;
            border-radius: 12px;
            background: linear-gradient(135deg, var(--card-bg) 0%, rgba(99, 102, 241, 0.1) 100%);
            border: 1px solid rgba(255,255,255,0.1);
        }

        .stat-card h3 {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 0;
        }

        .stat-card p {
            color: var(--text-secondary);
            margin-bottom: 0;
        }

        .stat-card.success { border-left: 4px solid var(--success-color); }
        .stat-card.warning { border-left: 4px solid var(--warning-color); }
        .stat-card.danger { border-left: 4px solid var(--danger-color); }
        .stat-card.info { border-left: 4px solid var(--primary-color); }

        .table {
            color: var(--text-primary);
        }

        .table thead th {
            border-bottom-color: rgba(255,255,255,0.1);
            color: var(--text-secondary);
            font-weight: 500;
        }

        .table td {
            border-bottom-color: rgba(255,255,255,0.1);
            vertical-align: middle;
        }

        .badge-online { background: var(--success-color); }
        .badge-offline { background: var(--danger-color); }
        .badge-error { background: var(--warning-color); }
        .badge-unknown { background: var(--text-secondary); }

        .btn-primary {
            background: var(--primary-color);
            border-color: var(--primary-color);
        }

        .btn-primary:hover {
            background: var(--secondary-color);
            border-color: var(--secondary-color);
        }

        .form-control, .form-select {
            background: var(--dark-bg);
            border-color: rgba(255,255,255,0.1);
            color: var(--text-primary);
        }

        .form-control:focus, .form-select:focus {
            background: var(--dark-bg);
            border-color: var(--primary-color);
            color: var(--text-primary);
            box-shadow: 0 0 0 0.25rem rgba(99, 102, 241, 0.25);
        }

        .modal-content {
            background: var(--card-bg);
            border: 1px solid rgba(255,255,255,0.1);
        }

        .modal-header {
            border-bottom-color: rgba(255,255,255,0.1);
        }

        .modal-footer {
            border-top-color: rgba(255,255,255,0.1);
        }

        .progress {
            background: var(--dark-bg);
            border-radius: 10px;
        }

        .progress-bar {
            background: var(--primary-color);
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            padding: 1rem;
            margin-bottom: 1rem;
        }

        .logo-icon {
            width: 45px;
            height: 45px;
            background: linear-gradient(135deg, var(--primary-color) 0%, #8b5cf6 100%);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 4px 15px rgba(99, 102, 241, 0.4);
        }

        .logo-icon i {
            font-size: 1.5rem;
            color: #ffffff;
        }

        .logo-text {
            display: flex;
            flex-direction: column;
            line-height: 1.1;
        }

        .logo-main {
            font-size: 1.25rem;
            font-weight: 800;
            color: var(--text-primary);
            letter-spacing: -0.02em;
        }

        .logo-year {
            font-size: 0.7rem;
            font-weight: 600;
            color: var(--primary-color);
            letter-spacing: 0.15em;
            text-transform: uppercase;
        }

        .logo i {
            margin-right: 0.5rem;
        }

        .section-title {
            font-size: 1.25rem;
            font-weight: 600;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .loading {
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 3rem;
        }

        .spinner-border {
            color: var(--primary-color);
        }

        .movie-card {
            background: var(--card-bg);
            border-radius: 12px;
            overflow: hidden;
            transition: transform 0.3s, box-shadow 0.3s;
            box-shadow: 0 4px 6px rgba(0,0,0,0.15);
        }

        .movie-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0,0,0,0.25);
        }

        .movie-poster {
            width: 100%;
            height: 200px;
            object-fit: cover;
            background: var(--dark-bg);
        }

        .movie-info {
            padding: 1rem 0.9rem 0.75rem;
            padding-top: 1.1rem;
            background: linear-gradient(180deg, #1e293b 0%, #0f172a 100%);
            border-top: 1px solid rgba(255,255,255,0.05);
            position: relative;
        }

        [data-theme="light"] .movie-card {
            box-shadow: 0 2px 8px rgba(0,0,0,0.08);
        }

        [data-theme="light"] .movie-card:hover {
            box-shadow: 0 8px 20px rgba(0,0,0,0.12);
        }

        .movie-title {
            font-weight: 700;
            font-size: 0.85rem;
            margin-bottom: 0.4rem;
            color: #ffffff !important;
            letter-spacing: 0.01em;
            line-height: 1.4;
            text-shadow: 0 1px 2px rgba(0,0,0,0.4);
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            /* Show full name - no truncation */
            white-space: normal;
            word-wrap: break-word;
            overflow-wrap: break-word;
        }

        [data-theme="light"] .movie-title {
            color: #0f172a !important;
            text-shadow: none;
        }

        .movie-year {
            color: #94a3b8 !important;
            font-size: 0.8rem;
            font-weight: 500;
            letter-spacing: 0.02em;
        }

        [data-theme="light"] .movie-year {
            color: #64748b !important;
        }

        /* TMDB-style movie card */
        .movie-info {
            position: relative;
        }

        .movie-rating {
            position: absolute;
            top: -14px;
            left: 10px;
            width: 28px;
            height: 28px;
            background: linear-gradient(135deg, #21d07a 0%, #1ca868 100%);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.65rem;
            font-weight: 700;
            color: #fff;
            border: 2px solid #0f172a;
            box-shadow: 0 2px 4px rgba(0,0,0,0.3);
        }

        [data-theme="light"] .movie-rating {
            border-color: #fff;
        }

        [data-theme="light"] .movie-info {
            background: linear-gradient(180deg, #ffffff 0%, #f1f5f9 100%);
            border-top: 1px solid rgba(0,0,0,0.05);
        }

        .log-entry {
            padding: 0.5rem;
            border-radius: 4px;
            margin-bottom: 0.25rem;
            font-size: 0.875rem;
            font-family: monospace;
        }

        .log-entry.error { background: rgba(239, 68, 68, 0.2); }
        .log-entry.warning { background: rgba(245, 158, 11, 0.2); }
        .log-entry.info { background: rgba(99, 102, 241, 0.2); }
        .log-entry.success { background: rgba(34, 197, 94, 0.2); }

        @media (max-width: 992px) {
            .sidebar {
                position: relative;
                width: 100%;
                min-height: auto;
            }
            .main-content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <nav class="sidebar">
        <div class="logo">
            <div class="logo-icon">
                <i class="bi bi-play-circle-fill"></i>
            </div>
            <div class="logo-text">
                <span class="logo-main">VOD Panel</span>
                <span class="logo-year">2025</span>
            </div>
        </div>

        <!-- User Info -->
        <div class="user-info mb-3 p-2 rounded" style="background: rgba(255,255,255,0.05);">
            <div class="d-flex align-items-center">
                <div class="avatar me-2">
                    <i class="bi bi-person-circle fs-4"></i>
                </div>
                <div class="flex-grow-1">
                    <div class="fw-bold"><?= htmlspecialchars($currentUser['username']) ?></div>
                    <small class="text-secondary"><?= ucfirst($currentUser['role']) ?></small>
                </div>
            </div>
        </div>

        <ul class="nav flex-column">
            <li class="nav-item">
                <a class="nav-link active" href="#" data-page="dashboard">
                    <i class="bi bi-speedometer2"></i>
                    Dashboard
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="#" data-page="servers">
                    <i class="bi bi-hdd-network"></i>
                    Source Servers
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="#" data-page="remote-servers">
                    <i class="bi bi-cloud-upload"></i>
                    Remote Servers
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="#" data-page="movies">
                    <i class="bi bi-camera-reels"></i>
                    Movies
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="#" data-page="series">
                    <i class="bi bi-collection-play"></i>
                    Series
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="#" data-page="download-jobs">
                    <i class="bi bi-download"></i>
                    Downloads
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="#" data-page="logs">
                    <i class="bi bi-journal-text"></i>
                    Logs
                </a>
            </li>
            <?php if ($currentUser['role'] === 'admin'): ?>
            <li class="nav-item">
                <a class="nav-link" href="#" data-page="users">
                    <i class="bi bi-people"></i>
                    Users
                </a>
            </li>
            <?php endif; ?>
            <li class="nav-item">
                <a class="nav-link" href="#" data-page="settings">
                    <i class="bi bi-gear"></i>
                    Settings
                </a>
            </li>
        </ul>

        <hr style="border-color: rgba(255,255,255,0.1);">

        <!-- Theme Toggle -->
        <div class="theme-toggle mb-3">
            <button class="btn btn-outline-secondary w-100 d-flex align-items-center justify-content-center gap-2" onclick="toggleTheme()" id="theme-toggle-btn">
                <i class="bi bi-sun-fill" id="theme-icon"></i>
                <span id="theme-text">Light Mode</span>
            </button>
        </div>

        <a href="logout.php" class="nav-link text-danger">
            <i class="bi bi-box-arrow-right"></i>
            Logout
        </a>
    </nav>

    <!-- Main Content -->
    <main class="main-content">
        <div id="page-content">
            <!-- Dashboard -->
            <div id="page-dashboard" class="page">
                <h2 class="section-title">
                    <i class="bi bi-speedometer2"></i>
                    Dashboard
                </h2>

                <div class="row g-4 mb-4">
                    <div class="col-md-2">
                        <div class="stat-card info">
                            <h3 id="stat-movies">0</h3>
                            <p>Total Movies</p>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="stat-card" style="border-left: 4px solid #8b5cf6;">
                            <h3 id="stat-series">0</h3>
                            <p>Total Series</p>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="stat-card" style="border-left: 4px solid #06b6d4;">
                            <h3 id="stat-episodes">0</h3>
                            <p>Episodes</p>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="stat-card success">
                            <h3 id="stat-servers-online">0</h3>
                            <p>Active Servers</p>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="stat-card warning">
                            <h3 id="stat-links">0</h3>
                            <p>Total Links</p>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="stat-card danger">
                            <h3 id="stat-queue">0</h3>
                            <p>In Queue</p>
                        </div>
                    </div>
                </div>

                <div class="row g-4">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <span>Latest Movies</span>
                                <button class="btn btn-sm btn-primary" onclick="loadNewestMovies()">
                                    <i class="bi bi-arrow-clockwise"></i>
                                </button>
                            </div>
                            <div class="card-body">
                                <div id="newest-movies" class="row g-3">
                                    <div class="loading">
                                        <div class="spinner-border"></div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header d-flex justify-content-between align-items-center">
                                <span>Latest Series</span>
                                <button class="btn btn-sm btn-primary" onclick="loadNewestSeries()">
                                    <i class="bi bi-arrow-clockwise"></i>
                                </button>
                            </div>
                            <div class="card-body">
                                <div id="newest-series" class="row g-3">
                                    <div class="loading">
                                        <div class="spinner-border"></div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="row g-4 mt-2">
                    <div class="col-12">
                        <div class="card">
                            <div class="card-header">Server Status</div>
                            <div class="card-body">
                                <div id="servers-status">
                                    <div class="loading">
                                        <div class="spinner-border"></div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Servers Page -->
            <div id="page-servers" class="page" style="display: none;">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2 class="section-title mb-0">
                        <i class="bi bi-hdd-network"></i>
                        Source Servers
                    </h2>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addServerModal">
                        <i class="bi bi-plus-lg"></i>
                        Add Server
                    </button>
                </div>

                <div class="card">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table">
                                <thead>
                                    <tr>
                                        <th>Name</th>
                                        <th>Type</th>
                                        <th>URL</th>
                                        <th>Status</th>
                                        <th>Last Check</th>
                                        <th>Movies</th>
                                        <th>Series</th>
                                        <th>Live</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="servers-table">
                                    <tr>
                                        <td colspan="9" class="text-center">Loading...</td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Server Info Modal -->
            <div class="modal fade" id="serverInfoModal" tabindex="-1">
                <div class="modal-dialog modal-xl">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">
                                <i class="bi bi-info-circle"></i>
                                Server Information: <span id="server-info-name"></span>
                            </h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <!-- Server Stats -->
                            <div class="row g-3 mb-4">
                                <div class="col-md-3">
                                    <div class="card bg-primary text-white">
                                        <div class="card-body text-center">
                                            <h3 id="server-info-movies">0</h3>
                                            <small>Movies</small>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="card bg-info text-white">
                                        <div class="card-body text-center">
                                            <h3 id="server-info-series">0</h3>
                                            <small>Series</small>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="card bg-success text-white">
                                        <div class="card-body text-center">
                                            <h3 id="server-info-live">0</h3>
                                            <small>Live Channels</small>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="card bg-warning text-dark">
                                        <div class="card-body text-center">
                                            <h3 id="server-info-categories">0</h3>
                                            <small>Categories</small>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Tabs for Categories -->
                            <ul class="nav nav-tabs" id="serverInfoTabs">
                                <li class="nav-item">
                                    <a class="nav-link active" href="#" onclick="loadServerCategories('movie')">
                                        <i class="bi bi-camera-reels"></i> Movie Categories
                                    </a>
                                </li>
                                <li class="nav-item">
                                    <a class="nav-link" href="#" onclick="loadServerCategories('series')">
                                        <i class="bi bi-collection-play"></i> Series Categories
                                    </a>
                                </li>
                                <li class="nav-item">
                                    <a class="nav-link" href="#" onclick="loadServerCategories('live')">
                                        <i class="bi bi-broadcast"></i> Live Categories
                                    </a>
                                </li>
                            </ul>

                            <!-- Categories List -->
                            <div class="mt-3">
                                <div class="d-flex justify-content-between align-items-center mb-2">
                                    <span id="server-categories-title">Movie Categories</span>
                                    <div>
                                        <button class="btn btn-sm btn-success" onclick="importSelectedCategories()">
                                            <i class="bi bi-download"></i> Import Selected
                                        </button>
                                        <button class="btn btn-sm btn-outline-primary" onclick="selectAllServerCategories()">
                                            Select All
                                        </button>
                                    </div>
                                </div>
                                <div id="server-categories-list" style="max-height: 400px; overflow-y: auto;">
                                    <p class="text-secondary">Select a tab to view categories</p>
                                </div>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                            <button type="button" class="btn btn-primary" onclick="syncServerCategories()">
                                <i class="bi bi-arrow-repeat"></i> Sync All Categories
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Movies Page -->
            <div id="page-movies" class="page" style="display: none;">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2 class="section-title mb-0">
                        <i class="bi bi-camera-reels"></i>
                        Movie Library
                    </h2>
                    <div class="d-flex gap-2">
                        <button class="btn btn-success" id="download-selected-btn" onclick="showDownloadModal()" style="display: none;">
                            <i class="bi bi-download"></i>
                            Download Selected (<span id="selected-count">0</span>)
                        </button>
                    </div>
                </div>

                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-body">
                        <div class="row g-3 align-items-end">
                            <div class="col-md-3">
                                <label class="form-label">Category</label>
                                <select class="form-select" id="movie-category-filter" onchange="filterMovies()">
                                    <option value="">All Categories</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Year</label>
                                <select class="form-select" id="movie-year-filter" onchange="filterMovies()">
                                    <option value="">All Years</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Search</label>
                                <input type="text" class="form-control" id="movie-search" placeholder="Search movies...">
                            </div>
                            <div class="col-md-2">
                                <button class="btn btn-primary w-100" onclick="searchMovies()">
                                    <i class="bi bi-search"></i> Search
                                </button>
                            </div>
                            <div class="col-md-2">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="select-all-movies" onchange="toggleSelectAll()">
                                    <label class="form-check-label" for="select-all-movies">Select All</label>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Movies Stats -->
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <span class="text-secondary" id="movies-stats">Loading...</span>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-secondary active" onclick="setMovieView('grid')" id="view-grid">
                            <i class="bi bi-grid-3x3-gap"></i>
                        </button>
                        <button class="btn btn-outline-secondary" onclick="setMovieView('list')" id="view-list">
                            <i class="bi bi-list"></i>
                        </button>
                    </div>
                </div>

                <div class="row g-3" id="movies-grid">
                    <div class="col-12 text-center">
                        <div class="spinner-border"></div>
                    </div>
                </div>

                <div class="d-flex justify-content-center mt-4">
                    <button class="btn btn-outline-primary" id="load-more-movies" onclick="loadMoreMovies()">
                        Load More
                    </button>
                </div>
            </div>

            <!-- Series Page -->
            <div id="page-series" class="page" style="display: none;">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2 class="section-title mb-0">
                        <i class="bi bi-collection-play"></i>
                        Series Library
                    </h2>
                    <div class="d-flex gap-2">
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#importSeriesUrlModal">
                            <i class="bi bi-link-45deg"></i>
                            Import from URL
                        </button>
                        <button class="btn btn-success" id="download-selected-series-btn" onclick="showSeriesDownloadModal()" style="display: none;">
                            <i class="bi bi-download"></i>
                            Download Selected (<span id="selected-series-count">0</span>)
                        </button>
                    </div>
                </div>

                <!-- Filters -->
                <div class="card mb-4">
                    <div class="card-body">
                        <div class="row g-3 align-items-end">
                            <div class="col-md-3">
                                <label class="form-label">Category</label>
                                <select class="form-select" id="series-category-filter" onchange="filterSeries()">
                                    <option value="">All Categories</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Year</label>
                                <select class="form-select" id="series-year-filter" onchange="filterSeries()">
                                    <option value="">All Years</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Search</label>
                                <input type="text" class="form-control" id="series-search" placeholder="Search series...">
                            </div>
                            <div class="col-md-2">
                                <button class="btn btn-primary w-100" onclick="searchSeries()">
                                    <i class="bi bi-search"></i> Search
                                </button>
                            </div>
                            <div class="col-md-2">
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="select-all-series" onchange="toggleSelectAllSeries()">
                                    <label class="form-check-label" for="select-all-series">Select All</label>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Series Stats -->
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <span class="text-secondary" id="series-stats">Loading...</span>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-secondary active" onclick="setSeriesView('grid')" id="series-view-grid">
                            <i class="bi bi-grid-3x3-gap"></i>
                        </button>
                        <button class="btn btn-outline-secondary" onclick="setSeriesView('list')" id="series-view-list">
                            <i class="bi bi-list"></i>
                        </button>
                    </div>
                </div>

                <div class="row g-3" id="series-grid">
                    <div class="col-12 text-center">
                        <div class="spinner-border"></div>
                    </div>
                </div>

                <div class="d-flex justify-content-center mt-4">
                    <button class="btn btn-outline-primary" id="load-more-series" onclick="loadMoreSeries()">
                        Load More
                    </button>
                </div>
            </div>

            <!-- Import Page -->
            <div id="page-import" class="page" style="display: none;">
                <h2 class="section-title">
                    <i class="bi bi-cloud-download"></i>
                    Import Content
                </h2>

                <!-- Content Type Tabs -->
                <ul class="nav nav-tabs mb-4" id="import-tabs">
                    <li class="nav-item">
                        <a class="nav-link active" href="#" onclick="setImportType('movie')">
                            <i class="bi bi-camera-reels"></i> Movies
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="#" onclick="setImportType('series')">
                            <i class="bi bi-collection-play"></i> Series
                        </a>
                    </li>
                </ul>

                <div class="row g-4">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">
                                <span id="import-card-title">Import Movies from Server</span>
                            </div>
                            <div class="card-body">
                                <div class="mb-3">
                                    <label class="form-label">Select Server</label>
                                    <select class="form-select" id="import-server" onchange="loadImportCategories()">
                                        <option value="">-- Select Server --</option>
                                    </select>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Category (Optional)</label>
                                    <select class="form-select" id="import-category">
                                        <option value="">All Categories</option>
                                    </select>
                                </div>
                                <div class="form-check mb-3">
                                    <input class="form-check-input" type="checkbox" id="import-force">
                                    <label class="form-check-label" for="import-force">
                                        Force Update Existing
                                    </label>
                                </div>
                                <button class="btn btn-primary w-100" onclick="startImport()">
                                    <i class="bi bi-download"></i>
                                    Start Import
                                </button>
                            </div>
                        </div>
                    </div>

                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">Smart Sync</div>
                            <div class="card-body">
                                <p class="text-secondary">
                                    Compares existing content with server and imports only new items.
                                </p>
                                <div class="mb-3">
                                    <label class="form-label">Select Server</label>
                                    <select class="form-select" id="sync-server">
                                        <option value="">-- Select Server --</option>
                                    </select>
                                </div>
                                <button class="btn btn-success w-100" onclick="startSmartSync()">
                                    <i class="bi bi-arrow-repeat"></i>
                                    Smart Sync
                                </button>
                            </div>
                        </div>
                    </div>

                    <div class="col-12">
                        <div class="card">
                            <div class="card-header">
                                <span id="check-new-title">Check New Content</span>
                            </div>
                            <div class="card-body">
                                <p class="text-secondary" id="check-new-desc">
                                    Scans all servers for new content.
                                </p>
                                <button class="btn btn-info w-100" onclick="checkNewContent()">
                                    <i class="bi bi-search"></i>
                                    <span id="check-new-btn-text">Check for New Content</span>
                                </button>
                                <div id="new-movies-result" class="mt-3" style="display: none;"></div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="card mt-4">
                    <div class="card-header">Import Log</div>
                    <div class="card-body">
                        <div id="import-log" style="max-height: 300px; overflow-y: auto;">
                            <p class="text-secondary">No import started yet.</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- FolderWatch Page -->
            <div id="page-folderwatch" class="page" style="display: none;">
                <h2 class="section-title">
                    <i class="bi bi-folder-check"></i>
                    FolderWatch
                </h2>

                <div class="row g-4 mb-4">
                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">Scan Watch Folder</div>
                            <div class="card-body">
                                <button class="btn btn-primary w-100 mb-3" onclick="scanWatchFolder()">
                                    <i class="bi bi-folder-symlink"></i>
                                    Scan Folder
                                </button>
                                <button class="btn btn-success w-100" onclick="processWatchFolder()">
                                    <i class="bi bi-check-all"></i>
                                    Process Valid Files
                                </button>
                            </div>
                        </div>
                    </div>

                    <div class="col-md-6">
                        <div class="card">
                            <div class="card-header">Validate Filename</div>
                            <div class="card-body">
                                <input type="text" class="form-control mb-3" id="validate-filename"
                                    placeholder="Enter filename to validate...">
                                <button class="btn btn-info w-100" onclick="validateFilename()">
                                    <i class="bi bi-check-circle"></i>
                                    Validate
                                </button>
                                <div id="validate-result" class="mt-3" style="display: none;"></div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <span>Scan Results</span>
                        <span class="badge bg-secondary" id="scan-count">0 files</span>
                    </div>
                    <div class="card-body">
                        <ul class="nav nav-tabs mb-3">
                            <li class="nav-item">
                                <a class="nav-link active" data-bs-toggle="tab" href="#valid-files">
                                    Valid <span class="badge bg-success" id="valid-count">0</span>
                                </a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link" data-bs-toggle="tab" href="#invalid-files">
                                    Invalid <span class="badge bg-danger" id="invalid-count">0</span>
                                </a>
                            </li>
                        </ul>

                        <div class="tab-content">
                            <div class="tab-pane fade show active" id="valid-files">
                                <div class="table-responsive">
                                    <table class="table table-sm">
                                        <thead>
                                            <tr>
                                                <th>Filename</th>
                                                <th>Type</th>
                                                <th>Title</th>
                                                <th>Year</th>
                                            </tr>
                                        </thead>
                                        <tbody id="valid-files-table">
                                            <tr><td colspan="4" class="text-center">Not scanned yet</td></tr>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                            <div class="tab-pane fade" id="invalid-files">
                                <div class="table-responsive">
                                    <table class="table table-sm">
                                        <thead>
                                            <tr>
                                                <th>Filename</th>
                                                <th>Error</th>
                                                <th>Suggestion</th>
                                            </tr>
                                        </thead>
                                        <tbody id="invalid-files-table">
                                            <tr><td colspan="3" class="text-center">Not scanned yet</td></tr>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Logs Page -->
            <div id="page-logs" class="page" style="display: none;">
                <h2 class="section-title">
                    <i class="bi bi-journal-text"></i>
                    System Logs
                </h2>

                <!-- Log Statistics Cards -->
                <div class="row g-3 mb-4">
                    <div class="col-md-3 col-6">
                        <div class="stat-card" style="background: linear-gradient(135deg, #1e3a5f 0%, #0d1b2a 100%); border-left: 4px solid #3b82f6;">
                            <h3 id="log-stat-total" class="text-info">0</h3>
                            <p class="text-secondary mb-0">Total Logs</p>
                        </div>
                    </div>
                    <div class="col-md-3 col-6">
                        <div class="stat-card" style="background: linear-gradient(135deg, #1a3d2e 0%, #0d1b2a 100%); border-left: 4px solid #22c55e;">
                            <h3 id="log-stat-success" class="text-success">0</h3>
                            <p class="text-secondary mb-0">Success</p>
                        </div>
                    </div>
                    <div class="col-md-3 col-6">
                        <div class="stat-card" style="background: linear-gradient(135deg, #3d3a1a 0%, #0d1b2a 100%); border-left: 4px solid #eab308;">
                            <h3 id="log-stat-warning" class="text-warning">0</h3>
                            <p class="text-secondary mb-0">Warnings</p>
                        </div>
                    </div>
                    <div class="col-md-3 col-6">
                        <div class="stat-card" style="background: linear-gradient(135deg, #3d1a1a 0%, #0d1b2a 100%); border-left: 4px solid #ef4444;">
                            <h3 id="log-stat-error" class="text-danger">0</h3>
                            <p class="text-secondary mb-0">Errors</p>
                        </div>
                    </div>
                </div>

                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center flex-wrap gap-2">
                        <div class="d-flex gap-2 flex-wrap">
                            <select class="form-select form-select-sm" id="log-status" style="width: auto;">
                                <option value="">All Status</option>
                                <option value="success">Success</option>
                                <option value="warning">Warning</option>
                                <option value="error">Error</option>
                                <option value="info">Info</option>
                            </select>
                            <select class="form-select form-select-sm" id="log-type" style="width: auto;">
                                <option value="">All Types</option>
                                <option value="movie">Movies</option>
                                <option value="series">Series</option>
                                <option value="server">Servers</option>
                                <option value="download">Downloads</option>
                                <option value="system">System</option>
                            </select>
                            <select class="form-select form-select-sm" id="log-limit" style="width: auto;">
                                <option value="50">50 logs</option>
                                <option value="100" selected>100 logs</option>
                                <option value="250">250 logs</option>
                                <option value="500">500 logs</option>
                            </select>
                        </div>
                        <div class="d-flex gap-2">
                            <button class="btn btn-sm btn-outline-info" onclick="loadLogs()" title="Refresh">
                                <i class="bi bi-arrow-clockwise"></i>
                            </button>
                            <div class="dropdown">
                                <button class="btn btn-sm btn-outline-danger dropdown-toggle" type="button" data-bs-toggle="dropdown" aria-expanded="false">
                                    <i class="bi bi-trash"></i> Clear
                                </button>
                                <ul class="dropdown-menu dropdown-menu-end dropdown-menu-dark">
                                    <li><a class="dropdown-item text-success" href="#" onclick="clearLogs('success'); return false;"><i class="bi bi-check-circle me-2"></i>Clear Success Logs</a></li>
                                    <li><a class="dropdown-item text-warning" href="#" onclick="clearLogs('warning'); return false;"><i class="bi bi-exclamation-triangle me-2"></i>Clear Warning Logs</a></li>
                                    <li><a class="dropdown-item text-danger" href="#" onclick="clearLogs('error'); return false;"><i class="bi bi-x-circle me-2"></i>Clear Error Logs</a></li>
                                    <li><a class="dropdown-item text-info" href="#" onclick="clearLogs('info'); return false;"><i class="bi bi-info-circle me-2"></i>Clear Info Logs</a></li>
                                    <li><hr class="dropdown-divider"></li>
                                    <li><a class="dropdown-item" href="#" onclick="clearLogs('old'); return false;"><i class="bi bi-calendar-minus me-2"></i>Clear Logs Older Than 7 Days</a></li>
                                    <li><hr class="dropdown-divider"></li>
                                    <li><a class="dropdown-item text-danger fw-bold" href="#" onclick="clearLogs('clear'); return false;"><i class="bi bi-trash-fill me-2"></i>Clear ALL Logs</a></li>
                                </ul>
                            </div>
                            <button class="btn btn-sm btn-outline-secondary" onclick="exportLogs()" title="Export">
                                <i class="bi bi-download"></i> Export
                            </button>
                        </div>
                    </div>
                    <div class="card-body">
                        <div id="logs-container" style="max-height: 600px; overflow-y: auto;">
                            <div class="loading">
                                <div class="spinner-border"></div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Settings Page -->
            <div id="page-settings" class="page" style="display: none;">
                <h2 class="section-title">
                    <i class="bi bi-gear"></i>
                    Settings
                </h2>

                <!-- Appearance Settings -->
                <div class="card mb-4">
                    <div class="card-header">
                        <i class="bi bi-palette me-2"></i>Appearance
                    </div>
                    <div class="card-body">
                        <label class="form-label fw-bold mb-3">Theme Mode</label>
                        <div class="btn-group w-100" role="group">
                            <button type="button" class="btn btn-outline-primary theme-btn" id="theme-btn-dark"
                                onclick="document.documentElement.setAttribute('data-theme','dark'); localStorage.setItem('vod-theme-mode','dark'); document.querySelectorAll('.theme-btn').forEach(b=>b.classList.remove('active')); this.classList.add('active');">
                                <i class="bi bi-moon-stars-fill me-2"></i>Dark
                            </button>
                            <button type="button" class="btn btn-outline-primary theme-btn" id="theme-btn-light"
                                onclick="document.documentElement.setAttribute('data-theme','light'); localStorage.setItem('vod-theme-mode','light'); document.querySelectorAll('.theme-btn').forEach(b=>b.classList.remove('active')); this.classList.add('active');">
                                <i class="bi bi-sun-fill me-2"></i>Light
                            </button>
                            <button type="button" class="btn btn-outline-primary theme-btn" id="theme-btn-auto"
                                onclick="var t=window.matchMedia('(prefers-color-scheme:dark)').matches?'dark':'light'; document.documentElement.setAttribute('data-theme',t); localStorage.setItem('vod-theme-mode','auto'); document.querySelectorAll('.theme-btn').forEach(b=>b.classList.remove('active')); this.classList.add('active');">
                                <i class="bi bi-circle-half me-2"></i>Auto
                            </button>
                        </div>
                        <small class="text-secondary mt-2 d-block">Auto mode follows your system preference</small>
                    </div>
                </div>

                <!-- Auto Download Settings -->
                <div class="card mb-4">
                    <div class="card-header">
                        <i class="bi bi-cloud-download me-2"></i>Auto Download
                    </div>
                    <div class="card-body">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" id="auto-download-movies">
                                    <label class="form-check-label" for="auto-download-movies">
                                        <strong>Auto Download New Movies</strong>
                                    </label>
                                </div>
                                <small class="text-muted">Automatically download new movies when imported</small>
                            </div>
                            <div class="col-md-6">
                                <div class="form-check form-switch">
                                    <input class="form-check-input" type="checkbox" id="auto-download-series">
                                    <label class="form-check-label" for="auto-download-series">
                                        <strong>Auto Download New Series</strong>
                                    </label>
                                </div>
                                <small class="text-muted">Automatically download new episodes when imported</small>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Default Remote Server</label>
                                <select class="form-select" id="auto-download-server" onchange="loadServerFolders()">
                                    <option value="">Select server...</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Download Schedule</label>
                                <select class="form-select" id="auto-download-schedule">
                                    <option value="immediate">Immediate</option>
                                    <option value="hourly">Every Hour</option>
                                    <option value="daily">Daily (at midnight)</option>
                                    <option value="manual">Manual Only</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label"><i class="bi bi-film me-1"></i>Movies Folder</label>
                                <div class="input-group">
                                    <input type="text" class="form-control" id="auto-download-movies-folder" placeholder="/path/to/movies">
                                    <button class="btn btn-outline-primary" type="button" onclick="browseFolder('movies')">
                                        <i class="bi bi-folder"></i>
                                    </button>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label"><i class="bi bi-tv me-1"></i>Series Folder</label>
                                <div class="input-group">
                                    <input type="text" class="form-control" id="auto-download-series-folder" placeholder="/path/to/series">
                                    <button class="btn btn-outline-primary" type="button" onclick="browseFolder('series')">
                                        <i class="bi bi-folder"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                        <div class="mt-3">
                            <button class="btn btn-primary" onclick="saveAutoDownloadSettings()">
                                <i class="bi bi-check-lg me-1"></i>Save Auto Download Settings
                            </button>
                            <button class="btn btn-success ms-2" onclick="runAutoDownload()">
                                <i class="bi bi-play-fill me-1"></i>Run Now
                            </button>
                        </div>
                    </div>
                </div>

                <!-- System Settings -->
                <div class="card">
                    <div class="card-header">
                        <i class="bi bi-sliders me-2"></i>System Settings
                    </div>
                    <div class="card-body">
                        <form id="settings-form">
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <label class="form-label">Download Path</label>
                                    <input type="text" class="form-control" name="download_path">
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">FolderWatch Path</label>
                                    <input type="text" class="form-control" name="folderwatch_path">
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Max Concurrent Downloads</label>
                                    <input type="number" class="form-control" name="max_concurrent_downloads">
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Auto Sync Interval (seconds)</label>
                                    <input type="number" class="form-control" name="auto_sync_interval">
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">TMDB API Key</label>
                                    <input type="text" class="form-control" name="tmdb_api_key">
                                </div>
                                <div class="col-md-6">
                                    <div class="form-check mt-4">
                                        <input class="form-check-input" type="checkbox" name="debug_mode" id="debug-mode">
                                        <label class="form-check-label" for="debug-mode">Debug Mode</label>
                                    </div>
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" name="auto_cleanup_orphans" id="auto-cleanup">
                                        <label class="form-check-label" for="auto-cleanup">Auto Cleanup Orphan Episodes</label>
                                    </div>
                                </div>
                            </div>
                            <div class="mt-4">
                                <button type="submit" class="btn btn-primary">
                                    <i class="bi bi-check-lg"></i>
                                    Save Settings
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Remote Servers Page -->
            <div id="page-remote-servers" class="page" style="display: none;">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2 class="section-title mb-0">
                        <i class="bi bi-cloud-upload"></i>
                        Remote Servers (SSH/FTP)
                    </h2>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addRemoteServerModal">
                        <i class="bi bi-plus-lg"></i>
                        Add Remote Server
                    </button>
                </div>

                <div class="card">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table">
                                <thead>
                                    <tr>
                                        <th>Name</th>
                                        <th>Host</th>
                                        <th>System Stats</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="remote-servers-table">
                                    <tr>
                                        <td colspan="5" class="text-center">Loading...</td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Download Jobs Page -->
            <div id="page-download-jobs" class="page" style="display: none;">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2 class="section-title mb-0">
                        <i class="bi bi-cloud-download"></i>
                        Download Manager
                    </h2>
                    <button class="btn btn-primary" onclick="loadDownloadJobs()">
                        <i class="bi bi-arrow-clockwise"></i> Refresh
                    </button>
                </div>

                <!-- Stats Cards -->
                <div class="row g-3 mb-4">
                    <div class="col-6 col-md-3">
                        <div class="card text-center" style="background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); border: none;">
                            <div class="card-body py-3">
                                <i class="bi bi-hourglass-split fs-3 text-white opacity-75"></i>
                                <h2 class="text-white mb-0 mt-2" id="jobs-pending">0</h2>
                                <small class="text-white-50">Pending</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-6 col-md-3">
                        <div class="card text-center" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); border: none;">
                            <div class="card-body py-3">
                                <i class="bi bi-arrow-down-circle fs-3 text-white opacity-75"></i>
                                <h2 class="text-white mb-0 mt-2" id="jobs-downloading">0</h2>
                                <small class="text-white-50">Downloading</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-6 col-md-3">
                        <div class="card text-center" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%); border: none;">
                            <div class="card-body py-3">
                                <i class="bi bi-check-circle fs-3 text-white opacity-75"></i>
                                <h2 class="text-white mb-0 mt-2" id="jobs-completed">0</h2>
                                <small class="text-white-50">Completed</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-6 col-md-3">
                        <div class="card text-center" style="background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); border: none;">
                            <div class="card-body py-3">
                                <i class="bi bi-x-circle fs-3 text-white opacity-75"></i>
                                <h2 class="text-white mb-0 mt-2" id="jobs-failed">0</h2>
                                <small class="text-white-50">Failed</small>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Jobs Table -->
                <div class="card" style="background: var(--card-bg); border: 1px solid var(--border-color);">
                    <div class="card-header d-flex flex-wrap justify-content-between align-items-center gap-2" style="background: rgba(255,255,255,0.03); border-bottom: 1px solid var(--border-color);">
                        <div class="d-flex align-items-center gap-2">
                            <i class="bi bi-list-task text-primary"></i>
                            <span class="text-light fw-semibold">Job Queue</span>
                        </div>
                        <div class="d-flex flex-wrap gap-2 align-items-center">
                            <select class="form-select form-select-sm" id="jobs-status-filter" onchange="loadDownloadJobs()" style="width: 140px; background: var(--dark-bg); border-color: var(--border-color); color: var(--text-primary);">
                                <option value="">All Status</option>
                                <option value="pending">Pending</option>
                                <option value="downloading">Downloading</option>
                                <option value="completed">Completed</option>
                                <option value="failed">Failed</option>
                            </select>
                            <div class="btn-group btn-group-sm">
                                <button class="btn btn-success" onclick="clearCompletedJobs()" title="Clear Completed">
                                    <i class="bi bi-check2-all"></i>
                                </button>
                                <button class="btn btn-warning" onclick="clearFailedJobs()" title="Clear Failed">
                                    <i class="bi bi-x-lg"></i>
                                </button>
                                <button class="btn btn-danger" onclick="clearAllJobs()" title="Clear All">
                                    <i class="bi bi-trash3"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                    <div class="card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-dark table-hover mb-0" style="background: transparent;">
                                <thead style="background: rgba(255,255,255,0.05);">
                                    <tr>
                                        <th class="text-secondary" style="width: 50px;">#</th>
                                        <th class="text-secondary">Server</th>
                                        <th class="text-secondary">File</th>
                                        <th class="text-secondary" style="width: 200px;">Progress</th>
                                        <th class="text-secondary" style="width: 100px;">Status</th>
                                        <th class="text-secondary" style="width: 150px;">Time</th>
                                        <th class="text-secondary" style="width: 80px;">Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="download-jobs-table">
                                    <tr>
                                        <td colspan="7" class="text-center py-4 text-secondary">
                                            <div class="spinner-border spinner-border-sm me-2"></div>
                                            Loading...
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Users Page (Admin Only) -->
            <?php if ($currentUser['role'] === 'admin'): ?>
            <div id="page-users" class="page" style="display: none;">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2 class="section-title mb-0">
                        <i class="bi bi-people"></i>
                        User Management
                    </h2>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addUserModal">
                        <i class="bi bi-plus-lg"></i>
                        Add User
                    </button>
                </div>

                <!-- User Statistics Cards -->
                <div class="row g-3 mb-4">
                    <div class="col-md-3 col-6">
                        <div class="stat-card" style="background: linear-gradient(135deg, #1e3a5f 0%, #0d1b2a 100%); border-left: 4px solid #3b82f6;">
                            <h3 id="user-stat-total" class="text-info">0</h3>
                            <p class="text-secondary mb-0">Total Users</p>
                        </div>
                    </div>
                    <div class="col-md-3 col-6">
                        <div class="stat-card" style="background: linear-gradient(135deg, #3d1a1a 0%, #0d1b2a 100%); border-left: 4px solid #ef4444;">
                            <h3 id="user-stat-admin" class="text-danger">0</h3>
                            <p class="text-secondary mb-0">Admins</p>
                        </div>
                    </div>
                    <div class="col-md-3 col-6">
                        <div class="stat-card" style="background: linear-gradient(135deg, #1a3d2e 0%, #0d1b2a 100%); border-left: 4px solid #22c55e;">
                            <h3 id="user-stat-active" class="text-success">0</h3>
                            <p class="text-secondary mb-0">Active</p>
                        </div>
                    </div>
                    <div class="col-md-3 col-6">
                        <div class="stat-card" style="background: linear-gradient(135deg, #3d3a1a 0%, #0d1b2a 100%); border-left: 4px solid #64748b;">
                            <h3 id="user-stat-inactive" class="text-secondary">0</h3>
                            <p class="text-secondary mb-0">Inactive</p>
                        </div>
                    </div>
                </div>

                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <span><i class="bi bi-people-fill me-2"></i>Users List</span>
                        <button class="btn btn-sm btn-outline-info" onclick="loadUsers()">
                            <i class="bi bi-arrow-clockwise"></i>
                        </button>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Username</th>
                                        <th>Email</th>
                                        <th>Role</th>
                                        <th>Status</th>
                                        <th>Last Login</th>
                                        <th>Login IP</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="users-table">
                                    <tr>
                                        <td colspan="7" class="text-center">Loading...</td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </main>

    <!-- Add Server Modal -->
    <div class="modal fade" id="addServerModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Add New Server</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="add-server-form">
                        <div class="mb-3">
                            <label class="form-label">Server Name</label>
                            <input type="text" class="form-control" name="name" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Type</label>
                            <select class="form-select" name="type" required>
                                <option value="xtream">Xtream Codes</option>
                                <option value="m3u">M3U Playlist</option>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">URL</label>
                            <input type="url" class="form-control" name="url" required
                                placeholder="http://example.com">
                        </div>
                        <div class="row xtream-fields">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Username</label>
                                <input type="text" class="form-control" name="username">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Password</label>
                                <input type="text" class="form-control" name="password">
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Port</label>
                                <input type="number" class="form-control" name="port" value="80">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Priority</label>
                                <input type="number" class="form-control" name="priority" value="1">
                            </div>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" onclick="addServer()">Add</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Edit Server Modal -->
    <div class="modal fade" id="editServerModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Edit Server</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Server Name</label>
                        <input type="text" class="form-control" id="edit-server-name" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Type</label>
                        <select class="form-select" id="edit-server-type">
                            <option value="xtream">Xtream Codes</option>
                            <option value="m3u">M3U Playlist</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">URL</label>
                        <input type="url" class="form-control" id="edit-server-url" required placeholder="http://example.com">
                    </div>
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Username</label>
                            <input type="text" class="form-control" id="edit-server-username">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Password</label>
                            <input type="text" class="form-control" id="edit-server-password">
                        </div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Port</label>
                        <input type="number" class="form-control" id="edit-server-port" value="80">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" onclick="saveServerEdit()">Save Changes</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Remote Server Modal -->
    <div class="modal fade" id="addRemoteServerModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Add Remote Server (SSH/FTP)</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="add-remote-server-form">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Server Name</label>
                                <input type="text" class="form-control" name="name" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Connection Type</label>
                                <select class="form-select" name="type" required>
                                    <option value="ssh">SSH</option>
                                    <option value="sftp">SFTP</option>
                                    <option value="ftp">FTP</option>
                                </select>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-8 mb-3">
                                <label class="form-label">Host / IP</label>
                                <input type="text" class="form-control" name="host" required
                                    placeholder="192.168.1.100 or server.domain.com">
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Port</label>
                                <input type="number" class="form-control" name="port" value="22">
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Username</label>
                                <input type="text" class="form-control" name="username" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Password</label>
                                <input type="password" class="form-control" name="password">
                            </div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Private Key (SSH Key) - Optional</label>
                            <textarea class="form-control" name="private_key" rows="3"
                                placeholder="-----BEGIN RSA PRIVATE KEY-----"></textarea>
                            <small class="text-secondary">Leave empty if using password only</small>
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Remote Path</label>
                                <input type="text" class="form-control" name="remote_path"
                                    value="/var/www/html/media" placeholder="/path/to/media">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Priority</label>
                                <input type="number" class="form-control" name="priority" value="1">
                            </div>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" onclick="addRemoteServer()">Add</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Edit Remote Server Modal -->
    <div class="modal fade" id="editRemoteServerModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Edit Remote Server</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="edit-remote-server-form">
                        <input type="hidden" name="id" id="edit-remote-server-id">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Server Name</label>
                                <input type="text" class="form-control" name="name" id="edit-rs-name" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Connection Type</label>
                                <select class="form-select" name="type" id="edit-rs-type" required>
                                    <option value="ssh">SSH</option>
                                    <option value="sftp">SFTP</option>
                                    <option value="ftp">FTP</option>
                                </select>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-8 mb-3">
                                <label class="form-label">Host / IP</label>
                                <input type="text" class="form-control" name="host" id="edit-rs-host" required>
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Port</label>
                                <input type="number" class="form-control" name="port" id="edit-rs-port" value="22">
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Username</label>
                                <input type="text" class="form-control" name="username" id="edit-rs-username" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Password (leave empty to keep current)</label>
                                <input type="password" class="form-control" name="password" id="edit-rs-password">
                            </div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Private Key (leave empty to keep current)</label>
                            <textarea class="form-control" name="private_key" id="edit-rs-private-key" rows="3"
                                placeholder="-----BEGIN RSA PRIVATE KEY-----"></textarea>
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Remote Path</label>
                                <input type="text" class="form-control" name="remote_path" id="edit-rs-remote-path">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Priority</label>
                                <input type="number" class="form-control" name="priority" id="edit-rs-priority" value="1">
                            </div>
                        </div>
                        <div class="form-check mb-3">
                            <input class="form-check-input" type="checkbox" name="is_active" id="edit-rs-active" checked>
                            <label class="form-check-label" for="edit-rs-active">Active</label>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" onclick="updateRemoteServer()">Save Changes</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Add User Modal -->
    <?php if ($currentUser['role'] === 'admin'): ?>
    <div class="modal fade" id="addUserModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Add New User</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="add-user-form">
                        <div class="mb-3">
                            <label class="form-label">Username</label>
                            <input type="text" class="form-control" name="username" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Email</label>
                            <input type="email" class="form-control" name="email">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Password</label>
                            <input type="password" class="form-control" name="password" required minlength="6">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Role</label>
                            <select class="form-select" name="role" required>
                                <option value="user">User</option>
                                <option value="admin">Admin</option>
                                <option value="viewer">Viewer</option>
                            </select>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" onclick="addUser()">Add</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Edit User Modal -->
    <div class="modal fade" id="editUserModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="bi bi-pencil-square me-2"></i>Edit User</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="edit-user-form">
                        <input type="hidden" name="id" id="edit-user-id">
                        <div class="mb-3">
                            <label class="form-label">Username</label>
                            <input type="text" class="form-control" name="username" id="edit-user-username" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Email</label>
                            <input type="email" class="form-control" name="email" id="edit-user-email">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Role</label>
                            <select class="form-select" name="role" id="edit-user-role" required>
                                <option value="user">User</option>
                                <option value="admin">Admin</option>
                                <option value="viewer">Viewer</option>
                            </select>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" onclick="saveEditUser()">Save Changes</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Reset Password Modal -->
    <div class="modal fade" id="resetPasswordModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="bi bi-key me-2"></i>Reset Password</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="reset-password-form">
                        <input type="hidden" name="id" id="reset-password-user-id">
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle me-2"></i>
                            Resetting password for: <strong id="reset-password-username"></strong>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">New Password</label>
                            <input type="password" class="form-control" name="password" id="reset-password-input" required minlength="6">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Confirm Password</label>
                            <input type="password" class="form-control" name="confirm_password" id="reset-password-confirm" required minlength="6">
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-warning" onclick="saveResetPassword()">Reset Password</button>
                </div>
            </div>
        </div>
    </div>

    <!-- API Token Modal -->
    <div class="modal fade" id="apiTokenModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="bi bi-key-fill me-2"></i>API Token</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-warning">
                        <i class="bi bi-exclamation-triangle me-2"></i>
                        This token will only be shown once. Please save it securely.
                    </div>
                    <div class="mb-3">
                        <label class="form-label">API Token for: <strong id="api-token-username"></strong></label>
                        <div class="input-group">
                            <input type="text" class="form-control" id="api-token-value" readonly>
                            <button class="btn btn-outline-info" onclick="copyApiToken()">
                                <i class="bi bi-clipboard"></i>
                            </button>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Download to Remote Server Modal -->
    <div class="modal fade" id="downloadModal" tabindex="-1">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Download to Remote Server</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-info">
                        <i class="bi bi-info-circle"></i>
                        <span id="download-movie-count">0</span> <span id="download-type-label">movies</span> selected for download
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Select Remote Server</label>
                                <select class="form-select" id="download-remote-server" required onchange="loadServerFolders()">
                                    <option value="">-- Select Server --</option>
                                </select>
                            </div>

                            <div class="mb-3">
                                <label class="form-label">Destination Folder</label>
                                <select class="form-select" id="download-folder-type" onchange="toggleCustomFolder()">
                                    <option value="default">Default Server Path</option>
                                    <option value="browse">Browse Server Folders</option>
                                    <option value="category">Organize by Category</option>
                                    <option value="custom">Custom Folder</option>
                                </select>
                            </div>

                            <!-- Browse Folders -->
                            <div class="mb-3" id="browse-folder-group" style="display: none;">
                                <label class="form-label">Select Folder from Server</label>
                                <div class="input-group mb-2">
                                    <span class="input-group-text"><i class="bi bi-folder"></i></span>
                                    <input type="text" class="form-control" id="current-browse-path" readonly>
                                    <button class="btn btn-outline-secondary" onclick="browseParentFolder()">
                                        <i class="bi bi-arrow-up"></i>
                                    </button>
                                    <button class="btn btn-outline-primary" onclick="refreshBrowseFolders()">
                                        <i class="bi bi-arrow-clockwise"></i>
                                    </button>
                                </div>
                                <div id="remote-folders-list" class="list-group" style="max-height: 200px; overflow-y: auto;">
                                    <p class="text-secondary p-2">Select a server first</p>
                                </div>
                                <div class="mt-2">
                                    <div class="input-group">
                                        <input type="text" class="form-control" id="new-folder-name" placeholder="New folder name">
                                        <button class="btn btn-success" onclick="createRemoteFolder()">
                                            <i class="bi bi-folder-plus"></i> Create
                                        </button>
                                    </div>
                                </div>
                            </div>

                            <div class="mb-3" id="category-folder-group" style="display: none;">
                                <label class="form-label">Select Category Folder</label>
                                <select class="form-select" id="download-category-folder">
                                    <option value="">-- Select Category --</option>
                                </select>
                            </div>

                            <div class="mb-3" id="custom-folder-group" style="display: none;">
                                <label class="form-label">Custom Folder Path</label>
                                <div class="input-group">
                                    <span class="input-group-text" id="base-path-label">/var/www/html/media/</span>
                                    <input type="text" class="form-control" id="download-custom-folder" placeholder="movies/action">
                                </div>
                                <small class="text-secondary">Enter subfolder path (will be created if not exists)</small>
                            </div>

                            <div class="mb-3">
                                <label class="form-label">Download Options</label>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="download-organize" checked>
                                    <label class="form-check-label" for="download-organize">
                                        Create subfolders by name (Series/Movie Name/)
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="download-rename" checked>
                                    <label class="form-check-label" for="download-rename">
                                        Clean filename (Name.Year.ext)
                                    </label>
                                </div>
                            </div>
                        </div>

                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Selected Items</label>
                                <div id="selected-movies-list" style="max-height: 300px; overflow-y: auto;" class="border rounded p-2">
                                    <small class="text-secondary">No items selected</small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-success" onclick="startRemoteDownload()">
                        <i class="bi bi-download"></i> Start Download
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Import Series from URL Modal -->
    <div class="modal fade" id="importSeriesUrlModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="bi bi-link-45deg"></i>
                        Import Series from URL
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-info">
                        <i class="bi bi-info-circle"></i>
                        Enter an Xtream Codes M3U URL or API credentials to import series.
                    </div>

                    <ul class="nav nav-tabs mb-3" id="importSeriesUrlTabs">
                        <li class="nav-item">
                            <a class="nav-link active" href="#" onclick="setImportSeriesUrlMode('url')">M3U URL</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link" href="#" onclick="setImportSeriesUrlMode('api')">API Credentials</a>
                        </li>
                    </ul>

                    <form id="import-series-url-form">
                        <!-- M3U URL Mode -->
                        <div id="import-series-url-mode">
                            <div class="mb-3">
                                <label class="form-label">M3U URL</label>
                                <input type="text" class="form-control" name="m3u_url" id="import-series-m3u-url"
                                    placeholder="http://server.com:port/get.php?username=xxx&password=xxx&type=m3u_plus">
                                <small class="text-secondary">Paste full M3U URL - credentials will be extracted automatically</small>
                            </div>
                        </div>

                        <!-- API Mode -->
                        <div id="import-series-api-mode" style="display: none;">
                            <div class="row">
                                <div class="col-md-8 mb-3">
                                    <label class="form-label">Server URL</label>
                                    <input type="text" class="form-control" name="server_url" id="import-series-server-url"
                                        placeholder="http://server.com:port">
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label class="form-label">Port</label>
                                    <input type="text" class="form-control" name="port" value="80">
                                </div>
                            </div>
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Username</label>
                                    <input type="text" class="form-control" name="username" id="import-series-username">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Password</label>
                                    <input type="text" class="form-control" name="password" id="import-series-password">
                                </div>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Server Name (for reference)</label>
                            <input type="text" class="form-control" name="server_name" id="import-series-server-name"
                                placeholder="My IPTV Server">
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Category Filter (Optional)</label>
                            <select class="form-select" id="import-series-category">
                                <option value="">All Series Categories</option>
                            </select>
                            <small class="text-secondary">Click "Load Categories" to fetch available categories</small>
                        </div>

                        <div class="d-flex gap-2">
                            <button type="button" class="btn btn-outline-primary" onclick="loadSeriesUrlCategories()">
                                <i class="bi bi-folder"></i> Load Categories
                            </button>
                            <button type="button" class="btn btn-outline-info" onclick="previewSeriesUrl()">
                                <i class="bi bi-eye"></i> Preview
                            </button>
                        </div>
                    </form>

                    <div id="import-series-url-preview" class="mt-3" style="display: none;">
                        <hr>
                        <h6>Preview</h6>
                        <div id="import-series-url-preview-content"></div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-success" onclick="importSeriesFromUrl()">
                        <i class="bi bi-download"></i> Import Series
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Series Details Modal -->
    <div class="modal fade" id="seriesDetailsModal" tabindex="-1">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="series-detail-title">Series Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-3">
                            <img id="series-detail-poster" src="" class="img-fluid rounded" alt="Poster">
                        </div>
                        <div class="col-md-9">
                            <h4 id="series-detail-name"></h4>
                            <p class="text-secondary" id="series-detail-year"></p>
                            <p id="series-detail-plot"></p>
                            <div class="row mb-3">
                                <div class="col-md-4">
                                    <strong>Genre:</strong> <span id="series-detail-genre">-</span>
                                </div>
                                <div class="col-md-4">
                                    <strong>Rating:</strong> <span id="series-detail-rating">-</span>
                                </div>
                                <div class="col-md-4">
                                    <strong>Seasons:</strong> <span id="series-detail-seasons">-</span>
                                </div>
                            </div>
                            <hr>
                            <div class="d-flex justify-content-between align-items-center mb-3">
                                <h6>Episodes</h6>
                                <div>
                                    <select class="form-select form-select-sm d-inline-block w-auto" id="series-season-select" onchange="loadSeasonEpisodes()">
                                        <option value="">All Seasons</option>
                                    </select>
                                    <button class="btn btn-sm btn-success ms-2" onclick="selectAllEpisodes()">
                                        <i class="bi bi-check-all"></i> Select All
                                    </button>
                                </div>
                            </div>
                            <div id="series-episodes-list" style="max-height: 300px; overflow-y: auto;">
                                <p class="text-secondary">Loading episodes...</p>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-success" id="series-detail-download-btn" onclick="downloadSelectedEpisodes()">
                        <i class="bi bi-download"></i> Download Selected (<span id="selected-episodes-count">0</span>)
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Movie Details Modal -->
    <div class="modal fade" id="movieDetailsModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="movie-detail-title">Movie Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-4">
                            <img id="movie-detail-poster" src="" class="img-fluid rounded" alt="Poster">
                        </div>
                        <div class="col-md-8">
                            <h4 id="movie-detail-name"></h4>
                            <p class="text-secondary" id="movie-detail-year"></p>
                            <p id="movie-detail-plot"></p>
                            <div class="mb-2">
                                <strong>Genre:</strong> <span id="movie-detail-genre">-</span>
                            </div>
                            <div class="mb-2">
                                <strong>Rating:</strong> <span id="movie-detail-rating">-</span>
                            </div>
                            <div class="mb-2">
                                <strong>Duration:</strong> <span id="movie-detail-duration">-</span>
                            </div>
                            <hr>
                            <h6>Stream Links (<span id="movie-detail-links-count">0</span>)</h6>
                            <div id="movie-detail-links" style="max-height: 150px; overflow-y: auto;"></div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-success" id="movie-detail-download-btn">
                        <i class="bi bi-download"></i> Download
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Folder Browser Modal -->
    <div class="modal fade" id="folderBrowserModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title"><i class="bi bi-folder me-2"></i>Select Folder</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="d-flex align-items-center mb-3 p-2 bg-dark rounded">
                        <i class="bi bi-geo-alt me-2"></i>
                        <code id="folder-browser-path" data-path="/">/</code>
                    </div>
                    <div id="folder-browser-list" class="list-group" style="max-height: 400px; overflow-y: auto;">
                        <div class="text-center py-3">Select a server first</div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" onclick="selectCurrentFolder()">
                        <i class="bi bi-check-lg me-1"></i>Select Current Folder
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="assets/js/app.js?v=36"></script>
</body>
</html>
