<?php
/**
 * VOD Import System - Installation Script
 * Access this file once to set up the database
 */

$dbHost = 'localhost';
$dbUser = 'root';
$dbPass = '';
$dbName = 'vod_import';

$errors = [];
$success = [];

// Check if form was submitted
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $dbHost = $_POST['db_host'] ?? 'localhost';
    $dbUser = $_POST['db_user'] ?? 'root';
    $dbPass = $_POST['db_pass'] ?? '';
    $dbName = $_POST['db_name'] ?? 'vod_import';

    try {
        // Connect to MySQL server
        $pdo = new PDO(
            "mysql:host={$dbHost}",
            $dbUser,
            $dbPass,
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );

        // Create database
        $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$dbName}` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        $success[] = "Database '{$dbName}' created successfully";

        // Select database
        $pdo->exec("USE `{$dbName}`");

        // Read and execute SQL file
        $sqlFile = __DIR__ . '/config/database.sql';
        if (file_exists($sqlFile)) {
            $sql = file_get_contents($sqlFile);

            // Remove CREATE DATABASE and USE statements (we already did that)
            $sql = preg_replace('/CREATE DATABASE.*?;/i', '', $sql);
            $sql = preg_replace('/USE.*?;/i', '', $sql);

            // Execute multi-query
            $statements = array_filter(
                array_map('trim', explode(';', $sql)),
                function ($s) { return !empty($s) && strpos($s, '--') !== 0; }
            );

            foreach ($statements as $statement) {
                if (stripos($statement, 'SET') === 0 ||
                    stripos($statement, 'CREATE') === 0 ||
                    stripos($statement, 'INSERT') === 0 ||
                    stripos($statement, 'DROP') === 0) {
                    try {
                        $pdo->exec($statement);
                    } catch (PDOException $e) {
                        // Ignore "table already exists" errors
                        if (strpos($e->getMessage(), '1050') === false) {
                            $errors[] = "SQL Error: " . substr($e->getMessage(), 0, 200);
                        }
                    }
                }
            }

            $success[] = "Database tables created successfully";
        } else {
            $errors[] = "SQL file not found: {$sqlFile}";
        }

        // Update config file with database credentials
        $configFile = __DIR__ . '/config/config.php';
        if (file_exists($configFile)) {
            $config = file_get_contents($configFile);

            $config = preg_replace("/define\('DB_HOST',\s*'[^']*'\);/", "define('DB_HOST', '{$dbHost}');", $config);
            $config = preg_replace("/define\('DB_NAME',\s*'[^']*'\);/", "define('DB_NAME', '{$dbName}');", $config);
            $config = preg_replace("/define\('DB_USER',\s*'[^']*'\);/", "define('DB_USER', '{$dbUser}');", $config);
            $config = preg_replace("/define\('DB_PASS',\s*'[^']*'\);/", "define('DB_PASS', '{$dbPass}');", $config);

            file_put_contents($configFile, $config);
            $success[] = "Config file updated successfully";
        }

        // Create necessary directories
        $dirs = ['cache', 'logs', 'media', 'media/movies', 'media/series', 'watch', 'watch/invalid'];
        foreach ($dirs as $dir) {
            $path = __DIR__ . '/' . $dir;
            if (!is_dir($path)) {
                mkdir($path, 0755, true);
                $success[] = "Directory created: {$dir}";
            }
        }

        // Set permissions
        chmod(__DIR__ . '/cache', 0755);
        chmod(__DIR__ . '/logs', 0755);
        chmod(__DIR__ . '/media', 0755);
        chmod(__DIR__ . '/watch', 0755);

        $success[] = "Installation completed successfully!";

    } catch (PDOException $e) {
        $errors[] = "Database connection failed: " . $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>VOD Import System - Installation</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
    <style>
        body {
            background: #0f172a;
            color: #f8fafc;
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .install-card {
            background: #1e293b;
            border-radius: 16px;
            padding: 2rem;
            max-width: 500px;
            width: 100%;
        }
        .form-control {
            background: #0f172a;
            border-color: rgba(255,255,255,0.1);
            color: #f8fafc;
        }
        .form-control:focus {
            background: #0f172a;
            border-color: #6366f1;
            color: #f8fafc;
        }
        .btn-primary {
            background: #6366f1;
            border-color: #6366f1;
        }
        .logo {
            font-size: 2rem;
            font-weight: 700;
            color: #6366f1;
            text-align: center;
            margin-bottom: 1.5rem;
        }
    </style>
</head>
<body>
    <div class="install-card">
        <div class="logo">
            <i class="bi bi-film"></i>
            VOD Import System
        </div>

        <?php if (!empty($errors)): ?>
            <div class="alert alert-danger">
                <strong>Errors:</strong>
                <ul class="mb-0">
                    <?php foreach ($errors as $error): ?>
                        <li><?= htmlspecialchars($error) ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <?php if (!empty($success)): ?>
            <div class="alert alert-success">
                <strong>Success:</strong>
                <ul class="mb-0">
                    <?php foreach ($success as $msg): ?>
                        <li><?= htmlspecialchars($msg) ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>

            <?php if (in_array("Installation completed successfully!", $success)): ?>
                <div class="text-center mt-4">
                    <a href="index.php" class="btn btn-primary btn-lg">
                        Go to Dashboard
                    </a>
                </div>
                <div class="alert alert-warning mt-3">
                    <strong>Important:</strong> Delete this file (install.php) after installation for security!
                </div>
            <?php endif; ?>
        <?php else: ?>
            <form method="POST">
                <div class="mb-3">
                    <label class="form-label">Database Host</label>
                    <input type="text" class="form-control" name="db_host" value="<?= htmlspecialchars($dbHost) ?>" required>
                </div>
                <div class="mb-3">
                    <label class="form-label">Database User</label>
                    <input type="text" class="form-control" name="db_user" value="<?= htmlspecialchars($dbUser) ?>" required>
                </div>
                <div class="mb-3">
                    <label class="form-label">Database Password</label>
                    <input type="password" class="form-control" name="db_pass" value="">
                </div>
                <div class="mb-3">
                    <label class="form-label">Database Name</label>
                    <input type="text" class="form-control" name="db_name" value="<?= htmlspecialchars($dbName) ?>" required>
                </div>
                <button type="submit" class="btn btn-primary w-100">
                    Install
                </button>
            </form>
        <?php endif; ?>

        <hr class="my-4" style="border-color: rgba(255,255,255,0.1);">

        <div class="small text-secondary">
            <strong>System Requirements:</strong>
            <ul>
                <li>PHP 7.4+ with PDO, cURL extensions</li>
                <li>MySQL 5.7+ or MariaDB 10.3+</li>
                <li>Apache with mod_rewrite</li>
            </ul>

            <strong>Cron Setup:</strong>
            <pre style="background: #0f172a; padding: 10px; border-radius: 4px; font-size: 0.75rem;">* * * * * php <?= __DIR__ ?>/workers/cron.php</pre>
        </div>
    </div>
</body>
</html>
