#!/bin/bash
#############################################
#  VOD Import System - Full Installation
#  Run: wget -O install.sh URL && bash install.sh
#############################################

set -e

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m'

clear
echo -e "${CYAN}"
echo "╔════════════════════════════════════════════════════════════╗"
echo "║          VOD Import System - Installation Script           ║"
echo "║                      Version 2.0                           ║"
echo "╚════════════════════════════════════════════════════════════╝"
echo -e "${NC}"

# Check root
if [ "$EUID" -ne 0 ]; then
    echo -e "${RED}[ERROR] Please run as root: sudo bash install.sh${NC}"
    exit 1
fi

# Configuration
DB_NAME="vod_import"
DB_USER="vod_user"
DB_PASS="VodSecure@$(date +%s | sha256sum | base64 | head -c 8)"
INSTALL_DIR="/var/www/html/VOD"
DOMAIN=""

echo -e "${YELLOW}Enter your domain (or press Enter for IP access):${NC}"
read -r DOMAIN

#############################################
# Step 1: Update System
#############################################
echo -e "\n${GREEN}[1/8] Updating system...${NC}"
apt-get update -y && apt-get upgrade -y

#############################################
# Step 2: Install Packages
#############################################
echo -e "\n${GREEN}[2/8] Installing required packages...${NC}"

apt-get install -y \
    nginx \
    mysql-server \
    php-fpm \
    php-mysql \
    php-curl \
    php-json \
    php-mbstring \
    php-xml \
    php-zip \
    php-gd \
    php-ssh2 \
    curl \
    wget \
    git \
    unzip \
    zip \
    sshpass \
    htop \
    net-tools \
    certbot \
    python3-certbot-nginx

# Get PHP version
PHP_VER=$(php -r "echo PHP_MAJOR_VERSION.'.'.PHP_MINOR_VERSION;")
echo -e "${CYAN}PHP Version: ${PHP_VER}${NC}"

#############################################
# Step 3: Configure MySQL
#############################################
echo -e "\n${GREEN}[3/8] Configuring MySQL...${NC}"

systemctl start mysql
systemctl enable mysql

mysql -e "CREATE DATABASE IF NOT EXISTS ${DB_NAME} CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
mysql -e "DROP USER IF EXISTS '${DB_USER}'@'localhost';"
mysql -e "CREATE USER '${DB_USER}'@'localhost' IDENTIFIED BY '${DB_PASS}';"
mysql -e "GRANT ALL PRIVILEGES ON ${DB_NAME}.* TO '${DB_USER}'@'localhost';"
mysql -e "FLUSH PRIVILEGES;"

#############################################
# Step 4: Create Directory Structure
#############################################
echo -e "\n${GREEN}[4/8] Creating directories...${NC}"

mkdir -p ${INSTALL_DIR}/{api,assets/{css,js,img},classes,config,cache,logs,media,watch,temp}

#############################################
# Step 5: Create Database Schema
#############################################
echo -e "\n${GREEN}[5/8] Creating database schema...${NC}"

mysql ${DB_NAME} << 'EOSQL'
-- Users
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100),
    role ENUM('admin', 'user', 'viewer') DEFAULT 'user',
    is_active TINYINT(1) DEFAULT 1,
    api_token VARCHAR(64),
    last_login DATETIME,
    login_ip VARCHAR(45),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Source Servers (IPTV)
CREATE TABLE IF NOT EXISTS servers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    url VARCHAR(255) NOT NULL,
    port INT DEFAULT 80,
    username VARCHAR(100),
    password VARCHAR(100),
    type ENUM('xtream', 'm3u', 'stalker') DEFAULT 'xtream',
    is_active TINYINT(1) DEFAULT 1,
    status ENUM('online', 'offline', 'unknown') DEFAULT 'unknown',
    last_check DATETIME,
    movies_count INT DEFAULT 0,
    series_count INT DEFAULT 0,
    live_count INT DEFAULT 0,
    priority INT DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Remote Servers (Download destinations)
CREATE TABLE IF NOT EXISTS remote_servers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    type ENUM('ssh', 'ftp', 'sftp', 'local') DEFAULT 'ssh',
    host VARCHAR(255) NOT NULL,
    port INT DEFAULT 22,
    username VARCHAR(100),
    password VARCHAR(255),
    ssh_key TEXT,
    remote_path VARCHAR(255) DEFAULT '/media',
    is_active TINYINT(1) DEFAULT 1,
    status ENUM('online', 'offline', 'unknown') DEFAULT 'unknown',
    last_check DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Categories
CREATE TABLE IF NOT EXISTS categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    server_id INT,
    external_id INT,
    name VARCHAR(100) NOT NULL,
    type ENUM('movie', 'series', 'live') DEFAULT 'movie',
    parent_id INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (server_id) REFERENCES servers(id) ON DELETE CASCADE,
    UNIQUE KEY unique_cat (server_id, external_id, type)
) ENGINE=InnoDB;

-- Movies
CREATE TABLE IF NOT EXISTS movies (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tmdb_id INT,
    title VARCHAR(255) NOT NULL,
    clean_name VARCHAR(255),
    original_title VARCHAR(255),
    year INT,
    plot TEXT,
    genre VARCHAR(255),
    director VARCHAR(255),
    cast TEXT,
    rating DECIMAL(3,1),
    duration INT,
    poster VARCHAR(500),
    backdrop VARCHAR(500),
    trailer_url VARCHAR(500),
    is_downloaded TINYINT(1) DEFAULT 0,
    local_path VARCHAR(500),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_title (title),
    INDEX idx_year (year),
    INDEX idx_tmdb (tmdb_id)
) ENGINE=InnoDB;

-- Movie Links
CREATE TABLE IF NOT EXISTS movie_links (
    id INT AUTO_INCREMENT PRIMARY KEY,
    movie_id INT NOT NULL,
    server_id INT NOT NULL,
    stream_id INT,
    category_id INT,
    url VARCHAR(500) NOT NULL,
    container_extension VARCHAR(10) DEFAULT 'mp4',
    quality VARCHAR(20),
    is_working TINYINT(1) DEFAULT 1,
    last_check DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (movie_id) REFERENCES movies(id) ON DELETE CASCADE,
    FOREIGN KEY (server_id) REFERENCES servers(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Series
CREATE TABLE IF NOT EXISTS series (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tmdb_id INT,
    imdb_id VARCHAR(20),
    title VARCHAR(255) NOT NULL,
    original_title VARCHAR(255),
    year INT,
    clean_name VARCHAR(255),
    description TEXT,
    plot TEXT,
    genre VARCHAR(255),
    cast TEXT,
    rating DECIMAL(3,1),
    poster VARCHAR(500),
    backdrop VARCHAR(500),
    seasons_count INT DEFAULT 0,
    episodes_count INT DEFAULT 0,
    category_id INT,
    status VARCHAR(20) DEFAULT 'ongoing',
    last_episode_date DATE,
    is_adult TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_title (title),
    INDEX idx_tmdb (tmdb_id)
) ENGINE=InnoDB;

-- Episodes
CREATE TABLE IF NOT EXISTS episodes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    series_id INT NOT NULL,
    server_id INT,
    stream_id INT,
    season_number INT NOT NULL,
    episode_number INT NOT NULL,
    title VARCHAR(255),
    clean_name VARCHAR(255),
    description TEXT,
    duration INT DEFAULT 0,
    stream_url VARCHAR(500),
    container_extension VARCHAR(10) DEFAULT 'mp4',
    poster VARCHAR(500),
    air_date DATE,
    rating DECIMAL(3,1),
    is_downloaded TINYINT(1) DEFAULT 0,
    download_path VARCHAR(500),
    is_orphan TINYINT(1) DEFAULT 0,
    status VARCHAR(20) DEFAULT 'available',
    added_timestamp INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (series_id) REFERENCES series(id) ON DELETE CASCADE,
    INDEX idx_series (series_id),
    INDEX idx_season (season_number)
) ENGINE=InnoDB;

-- Download Jobs
CREATE TABLE IF NOT EXISTS download_jobs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    remote_server_id INT NOT NULL,
    movie_id INT,
    episode_id INT,
    source_url VARCHAR(500) NOT NULL,
    filename VARCHAR(255) NOT NULL,
    destination_path VARCHAR(500),
    status ENUM('pending', 'downloading', 'completed', 'failed', 'cancelled') DEFAULT 'pending',
    progress INT DEFAULT 0,
    file_size BIGINT,
    downloaded_size BIGINT DEFAULT 0,
    error_message TEXT,
    started_at DATETIME,
    completed_at DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (remote_server_id) REFERENCES remote_servers(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Import Logs
CREATE TABLE IF NOT EXISTS import_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    server_id INT,
    action VARCHAR(50),
    type VARCHAR(50),
    target_id INT,
    target_name VARCHAR(255),
    status ENUM('success', 'error', 'warning', 'info') DEFAULT 'info',
    message TEXT,
    details JSON,
    ip_address VARCHAR(45),
    user_agent VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_status (status),
    INDEX idx_created (created_at)
) ENGINE=InnoDB;

-- Settings
CREATE TABLE IF NOT EXISTS settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    `key` VARCHAR(50) NOT NULL UNIQUE,
    `value` TEXT,
    `type` ENUM('string', 'int', 'bool', 'json') DEFAULT 'string',
    description VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Default Admin (password: admin123)
INSERT INTO users (username, password, email, role, is_active) VALUES
('admin', '$2y$12$LQv3c1yqBWVHxkd0LHAkCOYz6TtxMQJqhN8/X4x5gL1GpLqEKRnwq', 'admin@localhost', 'admin', 1)
ON DUPLICATE KEY UPDATE username=username;

-- Default Settings
INSERT INTO settings (`key`, `value`, `type`, description) VALUES
('site_name', 'VOD Import System', 'string', 'Site name'),
('items_per_page', '24', 'int', 'Items per page'),
('auto_check_links', '1', 'bool', 'Auto check links'),
('tmdb_api_key', '', 'string', 'TMDB API Key'),
('default_quality', '1080p', 'string', 'Default quality'),
('auto_download_movies', '0', 'bool', 'Auto download movies'),
('auto_download_series', '0', 'bool', 'Auto download series')
ON DUPLICATE KEY UPDATE `key`=`key`;
EOSQL

echo -e "${GREEN}Database schema created${NC}"

#############################################
# Step 6: Create Config File
#############################################
echo -e "\n${GREEN}[6/8] Creating configuration...${NC}"

cat > ${INSTALL_DIR}/config/config.php << PHPEOF
<?php
/**
 * VOD Import System Configuration
 * Generated: $(date)
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/../logs/php_errors.log');

date_default_timezone_set('UTC');

// Database
define('DB_HOST', 'localhost');
define('DB_NAME', '${DB_NAME}');
define('DB_USER', '${DB_USER}');
define('DB_PASS', '${DB_PASS}');
define('DB_CHARSET', 'utf8mb4');

// Paths
define('BASE_PATH', dirname(__DIR__));
define('CONFIG_PATH', BASE_PATH . '/config');
define('CACHE_PATH', BASE_PATH . '/cache');
define('LOGS_PATH', BASE_PATH . '/logs');
define('MEDIA_PATH', BASE_PATH . '/media');
define('WATCH_PATH', BASE_PATH . '/watch');

// API
define('API_VERSION', '2.0');
define('API_RATE_LIMIT', 100);

// Cache
define('CACHE_ENABLED', true);
define('CACHE_TTL', 300);

// Downloads
define('MAX_CONCURRENT_DOWNLOADS', 3);
define('DOWNLOAD_TIMEOUT', 7200);

// Debug
define('DEBUG_MODE', false);

/**
 * Database Connection
 */
function getDB(): PDO {
    static \$pdo = null;
    if (\$pdo === null) {
        \$dsn = sprintf('mysql:host=%s;dbname=%s;charset=%s', DB_HOST, DB_NAME, DB_CHARSET);
        \$pdo = new PDO(\$dsn, DB_USER, DB_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false
        ]);
    }
    return \$pdo;
}

function getSetting(string \$key, \$default = null) {
    static \$settings = null;
    if (\$settings === null) {
        try {
            \$stmt = getDB()->query("SELECT \`key\`, \`value\`, \`type\` FROM settings");
            \$settings = [];
            while (\$row = \$stmt->fetch()) {
                \$value = \$row['value'];
                switch (\$row['type']) {
                    case 'int': \$value = (int) \$value; break;
                    case 'bool': \$value = (bool) \$value; break;
                    case 'json': \$value = json_decode(\$value, true); break;
                }
                \$settings[\$row['key']] = \$value;
            }
        } catch (Exception \$e) { return \$default; }
    }
    return \$settings[\$key] ?? \$default;
}

function setSetting(string \$key, \$value, string \$type = 'string'): bool {
    if (\$type === 'json') \$value = json_encode(\$value);
    elseif (\$type === 'bool') \$value = \$value ? '1' : '0';
    \$stmt = getDB()->prepare("INSERT INTO settings (\`key\`, \`value\`, \`type\`) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE \`value\` = VALUES(\`value\`)");
    return \$stmt->execute([\$key, (string) \$value, \$type]);
}

spl_autoload_register(function (\$class) {
    \$file = BASE_PATH . '/classes/' . \$class . '.php';
    if (file_exists(\$file)) require_once \$file;
});

foreach ([CACHE_PATH, LOGS_PATH, MEDIA_PATH, WATCH_PATH] as \$dir) {
    if (!is_dir(\$dir)) mkdir(\$dir, 0755, true);
}
PHPEOF

#############################################
# Step 7: Configure PHP & Nginx
#############################################
echo -e "\n${GREEN}[7/8] Configuring PHP & Nginx...${NC}"

# PHP Config
PHP_INI="/etc/php/${PHP_VER}/fpm/php.ini"
if [ -f "$PHP_INI" ]; then
    sed -i 's/upload_max_filesize = .*/upload_max_filesize = 1024M/' $PHP_INI
    sed -i 's/post_max_size = .*/post_max_size = 1024M/' $PHP_INI
    sed -i 's/max_execution_time = .*/max_execution_time = 600/' $PHP_INI
    sed -i 's/memory_limit = .*/memory_limit = 512M/' $PHP_INI
fi

# Nginx Config
PHP_SOCK="/var/run/php/php${PHP_VER}-fpm.sock"

if [ -n "$DOMAIN" ]; then
    SERVER_NAME="$DOMAIN"
else
    SERVER_NAME="_"
fi

cat > /etc/nginx/sites-available/vod << NGINXEOF
server {
    listen 80;
    server_name ${SERVER_NAME};
    root /var/www/html;
    index index.php index.html;
    client_max_body_size 1024M;

    location /VOD {
        alias ${INSTALL_DIR};
        index index.php;

        location ~ \.php\$ {
            fastcgi_pass unix:${PHP_SOCK};
            fastcgi_param SCRIPT_FILENAME \$request_filename;
            include fastcgi_params;
            fastcgi_read_timeout 600;
        }
    }

    location ~ \.php\$ {
        include snippets/fastcgi-php.conf;
        fastcgi_pass unix:${PHP_SOCK};
        fastcgi_read_timeout 600;
    }

    location ~ /\.ht { deny all; }
}
NGINXEOF

ln -sf /etc/nginx/sites-available/vod /etc/nginx/sites-enabled/
rm -f /etc/nginx/sites-enabled/default

nginx -t && systemctl restart nginx
systemctl restart php${PHP_VER}-fpm
systemctl enable nginx php${PHP_VER}-fpm

#############################################
# Step 8: Set Permissions & Finish
#############################################
echo -e "\n${GREEN}[8/8] Setting permissions...${NC}"

chown -R www-data:www-data ${INSTALL_DIR}
chmod -R 755 ${INSTALL_DIR}
chmod -R 777 ${INSTALL_DIR}/{cache,logs,media,watch,temp}

# Get Server IP
SERVER_IP=$(hostname -I | awk '{print $1}')

# Save credentials
cat > /root/vod_credentials.txt << CREDEOF
============================================
VOD Import System - Installation Complete
============================================
Date: $(date)

ACCESS URL:
  http://${SERVER_IP}/VOD/

ADMIN LOGIN:
  Username: admin
  Password: admin123

DATABASE:
  Host: localhost
  Name: ${DB_NAME}
  User: ${DB_USER}
  Pass: ${DB_PASS}

FILES:
  Install Dir: ${INSTALL_DIR}
  Config: ${INSTALL_DIR}/config/config.php
  Logs: ${INSTALL_DIR}/logs/

IMPORTANT:
  1. Change admin password after first login!
  2. Copy PHP files to: ${INSTALL_DIR}/
============================================
CREDEOF

chmod 600 /root/vod_credentials.txt

# Summary
echo ""
echo -e "${CYAN}╔════════════════════════════════════════════════════════════╗"
echo -e "║              Installation Complete!                        ║"
echo -e "╚════════════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "${GREEN}Access URL:${NC} http://${SERVER_IP}/VOD/"
echo ""
echo -e "${GREEN}Admin Login:${NC}"
echo "  Username: admin"
echo "  Password: admin123"
echo ""
echo -e "${GREEN}Database:${NC}"
echo "  Name: ${DB_NAME}"
echo "  User: ${DB_USER}"
echo "  Pass: ${DB_PASS}"
echo ""
echo -e "${YELLOW}Credentials saved to: /root/vod_credentials.txt${NC}"
echo ""
echo -e "${RED}NEXT STEPS:${NC}"
echo "  1. Copy your PHP application files to: ${INSTALL_DIR}/"
echo "  2. Change admin password after login!"
echo ""
