#!/usr/bin/env python3
"""
VOD Smart Downloader
====================
Advanced download system with:
- Smart retry mechanism (10 retries, 5s delay)
- Dynamic protocol switching (Python → wget)
- Custom User-Agent (TiviMate)
- Progress tracking
- Error handling for 403/429/timeout/connection reset
"""

import os
import sys
import time
import json
import subprocess
import argparse
import hashlib
from pathlib import Path
from datetime import datetime
from typing import Optional, Tuple, Dict, Any

# Try httpx first, fallback to requests
try:
    import httpx
    USE_HTTPX = True
except ImportError:
    import requests
    USE_HTTPX = False

# Configuration
CONFIG = {
    'max_retries': 10,
    'retry_delay': 5,
    'timeout': 300,
    'chunk_size': 8192 * 4,  # 32KB chunks
    'user_agent': 'TiviMate/5.1.6 (Android 7.1.2)',
    'temp_dir': '/tmp/vod_downloads',
    'log_file': '/var/www/html/VOD/logs/downloader.log'
}

# Retry-able errors
RETRYABLE_ERRORS = [
    'Connection reset',
    'Connection refused',
    'Timeout',
    'timed out',
    'temporarily unavailable',
    'Service Unavailable',
    'Bad Gateway',
    'Gateway Timeout'
]

RETRYABLE_STATUS_CODES = [403, 429, 500, 502, 503, 504]


class SmartDownloader:
    """Smart VOD Downloader with retry and protocol switching"""

    def __init__(self, config: Dict = None):
        self.config = {**CONFIG, **(config or {})}
        self.stats = {
            'started_at': None,
            'completed_at': None,
            'total_bytes': 0,
            'downloaded_bytes': 0,
            'retries': 0,
            'method': 'python',
            'switched_to_wget': False,
            'errors': []
        }

        # Create temp directory
        Path(self.config['temp_dir']).mkdir(parents=True, exist_ok=True)

        # Setup logging
        log_dir = Path(self.config['log_file']).parent
        log_dir.mkdir(parents=True, exist_ok=True)

    def log(self, message: str, level: str = 'INFO'):
        """Log message to file and console"""
        timestamp = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        log_line = f"[{timestamp}] [{level}] {message}"
        print(log_line)

        try:
            with open(self.config['log_file'], 'a') as f:
                f.write(log_line + '\n')
        except:
            pass

    def is_retryable_error(self, error: str) -> bool:
        """Check if error is retryable"""
        error_lower = error.lower()
        return any(err.lower() in error_lower for err in RETRYABLE_ERRORS)

    def get_headers(self) -> Dict[str, str]:
        """Get request headers"""
        return {
            'User-Agent': self.config['user_agent'],
            'Accept': '*/*',
            'Accept-Encoding': 'gzip, deflate',
            'Connection': 'keep-alive',
            'Referer': 'http://localhost/'
        }

    def download_with_python(self, url: str, output_path: str,
                             progress_callback=None) -> Tuple[bool, str]:
        """Download using Python (httpx/requests)"""
        self.log(f"Starting Python download: {url}")
        self.stats['method'] = 'python'

        headers = self.get_headers()
        temp_file = f"{output_path}.part"

        try:
            if USE_HTTPX:
                return self._download_httpx(url, temp_file, output_path,
                                           headers, progress_callback)
            else:
                return self._download_requests(url, temp_file, output_path,
                                              headers, progress_callback)
        except Exception as e:
            error_msg = str(e)
            self.stats['errors'].append(error_msg)
            return False, error_msg

    def _download_httpx(self, url: str, temp_file: str, output_path: str,
                        headers: Dict, progress_callback) -> Tuple[bool, str]:
        """Download using httpx"""
        with httpx.stream('GET', url, headers=headers,
                         timeout=self.config['timeout'],
                         follow_redirects=True) as response:

            if response.status_code in RETRYABLE_STATUS_CODES:
                return False, f"HTTP {response.status_code}"

            response.raise_for_status()

            total_size = int(response.headers.get('content-length', 0))
            self.stats['total_bytes'] = total_size
            downloaded = 0

            with open(temp_file, 'wb') as f:
                for chunk in response.iter_bytes(chunk_size=self.config['chunk_size']):
                    f.write(chunk)
                    downloaded += len(chunk)
                    self.stats['downloaded_bytes'] = downloaded

                    if progress_callback and total_size > 0:
                        progress = int((downloaded / total_size) * 100)
                        progress_callback(progress, downloaded, total_size)

        # Move temp to final
        os.rename(temp_file, output_path)
        return True, "Success"

    def _download_requests(self, url: str, temp_file: str, output_path: str,
                          headers: Dict, progress_callback) -> Tuple[bool, str]:
        """Download using requests"""
        response = requests.get(url, headers=headers, stream=True,
                               timeout=self.config['timeout'],
                               allow_redirects=True)

        if response.status_code in RETRYABLE_STATUS_CODES:
            return False, f"HTTP {response.status_code}"

        response.raise_for_status()

        total_size = int(response.headers.get('content-length', 0))
        self.stats['total_bytes'] = total_size
        downloaded = 0

        with open(temp_file, 'wb') as f:
            for chunk in response.iter_content(chunk_size=self.config['chunk_size']):
                if chunk:
                    f.write(chunk)
                    downloaded += len(chunk)
                    self.stats['downloaded_bytes'] = downloaded

                    if progress_callback and total_size > 0:
                        progress = int((downloaded / total_size) * 100)
                        progress_callback(progress, downloaded, total_size)

        os.rename(temp_file, output_path)
        return True, "Success"

    def download_with_wget(self, url: str, output_path: str,
                          progress_callback=None) -> Tuple[bool, str]:
        """Download using wget as fallback"""
        self.log("Switching to wget download")
        self.stats['method'] = 'wget'
        self.stats['switched_to_wget'] = True

        temp_file = f"{output_path}.wget"

        cmd = [
            'wget',
            '--no-check-certificate',
            '--tries=3',
            '--timeout=60',
            '--waitretry=5',
            '--retry-connrefused',
            f'--user-agent={self.config["user_agent"]}',
            '--header=Accept: */*',
            '--header=Connection: keep-alive',
            '-O', temp_file,
            '-q',
            '--show-progress',
            url
        ]

        try:
            self.log(f"wget command: {' '.join(cmd)}")
            result = subprocess.run(cmd, capture_output=True, text=True,
                                   timeout=self.config['timeout'] * 2)

            if result.returncode == 0 and os.path.exists(temp_file):
                file_size = os.path.getsize(temp_file)
                if file_size > 0:
                    os.rename(temp_file, output_path)
                    self.stats['downloaded_bytes'] = file_size
                    self.stats['total_bytes'] = file_size
                    return True, "Success (wget)"
                else:
                    return False, "Empty file downloaded"
            else:
                error = result.stderr or f"wget failed with code {result.returncode}"
                return False, error

        except subprocess.TimeoutExpired:
            return False, "wget timeout"
        except Exception as e:
            return False, str(e)
        finally:
            # Cleanup temp file
            if os.path.exists(temp_file):
                try:
                    os.remove(temp_file)
                except:
                    pass

    def download_with_curl(self, url: str, output_path: str) -> Tuple[bool, str]:
        """Download using curl as last resort"""
        self.log("Trying curl as last resort")
        self.stats['method'] = 'curl'

        temp_file = f"{output_path}.curl"

        cmd = [
            'curl',
            '-L',  # Follow redirects
            '-k',  # Insecure (ignore SSL)
            '--retry', '3',
            '--retry-delay', '5',
            '-A', self.config['user_agent'],
            '-H', 'Accept: */*',
            '-o', temp_file,
            '-s',  # Silent
            '--progress-bar',
            url
        ]

        try:
            result = subprocess.run(cmd, capture_output=True, text=True,
                                   timeout=self.config['timeout'] * 2)

            if result.returncode == 0 and os.path.exists(temp_file):
                file_size = os.path.getsize(temp_file)
                if file_size > 0:
                    os.rename(temp_file, output_path)
                    self.stats['downloaded_bytes'] = file_size
                    return True, "Success (curl)"

            return False, f"curl failed: {result.stderr}"

        except Exception as e:
            return False, str(e)
        finally:
            if os.path.exists(temp_file):
                try:
                    os.remove(temp_file)
                except:
                    pass

    def download(self, url: str, output_path: str,
                progress_callback=None) -> Dict[str, Any]:
        """
        Main download method with smart retry and protocol switching

        Returns dict with:
            - success: bool
            - message: str
            - stats: dict
        """
        self.stats['started_at'] = datetime.now().isoformat()
        self.log(f"Starting download: {url}")
        self.log(f"Output: {output_path}")

        # Ensure output directory exists
        Path(output_path).parent.mkdir(parents=True, exist_ok=True)

        # Try Python first with retries
        for attempt in range(1, self.config['max_retries'] + 1):
            self.log(f"Attempt {attempt}/{self.config['max_retries']} (Python)")
            self.stats['retries'] = attempt

            success, error = self.download_with_python(url, output_path,
                                                       progress_callback)

            if success:
                self.stats['completed_at'] = datetime.now().isoformat()
                self.log(f"Download completed successfully via Python")
                return {
                    'success': True,
                    'message': 'Download completed',
                    'stats': self.stats
                }

            self.log(f"Python download failed: {error}", 'ERROR')

            # Check if error is retryable
            if not self.is_retryable_error(error) and 'HTTP' not in error:
                # Non-retryable error, switch to wget immediately
                break

            # Check if should switch to wget (after 3 Python failures)
            if attempt >= 3 and ('403' in error or '429' in error or
                                'reset' in error.lower()):
                self.log("Multiple failures detected, switching to wget")
                break

            # Wait before retry
            if attempt < self.config['max_retries']:
                self.log(f"Waiting {self.config['retry_delay']}s before retry...")
                time.sleep(self.config['retry_delay'])

        # Try wget with retries
        self.log("Switching to wget protocol")
        for attempt in range(1, self.config['max_retries'] + 1):
            self.log(f"Attempt {attempt}/{self.config['max_retries']} (wget)")

            success, error = self.download_with_wget(url, output_path,
                                                     progress_callback)

            if success:
                self.stats['completed_at'] = datetime.now().isoformat()
                self.log(f"Download completed successfully via wget")
                return {
                    'success': True,
                    'message': 'Download completed (wget)',
                    'stats': self.stats
                }

            self.log(f"wget download failed: {error}", 'ERROR')

            if attempt < self.config['max_retries']:
                self.log(f"Waiting {self.config['retry_delay']}s before retry...")
                time.sleep(self.config['retry_delay'])

        # Last resort: try curl
        self.log("Trying curl as last resort")
        success, error = self.download_with_curl(url, output_path)

        if success:
            self.stats['completed_at'] = datetime.now().isoformat()
            return {
                'success': True,
                'message': 'Download completed (curl)',
                'stats': self.stats
            }

        # All methods failed
        self.stats['completed_at'] = datetime.now().isoformat()
        self.log(f"All download methods failed", 'ERROR')

        return {
            'success': False,
            'message': f'Download failed after all retries: {error}',
            'stats': self.stats
        }


def progress_bar(progress: int, downloaded: int, total: int):
    """Display progress bar"""
    bar_length = 40
    filled = int(bar_length * progress / 100)
    bar = '█' * filled + '░' * (bar_length - filled)

    # Format sizes
    dl_mb = downloaded / (1024 * 1024)
    total_mb = total / (1024 * 1024)

    sys.stdout.write(f'\r[{bar}] {progress}% ({dl_mb:.1f}/{total_mb:.1f} MB)')
    sys.stdout.flush()

    if progress >= 100:
        print()


def main():
    parser = argparse.ArgumentParser(description='VOD Smart Downloader')
    parser.add_argument('url', help='URL to download')
    parser.add_argument('output', help='Output file path')
    parser.add_argument('--retries', type=int, default=10,
                       help='Max retries (default: 10)')
    parser.add_argument('--delay', type=int, default=5,
                       help='Delay between retries in seconds (default: 5)')
    parser.add_argument('--timeout', type=int, default=300,
                       help='Timeout in seconds (default: 300)')
    parser.add_argument('--quiet', action='store_true',
                       help='Quiet mode (no progress bar)')
    parser.add_argument('--json', action='store_true',
                       help='Output result as JSON')

    args = parser.parse_args()

    config = {
        'max_retries': args.retries,
        'retry_delay': args.delay,
        'timeout': args.timeout
    }

    downloader = SmartDownloader(config)

    callback = None if args.quiet else progress_bar
    result = downloader.download(args.url, args.output, callback)

    if args.json:
        print(json.dumps(result, indent=2))
    else:
        if result['success']:
            print(f"\n✓ Download completed: {args.output}")
            print(f"  Method: {result['stats']['method']}")
            print(f"  Size: {result['stats']['downloaded_bytes'] / (1024*1024):.2f} MB")
            print(f"  Retries: {result['stats']['retries']}")
        else:
            print(f"\n✗ Download failed: {result['message']}")
            sys.exit(1)


if __name__ == '__main__':
    main()
