<?php
/**
 * FolderWatch Integration
 * Handles file naming validation and folder structure for media management
 */

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/Logger.php';

class FolderWatch
{
    private PDO $db;
    private Logger $logger;
    private string $watchPath;
    private string $mediaPath;

    // Accepted file extensions
    private const VIDEO_EXTENSIONS = ['mp4', 'mkv', 'avi', 'mov', 'wmv', 'flv', 'webm', 'm4v', 'ts', 'm3u8'];

    // Name patterns
    private const MOVIE_PATTERN = '/^(.+)\s*\((\d{4})\)\.([a-zA-Z0-9]+)$/';
    private const SERIES_PATTERN = '/^(.+)\s*\((\d{4})\)\.s(\d{2})\.e(\d{2,3})\.([a-zA-Z0-9]+)$/i';

    public function __construct()
    {
        $this->db = getDB();
        $this->logger = new Logger('folderwatch', $this->db);
        $this->watchPath = getSetting('folderwatch_path', WATCH_PATH);
        $this->mediaPath = getSetting('download_path', MEDIA_PATH);

        // Ensure directories exist
        $this->ensureDirectories();
    }

    /**
     * Ensure required directories exist
     */
    private function ensureDirectories(): void
    {
        $dirs = [
            $this->watchPath,
            $this->mediaPath,
            $this->mediaPath . '/movies',
            $this->mediaPath . '/series',
            $this->watchPath . '/invalid'
        ];

        foreach ($dirs as $dir) {
            if (!is_dir($dir)) {
                mkdir($dir, 0755, true);
            }
        }
    }

    /**
     * Validate filename for FolderWatch compatibility
     */
    public function validateFilename(string $filename): array
    {
        $result = [
            'valid' => false,
            'type' => null,
            'title' => null,
            'year' => null,
            'season' => null,
            'episode' => null,
            'extension' => null,
            'errors' => [],
            'suggestion' => null
        ];

        // Check extension
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        if (!in_array($extension, self::VIDEO_EXTENSIONS)) {
            $result['errors'][] = "Invalid extension: {$extension}. Accepted: " . implode(', ', self::VIDEO_EXTENSIONS);
            return $result;
        }

        $result['extension'] = $extension;

        // Try to match as series episode first
        if (preg_match(self::SERIES_PATTERN, $filename, $matches)) {
            $result['valid'] = true;
            $result['type'] = 'episode';
            $result['title'] = trim($matches[1]);
            $result['year'] = (int) $matches[2];
            $result['season'] = (int) $matches[3];
            $result['episode'] = (int) $matches[4];
            return $result;
        }

        // Try to match as movie
        if (preg_match(self::MOVIE_PATTERN, $filename, $matches)) {
            $result['valid'] = true;
            $result['type'] = 'movie';
            $result['title'] = trim($matches[1]);
            $result['year'] = (int) $matches[2];
            return $result;
        }

        // Invalid format - try to suggest a fix
        $result['errors'][] = "Invalid filename format. Expected: 'Title (Year).ext' for movies or 'Title (Year).sXX.eXX.ext' for episodes";
        $result['suggestion'] = $this->suggestFilename($filename);

        return $result;
    }

    /**
     * Suggest a valid filename
     */
    public function suggestFilename(string $filename): ?string
    {
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        $basename = pathinfo($filename, PATHINFO_FILENAME);

        // Clean the basename
        $clean = $basename;

        // Replace common separators with spaces
        $clean = preg_replace('/[._-]+/', ' ', $clean);

        // Try to extract year
        $year = null;
        if (preg_match('/\b(19\d{2}|20\d{2})\b/', $clean, $yearMatch)) {
            $year = $yearMatch[1];
            $clean = preg_replace('/\b' . $year . '\b/', '', $clean);
        }

        // Try to extract season/episode
        $season = null;
        $episode = null;

        $seasonEpPatterns = [
            '/[Ss](\d{1,2})\s*[Ee](\d{1,3})/',
            '/[Ss]eason\s*(\d{1,2}).*?[Ee]pisode\s*(\d{1,3})/i',
            '/(\d{1,2})[xX](\d{1,3})/',
        ];

        foreach ($seasonEpPatterns as $pattern) {
            if (preg_match($pattern, $clean, $matches)) {
                $season = (int) $matches[1];
                $episode = (int) $matches[2];
                $clean = preg_replace($pattern, '', $clean);
                break;
            }
        }

        // Remove quality tags
        $clean = preg_replace('/\b(720p|1080p|2160p|4K|HD|SD|CAM|TS|HDCAM|BluRay|WEB-?DL|WEBRip|HDRip|DVDRip|HEVC|x264|x265|AAC|AC3)\b/i', '', $clean);

        // Remove brackets content
        $clean = preg_replace('/\[.*?\]|\(.*?\)/', '', $clean);

        // Clean up multiple spaces
        $clean = preg_replace('/\s+/', ' ', trim($clean));

        if (empty($clean)) {
            return null;
        }

        // Build suggested filename
        $year = $year ?? date('Y');

        if ($season !== null && $episode !== null) {
            return sprintf('%s (%s).s%02d.e%02d.%s', $clean, $year, $season, $episode, $extension);
        }

        return sprintf('%s (%s).%s', $clean, $year, $extension);
    }

    /**
     * Scan watch folder for new files
     */
    public function scanWatchFolder(): array
    {
        $result = [
            'valid' => [],
            'invalid' => [],
            'total' => 0
        ];

        if (!is_dir($this->watchPath)) {
            $this->logger->error("Watch path does not exist", ['path' => $this->watchPath]);
            return $result;
        }

        $files = scandir($this->watchPath);

        foreach ($files as $file) {
            if ($file === '.' || $file === '..' || $file === 'invalid') {
                continue;
            }

            $fullPath = $this->watchPath . '/' . $file;

            if (!is_file($fullPath)) {
                continue;
            }

            $result['total']++;
            $validation = $this->validateFilename($file);

            if ($validation['valid']) {
                $result['valid'][] = [
                    'filename' => $file,
                    'path' => $fullPath,
                    'info' => $validation
                ];
            } else {
                $result['invalid'][] = [
                    'filename' => $file,
                    'path' => $fullPath,
                    'errors' => $validation['errors'],
                    'suggestion' => $validation['suggestion']
                ];
            }
        }

        $this->logger->info("Watch folder scanned", [
            'valid' => count($result['valid']),
            'invalid' => count($result['invalid'])
        ]);

        return $result;
    }

    /**
     * Process valid files from watch folder
     */
    public function processWatchFolder(): array
    {
        $scan = $this->scanWatchFolder();
        $processed = [];
        $errors = [];

        foreach ($scan['valid'] as $item) {
            try {
                $moved = $this->moveToMediaFolder($item['path'], $item['info']);

                if ($moved) {
                    $processed[] = [
                        'filename' => $item['filename'],
                        'destination' => $moved
                    ];

                    // Update database if movie/episode exists
                    $this->updateMediaRecord($item['info'], $moved);
                }

            } catch (Exception $e) {
                $errors[] = [
                    'filename' => $item['filename'],
                    'error' => $e->getMessage()
                ];
            }
        }

        // Move invalid files to invalid folder
        foreach ($scan['invalid'] as $item) {
            $invalidPath = $this->watchPath . '/invalid/' . $item['filename'];
            if (!file_exists($invalidPath)) {
                rename($item['path'], $invalidPath);
            }
        }

        return [
            'processed' => $processed,
            'errors' => $errors,
            'invalid_moved' => count($scan['invalid'])
        ];
    }

    /**
     * Move file to appropriate media folder
     */
    private function moveToMediaFolder(string $sourcePath, array $info): ?string
    {
        if ($info['type'] === 'movie') {
            $destDir = $this->mediaPath . '/movies';
            $destFile = $info['title'] . ' (' . $info['year'] . ').' . $info['extension'];
        } else {
            // Series - create series folder
            $seriesDir = $info['title'] . ' (' . $info['year'] . ')';
            $seasonDir = 'Season ' . str_pad($info['season'], 2, '0', STR_PAD_LEFT);
            $destDir = $this->mediaPath . '/series/' . $seriesDir . '/' . $seasonDir;

            $destFile = sprintf(
                '%s (%d).s%02d.e%02d.%s',
                $info['title'],
                $info['year'],
                $info['season'],
                $info['episode'],
                $info['extension']
            );
        }

        // Create destination directory
        if (!is_dir($destDir)) {
            mkdir($destDir, 0755, true);
        }

        $destPath = $destDir . '/' . $destFile;

        // Don't overwrite existing files
        if (file_exists($destPath)) {
            $this->logger->warning("File already exists", ['path' => $destPath]);
            return null;
        }

        if (rename($sourcePath, $destPath)) {
            $this->logger->info("File moved", [
                'source' => $sourcePath,
                'destination' => $destPath
            ]);
            return $destPath;
        }

        return null;
    }

    /**
     * Update database record with download path
     */
    private function updateMediaRecord(array $info, string $path): void
    {
        if ($info['type'] === 'movie') {
            $stmt = $this->db->prepare("
                UPDATE movies SET
                    is_downloaded = 1,
                    download_path = :path,
                    download_date = NOW(),
                    status = 'downloaded'
                WHERE title = :title AND year = :year
            ");

            $stmt->execute([
                'path' => $path,
                'title' => $info['title'],
                'year' => $info['year']
            ]);

        } else {
            // Find series and update episode
            $stmt = $this->db->prepare("
                SELECT id FROM series WHERE title = :title AND year = :year
            ");
            $stmt->execute([
                'title' => $info['title'],
                'year' => $info['year']
            ]);

            $series = $stmt->fetch();

            if ($series) {
                $stmt = $this->db->prepare("
                    UPDATE episodes SET
                        is_downloaded = 1,
                        download_path = :path,
                        status = 'downloaded'
                    WHERE series_id = :series_id
                      AND season_number = :season
                      AND episode_number = :episode
                ");

                $stmt->execute([
                    'path' => $path,
                    'series_id' => $series['id'],
                    'season' => $info['season'],
                    'episode' => $info['episode']
                ]);
            }
        }
    }

    /**
     * Find and report orphan episodes
     */
    public function findOrphanEpisodes(): array
    {
        $stmt = $this->db->query("
            SELECT e.*, s.name as server_name
            FROM episodes e
            JOIN servers s ON e.server_id = s.id
            WHERE e.series_id IS NULL OR e.is_orphan = 1
            ORDER BY e.clean_name
        ");

        return $stmt->fetchAll();
    }

    /**
     * Suggest series for orphan episode
     */
    public function suggestSeriesForOrphan(int $episodeId): array
    {
        $stmt = $this->db->prepare("SELECT * FROM episodes WHERE id = ?");
        $stmt->execute([$episodeId]);
        $episode = $stmt->fetch();

        if (!$episode) {
            return [];
        }

        // Parse episode name to get potential series title
        $validation = $this->validateFilename($episode['clean_name']);

        if (!$validation['valid']) {
            return [];
        }

        $title = $validation['title'];

        // Search for matching series
        $stmt = $this->db->prepare("
            SELECT *, MATCH(title) AGAINST(:title) as relevance
            FROM series
            WHERE MATCH(title) AGAINST(:title IN BOOLEAN MODE)
               OR title LIKE :like_title
            ORDER BY relevance DESC
            LIMIT 5
        ");

        $stmt->execute([
            'title' => $title,
            'like_title' => "%{$title}%"
        ]);

        return $stmt->fetchAll();
    }

    /**
     * Link orphan episode to series
     */
    public function linkOrphanToSeries(int $episodeId, int $seriesId): bool
    {
        $stmt = $this->db->prepare("
            UPDATE episodes SET
                series_id = :series_id,
                is_orphan = 0
            WHERE id = :episode_id
        ");

        $result = $stmt->execute([
            'series_id' => $seriesId,
            'episode_id' => $episodeId
        ]);

        if ($result) {
            // Remove from orphan_episodes table
            $stmt = $this->db->prepare("
                UPDATE orphan_episodes SET
                    resolution_status = 'linked',
                    resolved_at = NOW()
                WHERE episode_id = ?
            ");
            $stmt->execute([$episodeId]);

            $this->logger->info("Orphan episode linked to series", [
                'episode_id' => $episodeId,
                'series_id' => $seriesId
            ]);
        }

        return $result;
    }

    /**
     * Delete orphan episodes
     */
    public function deleteOrphanEpisodes(array $episodeIds = null): int
    {
        if ($episodeIds === null) {
            // Delete all orphans
            $stmt = $this->db->query("DELETE FROM episodes WHERE is_orphan = 1");
        } else {
            $placeholders = implode(',', array_fill(0, count($episodeIds), '?'));
            $stmt = $this->db->prepare("DELETE FROM episodes WHERE id IN ({$placeholders}) AND is_orphan = 1");
            $stmt->execute($episodeIds);
        }

        $deleted = $stmt->rowCount();

        $this->logger->info("Deleted orphan episodes", ['count' => $deleted]);

        return $deleted;
    }

    /**
     * Auto cleanup orphans based on settings
     */
    public function autoCleanupOrphans(): int
    {
        if (!getSetting('auto_cleanup_orphans', false)) {
            return 0;
        }

        return $this->deleteOrphanEpisodes();
    }

    /**
     * Generate batch rename script
     */
    public function generateRenameScript(array $files): string
    {
        $script = "#!/bin/bash\n";
        $script .= "# FolderWatch Rename Script\n";
        $script .= "# Generated: " . date('Y-m-d H:i:s') . "\n\n";

        foreach ($files as $file) {
            if (isset($file['suggestion']) && $file['suggestion']) {
                $oldName = escapeshellarg($file['filename']);
                $newName = escapeshellarg($file['suggestion']);
                $script .= "mv {$oldName} {$newName}\n";
            }
        }

        return $script;
    }

    /**
     * Get folder structure
     */
    public function getFolderStructure(): array
    {
        $structure = [
            'watch' => [
                'path' => $this->watchPath,
                'exists' => is_dir($this->watchPath),
                'writable' => is_writable($this->watchPath),
                'files' => 0
            ],
            'media' => [
                'path' => $this->mediaPath,
                'exists' => is_dir($this->mediaPath),
                'writable' => is_writable($this->mediaPath),
                'movies' => 0,
                'series' => 0
            ]
        ];

        // Count watch folder files
        if ($structure['watch']['exists']) {
            $files = glob($this->watchPath . '/*.*');
            $structure['watch']['files'] = count($files);
        }

        // Count media files
        if ($structure['media']['exists']) {
            $movies = glob($this->mediaPath . '/movies/*.*');
            $structure['media']['movies'] = count($movies);

            $seriesDirs = glob($this->mediaPath . '/series/*', GLOB_ONLYDIR);
            $structure['media']['series'] = count($seriesDirs);
        }

        return $structure;
    }
}
