<?php
/**
 * VOD Import System Configuration
 * Version: 1.0
 */

// Error Reporting (disable in production)
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/../logs/php_errors.log');

// Timezone
date_default_timezone_set('UTC');

// Database Configuration
define('DB_HOST', 'localhost');
define('DB_NAME', 'vod_import');
define('DB_USER', 'root');
define('DB_PASS', 'Admin@123');
define('DB_CHARSET', 'utf8mb4');

// Paths
define('BASE_PATH', dirname(__DIR__));
define('CONFIG_PATH', BASE_PATH . '/config');
define('CACHE_PATH', BASE_PATH . '/cache');
define('LOGS_PATH', BASE_PATH . '/logs');
define('MEDIA_PATH', BASE_PATH . '/media');
define('WATCH_PATH', BASE_PATH . '/watch');

// API Settings
define('API_VERSION', '1.0');
define('API_RATE_LIMIT', 100); // requests per minute

// Cache Settings
define('CACHE_ENABLED', true);
define('CACHE_TTL', 300); // 5 minutes
define('CACHE_CATEGORIES_TTL', 3600); // 1 hour

// Download Settings
define('MAX_CONCURRENT_DOWNLOADS', 3);
define('DOWNLOAD_TIMEOUT', 7200); // 2 hours
define('DOWNLOAD_CHUNK_SIZE', 8192); // 8KB

// Link Check Settings
define('LINK_CHECK_TIMEOUT', 10); // seconds
define('LINK_CHECK_RETRIES', 3);

// TMDB API (optional)
define('TMDB_API_KEY', '');
define('TMDB_BASE_URL', 'https://api.themoviedb.org/3');
define('TMDB_IMAGE_BASE', 'https://image.tmdb.org/t/p/');

// Debug Mode
define('DEBUG_MODE', true);

// Admin Credentials
define('ADMIN_USER', 'admin');
define('ADMIN_PASS', 'admin123');

// CORS Settings
define('CORS_ALLOWED_ORIGINS', '*');

/**
 * Get Database Connection (PDO)
 */
function getDB(): PDO {
    static $pdo = null;

    if ($pdo === null) {
        try {
            $dsn = sprintf(
                'mysql:host=%s;dbname=%s;charset=%s',
                DB_HOST,
                DB_NAME,
                DB_CHARSET
            );

            $pdo = new PDO($dsn, DB_USER, DB_PASS, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci"
            ]);
        } catch (PDOException $e) {
            if (DEBUG_MODE) {
                die("Database connection failed: " . $e->getMessage());
            }
            die("Database connection failed");
        }
    }

    return $pdo;
}

/**
 * Get Setting Value
 */
function getSetting(string $key, $default = null) {
    static $settings = null;

    if ($settings === null) {
        try {
            $pdo = getDB();
            $stmt = $pdo->query("SELECT `key`, `value`, `type` FROM settings");
            $settings = [];
            while ($row = $stmt->fetch()) {
                $value = $row['value'];
                switch ($row['type']) {
                    case 'int':
                        $value = (int) $value;
                        break;
                    case 'bool':
                        $value = (bool) $value;
                        break;
                    case 'json':
                        $value = json_decode($value, true);
                        break;
                }
                $settings[$row['key']] = $value;
            }
        } catch (Exception $e) {
            return $default;
        }
    }

    return $settings[$key] ?? $default;
}

/**
 * Update Setting Value
 */
function setSetting(string $key, $value, string $type = 'string'): bool {
    try {
        $pdo = getDB();

        if ($type === 'json') {
            $value = json_encode($value);
        } elseif ($type === 'bool') {
            $value = $value ? '1' : '0';
        }

        $stmt = $pdo->prepare("
            INSERT INTO settings (`key`, `value`, `type`)
            VALUES (?, ?, ?)
            ON DUPLICATE KEY UPDATE `value` = VALUES(`value`), `type` = VALUES(`type`)
        ");

        return $stmt->execute([$key, (string) $value, $type]);
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Autoloader
 */
spl_autoload_register(function ($class) {
    $file = BASE_PATH . '/classes/' . $class . '.php';
    if (file_exists($file)) {
        require_once $file;
    }
});

// Create required directories
$dirs = [CACHE_PATH, LOGS_PATH, MEDIA_PATH, WATCH_PATH];
foreach ($dirs as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}
