# Bitrate Information - Technical Notes

## Current Status

✅ **Codec information is working perfectly** - Video/audio codecs, resolution, FPS all display correctly

⚠️ **Bitrate has limitations for codec copy streams**

---

## Why Bitrate May Not Display for Some Streams

### Technical Explanation

When ffmpeg uses **codec copy** mode (`-vcodec copy -acodec copy`), it doesn't decode or transcode the stream - it just copies the data as-is. This means:

1. **No bitrate in progress files** - Progress files show `bitrate=N/A` for copy streams
2. **No bitrate in HLS output** - The m3u8 and .ts segment files don't contain embedded bitrate metadata
3. **Source probing is slow** - Probing the live source URL with ffprobe can take 10-30 seconds and may timeout

### Which Streams Are Affected

- **Codec Copy Streams** (most efficient, lowest CPU):
  - `video_codec: "copy"` or actual codec like `"h264"`
  - `audio_codec: "copy"` or actual codec like `"aac"`
  - Bitrate: May be NULL or delayed

- **Transcoded Streams** (higher CPU usage):
  - `video_codec: "libx264"`, `"libx265"`, etc.
  - Bitrate: Available immediately from progress file

---

## Current Bitrate Status

```sql
SELECT stream_id, bitrate,
       JSON_EXTRACT(stream_info, '$.codecs.video.codec_name') as vcodec,
       JSON_EXTRACT(stream_info, '$.codecs.audio.codec_name') as acodec
FROM streams_sys
WHERE stream_status = 0;
```

**Example Results**:
- Stream 26: **4269 kbps** ✓ (transcoded or had source bitrate)
- Stream 3: **8000 kbps** ✓ (manually set for testing)

---

## Solution Options

### Option 1: Wait for Source Probing (Current Implementation)
The codec daemon tries to probe the source URL to get bitrate. This can take 10-30 seconds after a stream starts.

**Pros**:
- Accurate source bitrate
- No manual intervention

**Cons**:
- Delayed (may take up to 30 seconds)
- May fail for some URLs (geo-restricted, auth-protected, etc.)

### Option 2: Manual Bitrate Configuration (Recommended)
For important streams where you need exact bitrate display:

```sql
-- Set bitrate manually for a specific stream (in kbps)
UPDATE streams_sys SET bitrate = 8000 WHERE stream_id = 3;
```

**Pros**:
- Immediate display
- Reliable
- Can set expected/target bitrate

**Cons**:
- Requires manual setup
- Bitrate won't update if source changes

### Option 3: Calculate from Segment Files
Monitor the actual .ts segment file sizes to calculate real-time bitrate.

**Implementation** (advanced):
```bash
# Get segment size and duration to calculate bitrate
ls -lh /path/to/streams/3_1_*.ts | tail -5
# (size in bytes * 8 / 10 seconds) = bitrate in bps
```

---

## Recommendation

For your current setup:

1. **Keep the codec daemon running** - It will eventually get bitrate for most streams
2. **Manually set bitrate for critical streams** - Use UPDATE query above
3. **Monitor the log** to see which streams successfully get bitrate:
   ```bash
   tail -f /var/log/sync_codec_daemon.log
   ```

---

## What's Working Perfectly

Even without bitrate, you have complete codec information:

✅ **Video Codec**: h264, h265, mpeg2, etc.
✅ **Audio Codec**: aac, mp3, ac3, etc.
✅ **Resolution**: 1920x1080, 1280x720, etc.
✅ **FPS**: 25, 30, 60, etc.
✅ **Real-time Speed**: 1.02x, 0.98x, etc.
✅ **Frame Count**: Updated every 3 seconds

---

## Bitrate Display in Web Interface

The web interface shows bitrate when available:

```
┌─────────────────────┐
│  8000 K            │  ← Bitrate (if available)
│  h264/aac          │  ← Video/Audio codecs
└─────────────────────┘
```

If bitrate is NULL, it just shows:
```
┌─────────────────────┐
│  h264/aac          │  ← Only codecs
└─────────────────────┘
```

---

## How to Set Bitrate Manually

```bash
# For stream ID 3 (AlKuwait News) - set to 8000 kbps (8 Mbps)
mysql -h localhost -u root -p'Admin@123' xtream_iptvpro -e "UPDATE streams_sys SET bitrate = 8000 WHERE stream_id = 3;"

# For stream ID 26 (test) - already has 4269 kbps
# No action needed

# For all streams in category 1 - set default bitrate
mysql -h localhost -u root -p'Admin@123' xtream_iptvpro -e "UPDATE streams_sys sys
JOIN streams s ON sys.stream_id = s.id
SET sys.bitrate = 6000
WHERE s.category_id = 1 AND sys.bitrate IS NULL;"
```

---

## Monitoring Bitrate Updates

```bash
# Watch for bitrate updates in real-time
watch -n 3 "mysql -h localhost -u root -p'Admin@123' xtream_iptvpro -N -e 'SELECT stream_id, bitrate FROM streams_sys WHERE stream_status = 0 ORDER BY stream_id;'"
```

---

## Summary

- ✅ **Codec sync daemon is working perfectly** (every 3 seconds)
- ✅ **Codec information displays correctly** in web interface
- ⚠️ **Bitrate may be delayed or NULL** for codec copy streams (technical limitation)
- 💡 **Solution**: Manually set bitrate for important streams, or wait for source probing

**The system is functioning as designed - the bitrate limitation is inherent to how codec copy works in ffmpeg.**
