<?php
/**
 * MidnightStreamer API Adapter
 * Fetches data from MidnightStreamer panel and transforms to V6APK format
 */
class MidnightStreamerAdapter {
    
    private $panel_url;
    private $username;
    private $password;
    
    public function __construct($username, $password, $panel_url = 'https://flix-panel.xyz:2087/') {
        $this->username = $username;
        $this->password = $password;
        $this->panel_url = rtrim($panel_url, '/') . '/';
    }
    
    /**
     * Authenticate user and get account info
     */
    public function authenticate() {
        $url = $this->panel_url . "api/player/{$this->username}/{$this->password}";
        
        $response = $this->makeRequest($url);
        
        if ($response === false) {
            return [
                'status' => 7,
                'message' => 'Connection failed to IPTV server',
                'username' => '',
                'password' => '',
                'expire' => '',
                'allowed_output_formats' => [],
                'max_connections' => 0,
                'host' => '',
                'player_api' => '',
                'epg_api' => ''
            ];
        }
        
        $data = json_decode($response, true);
        
        if (!is_array($data) || !isset($data['user_info'])) {
            return [
                'status' => 7,
                'message' => 'Invalid credentials',
                'username' => '',
                'password' => ''
            ];
        }
        
        return $this->transformAuth($data);
    }
    
    /**
     * Get latest movies
     */
    public function getLatestMovies($limit = 50) {
        // Use pagination: offset/items_per_page/page - get first page of latest
        $url = $this->panel_url . "api/player/{$this->username}/{$this->password}/vod/streams/0/{$limit}/1";

        $response = $this->makeRequest($url);

        if ($response === false) {
            return [];
        }

        $data = json_decode($response, true);

        if (!is_array($data)) {
            return [];
        }

        return $this->transformMovies($data);
    }
    
    /**
     * Get movies by category
     */
    public function getMoviesByCategory($category_id, $limit = 200) {
        // Use XC API for movies by category
        $url = $this->panel_url . "player_api.php?username={$this->username}&password={$this->password}&action=get_vod_streams&category_id={$category_id}";

        $response = $this->makeRequest($url);

        if ($response === false) {
            return [];
        }

        $data = json_decode($response, true);

        if (!is_array($data)) {
            return [];
        }

        return $this->transformMovies($data);
    }

    /**
     * Get movie info by stream ID
     */
    public function getMovieInfo($stream_id) {
        // Use XC API for movie info - much more efficient
        $url = $this->panel_url . "player_api.php?username={$this->username}&password={$this->password}&action=get_vod_info&vod_id={$stream_id}";

        $response = $this->makeRequest($url);

        if ($response === false) {
            return [];
        }

        $data = json_decode($response, true);

        if (!is_array($data) || !isset($data['info'])) {
            return [];
        }

        $info = $data['info'];
        $movie_data = $data['movie_data'] ?? [];

        // Extract trailer YouTube ID
        $trailer = '';
        if (isset($info['youtube_trailer']) && !empty($info['youtube_trailer'])) {
            if (preg_match('/watch\?v=([^&]+)/', $info['youtube_trailer'], $matches)) {
                $trailer = $matches[1];
            } elseif (preg_match('/youtu\.be\/([^?]+)/', $info['youtube_trailer'], $matches)) {
                $trailer = $matches[1];
            } else {
                $trailer = $info['youtube_trailer'];
            }
        }

        // Build stream URL
        $container = $movie_data['container_extension'] ?? 'mp4';
        $stream_url = $this->buildStreamUrl($stream_id, 'movie', $container);

        // Format duration
        $duration_formatted = $info['duration'] ?? '';
        if (isset($info['duration_secs']) && $info['duration_secs'] > 0) {
            $total_seconds = (int)$info['duration_secs'];
            $hours = floor($total_seconds / 3600);
            $minutes = floor(($total_seconds % 3600) / 60);
            $seconds = $total_seconds % 60;
            $duration_formatted = sprintf('%02d:%02d:%02d', $hours, $minutes, $seconds);
        }

        return [[
            'id' => $stream_id,
            'title' => $movie_data['name'] ?? $info['name'] ?? '',
            'trailer' => $trailer,
            'catid' => isset($movie_data['category_id']) ? intval($movie_data['category_id']) : 0,
            'icon' => $info['movie_image'] ?? $info['cover_big'] ?? '',
            'backdrop' => [$info['backdrop_path'][0] ?? $info['movie_image'] ?? ''],
            'stream' => $stream_url,
            'stream_url' => [
                '480p' => '',
                '720p' => $stream_url,
                '1080p' => '',
                '4k' => ''
            ],
            'subtitle' => '',
            'genre' => $info['genre'] ?? '',
            'MPAA' => $info['age'] ?? 'n/a',
            'release_date' => $info['releasedate'] ?? $info['release_date'] ?? '',
            'plot' => $info['plot'] ?? $info['description'] ?? '',
            'cast' => $info['cast'] ?? '',
            'cast_images' => [],
            'duration' => $duration_formatted,
            'rating' => (string)($info['rating'] ?? '0'),
            'director' => $info['director'] ?? '',
            'year' => $movie_data['year'] ?? 'N/A',
            'user_rating' => 0,
            'likes' => 0,
            'dislikes' => 0,
            'tmdb_id' => (string)($info['tmdb_id'] ?? '')
        ]];
    }

    /**
     * Get movie categories
     */
    public function getMovieCategories() {
        $url = $this->panel_url . "api/player/{$this->username}/{$this->password}/vod/categories";
        
        $response = $this->makeRequest($url);
        
        if ($response === false) {
            return [];
        }
        
        $data = json_decode($response, true);
        
        if (!is_array($data)) {
            return [];
        }
        
        return $this->transformCategories($data, 'movie');
    }
    
    /**
     * Get latest series
     */
    public function getLatestSeries($limit = 50) {
        // Use XC API for series (MidnightStreamer series_cat returns categories only)
        $url = $this->panel_url . "player_api.php?username={$this->username}&password={$this->password}&action=get_series";

        $response = $this->makeRequest($url);

        if ($response === false) {
            return [];
        }

        $data = json_decode($response, true);

        if (!is_array($data) || empty($data)) {
            return [];
        }

        // Sort by last_modified descending and limit
        usort($data, function($a, $b) {
            return ($b['last_modified'] ?? 0) - ($a['last_modified'] ?? 0);
        });
        $data = array_slice($data, 0, $limit);

        return $this->transformSeries($data);
    }

    /**
     * Get series by category
     */
    public function getSeriesByCategory($category_id, $limit = 100) {
        // Use player_api.php endpoint for series with category filter
        $url = $this->panel_url . "player_api.php?username={$this->username}&password={$this->password}&action=get_series";

        if ($category_id && $category_id != 'all' && intval($category_id) > 0) {
            $url .= "&category_id=" . intval($category_id);
        }

        $response = $this->makeRequest($url);

        if ($response === false) {
            return [];
        }

        $data = json_decode($response, true);

        if (!is_array($data) || empty($data)) {
            return [];
        }

        // Limit results to prevent app crash
        if (count($data) > $limit) {
            $data = array_slice($data, 0, $limit);
        }

        return $this->transformSeries($data);
    }

    /**
     * Get series info with seasons and episodes
     */
    public function getSeriesInfo($series_id) {
        // Get series info
        $series_url = $this->panel_url . "player_api.php?username={$this->username}&password={$this->password}&action=get_series_info&series_id={$series_id}";

        $response = $this->makeRequest($series_url);

        if ($response === false) {
            return [];
        }

        $data = json_decode($response, true);

        if (!is_array($data) || empty($data)) {
            return [];
        }

        // Extract series info
        $info = $data['info'] ?? [];
        $episodes = $data['episodes'] ?? [];

        // Get icon from cover
        $icon = $info['cover'] ?? '';

        // Get backdrop - handle array format
        $backdrop = [];
        if (isset($info['backdrop_path'])) {
            if (is_array($info['backdrop_path'])) {
                $backdrop = $info['backdrop_path'];
            } else {
                $backdrop = [$info['backdrop_path']];
            }
        }
        if (empty($backdrop)) {
            $backdrop = [$icon];
        }

        // Format info section
        $formatted_info = [
            'id' => $series_id,
            'title' => $info['name'] ?? '',
            'icon' => $icon,
            'catid' => $info['category_id'] ?? '',
            'icon_big' => $icon,
            'backdrop' => $backdrop,
            'genre' => $info['genre'] ?? '',
            'plot' => $info['plot'] ?? '',
            'cast' => $info['cast'] ?? '',
            'cast_images' => [],
            'rating' => (string)($info['rating'] ?? '0'),
            'director' => $info['director'] ?? '',
            'releaseDate' => $info['releaseDate'] ?? $info['release_date'] ?? '',
            'last_modified' => (string)($info['last_modified'] ?? time()),
            'trailer' => '',
            'likes' => 0,
            'dislikes' => 0,
            'tmdb_id' => (string)($info['tmdb_id'] ?? '')
        ];

        // Format seasons and episodes
        $seasons = [];
        foreach ($episodes as $season_num => $season_episodes) {
            $formatted_episodes = [];

            foreach ($season_episodes as $episode) {
                $episode_num = $episode['episode_num'] ?? '';
                $container_ext = $episode['container_extension'] ?? 'mkv';
                $episode_id = $episode['id'] ?? '';

                $formatted_episodes[] = [
                    'episode_num' => (string)$episode_num,
                    'episode_name' => $episode['title'] ?? '',
                    'stream_url' => $this->buildStreamUrl($episode_id, 'series', $container_ext)
                ];
            }

            if (!empty($formatted_episodes)) {
                $seasons[] = [
                    'season_num' => intval($season_num),
                    'episodes' => $formatted_episodes
                ];
            }
        }

        return [
            'info' => $formatted_info,
            'seasons' => $seasons
        ];
    }

    /**
     * Get series categories
     */
    public function getSeriesCategories() {
        $url = $this->panel_url . "player_api.php?username={$this->username}&password={$this->password}&action=get_series_categories";

        $response = $this->makeRequest($url);

        if ($response === false) {
            return [];
        }

        $data = json_decode($response, true);

        if (!is_array($data)) {
            return [];
        }

        return $this->transformCategories($data, 'series');
    }
    
    /**
     * Get live categories (packages)
     */
    public function getLiveCategories() {
        $url = $this->panel_url . "api/player/{$this->username}/{$this->password}/live/categories";
        
        $response = $this->makeRequest($url);
        
        if ($response === false) {
            return [];
        }
        
        $data = json_decode($response, true);
        
        if (!is_array($data)) {
            return [];
        }
        
        return $this->transformCategories($data, 'live');
    }
    
    // ==================== TRANSFORMATION METHODS ====================
    
    private function transformAuth($data) {
        $user_info = $data['user_info'] ?? [];
        $server_info = $data['server_info'] ?? [];
        
        $expire_timestamp = $user_info['exp_date'] ?? null;
        $expire_days = 0;
        
        if ($expire_timestamp && is_numeric($expire_timestamp)) {
            $expire_days = max(0, ceil(($expire_timestamp - time()) / 86400));
        }
        
        return [
            'status' => 100,
            'server_name' => 'V6apk',
            'apk_ver_code' => '',
            'message' => 'The Code is active',
            'osd_msg' => '',
            'osd_msg_global' => '',
            'expire' => $expire_timestamp ? date('Y-m-d', $expire_timestamp) : 'Unlimited',
            'user_agent' => $user_info['allowed_user_agent'] ?? '',
            'username' => $user_info['username'] ?? $this->username,
            'password' => $user_info['password'] ?? $this->password,
            'allowed_output_formats' => $user_info['allowed_output_formats'] ?? ['m3u8', 'ts'],
            'max_connections' => $user_info['max_connections'] ?? '1',
            'active_connections' => $user_info['active_cons'] ?? '0',
            'host' => $this->panel_url,
            'player_api' => $this->panel_url . 'player_api.php',
            'xmltv_api' => $this->panel_url . 'xmltv.php?username={user}&password={pass}',
            'epg_api' => $this->panel_url . 'player_api.php?username={user}&password={pass}&action=get_short_epg&stream_id={stream_id}',
            'code_id' => '0',
            'force_update' => 0,
            'update_url' => '',
            'apk_page' => '',
            'update_ch' => 'true',
            'timezone' => $server_info['timezone'] ?? 'Europe/Berlin',
            'server_info' => [
                'server_protocol' => $server_info['server_protocol'] ?? 'http',
                'url' => $server_info['url'] ?? '204.188.233.170',
                'port' => $server_info['port'] ?? ':8000',
                'https_port' => $server_info['https_port'] ?? ':8000',
                'timezone' => $server_info['timezone'] ?? 'Europe/Berlin',
                'timestamp_now' => $server_info['timestamp_now'] ?? time(),
                'time_now' => $server_info['time_now'] ?? date('Y-m-d H:i:s')
            ],
            'total_streams' => 0,
            'total_movies' => 0,
            'total_series' => 0,
            'token' => ''
        ];
    }
    
    private function transformMovies($data) {
        $result = [];
        $counter = 1;

        foreach ($data as $movie) {
            // Get the stream ID from various possible fields
            $stream_id = $movie['stream_id'] ?? $movie['id'] ?? $movie['num'] ?? '';

            // Parse TMDB info if available
            $tmdb = [];
            if (isset($movie['tmdb'])) {
                $tmdb_json = is_string($movie['tmdb']) ? $movie['tmdb'] : '';
                $tmdb = json_decode($tmdb_json, true);
                if (!is_array($tmdb)) {
                    $tmdb = [];
                }
            }

            // Extract info from movie_data if available (fallback)
            $info = [];
            if (isset($movie['info']) && is_array($movie['info'])) {
                $info = $movie['info'];
            }

            // Determine the best source for each field
            $poster = $tmdb['poster_image'] ?? $movie['poster'] ?? $movie['logo'] ?? $movie['stream_icon'] ?? '';
            $backdrop = $tmdb['backdrop_path'] ?? $poster;

            // Extract trailer
            $trailer = '';
            if (isset($tmdb['video_trailer'])) {
                if (preg_match('/watch\?v=([^&]+)/', $tmdb['video_trailer'], $matches)) {
                    $trailer = $matches[1];
                }
            } else {
                $trailer = $info['youtube_trailer'] ?? $info['trailer'] ?? '';
            }

            // Get added timestamp - convert from various formats
            $added = '';
            if (isset($movie['added'])) {
                $added = (string)$movie['added'];
            } elseif (isset($movie['date_added'])) {
                $added = (string)$movie['date_added'];
            } else {
                $added = (string)time();
            }

            $result[] = [
                'id' => $stream_id,
                'stream_display_name' => $movie['name'] ?? '',
                'title' => $movie['name'] ?? '',
                'trailer' => $trailer,
                'category_id' => $movie['category_id'] ?? '',
                'catid' => $movie['category_id'] ?? '',
                'stream_icon' => $poster,
                'icon' => $poster,
                'icon_big' => $poster,
                'backdrop' => $backdrop,
                'poster' => $poster,
                'view_order' => $counter,
                'views' => 0,
                'plot' => $tmdb['plot'] ?? $info['plot'] ?? '',
                'rating' => (string)($tmdb['rating'] ?? $info['rating'] ?? '0'),
                'genre' => $tmdb['genre'] ?? $info['genre'] ?? '',
                'cast' => $tmdb['cast'] ?? $info['cast'] ?? '',
                'year' => $tmdb['release_year'] ?? $tmdb['first_air_year'] ?? $info['releasedate'] ?? $info['year'] ?? $movie['year'] ?? '',
                'added' => $added,
                'stream_url' => $this->buildStreamUrl($stream_id, 'movie', $movie['container_format'] ?? $movie['container_extension'] ?? 'mp4')
            ];
            $counter++;
        }

        return $result;
    }
    
    private function transformSeries($data) {
        $result = [];
        $counter = 1;

        foreach ($data as $series) {
            // Get icon from cover
            $icon = $series['cover'] ?? '';

            // Try to get backdrop from info or use cover as fallback
            $backdrop = '';
            if (isset($series['info']) && is_array($series['info'])) {
                $backdrop = $series['info']['backdrop_path'] ?? '';
            }
            if (empty($backdrop) && isset($series['backdrop_path'])) {
                $backdrop = is_array($series['backdrop_path']) ? ($series['backdrop_path'][0] ?? '') : $series['backdrop_path'];
            }
            if (empty($backdrop)) {
                $backdrop = $icon;
            }

            $result[] = [
                'id' => $series['series_id'] ?? $series['id'] ?? '',
                'title' => $series['name'] ?? '',
                'icon' => $icon,
                'stream_icon' => $icon,
                'poster' => $icon,
                'catid' => $series['category_id'] ?? '',
                'category_id' => $series['category_id'] ?? '',
                'icon_big' => $icon,
                'backdrop' => $backdrop,
                'cover' => $icon,
                'genre' => $series['genre'] ?? '',
                'plot' => $series['plot'] ?? '',
                'cast' => $series['cast'] ?? '',
                'rating' => (string)($series['rating'] ?? '0'),
                'director' => $series['director'] ?? '',
                'releaseDate' => $series['releaseDate'] ?? $series['release_date'] ?? '',
                'year' => $series['year'] ?? '',
                'last_modified' => (string)($series['last_modified'] ?? time()),
                'youtube_trailer' => $series['youtube_trailer'] ?? '',
                'view_order' => $counter,
                'views' => 0
            ];
            $counter++;
        }

        return $result;
    }
    
    private function transformCategories($data, $type = 'movie') {
        $result = [];

        foreach ($data as $category) {
            $result[] = [
                'id' => $category['id'] ?? $category['category_id'] ?? '0',
                'category_name' => $category['name'] ?? $category['category_name'] ?? 'Unknown',
                'category_type' => $type,
                'category_icon' => $category['category_icon'] ?? 'http://pro.netmos.ovh:6051/img/tsawer/default.png',
                'cat_order' => isset($category['cat_order']) ? (string)$category['cat_order'] : '1',
                'isLocked' => $category['parental_lock'] ?? false,
                'stream_count' => $category['stream_count'] ?? 0,
                'parent_id' => isset($category['parent_id']) ? (string)$category['parent_id'] : '0'
            ];
        }

        return $result;
    }
    
    private function buildStreamUrl($stream_id, $type = 'movie', $ext = 'mp4') {
        return $this->panel_url . "{$type}/{$this->username}/{$this->password}/{$stream_id}.{$ext}";
    }
    
    private function makeRequest($url) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($http_code !== 200) {
            return false;
        }
        
        return $response;
    }
}
