<?php
/**
 * Universal Live Stream Proxy
 * Works with XCIPTV and external panel authentication
 *
 * URL format: /live/username/password/stream_id.ts
 */

// Get URL parameters
$path_info = isset($_SERVER['PATH_INFO']) ? $_SERVER['PATH_INFO'] : '';
$request_uri = $_SERVER['REQUEST_URI'];

// Parse path: /live/username/password/stream_id.ts or /live.php/username/password/stream_id.ts
if (preg_match('#/live(?:\.php)?/([^/]+)/([^/]+)/([^/]+)#', $request_uri, $matches)) {
    $username = $matches[1];
    $password = $matches[2];
    $stream_file = $matches[3];
    $stream_id = (int)preg_replace('/\.[^.]+$/', '', $stream_file);
} else {
    header('HTTP/1.1 400 Bad Request');
    die('Invalid request format');
}

// External panel configuration
$external_panel_url = 'https://flix-panel.xyz:2087';

/**
 * Get stream URL from external panel
 * This URL will already have the token embedded
 */
function getStreamUrl($external_panel_url, $username, $password, $stream_id) {
    // Build external stream URL - this goes through flix-panel which adds the token
    $external_url = $external_panel_url . '/live/' . $username . '/' . $password . '/' . $stream_id . '.ts';

    // No need to check, just return it
    // The external panel (flix-panel.xyz) will handle authentication and redirect to the real source with token
    return $external_url;
}

/**
 * Proxy stream to client - OPTIMIZED FOR SPEED
 */
function proxyStream($url) {
    // CRITICAL: Stop script immediately when client disconnects (zapping)
    ignore_user_abort(false);

    // Disable output buffering completely for faster streaming
    while (ob_get_level() > 0) {
        ob_end_clean();
    }

    // Increase PHP limits for streaming
    @ini_set('memory_limit', '256M');
    @ini_set('max_execution_time', '0');
    @set_time_limit(0);

    $headers = [
        'User-Agent: ' . ($_SERVER['HTTP_USER_AGENT'] ?? 'XCIPTV/7.0'),
        'Accept-Encoding: identity',
        'Connection: Keep-Alive',
        'Icy-MetaData: 1'
    ];

    if (isset($_SERVER['HTTP_RANGE'])) {
        $headers[] = 'Range: ' . $_SERVER['HTTP_RANGE'];
    }

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => false,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 5,
        CURLOPT_TIMEOUT => 0,
        CURLOPT_CONNECTTIMEOUT => 5,  // Reduced from 10 to 5 seconds
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,  // Force HTTP/1.1 for better compatibility
        CURLOPT_HEADERFUNCTION => function($ch, $header) {
            $len = strlen($header);
            if (preg_match('/^HTTP\//', $header)) {
                return $len;
            }
            // Forward important headers
            $forward_headers = ['content-type', 'content-length', 'content-range', 'cache-control', 'accept-ranges'];
            $header_lower = strtolower(trim($header));
            foreach ($forward_headers as $fwd) {
                if (strpos($header_lower, $fwd) === 0) {
                    header($header, false);
                    break;
                }
            }
            return $len;
        },
        CURLOPT_WRITEFUNCTION => function($ch, $data) {
            // Check if client disconnected (zapping to another channel)
            if (connection_aborted() || connection_status() != CONNECTION_NORMAL) {
                return 0; // Stop streaming immediately
            }

            echo $data;
            flush(); // Immediate flush - no buffering

            // Double-check after output
            if (connection_aborted()) {
                return 0; // Stop streaming
            }

            return strlen($data);
        },
        CURLOPT_BUFFERSIZE => 65536,  // Increased from 16KB to 64KB
        CURLOPT_TCP_NODELAY => true,   // Disable Nagle's algorithm for lower latency
        CURLOPT_FORBID_REUSE => false  // Allow connection reuse
    ]);

    curl_exec($ch);
    $error = curl_error($ch);
    curl_close($ch);

    if ($error) {
        error_log("Stream proxy error: $error");
    }
}

// Get stream URL
$stream_url = getStreamUrl($external_panel_url, $username, $password, $stream_id);

if (!$stream_url) {
    header('HTTP/1.1 404 Not Found');
    die('Stream not found or unavailable');
}

// REDIRECT directly to external panel (NO PROXY - FASTER!)
header('HTTP/1.1 302 Found');
header('Location: ' . $stream_url);
header('Cache-Control: no-cache, must-revalidate');
exit;
