<?php
/**
 * External Panel Configuration
 * Manage settings for external IPTV panel integration
 */

class Panel_Config_AppAdmin
{
    public $appname = null;
    public $base = null;

    public function __construct($appname, $base)
    {
        global $intro;
        $this->appname = $appname;
        $this->base = $base;

        $intro->lang['panel_config_title'] = 'External Panel Configuration';
        $intro->lang['panel_config_desc'] = 'Configure external IPTV panel settings';
    }

    public function nav()
    {
        global $intro;
        echo "<div class=\"app_nav\">
            <a class=\"btn btn-info\" href=\"" . $this->base . "/index\"><i class=\"icon-cog\"></i> " . $intro->lang['panel_config_title'] . "</a>
            <a class=\"btn btn-warning\" href=\"" . $this->base . "/index?action=clear_cache\"><i class=\"icon-trash\"></i> Clear Cache</a>
        </div>";
    }

    public function index()
    {
        global $intro;

        $this->nav();

        // Handle form submission
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_settings'])) {
            $this->saveSettings();
            echo "<div class='alert alert-success'><i class='icon-ok'></i> Settings saved successfully!</div>";
        }

        // Clear cache action
        if (isset($_GET['action']) && $_GET['action'] === 'clear_cache') {
            $this->clearCache();
            echo "<div class='alert alert-success'><i class='icon-ok'></i> Cache cleared successfully!</div>";
        }

        // Get current settings
        $settings = $this->getSettings();

        echo "<div class='panel panel-default'>
            <div class='panel-heading'>
                <h3 class='panel-title'><i class='icon-cog'></i> " . $intro->lang['panel_config_title'] . "</h3>
            </div>
            <div class='panel-body'>";

        echo "<form method='POST' class='form-horizontal'>

            <fieldset>
                <legend><i class='icon-globe'></i> External Panel Connection</legend>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>Panel URL *</label>
                    <div class='col-sm-9'>
                        <input type='text' name='panel_url' class='form-control' value='" . htmlspecialchars($settings['panel_url'] ?? '') . "' placeholder='https://flix-panel.xyz:2087/' required>
                        <span class='help-block'>Full URL with port (https://panel.example.com:2087/)</span>
                    </div>
                </div>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>API Key</label>
                    <div class='col-sm-9'>
                        <input type='text' name='api_key' class='form-control' value='" . htmlspecialchars($settings['api_key'] ?? '') . "' placeholder='Your API Key'>
                    </div>
                </div>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>Default Username</label>
                    <div class='col-sm-9'>
                        <input type='text' name='panel_username' class='form-control' value='" . htmlspecialchars($settings['panel_username'] ?? '') . "' placeholder='Optional'>
                    </div>
                </div>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>Default Password</label>
                    <div class='col-sm-9'>
                        <input type='password' name='panel_password' class='form-control' value='" . htmlspecialchars($settings['panel_password'] ?? '') . "' placeholder='Optional'>
                    </div>
                </div>
            </fieldset>

            <fieldset>
                <legend><i class='icon-lock'></i> Security Settings</legend>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>XOR Encryption Key *</label>
                    <div class='col-sm-9'>
                        <input type='text' name='xor_key' class='form-control' value='" . htmlspecialchars($settings['xor_key'] ?? 'KvuZRm365wk5Sz&Xy.Zx') . "' required>
                        <span class='help-block'>Used for API encryption (must match app)</span>
                    </div>
                </div>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>Token Expiry (days)</label>
                    <div class='col-sm-9'>
                        <input type='number' name='token_expiry_days' class='form-control' value='" . htmlspecialchars($settings['token_expiry_days'] ?? '7') . "' min='1' max='365'>
                    </div>
                </div>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>Max Connections</label>
                    <div class='col-sm-9'>
                        <input type='number' name='max_connections' class='form-control' value='" . htmlspecialchars($settings['max_connections'] ?? '1') . "' min='1' max='10'>
                    </div>
                </div>
            </fieldset>

            <fieldset>
                <legend><i class='icon-desktop'></i> App Settings</legend>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>Portal Name</label>
                    <div class='col-sm-9'>
                        <input type='text' name='portal_name' class='form-control' value='" . htmlspecialchars($settings['portal_name'] ?? 'LYNX') . "'>
                    </div>
                </div>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>User Agent</label>
                    <div class='col-sm-9'>
                        <input type='text' name='user_agent' class='form-control' value='" . htmlspecialchars($settings['user_agent'] ?? '') . "' placeholder='Optional'>
                    </div>
                </div>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>Global OSD Message</label>
                    <div class='col-sm-9'>
                        <textarea name='global_message' class='form-control' rows='2' placeholder='Message shown to all users'>" . htmlspecialchars($settings['global_message'] ?? '') . "</textarea>
                    </div>
                </div>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>APK Download Page</label>
                    <div class='col-sm-9'>
                        <input type='text' name='apk_page' class='form-control' value='" . htmlspecialchars($settings['apk_page'] ?? '') . "' placeholder='https://example.com/app.apk'>
                    </div>
                </div>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>Update URL</label>
                    <div class='col-sm-9'>
                        <input type='text' name='update_url' class='form-control' value='" . htmlspecialchars($settings['update_url'] ?? '') . "' placeholder='Optional'>
                    </div>
                </div>
            </fieldset>

            <fieldset>
                <legend><i class='icon-picture'></i> Display Settings</legend>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>Default Channel Logo</label>
                    <div class='col-sm-9'>
                        <input type='text' name='default_logo' class='form-control' value='" . htmlspecialchars($settings['default_logo'] ?? '') . "'>
                        <span class='help-block'>URL for default logo when channel has no icon</span>
                    </div>
                </div>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>Stream Format</label>
                    <div class='col-sm-9'>
                        <select name='stream_format' class='form-control'>
                            <option value='ts'" . (($settings['stream_format'] ?? 'ts') == 'ts' ? ' selected' : '') . ">TS (MPEG-TS)</option>
                            <option value='m3u8'" . (($settings['stream_format'] ?? '') == 'm3u8' ? ' selected' : '') . ">M3U8 (HLS)</option>
                        </select>
                    </div>
                </div>
            </fieldset>

            <fieldset>
                <legend><i class='icon-clock'></i> Cache Settings</legend>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>Channels Cache (seconds)</label>
                    <div class='col-sm-9'>
                        <input type='number' name='cache_time_channels' class='form-control' value='" . htmlspecialchars($settings['cache_time_channels'] ?? '300') . "' min='0' max='3600'>
                        <span class='help-block'>0 = disabled, 300 = 5 minutes</span>
                    </div>
                </div>

                <div class='form-group'>
                    <label class='col-sm-3 control-label'>Packages Cache (seconds)</label>
                    <div class='col-sm-9'>
                        <input type='number' name='cache_time_packages' class='form-control' value='" . htmlspecialchars($settings['cache_time_packages'] ?? '600') . "' min='0' max='3600'>
                        <span class='help-block'>0 = disabled, 600 = 10 minutes</span>
                    </div>
                </div>
            </fieldset>

            <fieldset>
                <legend><i class='icon-toggle-on'></i> Features</legend>

                <div class='form-group'>
                    <div class='col-sm-offset-3 col-sm-9'>
                        <label class='checkbox-inline'>
                            <input type='checkbox' name='epg_enabled' value='1'" . (($settings['epg_enabled'] ?? '1') == '1' ? ' checked' : '') . "> Enable EPG
                        </label>
                        <label class='checkbox-inline'>
                            <input type='checkbox' name='vod_enabled' value='1'" . (($settings['vod_enabled'] ?? '1') == '1' ? ' checked' : '') . "> Enable VOD
                        </label>
                        <label class='checkbox-inline'>
                            <input type='checkbox' name='series_enabled' value='1'" . (($settings['series_enabled'] ?? '1') == '1' ? ' checked' : '') . "> Enable Series
                        </label>
                        <label class='checkbox-inline'>
                            <input type='checkbox' name='debug_mode' value='1'" . (($settings['debug_mode'] ?? '0') == '1' ? ' checked' : '') . "> Debug Mode
                        </label>
                        <label class='checkbox-inline'>
                            <input type='checkbox' name='force_update' value='1'" . (($settings['force_update'] ?? '0') == '1' ? ' checked' : '') . "> Force Update
                        </label>
                    </div>
                </div>
            </fieldset>

            <div class='form-group'>
                <div class='col-sm-offset-3 col-sm-9'>
                    <button type='submit' name='save_settings' class='btn btn-success btn-lg'>
                        <i class='icon-floppy'></i> Save Settings
                    </button>
                    <a href='" . $this->base . "/index?action=clear_cache' class='btn btn-warning btn-lg' onclick=\"return confirm('Clear all cache?');\">
                        <i class='icon-trash'></i> Clear Cache
                    </a>
                </div>
            </div>
        </form>";

        echo "</div></div>";

        // Show current config file
        $this->showConfigStatus($settings);
    }

    private function getSettings()
    {
        global $intro;
        $settings = [];

        $result = $intro->db->query("SELECT setting_key, setting_value FROM panel_settings");
        if ($result) {
            while ($row = $intro->db->fetch_assoc($result)) {
                $settings[$row['setting_key']] = $row['setting_value'];
            }
        }

        return $settings;
    }

    private function saveSettings()
    {
        global $intro;

        $fields = [
            'panel_url', 'panel_username', 'panel_password', 'api_key', 'xor_key',
            'default_logo', 'stream_format', 'portal_name', 'user_agent',
            'cache_time_channels', 'cache_time_packages',
            'epg_enabled', 'vod_enabled', 'series_enabled', 'debug_mode', 'force_update',
            'token_expiry_days', 'max_connections', 'global_message', 'apk_page', 'update_url'
        ];

        $checkboxes = ['epg_enabled', 'vod_enabled', 'series_enabled', 'debug_mode', 'force_update'];

        foreach ($fields as $field) {
            if (in_array($field, $checkboxes)) {
                $value = isset($_POST[$field]) ? '1' : '0';
            } else {
                $value = isset($_POST[$field]) ? trim($_POST[$field]) : '';
            }

            $value = $intro->db->escape($value);

            // Insert or update
            $intro->db->query("INSERT INTO panel_settings (setting_key, setting_value)
                              VALUES ('$field', '$value')
                              ON DUPLICATE KEY UPDATE setting_value = '$value'");
        }

        // Generate config file for API
        $this->generateConfigFile();

        // Clear cache after saving
        $this->clearCache();
    }

    private function generateConfigFile()
    {
        $settings = $this->getSettings();

        $config = "<?php\n";
        $config .= "// Auto-generated config - DO NOT EDIT\n";
        $config .= "// Generated: " . date('Y-m-d H:i:s') . "\n\n";
        $config .= "\$_CFG = array();\n\n";

        // Core settings
        $config .= "// Core Settings\n";
        $config .= "\$_CFG['panel_url'] = '" . addslashes($settings['panel_url'] ?? 'https://flix-panel.xyz:2087/') . "';\n";
        $config .= "\$_CFG['api_key'] = '" . addslashes($settings['api_key'] ?? '') . "';\n";
        $config .= "\$_CFG['portalName'] = '" . addslashes($settings['portal_name'] ?? 'LYNX') . "';\n";

        // Extract URL and port from panel_url
        $parsed = parse_url($settings['panel_url'] ?? 'https://flix-panel.xyz:2087/');
        $config .= "\$_CFG['url'] = '" . addslashes($parsed['host'] ?? 'flix-panel.xyz') . "';\n";
        $config .= "\$_CFG['port'] = '" . addslashes($parsed['port'] ?? '2087') . "';\n";

        // Security
        $config .= "\n// Security\n";
        $config .= "\$_CFG['XOR'] = 'yes';\n";
        $config .= "\$_CFG['XOR_KEY'] = '" . addslashes($settings['xor_key'] ?? 'KvuZRm365wk5Sz&Xy.Zx') . "';\n";
        $config .= "\$_CFG['token_expiry_days'] = " . intval($settings['token_expiry_days'] ?? 7) . ";\n";
        $config .= "\$_CFG['max_connections'] = " . intval($settings['max_connections'] ?? 1) . ";\n";
        $config .= "\$_CFG['verify_token_ip'] = false;\n";

        // App settings
        $config .= "\n// App Settings\n";
        $config .= "\$_CFG['user_agent'] = '" . addslashes($settings['user_agent'] ?? '') . "';\n";
        $config .= "\$_CFG['global_message'] = '" . addslashes($settings['global_message'] ?? '') . "';\n";
        $config .= "\$_CFG['apk_page'] = '" . addslashes($settings['apk_page'] ?? '') . "';\n";
        $config .= "\$_CFG['update_url'] = '" . addslashes($settings['update_url'] ?? '') . "';\n";
        $config .= "\$_CFG['force_update'] = " . intval($settings['force_update'] ?? 0) . ";\n";

        // Display
        $config .= "\n// Display Settings\n";
        $config .= "\$_CFG['default_logo'] = '" . addslashes($settings['default_logo'] ?? '') . "';\n";
        $config .= "\$_CFG['stream_format'] = '" . addslashes($settings['stream_format'] ?? 'ts') . "';\n";

        // Cache
        $config .= "\n// Cache Settings\n";
        $config .= "\$_CFG['cache_time_channels'] = " . intval($settings['cache_time_channels'] ?? 300) . ";\n";
        $config .= "\$_CFG['cache_time_packages'] = " . intval($settings['cache_time_packages'] ?? 600) . ";\n";

        // Features
        $config .= "\n// Features\n";
        $config .= "\$_CFG['epg_enabled'] = " . intval($settings['epg_enabled'] ?? 1) . ";\n";
        $config .= "\$_CFG['vod'] = '" . ($settings['vod_enabled'] ?? '1' ? 'Yes' : 'No') . "';\n";
        $config .= "\$_CFG['series_enabled'] = " . intval($settings['series_enabled'] ?? 1) . ";\n";

        // Debug
        $config .= "\n// Debug\n";
        $config .= "\$_CFG['debug'] = " . ($settings['debug_mode'] ?? '0' ? 'true' : 'false') . ";\n";
        $config .= "\$_CFG['debug_post'] = " . ($settings['debug_mode'] ?? '0' ? 'true' : 'false') . ";\n";
        $config .= "\$_CFG['allow_direct_json'] = true;\n";
        $config .= "\$_CFG['update_password'] = 'Yes';\n";
        $config .= "\$_CFG['restreamer'] = 0;\n";
        $config .= "\$_CFG['allow_m3u'] = 1;\n";
        $config .= "\$_CFG['subscription_plan'] = 1;\n";

        $config_path = dirname(__DIR__) . '/V6APK/api_cfg_v6.php';
        file_put_contents($config_path, $config);
        chmod($config_path, 0644);
    }

    private function clearCache()
    {
        $cache_dir = dirname(__DIR__) . '/V6APK/cache/';
        if (is_dir($cache_dir)) {
            $files = glob($cache_dir . '*.json');
            foreach ($files as $file) {
                @unlink($file);
            }
        }
    }

    private function showConfigStatus($settings)
    {
        echo "<div class='panel panel-info' style='margin-top:20px;'>
            <div class='panel-heading'>
                <h3 class='panel-title'><i class='icon-info'></i> Connection Status</h3>
            </div>
            <div class='panel-body'>
                <table class='table table-striped'>
                    <tr>
                        <td width='200'><strong>Panel URL</strong></td>
                        <td>" . htmlspecialchars($settings['panel_url'] ?? 'Not set') . "</td>
                        <td width='150'>";

        // Test connection
        if (!empty($settings['panel_url'])) {
            $test_url = rtrim($settings['panel_url'], '/') . '/player_api.php';
            $ctx = stream_context_create(['http' => ['timeout' => 5], 'ssl' => ['verify_peer' => false, 'verify_peer_name' => false]]);
            $test = @file_get_contents($test_url, false, $ctx);
            if ($test !== false) {
                echo "<span class='label label-success'><i class='icon-ok'></i> Connected</span>";
            } else {
                echo "<span class='label label-danger'><i class='icon-cancel'></i> Failed</span>";
            }
        }

        echo "</td>
                    </tr>
                    <tr>
                        <td><strong>XOR Key</strong></td>
                        <td>" . htmlspecialchars(substr($settings['xor_key'] ?? '', 0, 10) . '...') . "</td>
                        <td><span class='label label-info'>Configured</span></td>
                    </tr>
                    <tr>
                        <td><strong>Cache Files</strong></td>
                        <td>" . dirname(__DIR__) . "/V6APK/cache/</td>
                        <td>";

        $cache_dir = dirname(__DIR__) . '/V6APK/cache/';
        $cache_files = is_dir($cache_dir) ? count(glob($cache_dir . '*.json')) : 0;
        echo "<span class='label label-info'>" . $cache_files . " files</span>";

        echo "</td>
                    </tr>
                    <tr>
                        <td><strong>Config File</strong></td>
                        <td>" . dirname(__DIR__) . "/V6APK/api_cfg_v6.php</td>
                        <td>";

        $config_file = dirname(__DIR__) . '/V6APK/api_cfg_v6.php';
        if (file_exists($config_file)) {
            $mtime = date('Y-m-d H:i', filemtime($config_file));
            echo "<span class='label label-success' title='Last modified: $mtime'><i class='icon-ok'></i> OK</span>";
        } else {
            echo "<span class='label label-warning'><i class='icon-attention'></i> Missing</span>";
        }

        echo "</td>
                    </tr>
                </table>
            </div>
        </div>";
    }
}
