<?php
/**
 * Stream Checker and Auto-Fix Script
 * يفحص جميع القنوات ويحدث حالتها ويحذف الروابط المعطلة
 *
 * Usage: php check_and_fix_streams.php
 * Or: wget -O- "http://yourserver.com/iptv/check_and_fix_streams.php?key=your_secret_key"
 */

set_time_limit(0);
ini_set('memory_limit', '512M');

// Security key (change this!)
$SECRET_KEY = 'change_me_' . md5(__DIR__);

// If accessed via web, check security key
if (php_sapi_name() !== 'cli') {
    $provided_key = $_GET['key'] ?? '';
    if ($provided_key !== $SECRET_KEY) {
        die('Access denied. Invalid key.');
    }
    header('Content-Type: text/plain; charset=utf-8');
}

echo "=== Stream Checker Started ===\n";
echo "Time: " . date('Y-m-d H:i:s') . "\n\n";

// Load config
require_once __DIR__ . '/includes/Config.php';

try {
    $pdo = new PDO(
        "mysql:host={$config['db']['hostname']};dbname={$config['db']['database']};charset=utf8",
        $config['db']['username'],
        $config['db']['password']
    );
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Function to check if URL is accessible
    function checkUrl($url, $timeout = 5) {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_NOBODY => true,
            CURLOPT_TIMEOUT => $timeout,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
        ]);

        curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        return [
            'working' => ($http_code >= 200 && $http_code < 400),
            'http_code' => $http_code,
            'error' => $error
        ];
    }

    // Get all live streams
    $stmt = $pdo->query("SELECT id, stream_display_name, stream_source, status FROM streams WHERE type = 1 ORDER BY id ASC");
    $streams = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $total = count($streams);
    $checked = 0;
    $fixed = 0;
    $disabled = 0;
    $already_ok = 0;

    echo "Total streams to check: {$total}\n\n";

    foreach ($streams as $stream) {
        $checked++;
        $stream_id = $stream['id'];
        $name = $stream['stream_display_name'];
        $sources = json_decode($stream['stream_source'], true);

        echo "[{$checked}/{$total}] Stream #{$stream_id}: {$name}\n";

        if (!$sources || !is_array($sources) || count($sources) === 0) {
            echo "  ⚠ No sources found\n";
            // Disable stream with no sources
            $pdo->prepare("UPDATE streams SET status = 0 WHERE id = ?")->execute([$stream_id]);
            $disabled++;
            continue;
        }

        $working_sources = [];
        $broken_sources = [];

        // Check each source
        foreach ($sources as $index => $source_url) {
            if (empty($source_url)) continue;

            $source_url = stripslashes($source_url);
            echo "  Testing source " . ($index + 1) . "... ";

            $result = checkUrl($source_url, 5);

            if ($result['working']) {
                echo "✓ OK (HTTP {$result['http_code']})\n";
                $working_sources[] = $source_url;
            } else {
                echo "✗ FAILED (HTTP {$result['http_code']})\n";
                $broken_sources[] = $source_url;
            }
        }

        // Update stream based on results
        if (count($working_sources) > 0) {
            // Has working sources
            if (count($working_sources) < count($sources)) {
                // Some sources were broken, update with only working ones
                $new_sources = json_encode($working_sources);
                $pdo->prepare("UPDATE streams SET stream_source = ?, status = 1 WHERE id = ?")
                    ->execute([$new_sources, $stream_id]);
                echo "  ✓ Fixed: Removed " . count($broken_sources) . " broken source(s)\n";
                $fixed++;
            } else {
                echo "  ✓ All sources working\n";
                $already_ok++;
            }
        } else {
            // No working sources, disable stream
            $pdo->prepare("UPDATE streams SET status = 0 WHERE id = ?")
                ->execute([$stream_id]);
            echo "  ✗ Disabled: No working sources\n";
            $disabled++;
        }

        echo "\n";

        // Small delay to avoid overwhelming servers
        usleep(200000); // 0.2 seconds
    }

    echo "\n=== Summary ===\n";
    echo "Total checked: {$checked}\n";
    echo "Already OK: {$already_ok}\n";
    echo "Fixed: {$fixed}\n";
    echo "Disabled: {$disabled}\n";
    echo "\nCompleted at: " . date('Y-m-d H:i:s') . "\n";

} catch (Exception $e) {
    echo "ERROR: " . $e->getMessage() . "\n";
    exit(1);
}
