<?php
/**
 * Diagnostic Script for App Token Issues
 * This helps identify WHERE and HOW the token gets corrupted
 */

header('Content-Type: text/plain; charset=utf-8');

echo "╔════════════════════════════════════════════════════════════════════╗\n";
echo "║          APP TOKEN DIAGNOSTIC TOOL                                 ║\n";
echo "╚════════════════════════════════════════════════════════════════════╝\n\n";

// Known correct token from activation
$correct_token = "f3ed71d56579442cbdc114e8f77e57b4rueo5uPmlFslxvKta6fG9lMBRXMKLU0VDwvqXuJXnhJs/PpqTO7+kK0mtjIDjNwj1ZVOAER8JPeqKd5QljjBHQ==";

// Token that app sends (wrong)
$wrong_token = "4f8ecf06c3bc9c8e0ca6aca2129e427bulAkIuLgOU1h0suMbUZrFchMETQsZxUtSb6V32USiF6BI05fY8FAt2v0vyxg1S8br446KyYRDgjfe2Y/GDePeQ==";

echo "📊 TOKEN ANALYSIS\n";
echo str_repeat("─", 72) . "\n\n";

echo "Correct Token (from server):\n";
echo "  Full: $correct_token\n";
echo "  MD5 prefix: " . substr($correct_token, 0, 32) . "\n";
echo "  Length: " . strlen($correct_token) . " chars\n";
echo "  Has forward slash: " . (strpos($correct_token, '/') !== false ? 'YES' : 'NO') . "\n";
echo "  First 50 chars: " . substr($correct_token, 0, 50) . "\n\n";

echo "Wrong Token (from app):\n";
echo "  Full: $wrong_token\n";
echo "  MD5 prefix: " . substr($wrong_token, 0, 32) . "\n";
echo "  Length: " . strlen($wrong_token) . " chars\n";
echo "  Has escaped slash: " . (strpos($wrong_token, '\/') !== false ? 'YES' : 'NO') . "\n";
echo "  First 50 chars: " . substr($wrong_token, 0, 50) . "\n\n";

echo "🔍 POSSIBLE CAUSES:\n";
echo str_repeat("─", 72) . "\n\n";

// Check 1: Different MD5 prefix = completely different token
if (substr($correct_token, 0, 32) !== substr($wrong_token, 0, 32)) {
    echo "❌ ISSUE #1: Completely different tokens!\n";
    echo "   The MD5 prefixes don't match at all.\n";
    echo "   This means the app is NOT using the token from activation.\n\n";

    echo "   🔎 What to check in the app:\n";
    echo "   1. Does the app save the token IMMEDIATELY after receiving it?\n";
    echo "   2. Is there a cache/storage mechanism overwriting the token?\n";
    echo "   3. Is the app generating its own token instead of using server's?\n";
    echo "   4. Is there another activation happening in the background?\n\n";
}

// Check 2: Escaped slash issue
if (strpos($wrong_token, '\/') !== false) {
    echo "⚠️  ISSUE #2: Escaped forward slash detected!\n";
    echo "   The token has '\\/' instead of '/'\n";
    echo "   This usually happens with JSON escaping.\n\n";

    echo "   🔎 What to check in the app:\n";
    echo "   1. Are you using Gson with HTML escaping enabled?\n";
    echo "   2. Are you converting the JSON twice?\n";
    echo "   3. Check: new Gson().toJson() might escape slashes\n\n";
}

// Check 3: Test both tokens with database
require_once('/var/www/html/iptv/includes/Config.php');
$link = mysqli_connect(
    $config['db']['hostname'],
    $config['db']['username'],
    $config['db']['password'],
    $config['db']['database']
);

if ($link) {
    echo "🗄️  DATABASE CHECK:\n";
    echo str_repeat("─", 72) . "\n\n";

    // Check correct token
    $safe_correct = mysqli_real_escape_string($link, $correct_token);
    $result1 = mysqli_query($link, "SELECT user_id, FROM_UNIXTIME(created) as created FROM user_tokens WHERE token = '$safe_correct'");

    if ($result1 && mysqli_num_rows($result1) > 0) {
        $row = mysqli_fetch_assoc($result1);
        echo "✅ Correct token EXISTS in database\n";
        echo "   User ID: " . $row['user_id'] . "\n";
        echo "   Created: " . $row['created'] . "\n\n";
    } else {
        echo "❌ Correct token NOT FOUND in database\n\n";
    }

    // Check wrong token
    $safe_wrong = mysqli_real_escape_string($link, $wrong_token);
    $result2 = mysqli_query($link, "SELECT user_id FROM user_tokens WHERE token = '$safe_wrong'");

    if ($result2 && mysqli_num_rows($result2) > 0) {
        echo "⚠️  Wrong token EXISTS in database (unexpected!)\n\n";
    } else {
        echo "❌ Wrong token NOT FOUND in database (as expected)\n\n";
    }

    // Check for user 426687097695
    $result3 = mysqli_query($link, "SELECT token FROM user_tokens WHERE user_id = 41 ORDER BY created DESC LIMIT 1");
    if ($result3 && mysqli_num_rows($result3) > 0) {
        $latest = mysqli_fetch_assoc($result3);
        echo "📌 Latest token for user 426687097695:\n";
        echo "   " . substr($latest['token'], 0, 50) . "...\n\n";

        if ($latest['token'] === $correct_token) {
            echo "✅ Latest token matches the correct one!\n\n";
        } else {
            echo "⚠️  Latest token is different from correct token\n\n";
        }
    }

    mysqli_close($link);
}

echo "🎯 RECOMMENDATIONS:\n";
echo str_repeat("─", 72) . "\n\n";

echo "1️⃣  Add logging in the app to print the token RIGHT AFTER receiving it:\n";
echo "   Log.d(\"Token\", \"Received: \" + response.getString(\"token\"));\n\n";

echo "2️⃣  Add logging RIGHT BEFORE sending request:\n";
echo "   Log.d(\"Token\", \"Sending: \" + savedToken);\n\n";

echo "3️⃣  Check SharedPreferences/Storage:\n";
echo "   - Is the token being saved correctly?\n";
echo "   - Is something else overwriting it?\n\n";

echo "4️⃣  Check JSON parsing:\n";
echo "   // WRONG - may escape slashes:\n";
echo "   String json = new Gson().toJson(data);\n\n";
echo "   // CORRECT - parse response directly:\n";
echo "   String token = response.getString(\"token\"); // No modifications!\n\n";

echo "5️⃣  Make sure you're not:\n";
echo "   - Generating a token client-side\n";
echo "   - Using cached/old token from previous session\n";
echo "   - Making multiple activation requests\n\n";

echo "╔════════════════════════════════════════════════════════════════════╗\n";
echo "║  NEXT STEP: Add logging to your app and compare the tokens!       ║\n";
echo "╚════════════════════════════════════════════════════════════════════╝\n";
