<?php
require_once(dirname(__FILE__) . '/RemotePanelConfig.php');

class AdminStreams
{
    public $appname = null;
    public $base = null;
    public $img_path = null;
    public $error = null;
    public $app_name = '';
    public $app_color = 'info';
    public $msg = null;
    public $type = 0;
    public $array_stream_status = [
        'all' => 'All',
        'online' => 'Online',
        'offline' => 'Offline',
        'stopped' => 'Stopped'
    ];
    public $array_stream_status_qry = [
        'all' => '',
        'online' => ' AND sys.stream_status=0 ',
        'offline' => ' AND sys.stream_id IS NULL ',
        'stopped' => ' AND sys.stream_status=1 '
    ];

    // Remote Database Operations - Using RemotePanelConfig
    public function remoteDbExecute($sql)
    {
        // Get settings from database
        $remote_host = RemotePanelConfig::get('host');
        $remote_user = RemotePanelConfig::get('user');
        $remote_pass = RemotePanelConfig::get('pass');
        $remote_db = RemotePanelConfig::get('database');

        $tmp_file = '/tmp/remote_stream_sql_' . uniqid() . '.sql';
        file_put_contents($tmp_file, $sql);

        $cmd = "sshpass -p " . escapeshellarg($remote_pass) . " ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 " . $remote_user . "@" . $remote_host . " \"mysql " . $remote_db . " -N\" < " . escapeshellarg($tmp_file) . " 2>&1";

        $result = shell_exec($cmd);
        @unlink($tmp_file);

        error_log("Remote Stream DB Execute: $sql - Result: $result");
        return $result;
    }

    protected function addStreamToRemote($name, $type, $urls, $id = null)
    {
        $name_esc = addslashes($name);
        $urls_json = is_array($urls) ? json_encode($urls) : $urls;
        $urls_esc = addslashes($urls_json);

        // Convert type from local to remote:
        // Local type=1 (Live TV) -> Remote type=0 (Live TV)
        // Local type=2 (Movies) -> Remote type=1 (VOD)
        $remote_type = ($type == 1) ? 0 : 1;

        if($id !== null) {
            // Add with specific ID to keep IDs synchronized
            // Always set disabled=0 so stream is enabled on external panel
            $sql = "INSERT INTO streams (id, name, type, urls, disabled) VALUES ($id, '$name_esc', $remote_type, '$urls_esc', 0);";
        } else {
            // Always set disabled=0 so stream is enabled on external panel
            $sql = "INSERT INTO streams (name, type, urls, disabled) VALUES ('$name_esc', $remote_type, '$urls_esc', 0);";
        }
        $result = $this->remoteDbExecute($sql);
        return (empty($result) || strpos($result, 'ERROR') === false);
    }

    protected function updateStreamInRemote($old_name, $new_name, $type, $urls)
    {
        $old_name_esc = addslashes($old_name);
        $new_name_esc = addslashes($new_name);
        $urls_json = is_array($urls) ? json_encode($urls) : $urls;
        $urls_esc = addslashes($urls_json);

        $sql = "UPDATE streams SET name='$new_name_esc', type=$type, urls='$urls_esc' WHERE name='$old_name_esc';";
        $result = $this->remoteDbExecute($sql);
        return (empty($result) || strpos($result, 'ERROR') === false);
    }

    protected function updateStreamFieldsInRemote($stream_name, $fields)
    {
        $name_esc = addslashes($stream_name);
        $updates = [];
        foreach($fields as $key => $value) {
            if(is_numeric($value)) {
                $updates[] = "$key=" . intval($value);
            } else {
                $updates[] = "$key='" . addslashes($value) . "'";
            }
        }
        if(empty($updates)) return false;

        $sql = "UPDATE streams SET " . implode(', ', $updates) . " WHERE name='$name_esc';";
        $result = $this->remoteDbExecute($sql);
        return (empty($result) || strpos($result, 'ERROR') === false);
    }

    protected function deleteStreamFromRemote($name)
    {
        $name_esc = addslashes($name);
        $sql = "DELETE FROM streams WHERE name='$name_esc';";
        $result = $this->remoteDbExecute($sql);
        return (empty($result) || strpos($result, 'ERROR') === false);
    }

    protected function deleteStreamFromRemoteById($id)
    {
        $id = intval($id);
        $sql = "DELETE FROM streams WHERE id=$id;";
        $result = $this->remoteDbExecute($sql);
        return (empty($result) || strpos($result, 'ERROR') === false);
    }

    public function syncCategoryToRemote($stream_id, $local_category_id)
    {
        global $intro;

        $stream_id = intval($stream_id);
        $local_category_id = intval($local_category_id);

        error_log("syncCategoryToRemote: START - stream_id=$stream_id, local_category_id=$local_category_id");

        if($local_category_id == 0 || $stream_id == 0) {
            error_log("syncCategoryToRemote: Invalid IDs, returning false");
            return false;
        }

        // Get local category name and type
        $cat_sql = $intro->db->query("SELECT category_name, category_type FROM stream_categories WHERE id = $local_category_id");
        if($intro->db->returned_rows == 0) {
            error_log("syncCategoryToRemote: Category $local_category_id not found locally");
            return false;
        }

        $cat_row = $intro->db->fetch_assoc($cat_sql);
        $category_name = addslashes($cat_row['category_name']);
        $category_type = $cat_row['category_type'];

        error_log("syncCategoryToRemote: Category='$category_name', Type='$category_type'");

        // Map local category type to remote type
        // local: 'live' -> remote: 0, 'movie' -> remote: 1, 'series' -> remote: 2
        $remote_type = 0; // default to Live TV
        if($category_type == 'movie') {
            $remote_type = 1;
        } elseif($category_type == 'series') {
            $remote_type = 2;
        }

        error_log("syncCategoryToRemote: Remote type=$remote_type");

        // Find category in remote by name AND type
        $sql = "SELECT id FROM categories WHERE name = '$category_name' AND type = $remote_type;";
        $result = $this->remoteDbExecute($sql);

        error_log("syncCategoryToRemote: Search result='" . trim($result) . "'");

        if(empty(trim($result))) {
            error_log("syncCategoryToRemote: Creating new category on remote");
            // Create category in remote with correct type
            $sql = "INSERT INTO categories (name, type, parent, parental_lock, ordering) VALUES ('$category_name', $remote_type, 0, 0, 0);";
            $create_result = $this->remoteDbExecute($sql);
            error_log("syncCategoryToRemote: Create result='$create_result'");

            // Get the new ID
            $sql = "SELECT id FROM categories WHERE name = '$category_name' AND type = $remote_type;";
            $result = $this->remoteDbExecute($sql);
            error_log("syncCategoryToRemote: New category ID='" . trim($result) . "'");
        }

        $remote_category_id = intval(trim($result));

        if($remote_category_id > 0) {
            error_log("syncCategoryToRemote: Assigning stream $stream_id to remote category $remote_category_id");

            // Delete old category assignment
            $sql = "DELETE FROM categories_sids WHERE sid = $stream_id;";
            $this->remoteDbExecute($sql);

            // Add new category assignment
            $sql = "INSERT INTO categories_sids (sid, cid) VALUES ($stream_id, $remote_category_id);";
            $this->remoteDbExecute($sql);

            error_log("syncCategoryToRemote: SUCCESS");
            return true;
        }

        error_log("syncCategoryToRemote: FAILED - could not get remote category ID");
        return false;
    }

    protected function syncBouquetToRemote($stream_id, $local_bouquet_id)
    {
        global $intro;

        $stream_id = intval($stream_id);
        $local_bouquet_id = intval($local_bouquet_id);

        if($local_bouquet_id == 0 || $stream_id == 0) return false;

        // Get local bouquet name
        $bouquet_sql = $intro->db->query("SELECT bouquet_name FROM bouquets WHERE id = $local_bouquet_id");
        if($intro->db->returned_rows == 0) return false;

        $bouquet_row = $intro->db->fetch_assoc($bouquet_sql);
        $bouquet_name = addslashes($bouquet_row['bouquet_name']);

        // Find or create bouquet in remote
        $sql = "SELECT id FROM bouquets WHERE name = '$bouquet_name';";
        $result = $this->remoteDbExecute($sql);

        if(empty(trim($result))) {
            // Create bouquet in remote
            $sql = "INSERT INTO bouquets (name) VALUES ('$bouquet_name');";
            $this->remoteDbExecute($sql);

            // Get the new ID
            $sql = "SELECT id FROM bouquets WHERE name = '$bouquet_name';";
            $result = $this->remoteDbExecute($sql);
        }

        $remote_bouquet_id = intval(trim($result));

        if($remote_bouquet_id > 0) {
            // Delete ALL existing bouquet assignments for this stream first
            // This ensures the stream only has the currently selected bouquet
            $sql = "DELETE FROM bouquets_sids WHERE sid = $stream_id;";
            $this->remoteDbExecute($sql);

            // Add new bouquet assignment
            $sql = "INSERT INTO bouquets_sids (bid, sid) VALUES ($remote_bouquet_id, $stream_id);";
            $this->remoteDbExecute($sql);

            return true;
        }

        return false;
    }

    // MidnightStreamer API helper function
    protected function midnightStreamerApiCall($endpoint, $stream_ids = [])
    {
        // Load config from api_cfg_v6.php
        $_CFG = [];
        $config_file = dirname(__FILE__) . '/../V6APK/api_cfg_v6.php';
        if(file_exists($config_file)) {
            include($config_file);
        }

        $api_key = isset($_CFG['midnight_api_key']) ? $_CFG['midnight_api_key'] : 'eJIdy5sAgD';
        $api_base = isset($_CFG['midnight_api_url']) ? $_CFG['midnight_api_url'] : RemotePanelConfig::getApiUrl() . '/';

        // Build URL with stream IDs
        $url = $api_base . $endpoint . '/' . $api_key;
        if(!empty($stream_ids)) {
            if(is_array($stream_ids)) {
                $url .= '/' . implode('/', $stream_ids);
            } else {
                $url .= '/' . $stream_ids;
            }
        }

        // Use curl with X-Forwarded-For header
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'X-Forwarded-For: 127.0.0.1'
        ]);

        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        error_log("MidnightStreamer API: $url - HTTP: $http_code - Response: $response");

        if($error) {
            return json_encode(['result' => false, 'error' => $error]);
        }

        // Extract JSON from response (may contain HTML errors before JSON)
        // Look for {"result":true} or {"result":false} pattern
        if(preg_match('/\{"result":(true|false)[^}]*\}/', $response, $matches)) {
            return $matches[0];
        }

        // Fallback: try to find any JSON object
        if(preg_match('/\{[^}]+\}/', $response, $matches)) {
            return $matches[0];
        }

        return json_encode(['result' => false, 'error' => 'Invalid API response']);
    }

    // Start stream via MidnightStreamer API
    public function startStreamViaApi($stream_id)
    {
        return $this->midnightStreamerApiCall('streams/start', $stream_id);
    }

    // Stop stream via MidnightStreamer API
    public function stopStreamViaApi($stream_id)
    {
        return $this->midnightStreamerApiCall('streams/stop', $stream_id);
    }

    // Restart stream via MidnightStreamer API (stop then start)
    protected function restartStreamViaApi($stream_id)
    {
        $this->stopStreamViaApi($stream_id);
        sleep(1);
        return $this->startStreamViaApi($stream_id);
    }

    // Sync all stream data from remote streams table
    protected function syncOnDemandStatusFromRemote()
    {
        global $intro;

        // Get all stream data from remote (Live TV only - type=0) including redirect and flussonic settings
        $sql = "SELECT id, name, direct_streaming_on_demand, disabled, urls, redirect, stream_all_tracks, append_flussonic_token FROM streams WHERE type=0;";
        $result = $this->remoteDbExecute($sql);

        if(empty($result)) return;

        $lines = explode("\n", trim($result));
        foreach($lines as $line) {
            if(empty(trim($line))) continue;

            $parts = explode("\t", $line);
            if(count($parts) < 3) continue;

            $remote_id = intval($parts[0]);
            $stream_name = trim($parts[1]);
            $on_demand = intval($parts[2]);
            $disabled = isset($parts[3]) ? intval($parts[3]) : 0;
            $urls = isset($parts[4]) ? $parts[4] : '';
            $redirect = isset($parts[5]) ? intval($parts[5]) : 0;
            $stream_all_tracks = isset($parts[6]) ? intval($parts[6]) : 0;
            $append_flussonic_token = isset($parts[7]) ? intval($parts[7]) : 0;

            if(empty($stream_name)) continue;

            // Decode URLs and get first source
            $stream_source = '';
            if(!empty($urls) && $urls != 'NULL') {
                $urls_decoded = @json_decode($urls, true);
                if(is_array($urls_decoded)) {
                    $stream_source = json_encode($urls_decoded);
                }
            }

            // Find local stream by remote_id mapping or by name
            $local = $intro->db->query("SELECT id, stream_display_name FROM streams WHERE id = $remote_id OR stream_display_name='" . $intro->db->escape($stream_name) . "' LIMIT 1");
            if($intro->db->returned_rows > 0) {
                $local_row = $intro->db->fetch_assoc($local);
                $local_stream_id = $local_row['id'];

                // Update on_demand in streams_sys for all servers
                $intro->db->query("UPDATE streams_sys SET on_demand = $on_demand WHERE stream_id = $local_stream_id");

                // Update stream data (name, source)
                // NOTE: Do NOT sync status, redirect_stream, stream_all, or append_flussonic_token from remote
                // These fields should only be managed locally and synced TO remote, not FROM remote
                $update_data = [];
                if($local_row['stream_display_name'] != $stream_name) {
                    $update_data['stream_display_name'] = $stream_name;
                }
                if(!empty($stream_source)) {
                    $update_data['stream_source'] = $stream_source;
                }
                // DO NOT update redirect_stream from remote - it should be controlled locally
                // $update_data['redirect_stream'] = $redirect; // REMOVED - local only field
                // DO NOT update status from remote - it should be controlled locally
                // $update_data['status'] = ($disabled == 0 ? 1 : 0); // REMOVED - local only field
                // DO NOT update stream_all from remote - it should be controlled locally
                // $update_data['stream_all'] = $stream_all_tracks; // REMOVED - local only field
                // DO NOT update append_flussonic_token from remote - it should be controlled locally
                // $update_data['append_flussonic_token'] = $append_flussonic_token; // REMOVED - local only field

                if(!empty($update_data)) {
                    $intro->db->update('streams', $update_data, 'id=' . $local_stream_id);
                }
            }
        }
    }

    // Sync servers from external panel to local streaming_servers table
    protected function syncServersFromRemote()
    {
        global $intro;

        // Get all active servers from external panel
        $sql = "SELECT id, name, server_ip, disabled, http_port, https_port, rtmp_port, ssh_port FROM servers ORDER BY id ASC;";
        $result = $this->remoteDbExecute($sql);

        if(empty($result)) return;

        $lines = explode("\n", trim($result));
        $remote_server_ids = []; // Track remote server IDs

        foreach($lines as $line) {
            if(empty(trim($line))) continue;

            $parts = explode("\t", $line);
            if(count($parts) < 4) continue;

            $server_id = intval($parts[0]);
            $server_name = trim($parts[1]);
            $server_ip = trim($parts[2]);
            $disabled = intval($parts[3]);
            $http_port = isset($parts[4]) ? intval($parts[4]) : 8000;
            $https_port = isset($parts[5]) ? intval($parts[5]) : 8001;
            $rtmp_port = isset($parts[6]) ? intval($parts[6]) : 8002;
            $ssh_port = isset($parts[7]) ? intval($parts[7]) : 22;

            // Add to remote server IDs list
            $remote_server_ids[] = $server_id;

            // Convert disabled status (remote: 0=enabled, 1=disabled → local: 1=enabled, -1=disabled)
            $local_status = ($disabled == 0) ? 1 : -1;

            // Check if server exists in local database
            $check_sql = $intro->db->query("SELECT id FROM streaming_servers WHERE id = $server_id");

            if($intro->db->returned_rows > 0) {
                // Update existing server
                $intro->db->query("UPDATE streaming_servers SET
                    server_name = '" . $intro->db->escape($server_name) . "',
                    server_ip = '" . $intro->db->escape($server_ip) . "',
                    http_broadcast_port = $http_port,
                    https_broadcast_port = $https_port,
                    rtmp_port = $rtmp_port,
                    ssh_port = $ssh_port,
                    status = $local_status
                    WHERE id = $server_id");
            } else {
                // Insert new server
                $intro->db->query("INSERT INTO streaming_servers
                    (id, server_name, server_ip, http_broadcast_port, https_broadcast_port, rtmp_port, ssh_port, status, can_delete, view_order)
                    VALUES
                    ($server_id, '" . $intro->db->escape($server_name) . "', '" . $intro->db->escape($server_ip) . "',
                    $http_port, $https_port, $rtmp_port, $ssh_port, $local_status, 1, $server_id)");
            }
        }

        // Delete servers that don't exist in external panel anymore
        if(!empty($remote_server_ids)) {
            $ids_list = implode(',', $remote_server_ids);
            $intro->db->query("DELETE FROM streaming_servers WHERE id NOT IN ($ids_list)");
        }
    }

    // Change stream mode (Live/OnDemand) on remote server
    public function setStreamModeOnRemote($stream_name, $on_demand = 0, $servers = null, $probesize = 512000)
    {
        $stream_name_esc = addslashes($stream_name);
        // When setting to Live mode (on_demand=0), clear on_demand_servers and enable stream
        if($on_demand == 0) {
            $sql = "UPDATE streams SET
                    direct_streaming_on_demand = 0,
                    on_demand_servers = NULL,
                    on_demand_probesize = NULL,
                    disabled = 0
                    WHERE name = '$stream_name_esc';";
        } else {
            // When setting to OnDemand mode:
            // 1. Enable OnDemand flag
            // 2. Set probesize for better stream analysis
            // 3. Set selected servers (use provided servers or default to [1])
            // 4. Enable the stream (disabled = 0) so it's available via API

            // Build on_demand_servers JSON array
            if($servers && is_array($servers) && count($servers) > 0) {
                $servers_array = array_map('strval', $servers); // Convert to strings
                $on_demand_servers_json = json_encode($servers_array);
            } else {
                $on_demand_servers_json = '["1"]'; // Default to server 1
            }

            // Ensure probesize is valid (use provided or default to 512000)
            $probesize = intval($probesize);
            if($probesize == 0) {
                $probesize = 512000;
            }

            $on_demand_servers_esc = addslashes($on_demand_servers_json);

            $sql = "UPDATE streams SET
                    direct_streaming_on_demand = 1,
                    on_demand_servers = '$on_demand_servers_esc',
                    on_demand_probesize = $probesize,
                    disabled = 0
                    WHERE name = '$stream_name_esc';";
        }
        error_log("setStreamModeOnRemote SQL: $sql");
        return $this->remoteDbExecute($sql);
    }

    protected function ensureRemoteStreamServer($stream_id, $servers, $on_demand)
    {
        // Map local servers to remote (typically both use server_id=1)
        $remote_server_ids = is_array($servers) ? $servers : [1];

        // Use first server as primary
        $primary_server_id = intval($remote_server_ids[0]);

        // Build servers JSON in MidnightStreamer format: {"server_id":"parent_id"}
        // For main server, parent_id is always "0"
        $servers_obj = new stdClass();
        foreach($remote_server_ids as $sid) {
            $servers_obj->{$sid} = "0";
        }
        $servers_json = json_encode($servers_obj);
        $servers_json_esc = addslashes($servers_json);

        // Update stream with server assignment on remote
        $sql_update = "UPDATE streams SET server_id = $primary_server_id, servers = '$servers_json_esc' WHERE id = $stream_id;";
        $this->remoteDbExecute($sql_update);

        return true;
    }

    // Start stream manually via SSH ffmpeg (for streams that won't start via API)
    protected function startStreamManually($stream_id)
    {
        $stream_id = intval($stream_id);

        // Get stream URL from remote
        $sql = "SELECT urls FROM streams WHERE id = $stream_id";
        $result = $this->remoteDbExecute($sql);
        error_log("startStreamManually: stream_id=$stream_id, urls result=$result");
        if(empty($result)) return false;

        // Parse URLs - result may be JSON array string
        $urls = json_decode($result, true);
        if(empty($urls) || !is_array($urls)) {
            // Try to extract URL if it's in escaped format
            $urls = json_decode(stripslashes($result), true);
        }
        if(empty($urls) || !is_array($urls)) {
            error_log("startStreamManually: Failed to parse URLs");
            return false;
        }

        $url = $urls[0];
        // Convert HTTPS to HTTP if needed
        $url = str_replace('https://', 'http://', $url);
        $url_esc = escapeshellarg($url);
        error_log("startStreamManually: Starting stream with URL: $url");

        // First insert pids entry so watchdog can track it
        $pids_sql = "INSERT INTO pids (sid, server_id, ppid, url_num, url_num2, adaptive_level, url, start_time, watchdog) VALUES ($stream_id, 1, 0, 1, 1, 0, '" . addslashes($url) . "', UNIX_TIMESTAMP(), UNIX_TIMESTAMP())";
        $this->remoteDbExecute($pids_sql);

        // Build ffmpeg command
        $ffmpeg_cmd = "nohup /home/midnightstreamer/iptv_midnight_streamer/bin/ffmpeg " .
            "-progress /home/midnightstreamer/iptv_midnight_streamer/streams/{$stream_id}_1_.progress " .
            "-y -nostats -nostdin -hide_banner -loglevel error -err_detect ignore_err " .
            "-fflags +genpts -start_at_zero -copyts -vsync 0 -correct_ts_overflow 0 " .
            "-avoid_negative_ts disabled -max_interleave_delta 0 -async 1 " .
            "-probesize 5000000 -analyzeduration 5000000 -user_agent Mozilla/5.0 " .
            "-i $url_esc -strict -2 -ignore_unknown -acodec copy -vcodec copy -scodec copy " .
            "-map 0 -individual_header_trailer 0 -f hls -hls_segment_type mpegts -hls_time 10 " .
            "-hls_allow_cache 1 -hls_ts_options mpegts_flags=+initial_discontinuity:mpegts_copyts=1 " .
            "-hls_delete_threshold 6 -hls_list_size 6 " .
            "-hls_flags independent_segments+delete_segments+discont_start+omit_endlist " .
            "-hls_segment_filename /home/midnightstreamer/iptv_midnight_streamer/streams/{$stream_id}_1_%d.ts " .
            "/home/midnightstreamer/iptv_midnight_streamer/streams/{$stream_id}_1_.m3u8 > /dev/null 2>&1 &";

        $ssh_cmd = "sshpass -p " . escapeshellarg(RemotePanelConfig::get('pass')) .
                   " ssh -o StrictHostKeyChecking=no -o ConnectTimeout=10 " .
                   escapeshellarg(RemotePanelConfig::get('user')) . "@" . escapeshellarg(RemotePanelConfig::get('host')) .
                   " " . escapeshellarg($ffmpeg_cmd);
        shell_exec($ssh_cmd);
        error_log("startStreamManually: SSH command executed");

        return true;
    }

    // Sync stream status from remote pids table
    protected function syncStreamStatusFromRemote()
    {
        global $intro;

        // Also sync on_demand status
        $this->syncOnDemandStatusFromRemote();

        // Sync servers from external panel
        $this->syncServersFromRemote();

        // Get running streams from remote pids table
        $sql = "SELECT p.sid, p.pid, p.ffprobe, p.speed, p.start_time, p.url, s.name
                FROM pids p
                LEFT JOIN streams s ON p.sid = s.id
                WHERE p.pid IS NOT NULL;";
        $result = $this->remoteDbExecute($sql);

        if(empty($result)) return;

        $lines = explode("\n", trim($result));
        foreach($lines as $line) {
            if(empty(trim($line))) continue;

            $parts = explode("\t", $line);
            if(count($parts) < 5) continue;

            $remote_stream_id = intval($parts[0]);
            $pid = intval($parts[1]);
            $ffprobe = isset($parts[2]) ? $parts[2] : '';
            $speed = isset($parts[3]) ? $parts[3] : '';
            $start_time = isset($parts[4]) ? intval($parts[4]) : 0;
            $url = isset($parts[5]) ? $parts[5] : '';
            $stream_name = isset($parts[6]) ? trim($parts[6]) : '';

            if(empty($stream_name)) continue;

            // Find local stream by name
            $local = $intro->db->query("SELECT id FROM streams WHERE stream_display_name='" . $intro->db->escape($stream_name) . "'");
            if($intro->db->returned_rows > 0) {
                $local_row = $intro->db->fetch_assoc($local);
                $local_stream_id = $local_row['id'];

                // Parse codec info from ffprobe - format for StreamsEdit.php
                $stream_info = '';
                $video_width = 0;
                $video_height = 0;
                $video_codec = '-';
                $audio_codec = '-';
                $video_bitrate = 0;
                $start_time_codec = 0;

                $ffprobe_data = @json_decode($ffprobe, true);
                if($ffprobe_data && isset($ffprobe_data['streams'])) {
                    foreach($ffprobe_data['streams'] as $stream) {
                        if($stream['codec_type'] == 'video') {
                            $video_codec = isset($stream['codec_name']) ? $stream['codec_name'] : '-';
                            $video_width = isset($stream['width']) ? intval($stream['width']) : 0;
                            $video_height = isset($stream['height']) ? intval($stream['height']) : 0;
                            $start_time_codec = isset($stream['start_time']) ? floatval($stream['start_time']) : 0;
                        } elseif($stream['codec_type'] == 'audio') {
                            $audio_codec = isset($stream['codec_name']) ? $stream['codec_name'] : '-';
                        }
                    }
                    // Get bitrate from format
                    if(isset($ffprobe_data['format']['bit_rate'])) {
                        $video_bitrate = intval($ffprobe_data['format']['bit_rate'] / 1000); // Convert to Kbps
                    }
                }

                // Build stream_info in correct format for StreamsEdit.php
                $stream_info = json_encode([
                    'codecs' => [
                        'video' => [
                            'codec_name' => $video_codec,
                            'width' => $video_width,
                            'height' => $video_height,
                            'start_time' => $start_time_codec
                        ],
                        'audio' => [
                            'codec_name' => $audio_codec
                        ]
                    ],
                    'speed' => $speed
                ]);

                // Update streams_sys
                $intro->db->query("UPDATE streams_sys SET
                    pid = $pid,
                    stream_status = 0,
                    stream_started = $start_time,
                    stream_info = '" . $intro->db->escape($stream_info) . "',
                    current_source = '" . $intro->db->escape($url) . "',
                    bitrate = $video_bitrate
                    WHERE stream_id = $local_stream_id");
            }
        }

        // Mark streams not in running list as offline (stream_status = 1)
        // Get list of running stream names
        $running_names = [];
        foreach($lines as $line) {
            if(empty(trim($line))) continue;
            $parts = explode("\t", $line);
            if(isset($parts[6]) && !empty(trim($parts[6]))) {
                $running_names[] = "'" . $intro->db->escape(trim($parts[6])) . "'";
            }
        }

        if(!empty($running_names)) {
            $names_in = implode(',', $running_names);
            $intro->db->query("UPDATE streams_sys SET stream_status = 1, pid = -1
                              WHERE stream_id NOT IN (
                                  SELECT id FROM streams WHERE stream_display_name IN ($names_in)
                              )");
        } else {
            // No running streams, mark all as offline
            $intro->db->query("UPDATE streams_sys SET stream_status = 1, pid = -1");
        }
    }

    protected function autoSyncStreamsFromRemote()
    {
        global $intro;

        // Get only Live TV channels (type=0) from remote with logo_image
        $sql = "SELECT id, name, urls, logo_image FROM streams WHERE type=0 ORDER BY id;";
        $result = $this->remoteDbExecute($sql);

        if(empty($result)) return;

        // Build list of remote stream names
        $remote_stream_names = [];
        $lines = explode("\n", trim($result));
        foreach($lines as $line) {
            if(empty(trim($line))) continue;

            $parts = explode("\t", $line);
            if(count($parts) < 2) continue;

            $remote_id = intval($parts[0]);
            $remote_name = trim($parts[1]);
            $remote_urls = isset($parts[2]) ? $parts[2] : '[]';
            $remote_icon = isset($parts[3]) ? trim($parts[3]) : '';

            // Convert file path to web URL if needed
            if(!empty($remote_icon) && $remote_icon != 'photo' && $remote_icon != 'NULL' && !preg_match('/^x+$/', $remote_icon)) {
                if(strpos($remote_icon, '/home/midnightstreamer') !== false) {
                    // Extract filename from path
                    $filename = basename($remote_icon);
                    // Convert to web URL using RemotePanelConfig
                    $remote_host = RemotePanelConfig::get('host');
                    $remote_port = RemotePanelConfig::get('api_port', 8000);
                    $remote_icon = "http://{$remote_host}:{$remote_port}/images/" . $filename;
                }
            } else {
                // If empty, "photo", "NULL", or placeholder like "xxxxxxxxxx", set to empty string
                $remote_icon = '';
            }

            if(empty($remote_name)) continue;

            // Track remote stream name
            $remote_stream_names[] = $remote_name;

            // Check if stream exists locally (by ID or name)
            $check = $intro->db->query("SELECT id, stream_icon FROM streams WHERE (id = $remote_id OR stream_display_name='" . $intro->db->escape($remote_name) . "') AND type=1");
            if($intro->db->returned_rows == 0) {
                // Decode URLs
                $urls_decoded = json_decode($remote_urls, true);
                $stream_source = '';
                if(is_array($urls_decoded) && count($urls_decoded) > 0) {
                    $stream_source = $urls_decoded[0];
                }

                // Insert new stream with same ID as remote
                $data = [
                    'id' => $remote_id,
                    'type' => 1,
                    'stream_display_name' => $remote_name,
                    'stream_source' => json_encode([$stream_source]),
                    'stream_icon' => $remote_icon,  // Sync icon from remote
                    'notes' => '',
                    'category_id' => null,
                    'order' => $remote_id,
                    'added' => time()
                ];
                $intro->db->insert('streams', $data);
                $new_stream_id = $remote_id;
            } else {
                // Stream exists - update icon if changed
                $local_row = $intro->db->fetch_assoc($check);
                $local_icon = $local_row['stream_icon'];

                // Update icon if different from remote (and remote icon is not empty)
                if(!empty($remote_icon) && $local_icon != $remote_icon) {
                    $intro->db->update('streams', ['stream_icon' => $remote_icon], 'id=' . $remote_id);
                }

                $new_stream_id = $remote_id;
            }

            // Add to streams_sys with server_id=1 (only if not exists)
            $check_sys = $intro->db->query("SELECT stream_id FROM streams_sys WHERE stream_id = $new_stream_id AND server_id = 1");
            if($intro->db->returned_rows == 0) {
                $sys_data = [
                    'server_id' => 1,
                    'stream_id' => $new_stream_id,
                    'pid' => -1,
                    'monitor_pid' => -1,
                    'on_demand' => 0
                ];
                $intro->db->insert('streams_sys', $sys_data);
            }
        }

        // DISABLED: Auto-delete feature to prevent deletion of newly added channels
        // Channels added from local panel need time to sync to remote
        // Manual cleanup should be done if needed
        /*
        // Delete local streams that no longer exist on remote
        if(!empty($remote_stream_names)) {
            $names_list = "'" . implode("','", array_map([$intro->db, 'escape'], $remote_stream_names)) . "'";

            // Get IDs of streams to delete (exclude recently added streams - added in last 60 seconds)
            $to_delete = $intro->db->query("SELECT id FROM streams WHERE type=1 AND stream_display_name NOT IN ($names_list) AND added < " . (time() - 60));
            $delete_ids = [];
            while($row = $intro->db->fetch_assoc($to_delete)) {
                $delete_ids[] = intval($row['id']);
            }

            // Delete streams_sys entries first (foreign key)
            if(!empty($delete_ids)) {
                $ids_list = implode(',', $delete_ids);
                $intro->db->query("DELETE FROM streams_sys WHERE stream_id IN ($ids_list)");
                $intro->db->query("DELETE FROM streams WHERE id IN ($ids_list)");
                error_log("Auto-sync: Deleted " . count($delete_ids) . " streams that were removed from remote");
            }
        }
        */
    }

    public function index()
    {
        global $intro;
        global $array;

        // Auto-sync disabled - was causing "Duplicate entry" errors on every page load
        // Use manual sync button if needed to sync from remote
        // if($this->type == 1) {
        //     $this->autoSyncStreamsFromRemote();
        // }

        // Sync stream status from remote pids table - disabled to prevent errors on page load
        // $this->syncStreamStatusFromRemote();

        $edit = new StreamsEdit($this->base, $this->type);
        $qry = $params = '';
        $JOIN = 'LEFT JOIN';
        $JOIN = 'LEFT OUTER JOIN';
        $page = intval($intro->input->get_post('page'));
        $category_id = intval($intro->input->get_post('category_id'));
        $order = trim($intro->input->get_post('order'));
        $stream_display_name = trim($intro->input->get_post('stream_display_name'));
        $status = trim($intro->input->get_post('status'));
        $stream_status = trim($intro->input->get_post('stream_status'));
        $source = trim($intro->input->get_post('source'));
        $stream_icon = trim($intro->input->get_post('stream_icon'));
        $ids = trim($intro->input->get_post('ids'));
        $server = $intro->input->get_post('server');
        $view = trim($intro->input->get_post('view'));
        $ds = intval($intro->input->get_post('ds'));
        $rs = intval($intro->input->get_post('rs'));
        $ar = intval($intro->input->get_post('ar'));
        $et = intval($intro->input->get_post('et'));
        $tvr = intval($intro->input->get_post('tvr'));
        $this->nav();
        echo _obf_0D0713255B04072D042B135C2E233E1902393B0C1B2911();
        if( $intro->input->get('from') == 'edit' ) 
        {
            echo _obf_0D3D40321528110F062A0B0321102712170C15030F2232('<h4> Stream id (' . $intro->input->get_post('id') . (") updated successfully. \n\t\t\t\t<a class='btn btn-default RestartStreamSingle' href=\"" . $this->base . '/RestartCycle?NH=1&id=') . intval($intro->input->get_post('id')) . "\" OnCLick='return false;'>Click to Restart Stream</a>  \n\t\t\t\t<span id='span_single_status'></span></h4>", 'success');
        }
        $qry = ' WHERE s.type=' . $this->type;
        if( $rs == 1 ) 
        {
            $qry .= ' AND s.redirect_stream=1';
            $params .= '&rs=1';
        }
        if( $ds == 1 ) 
        {
            $qry .= ' AND s.direct_source=1';
            $params .= '&ds=1';
        }
        if( $ar == 1 ) 
        {
            $qry .= ' AND s.allow_record=1';
            $params .= '&ar=1';
        }
        if( $et != '' ) 
        {
            $qry .= ' AND s.enable_transcode=1';
            $params .= '&et=1';
        }
        if( $tvr != '' ) 
        {
            $qry .= ' AND s.tv_archive_duration!=0';
            $params .= '&tvr=1';
        }
        if( $ids != '' ) 
        {
            $qry .= (' AND s.id  in (' . $ids . ') ');
        }
        if( $stream_display_name != '' ) 
        {
            $qry .= (' AND stream_display_name  LIKE \'%' . $stream_display_name . '%\' ');
            $params .= ('&stream_display_name=' . $stream_display_name);
        }
        if( $view == 'stopped' ) 
        {
            $qry .= ' AND stream_status=1';
            $params .= '&view=stopped';
        }
        if( $category_id != 0 ) 
        {
            $qry = ' WHERE category_id=' . $category_id;
            $params .= ('&category_id=' . $category_id);
        }
        if( array_key_exists($stream_status, $this->array_stream_status) ) 
        {
            $qry .= $this->array_stream_status_qry[$stream_status];
            $params .= ('&stream_status=' . $stream_status);
        }
        if( $stream_status == 'online' || $stream_status == 'stopped' ) 
        {
            $JOIN = 'INNER JOIN';
        }
        if( $source != '' ) 
        {
            $qry .= (' AND stream_source LIKE \'%' . $source . '%\' ');
            $params .= ('&source=' . $source);
        }
        if( $stream_icon != '' ) 
        {
            $qry .= (' AND stream_icon LIKE \'%' . $stream_icon . '%\' ');
            $params .= ('&stream_icon=' . $stream_icon);
        }
        if( is_array($server) && count($server) >= 1 ) 
        {
            $allssid = [];
            foreach( $server as $ss ) 
            {
                $allssid[] = intval($ss);
            }
            $server = $allssid;
            $qry .= (' AND sys.server_id IN (' . implode(',', $server) . ') ');
            $query = http_build_query(['server' => $server]);
            $params .= ('&' . $query);
        }
        if( $stream_status == 'online' ) 
        {
            $qry .= ' AND sys.pid > 0 AND sys.stream_status = 0 ';
            $params .= '&view=online';
        }
        if( $view == 'missing' ) 
        {
            $qry .= ' AND s.direct_source=0 AND s.id NOT IN (select stream_id from streams_sys) ';
            $params .= ('&view=' . $view);
        }
        if( $order == '' ) 
        {
            $order = 's.`order`:asc';
        }
        $order = str_replace(':', ' ', $order);
        $order2 = $order;
        $rows_per_page = 100;
        if( $page == 0 ) 
        {
            $page = 1;
        }
        $nexlimit = $page * $rows_per_page - $rows_per_page;
        $result = $intro->db->query('SELECT s.*, sys.*, (select count(*) from user_activity_now where stream_id=s.id AND server_id=sys.server_id) as tot  from streams s ' . (' ' . $JOIN . ' `streams_sys` sys ON s.id=sys.stream_id AND sys.server_stream_id = (SELECT MIN(server_stream_id) FROM streams_sys WHERE stream_id=s.id) ') . ('   ' . $qry . ' order by ' . $order . ' limit ' . $nexlimit . ',' . $rows_per_page));
        $totrows = $intro->db->returned_rows;
        $sql_all_rows = $intro->db->query('SELECT DISTINCT s.id from streams s' . (' ' . $JOIN . ' `streams_sys` sys ON s.id=sys.stream_id ') . ('  ' . $qry . ' '));
        $totalrows = $intro->db->returned_rows;
        echo _obf_0D032526222A033D1A2F331C092F2C3636101E15182F22('<i class="' . $this->app_icon . '"></i> ' . $this->app_name . ' (' . $totalrows . ')', $this->app_color);
        echo "\n\t\t<fieldset>\n\t\t\t<form action=\"\" method=\"post\">\n\t\t\t\t<input type=\"text\" name=\"ids\" value=\"" . $ids . "\" placeholder=\"IDs\" size=\"15\">\n\t\t\t\t<input type=\"text\" name=\"stream_display_name\" value=\"" . $stream_display_name . "\" placeholder=\"Stream Name\" size=\"30\">\n\t\t\t\t" . _obf_0D3114132D0E1B5C14292309230C3F01222903383B2811('stream_status', $this->array_stream_status, $stream_status, 'Stream Status', '') . "\n\t\t\t\t<input type=\"text\" name=\"source\" value=\"" . $source . "\" placeholder=\"Stream Source\" size=\"20\">\n\t\t\t\t<input type=\"text\" name=\"stream_icon\" value=\"" . $stream_icon . "\" placeholder=\"Icon\" size=\"20\">\n\t\t\t\t" . _obf_0D311A13371B215B013B112303362D1032353D2E344022('category_id', 'ALL Categories', 'stream_categories', 0, 'id', 'category_name', 'where category_type=\'live\'', 'order by category_name ASC', 'chosen') . "\n\t\t\t\t<input name=\"name\" value=\" Search \" type=\"submit\">\n\t\t\t</form>\n\t\t</fieldset>";

        echo '<div style="margin:10px 0;padding:10px;background:#f5f5f5;border:1px solid #ddd;border-radius:4px;">';
        echo '<strong>Bulk Actions:</strong> ';
        echo '<button id="bulkDelete" class="btn btn-sm btn-danger" style="margin:0 5px;"><i class="icon-trash"></i> Delete Selected</button>';
        echo '<button id="bulkRestart" class="btn btn-sm btn-primary" style="margin:0 5px;"><i class="icon-arrows-cw"></i> Restart Selected</button>';
        echo '<button id="bulkStop" class="btn btn-sm btn-warning" style="margin:0 5px;"><i class="icon-stop"></i> Stop Selected</button>';
        echo '<button id="bulkDisable" class="btn btn-sm btn-secondary" style="margin:0 5px;"><i class="icon-ban"></i> Disable Selected</button>';
        echo ' <span id="bulkActionResult"></span>';
        echo '</div>';

        $edit->TableStart($params);
        $i = 0;
        while( $row = $intro->db->fetch_assoc($result) ) 
        {
            $i++;
            $edit->Row($row, $i);
        }
        echo '</tbody></table></div>';
        $order = str_replace(' ', ':', $order2);
        echo _obf_0D011E16010C0A3322370E3E072C312F130B400C152411();
        echo '<center>' . _obf_0D310332094006251F2A1D300709060C1C245B0E110B32($this->base . '/index?&amp;order=' . $order . $params, $totalrows, $rows_per_page, $page) . '</center>';
        echo '</fieldset>';
        $edit->JavaScript();
        echo "<script>
        \$(\"#category_id\").addClass('chosen').chosen({search_contains: true});
        // Bulk actions are handled by force_bulk_fix.js
        </script>
        <script src=\"" . admin_path . "style/js/force_bulk_fix.js?v=" . time() . "\"></script>";
    }
    public function getStreamStatus()
    {
        global $intro;
        global $array;
        $e = new StreamsEdit($this->base, $this->type);
        $e->getStreamStatus();
    }
    public function Start()
    {
        global $intro;
        global $sess_admin;
        $id = $streams = intval($intro->input->get_post('id'));

        error_log("START BUTTON CLICKED - Stream ID: $id");

        // Use MidnightStreamer API instead of XtreamApi
        $response = $this->startStreamViaApi($id);
        error_log("START API Response: $response");

        $result = json_decode($response, true);

        if(isset($result['result']) && $result['result'] === true) {
            echo '<span class="text-success">✓ Started (ID: ' . $id . ')</span>';
        } else {
            $error_msg = isset($result['error']) ? $result['error'] : 'Unknown error';
            echo '<span class="text-danger">✗ Failed (ID: ' . $id . '): ' . $error_msg . '</span>';
        }
    }
    public function Play()
    {
        global $intro;
        global $sess_admin;
        $id = $streams = intval($intro->input->get_post('id'));
        $bypass = intval($intro->input->get_post('bypass'));
        $host = _obf_0D30011C3F2A2E3F22302707253C09162F3D0E082D0E11('host_port') . ('live/' . $intro->option['play_strm_user'] . '/' . $intro->option['play_strm_pass'] . '/' . $id . '.m3u8');
        $sql = $intro->db->query('SELECT * FROM users where username=\'' . $intro->option['play_strm_user'] . '\' AND password=\'' . $intro->option['play_strm_pass'] . '\';');
        $row = $intro->db->fetch_assoc($sql);
        if( isset($row['id']) ) 
        {
            $userid = $row['id'];
            $x = new XtreamApi('EmptyFunc');
            $AllPids = [];
            if( $bypass == 0 ) 
            {
                $result = $intro->db->query('SELECT pid,server_id FROM `user_activity_now` WHERE user_id=' . $userid . ';');
                while( $myrow = $intro->db->fetch_assoc($result) ) 
                {
                    $AllPids[$myrow['server_id']][] = $myrow['pid'];
                }
                if( count($AllPids) > 0 ) 
                {
                    foreach( $AllPids as $server_id => $pids ) 
                    {
                        $res = $x->KillPids($server_id, $pids);
                        if( $res == 'success' ) 
                        {
                            $intro->db->query('DELETE FROM `user_activity_now` WHERE user_id=' . $userid . ';');
                        }
                    }
                    $intro->redirect('streams', 'Play', '?NH=1&id=' . $id . '&bypass=1&res=' . $res);
                    exit();
                }
            }
        }
        else
        {
            exit( _obf_0D3D40321528110F062A0B0321102712170C15030F2232("<h3>Please set: User/Pass to Play Streams <br/>Go to \n\t\t\t<a href=\"" . $intro->app_url('options', 'index', '#userToPlayStrm') . "\">Tools-> Codes Options</a>\n\t\t\t<br/> First create username from <a href=\"" . $intro->app_url('users', 'Form', '?t=add') . "\">Users</a> with 5 connections for HLS\n\t\t\t<br/> This user will be used to play streams.\n\t\t\t<br>Then set: User/Pass to Play Streams from Streams List</h3>", 'danger') );
        }
        echo "<html>\n\t\t<head>\n\t\t<script src=\"https://cdn.jsdelivr.net/npm/hls.js@latest\"></script>\n\t\t</head>\n\t\t<body>\n\t\t<video id=\"video\" width=\"100%\" height=\"100%\" controls></video>\n\t\t<script>\n\t\tvar video = document.getElementById('video');\n\t\tif(Hls.isSupported()) {\n\t\t\tvar hls = new Hls();\n\t\t\thls.loadSource('" . $host . "');\n\t\t\thls.attachMedia(video);\n\t\t\thls.on(Hls.Events.MANIFEST_PARSED,function() {video.play();});\n\t\t}\n\t\t</script>\n\t\t</body>\n\t\t</html>";
    }
    public function Stop()
    {
        global $intro;
        global $sess_admin;
        $id = $streams = intval($intro->input->get_post('id'));

        // Use MidnightStreamer API instead of XtreamApi
        $response = $this->stopStreamViaApi($id);
        $result = json_decode($response, true);

        if(isset($result['result']) && $result['result'] === true) {
            // Update local database to reflect stopped status
            // This prevents auto_sync from immediately restarting it
            $intro->db->query("UPDATE streams_sys SET stream_status=1, pid=NULL WHERE stream_id=$id");
            error_log("Stop: Updated local streams_sys for stream $id - set to stopped");

            echo '<span class="text-warning">⬛ Stopped</span>';
        } else {
            $error_msg = isset($result['error']) ? $result['error'] : 'Unknown error';
            echo '<span class="text-danger">✗ Failed: ' . $error_msg . '</span>';
        }
    }
    public function RestartCycle()
    {
        global $intro;
        global $sess_admin;
        $id = intval($intro->input->get_post('id'));

        // Use MidnightStreamer API instead of XtreamApi
        $response = $this->restartStreamViaApi($id);
        $result = json_decode($response, true);

        if(isset($result['result']) && $result['result'] === true) {
            echo '<span class="text-info">↻ Restarted</span>';
        } else {
            $error_msg = isset($result['error']) ? $result['error'] : 'Unknown error';
            echo '<span class="text-danger">✗ Failed: ' . $error_msg . '</span>';
        }
    }

    // Bulk Delete Streams
    public function BulkDelete()
    {
        global $intro;
        $ids = $intro->input->post('ids');
        if(!is_array($ids) || count($ids) == 0) {
            echo '<span class="text-danger">No streams selected</span>';
            return;
        }

        $deleted = 0;
        foreach($ids as $id) {
            $id = intval($id);
            $intro->db->query("DELETE FROM streams WHERE id = $id");
            $intro->db->query("DELETE FROM streams_sys WHERE stream_id = $id");
            $intro->db->query("DELETE FROM streams_options WHERE stream_id = $id");
            $deleted++;
            try {
                $this->deleteStreamViaApi($id);
            } catch(Exception $e) {
                error_log("BulkDelete: API error for stream $id: " . $e->getMessage());
            }
        }

        echo '<span class="text-success"><i class="icon-ok"></i> Deleted ' . $deleted . ' stream(s)</span>';
    }

    // Bulk Restart Streams
    public function BulkRestart()
    {
        global $intro;
        $ids = $intro->input->post('ids');
        if(!is_array($ids) || count($ids) == 0) {
            echo '<span class="text-danger">No streams selected</span>';
            return;
        }

        // Clean IDs
        $clean_ids = array_map('intval', $ids);

        // Use bulk API call - stop all streams first
        error_log("BulkRestart: Stopping streams: " . implode(',', $clean_ids));
        $stop_response = $this->midnightStreamerApiCall('streams/stop', $clean_ids);
        error_log("BulkRestart: Stop response: " . $stop_response);

        sleep(1);

        // Then start all streams
        error_log("BulkRestart: Starting streams: " . implode(',', $clean_ids));
        $start_response = $this->midnightStreamerApiCall('streams/start', $clean_ids);
        error_log("BulkRestart: Start response: " . $start_response);

        $result = json_decode($start_response, true);

        if(isset($result['result']) && $result['result'] === true) {
            // Note: PIDs and running status will be synced by auto_sync_daemon within 30 seconds
            echo '<span class="text-success"><i class="icon-ok"></i> Successfully restarted ' . count($clean_ids) . ' stream(s) on external panel<br><small>Status will update automatically within 30 seconds</small></span>';
        } else {
            $error_msg = isset($result['error']) ? $result['error'] : 'Unknown error';
            echo '<span class="text-warning"><i class="icon-info"></i> Restart command sent for ' . count($clean_ids) . ' stream(s)';
            if($error_msg != 'Unknown error') {
                echo '<br><small>API Response: ' . htmlspecialchars($error_msg) . '</small>';
            }
            echo '</span>';
        }
    }

    // Bulk Stop Streams
    public function BulkStop()
    {
        global $intro;
        $ids = $intro->input->post('ids');
        if(!is_array($ids) || count($ids) == 0) {
            echo '<span class="text-danger">No streams selected</span>';
            return;
        }

        // Clean IDs
        $clean_ids = array_map('intval', $ids);

        // Use bulk API call
        error_log("BulkStop: Stopping streams: " . implode(',', $clean_ids));
        $response = $this->midnightStreamerApiCall('streams/stop', $clean_ids);
        error_log("BulkStop: API response: " . $response);

        $result = json_decode($response, true);

        if(isset($result['result']) && $result['result'] === true) {
            // Update local database to reflect stopped status
            // This prevents auto_sync from immediately restarting them
            foreach($clean_ids as $stream_id) {
                $intro->db->query("UPDATE streams_sys SET stream_status=1, pid=NULL WHERE stream_id=$stream_id");
                error_log("BulkStop: Updated local streams_sys for stream $stream_id - set to stopped");
            }
            echo '<span class="text-success"><i class="icon-ok"></i> Successfully stopped ' . count($clean_ids) . ' stream(s) on external panel and updated local status</span>';
        } else {
            $error_msg = isset($result['error']) ? $result['error'] : 'Unknown error';
            echo '<span class="text-warning"><i class="icon-info"></i> Stop command sent for ' . count($clean_ids) . ' stream(s)';
            if($error_msg != 'Unknown error') {
                echo '<br><small>API Response: ' . htmlspecialchars($error_msg) . '</small>';
            }
            echo '</span>';
        }
    }

    // Bulk Disable Streams
    public function BulkDisable()
    {
        global $intro;
        $ids = $intro->input->post('ids');
        if(!is_array($ids) || count($ids) == 0) {
            echo '<span class="text-danger">No streams selected</span>';
            return;
        }

        $disabled = 0;
        foreach($ids as $id) {
            $id = intval($id);
            $intro->db->query("UPDATE streams SET status = 0 WHERE id = $id");
            $disabled++;
        }

        echo '<span class="text-success"><i class="icon-ok"></i> Disabled ' . $disabled . ' stream(s)</span>';
    }

    public function BulkToggleStatus()
    {
        global $intro;
        $ids = $intro->input->post('ids');
        if(!is_array($ids) || count($ids) == 0) {
            echo '<span class="text-danger">No streams selected</span>';
            return;
        }

        $enabled_count = 0;
        $disabled_count = 0;

        foreach($ids as $id) {
            $id = intval($id);

            // Get current status and stream name
            $sql = $intro->db->query("SELECT status, stream_display_name FROM streams WHERE id = $id");
            $row = $intro->db->fetch_assoc($sql);

            if($row) {
                $current_status = intval($row['status']);
                $stream_name = $row['stream_display_name'];

                // Toggle status: 1→0 or 0→1
                $new_status = ($current_status == 1) ? 0 : 1;

                // Update local database
                $intro->db->query("UPDATE streams SET status = $new_status WHERE id = $id");

                // Sync to remote panel
                // Local status=1 → Remote disabled=0 (enabled)
                // Local status=0 → Remote disabled=1 (disabled)
                $remote_disabled = ($new_status == 0) ? 1 : 0;
                $this->updateStreamFieldsInRemote($stream_name, [
                    'disabled' => $remote_disabled
                ]);

                if($new_status == 1) {
                    $enabled_count++;
                } else {
                    $disabled_count++;
                }

                error_log("BulkToggleStatus: Stream ID $id ($stream_name) toggled from status=$current_status to status=$new_status, remote disabled=$remote_disabled");
            }
        }

        echo '<span class="text-success"><i class="icon-ok"></i> Toggled status: ' . $enabled_count . ' enabled, ' . $disabled_count . ' disabled</span>';
    }

    // Set stream to Live mode on remote server
    public function SetLive()
    {
        global $intro;
        global $sess_admin;
        $id = intval($intro->input->get_post('id'));

        error_log("SetLive called for stream ID: $id");

        // Get stream name
        $sql = $intro->db->query("SELECT stream_display_name FROM streams WHERE id = $id");
        $row = $intro->db->fetch_assoc($sql);
        $stream_name = $row['stream_display_name'];

        error_log("SetLive: Stream name = $stream_name");

        // Update remote server (set direct_streaming_on_demand = 0)
        $result = $this->setStreamModeOnRemote($stream_name, 0);
        error_log("SetLive: Remote update result = $result");

        // Update local database (don't change status - it's independent from mode)
        // IMPORTANT: Ensure stream stays ENABLED (status=1) when switching to Live
        error_log("SetLive: About to set direct_source=1 and ensure status=1 for stream ID=$id");
        $intro->db->query("UPDATE streams SET direct_source = 1, status = 1 WHERE id = $id");

        // Update or insert into streams_sys
        $check_sys = $intro->db->query("SELECT stream_id FROM streams_sys WHERE stream_id = $id");
        if($intro->db->returned_rows > 0) {
            $intro->db->query("UPDATE streams_sys SET on_demand = 0 WHERE stream_id = $id");
        } else {
            $intro->db->query("INSERT INTO streams_sys (stream_id, on_demand) VALUES ($id, 0)");
        }

        error_log("SetLive: Local database updated - direct_source=1, on_demand=0");

        // Try to start stream via API
        $api_result = $this->startStreamViaApi($id);
        error_log("SetLive: API start result = $api_result");

        echo "OK"; // Simple response for JavaScript reload
    }

    // Set stream to OnDemand mode on remote server
    public function SetOnDemand()
    {
        global $intro;
        global $sess_admin;
        $id = intval($intro->input->get_post('id'));

        error_log("SetOnDemand called for stream ID: $id");

        // Get stream name
        $sql = $intro->db->query("SELECT stream_display_name FROM streams WHERE id = $id");
        $row = $intro->db->fetch_assoc($sql);
        $stream_name = $row['stream_display_name'];

        error_log("SetOnDemand: Stream name = $stream_name");

        // Update remote server (set direct_streaming_on_demand = 1)
        $result = $this->setStreamModeOnRemote($stream_name, 1);
        error_log("SetOnDemand: Remote update result = $result");

        // Update local database (don't change status - it's independent from mode)
        // IMPORTANT: Ensure stream stays ENABLED (status=1) when switching to OnDemand
        error_log("SetOnDemand: About to set direct_source=0 and ensure status=1 for stream ID=$id");
        $intro->db->query("UPDATE streams SET direct_source = 0, status = 1 WHERE id = $id");

        // Update or insert into streams_sys
        $check_sys = $intro->db->query("SELECT stream_id FROM streams_sys WHERE stream_id = $id");
        if($intro->db->returned_rows > 0) {
            $intro->db->query("UPDATE streams_sys SET on_demand = 1 WHERE stream_id = $id");
        } else {
            $intro->db->query("INSERT INTO streams_sys (stream_id, on_demand) VALUES ($id, 1)");
        }

        error_log("SetOnDemand: Local database updated - direct_source=0, on_demand=1");

        // Stop stream via API (OnDemand doesn't need to run continuously)
        $api_result = $this->stopStreamViaApi($id);
        error_log("SetOnDemand: API stop result = $api_result");

        // IMPORTANT: Re-enable stream on remote after stopping
        // The stop API sets disabled=1, but we need disabled=0 for OnDemand to work
        $stream_name_esc = addslashes($stream_name);
        $re_enable_sql = "UPDATE streams SET disabled = 0 WHERE name = '$stream_name_esc';";
        $this->remoteDbExecute($re_enable_sql);
        error_log("SetOnDemand: Re-enabled stream on remote panel after stop");

        echo "OK"; // Simple response for JavaScript reload
    }

    // Toggle stream status (enable/disable for external API)
    public function ToggleStatus()
    {
        global $intro;
        $id = intval($intro->input->get_post('id'));
        $new_status = intval($intro->input->get_post('status'));

        // Validate status value (0 or 1 only)
        if($new_status !== 0 && $new_status !== 1) {
            echo "ERROR: Invalid status value";
            return;
        }

        // Update local database
        $intro->db->query("UPDATE streams SET status = $new_status WHERE id = $id");

        // Get stream name for remote sync
        $sql = $intro->db->query("SELECT stream_display_name FROM streams WHERE id = $id");
        $row = $intro->db->fetch_assoc($sql);
        if($row) {
            $stream_name = $row['stream_display_name'];
            // Sync status to external panel (local status=1 → remote disabled=0)
            $this->updateStreamFieldsInRemote($stream_name, [
                'disabled' => ($new_status == 0 ? 1 : 0)
            ]);
            error_log("ToggleStatus: Stream ID $id status changed to $new_status - synced to external panel as disabled=" . ($new_status == 0 ? 1 : 0));
        } else {
            error_log("ToggleStatus: Stream ID $id status changed to $new_status (local only - stream not found)");
        }

        echo "OK";
    }

    public function EditChannel()
    {
        global $intro;
        global $array;
        $edit = new StreamsEdit($this->base, $this->type);
        $edit->EditChannel();
    }
    public function doEditChannel()
    {
        global $intro;
        global $array;
        $edit = new StreamsEdit($this->base, $this->type);
        $edit->doEditChannel();
    }
    public function update_status()
    {
        global $intro;
        global $error;
        global $sess_admin;
        $status = trim($intro->input->get_post('status'));
        $status = ($status == 'true' ? 1 : 0);
        $id = intval($intro->input->get_post('id'));
        $sql = $intro->db->query('update streams set status=' . $status . ' WHERE id=' . $id . '; ');
        echo 'Success';
    }
    public function single_multi()
    {
        global $intro;
        $add_type = intval($intro->input->get_post('add_type'));
        $html = "<select id='singleMulti' name='add_type'>\n\t\t\t<option value=\"1\" " . (($add_type == 1 ? 'selected=""' : '')) . ">Add Single Stream</option>\n\t\t\t<option value=\"2\" " . (($add_type == 2 ? 'selected=""' : '')) . ">Upload m3u file</option>\n\t\t</select>";
        return $html;
    }
    public function Form($t = '')
    {
        global $intro;
        global $error;
        global $sess_admin;
        global $array;
        global $category_id;
        global $direct_source;
        global $redirect_stream;
        global $notes;
        global $stream_display_name;
        global $stream_source;
        global $stream_icon;
        global $status;
        global $stream_servers;
        global $rtmp_output;
        global $gen_timestamps;
        global $read_native;
        global $stream_all;
        global $append_flussonic_token;
        global $custom_sid;
        global $custom_ffmpeg;
        global $custom_map;
        global $tv_archive_duration;
        global $tv_archive_server_id;
        global $number;
        global $allow_recordm;
        global $probesize_ondemand;
        global $allow_record;
        global $enable_transcode;
        global $transcode_profile_id;
        global $movie_location;
        global $delay_minutes;
        global $segment_time;

        // Auto-sync servers from external panel before loading form
        $this->syncServersFromRemote();

        $StreamsEdit = new StreamsEdit($this->base, $this->type);
        $on_demand = '';
        if( $error || $_POST != null ) 
        {
            @extract($_POST);
        }
        $IF = intval($intro->input->get_post('IF'));
        $id = intval($intro->input->get_post('id'));
        $page = intval($intro->input->get_post('page'));
        $t = ($t == '' ? $intro->input->get_post('t') : $t);
        if( $IF != 1 ) 
        {
            $this->nav();
        }
        if( $t == 'edit' )
        {
            policy($sess_admin['adminid'], $this->appname . '.php', 'edit');
            $sql = $intro->db->query('SELECT * ' . (' FROM streams where id=\'' . $id . '\''));
            $row = $intro->db->fetch_assoc($sql);
            @extract($row);

            // Log values after extract for debugging
            error_log("Form(edit) after extract: ID=$id, status=" . (isset($status) ? $status : 'NOT_SET') . ", direct_source=" . (isset($direct_source) ? $direct_source : 'NOT_SET'));

            $btn['legend_name'] = $this->app_edit . (' <b>' . $id . '</b> ' . $stream_display_name);
            $btn['legend_icon'] = 'icon-edit';
            $btn['name'] = $intro->lang['save_changes'];
            $btn['img_icon'] = 'icon-floppy';
            $btn['action'] = 'doEdit';
            $stream_source = json_decode($stream_source, true);
            $stream_servers = _obf_0D3B0A102B29073D0C153B282D0C1D183D1E3D27320122($id);
            $sql2 = $intro->db->query('SELECT on_demand FROM `streams_sys` where stream_id=\'' . $id . '\' LIMIT 1');
            $row2 = $intro->db->fetch_assoc($sql2);
            @extract($row2);
            if( $on_demand == 1 ) 
            {
                $on_demand = 'on';
            }
            else if( $on_demand == 0 ) 
            {
                $on_demand = 'off';
            }
            $newSruce = '';
            if( $this->type == 3 ) 
            {
                foreach( $stream_source as $strSource ) 
                {
                    $newSruce .= ($strSource . "\n");
                }
                unset($stream_source);
                $stream_source = $newSruce;
                $movie_location = $stream_servers[0];
            }
        }
        else if( $t == 'add' ) 
        {
            policy($sess_admin['adminid'], $this->appname . '.php', 'add');
            $btn['legend_name'] = $this->app_add;
            $btn['legend_icon'] = 'icon-plus-squared';
            $btn['name'] = $this->app_add;
            $btn['img_icon'] = 'icon-plus-squared';
            $btn['action'] = 'doAdd';
            if( isset($_POST['stream_source']) ) 
            {
                $stream_source = $intro->input->post('stream_source');
            }
            // Handle primary server selection (single server like Xtream Codes)
            $primary_server = intval($intro->input->post('primary_server'));

            if( isset($_POST['server']) )
            {
                $stream_servers = $intro->input->post('server');
            }
            if( is_null($stream_servers) || !is_array($stream_servers) )
            {
                $stream_servers = [];
            }

            // If primary_server is selected, add it to the beginning of stream_servers array
            if($primary_server > 0) {
                // Remove primary_server from array if it already exists
                $stream_servers = array_diff($stream_servers, [$primary_server]);
                // Add primary_server at the beginning
                array_unshift($stream_servers, $primary_server);
            }
            ($direct_source == '' ? 0 : $direct_source);
            ($redirect_stream == '' ? 0 : $redirect_stream);
            $on_demand = ($on_demand == '' ? 'off' : $on_demand);
            $probesize_ondemand = ($probesize_ondemand == '' ? 512000 : $probesize_ondemand);
            // Set default values for source options when adding new stream
            $status = ($status == '' ? 1 : $status); // Default: Enabled
            $gen_timestamps = ($gen_timestamps == '' ? 1 : $gen_timestamps);
            $read_native = ($read_native == '' ? 0 : $read_native);
            $stream_all = ($stream_all == '' ? 0 : $stream_all);
            $rtmp_output = ($rtmp_output == '' ? 0 : $rtmp_output);
            $segment_time = ($segment_time == '' ? 10 : $segment_time);
            if( $this->type == 3 && isset($stream_source[0]) ) 
            {
                $stream_source = $stream_source[0];
            }
        }
        if( $this->type == 3 ) 
        {
            echo "\n\t\t<script>var base = '" . $intro->app_url('movies', 'brws') . "/'; var type='created';</script>\n\t\t<link rel=\"stylesheet\" href=\"" . admin_path . "style/tree/proton/proton.min.css\">\n\t\t<script src=\"" . admin_path . "style/tree/jstree.min.js\"></script>\n\t\t<script src=\"" . admin_path . 'style/js/vod.js?v=2"></script>';
        }
        echo _obf_0D032526222A033D1A2F331C092F2C3636101E15182F22('<i class="' . $this->app_icon . '"></i> ' . $btn['legend_name'] . ' ', $this->app_color);
        $star = '<span style=\'color:#ff0000\'>*</span>';
        echo "\n\t\t<!-- Professional Design Wrapper -->\n\t\t<link rel=\"stylesheet\" href=\"" . admin_path . "style/css/streams-bulk-enhanced.css\">\n\t\t<link rel=\"stylesheet\" href=\"" . admin_path . "style/css/load-balance-enhanced.css\">\n\t\t<div class=\"form-edit-wrapper\">\n\t\t<form method=\"POST\" name=\"form_add\"  action=\"" . $this->base . '/' . $btn['action'] . "\" enctype=\"multipart/form-data\" >\n\t\t\n\t\t" . _obf_0D0713255B04072D042B135C2E233E1902393B0C1B2911() . "\n\t\t<table class=\"table table-bordered\" id=\"table_codes\">";
        if( $t == 'add' && $this->type != 3 ) 
        {
            echo "\n\t\t<tr>\n\t\t\t<td> </td>\n\t\t\t<td>" . $this->single_multi() . ("</td>\n\t\t</tr>\n\t\t<tr id='upload_m3u' style='display:none;'>\n\t\t\t<td>Upload M3U8 File: (Optional)</td>\n\t\t\t<td><input  type=\"file\" name=\"m3u_file\" accept=\".m3u,.m3u8\"> \n\t\t\t" . $this->error('m3u_file') . " \n\t\t\t<!-- <span style='color:red'>(file contents should starts with: #EXTM3U)</span>--> </td>\n\t\t</tr>");
        }
        echo "\n\t\t<tr>\n\t\t\t<td style='width:20%;'>Category  : " . $star . "</td>\n\t\t\t<td >" . _obf_0D311A13371B215B013B112303362D1032353D2E344022('category_id', 'Choose Category', 'stream_categories', $category_id, 'id', 'category_name', 'where category_type=\'live\'', 'order by category_name ASC') . ($this->error('category_id') . "</td>\n\t\t</tr>");

        // Add Bouquet selection right after Category
        $current_bouquet_id = 0;
        if($t == 'edit' && $id > 0) {
            // Check which bouquet this stream belongs to
            $bouquet_check = $intro->db->query("SELECT id FROM bouquets WHERE bouquet_status=1 AND JSON_CONTAINS(bouquet_channels, '$id')");
            if($intro->db->returned_rows > 0) {
                $bouquet_row = $intro->db->fetch_assoc($bouquet_check);
                $current_bouquet_id = intval($bouquet_row['id']);
            }
        }
        echo "\n\t\t<tr>\n\t\t\t\t<td>(Optional) " . ($t == 'add' ? 'Add to' : 'Change') . " Bouquet: </td>\n\t\t\t\t<td>\n\t\t\t\t" . _obf_0D311A13371B215B013B112303362D1032353D2E344022('bouqet_id', 'Choose bouquet', 'bouquets', $current_bouquet_id, 'id', 'bouquet_name', 'bouquet_status', 'where bouquet_status=1', 'order by bouquet_name asc') . "\n\t\t\t\t (This will " . ($t == 'add' ? 'add' : 'move') . " stream to selected Bouquet.)\n\t\t\t\t</td>\n\t\t</tr>";

        // Stream Status section - positioned at top after Bouquet
        echo "\n\t\t<tr>\n\t\t\t<th colspan='2' style='background:#e8f5e9;'>Stream Status</th>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Enable Stream (Available Externally): </td>\n\t\t\t<td>" . _obf_0D363B0530401A36170C40253C1B2D5C2B1D340A0D3D22('status', $status, 'yes') . " \n\t\t\t\t<span style='color:#666;font-size:11px;'>(When disabled, stream will NOT be available to users via API/Player)</span>\n\t\t\t</td>\n\t\t</tr>";

        // Flussonic Settings - positioned right after Stream Status
        echo "\n\t\t<tr>\n\t\t\t<th colspan='2' style='background:#e3f2fd;'>Flussonic Settings</th>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Redirect Stream : </td>\n\t\t\t<td>" . _obf_0D363B0530401A36170C40253C1B2D5C2B1D340A0D3D22('redirect_stream', $redirect_stream, 'yes') . "</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Append Token to Redirected Response : </td>\n\t\t\t<td>" . _obf_0D363B0530401A36170C40253C1B2D5C2B1D340A0D3D22('append_flussonic_token', $append_flussonic_token, 'yes') . "\n\t\t\t\t<span style='color:#666;font-size:11px;'>(Append authentication token to redirect response for Flussonic)</span>\n\t\t\t</td>\n\t\t</tr>";

        if( $this->type == 3 ) 
        {
            echo "\n\t\t<tr>\n\t\t\t<td>Channel Location:</td>\n\t\t\t<td>\n\t\t\t\t" . _obf_0D12262623113F1D022A011A3E1D1B253D294027090E01('movie_location', $movie_location) . (' ' . $this->error('movie_location') . " \n\t\t\t\t<span id=\"local\">\n\t\t\t\t\t<button type=\"button\" class=\"btn btn-danger\" id=\"PickFile\">Pick Files</button>\n\t\t\t\t</span>\n\t\t\t</td>\n\t\t</tr>");
        }
        echo "\n\t\t<tr class='singleTR'>\n\t\t\t<td>Stream Display Name :  <span style='color:#ff0000'>*</span></td>\n\t\t\t<td><input  type=\"text\" name=\"stream_display_name\" value=\"" . $stream_display_name . '" class=\'form-control\'> ' . $this->error('stream_display_name') . "</td>\n\t\t</tr>";
        if( $this->type == 3 ) 
        {
            echo "\n\t\t<tr id=\"movie_source_form\">\n\t\t\t<td>Channel Source : " . $star . " <span id='total_selected_from_tree'>0</span></td>\n\t\t\t<td><textarea name=\"stream_source[]\" id=\"stream_source\" class='form-control' style='height:200px;'>" . $stream_source . '</textarea>' . $this->error('stream_source') . "</td>\n\t\t</tr>";
        }
        if( $this->type != 3 ) 
        {
            echo "\n\t\t<tr class='singleTR'>\n\t\t\t<td>Stream Source:  <span style='color:#ff0000'>*</span></td>\n\t\t\t<td>\n\t\t\t<table class=\"table table-bordered table-hover table-striped\" id=\"tblStream" . $id . "\">\n\t\t\t<tbody>";
            if( is_array($stream_source) ) 
            {
                $i = 0;
                foreach( $stream_source as $val ) 
                {
                    $i++;
                    $val = stripslashes($val);
                    echo "\n\t\t\t\t\t<tr>\n\t\t\t\t\t\t<td>" . $i . " </td>\n\t\t\t\t\t\t<td><input type=\"text\" name=\"stream_source[]\" value=\"" . $val . "\" class='form-control'></td>\n\t\t\t\t\t\t<td><button type=\"button\" class='btn btn-danger btnDelStreamRow'><span class='icon-trash'></span></button></td>\n\t\t\t\t\t</tr>";
                }
            }
            else
            {
                echo "<tr class='singleTR'>\n\t\t\t\t\t<td>1 </td>\n\t\t\t\t\t<td><input type=\"text\" name=\"stream_source[]\" value=\"\" class='form-control'></td>\n\t\t\t\t\t<td><button type=\"button\" class='btn btn-danger btnDelStreamRow'><span class='icon-trash'></span></button></td>\n\t\t\t\t</tr>";
            }
            echo "\t\n\t\t\t</tbody>\n\t\t\t</table>\n\t\t\t<div>\n\t\t\t\t<button data-id=\"" . $id . "\" type=\"button\" class='btn btn-default btnAddNewStreamRowFirst'><span class='icon-plus-squared'></span>First</button>\n\t\t\t\t<button data-id=\"" . $id . "\" type=\"button\" class='btn btn-default btnAddNewStreamRowLast'><span class='icon-plus-squared'></span>Last</button>\n\t\t\t</div>\n\t\t\t</td>\n\t\t</tr>";
        }
        echo "\n\t\t<tr>\n\t\t\t<td>Stream Icon : </td>\n\t\t\t<td>\n\t\t\t\t<input type=\"text\" name=\"stream_icon\" id=\"stream_icon_input\" value=\"" . $stream_icon . "\" class=\"form-control\" placeholder=\"Enter icon URL or upload path\" onkeyup=\"updateIconPreview(this.value)\">\n\t\t\t\t" . $this->error('stream_icon') . "\n\t\t\t\t<div id=\"icon_preview_container\" style=\"margin-top:10px;\">";

        if( !empty($stream_icon) ) {
            echo "\n\t\t\t\t\t<img id=\"stream_icon_preview\" src=\"" . $stream_icon . "\" style=\"max-width:150px; max-height:150px; border:1px solid #ddd; padding:5px; border-radius:4px;\" onerror=\"this.style.display='none'; document.getElementById('icon_error').style.display='block';\">\n\t\t\t\t\t<div id=\"icon_error\" style=\"display:none; color:#d9534f; font-size:11px; margin-top:5px;\"><i class=\"icon-attention\"></i> Image could not be loaded</div>";
        } else {
            echo "\n\t\t\t\t\t<img id=\"stream_icon_preview\" style=\"display:none; max-width:150px; max-height:150px; border:1px solid #ddd; padding:5px; border-radius:4px;\" onerror=\"this.style.display='none'; document.getElementById('icon_error').style.display='block';\">\n\t\t\t\t\t<div id=\"icon_error\" style=\"display:none; color:#d9534f; font-size:11px; margin-top:5px;\"><i class=\"icon-attention\"></i> Image could not be loaded</div>";
        }

        echo "\n\t\t\t\t</div>\n\t\t\t\t<script>\n\t\t\t\tfunction updateIconPreview(url) {\n\t\t\t\t\tvar preview = document.getElementById('stream_icon_preview');\n\t\t\t\t\tvar errorDiv = document.getElementById('icon_error');\n\t\t\t\t\tif(url && url.trim() !== '') {\n\t\t\t\t\t\tpreview.src = url;\n\t\t\t\t\t\tpreview.style.display = 'inline-block';\n\t\t\t\t\t\terrorDiv.style.display = 'none';\n\t\t\t\t\t} else {\n\t\t\t\t\t\tpreview.style.display = 'none';\n\t\t\t\t\t\terrorDiv.style.display = 'none';\n\t\t\t\t\t}\n\t\t\t\t}\n\t\t\t\t</script>\n\t\t\t</td>\n\t\t</tr>";

        echo "\n\t\t<tr>\n\t\t\t<th colspan='2'>Custom Options</th>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Custom Channel SID : </td>\n\t\t\t<td><input  type=\"text\" name=\"custom_sid\" value=\"" . $custom_sid . '" class=\'form-control\'> ' . $this->error('custom_sid') . "</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Custom FFmpeg command : </td>\n\t\t\t<td><input  type=\"text\" name=\"custom_ffmpeg\" value=\"" . $custom_ffmpeg . '" class=\'form-control\'> ' . $this->error('custom_ffmpeg') . "</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Custom MAP Stream : </td>\n\t\t\t<td><input  type=\"text\" name=\"custom_map\" value=\"" . $custom_map . '" class=\'form-control\'> ' . $this->error('custom_map') . "</td>\n\t\t</tr>";
        echo "\n\t\t<tr>\n\t\t\t<th colspan='2'>Servers</th>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Server: <span style='color:#ff0000'>*</span></td>\n\t\t\t<td>";

        // Single server selection dropdown (like Xtream Codes)
        global $intro;
        $primary_server_id = 0;
        if(!empty($stream_servers) && is_array($stream_servers)) {
            $primary_server_id = $stream_servers[0]; // First server is primary
        }

        echo "<select name='primary_server' class='form-control searchable chosen'>";
        echo "<option value=''>Choose Server</option>";
        $servers_result = $intro->db->query('SELECT * FROM `streaming_servers` WHERE status=1 ORDER BY id ASC');
        while($server_row = $intro->db->fetch_assoc($servers_result)) {
            $selected = ($server_row['id'] == $primary_server_id) ? 'selected="selected"' : '';
            echo "<option value='" . $server_row['id'] . "' " . $selected . ">" . $server_row['server_name'] . "</option>";
        }
        echo "</select>";
        echo " <span style='color:#666;font-size:11px;'>(Primary server for this stream)</span>";

        echo "</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>OnDemand/Live:</td>\n\t\t\t<td>" . _obf_0D1029270D2B062E351F39253F1B39061037400E130401('on_demand', [
            'on' => 'Make streams OnDemand on selected servers',
            'live_ondemand' => 'Live mode with OnDemand (Auto-start + OnDemand)',
            'off' => 'Remove OnDemand (Live mode only)'
        ], $on_demand, 'Don\'t Change') . "</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Restart Stream After Editing:</td>\n\t\t\t<td>" . _obf_0D0E0C021E271714230F1135191C022F35343836305C01('restart', 1, 0, 'Yes', 'No', $onStyle = '', $offStyle = '') . "\n\t\t\t<span style='color:#666;font-size:11px;'>(Restart stream after saving changes)</span>\n\t\t\t</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td colspan='2' style='padding:0 !important;'>\n\t\t\t\t<div class='lb-server-section'>\n\t\t\t\t\t<div class='bulk-section-title' style='margin-bottom:0;'>\n\t\t\t\t\t\t<i class='icon-hdd'></i> LOAD BALANCE SERVER CONFIGURATION\n\t\t\t\t\t</div>\n\t\t\t\t\t<div class='lb-config-wrapper'>\n\t\t\t\t\t\t\n\t\t\t\t\t\t<div class='lb-type-selector'>\n\t\t\t\t\t\t\t<label class='control-label'><i class='icon-fork'></i> BALANCING TYPE:</label>\n\t\t\t\t\t\t\t" . _obf_0D1029270D2B062E351F39253F1B39061037400E130401('lb_tree_type', [
            'normal' => 'Normal Load Balancing',
            'tree' => 'Tree Mode'
        ], '', 'Choose Balancing Type') . "\n\t\t\t\t\t\t\t<div id='balanceTypeTXT_form' class='balance-info-box'></div>\n\t\t\t\t\t\t</div>\n\t\t\t\t\t\t\n\t\t\t\t\t\t<div class='lb-servers-wrapper'>\n\t\t\t\t\t\t\t<label class='control-label'>\n\t\t\t\t\t\t\t\t<i class='icon-server'></i> SELECT SERVERS:\n\t\t\t\t\t\t\t</label>\n\t\t\t\t\t\t\t<div style='background: linear-gradient(135deg, #fff9c4 0%, #fff59d 100%); padding: 14px 20px; border-radius: 10px; border: 3px solid #fbc02d; margin-bottom: 18px; box-shadow: 0 2px 8px rgba(251, 192, 45, 0.3);'>\n\t\t\t\t\t\t\t\t<span style='color: #f57f17; font-weight: 700; font-size: 16px; display: block;'>\n\t\t\t\t\t\t\t\t\t<i class='icon-info-circled' style='font-size: 20px; margin-right: 10px; vertical-align: middle;'></i>\n\t\t\t\t\t\t\t\t\tChoose one or more servers for load balancing\n\t\t\t\t\t\t\t\t</span>\n\t\t\t\t\t\t\t\t<span style='color: #e65100; font-size: 14px; font-weight: 600; display: block; margin-top: 6px; margin-left: 30px;'>\n\t\t\t\t\t\t\t\t\t📌 Leave empty to keep current servers unchanged\n\t\t\t\t\t\t\t\t</span>\n\t\t\t\t\t\t\t</div>\n\t\t\t\t\t\t\t<div class='server-checkbox-group-form' id='sortable'>";
        echo _obf_0D31033702041D1F0B1015180339183714070C36032811('server[]', $stream_servers);
        echo $this->error('server') . "\n\t\t\t\t\t\t</div>\n\t\t\t\t\t</div>\n\t\t\t\t</div>\n\t\t\t</td>\n\t\t</tr>";
        if( $this->type == 1 ) 
        {
            echo "\t\n\t\t<tr>\n\t\t\t<td>Probesize On-Demand:</td>\n\t\t\t<td><input type=\"text\" name=\"probesize_ondemand\" value=\"" . $probesize_ondemand . "\"> Default: 128000 or 512000</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Delay Stream:</td>\n\t\t\t<td><input type=\"text\" name=\"delay_minutes\" value=\"" . $delay_minutes . "\"> Delay stream to start after X minutes (Only for live streams)</td>\n\t\t</tr>\n\t\t\n\t\t<tr>\n\t\t\t<th colspan='2'>TV Archive</th>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Number of Days to keep recording this stream : </td>\n\t\t\t<td><input  type=\"text\" name=\"tv_archive_duration\" value=\"" . $tv_archive_duration . "\"></td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Server to Save the Recordings In : </td>\n\t\t\t<td>" . _obf_0D311A13371B215B013B112303362D1032353D2E344022('tv_archive_server_id', 'None', 'streaming_servers', $tv_archive_server_id, 'id', 'server_name', 'bouquet_status', 'where `status`=1', 'order by server_name asc') . "\n\t\t\t<br/> This server must be the same as load balance server. </td>\n\t\t</tr>";
        }
        $arguments = [];
        $arguments[1] = '';
        $arguments[2] = '';
        $arguments[17] = '';
        if( $t == 'edit' ) 
        {
            $sql6 = $intro->db->query('select id,stream_id,argument_id,value from `streams_options` WHERE stream_id=' . $id);
            while( $row6 = $intro->db->fetch_assoc($sql6) ) 
            {
                $argument_id = $row6['argument_id'];
                $arguments[$argument_id] = $row6['value'];
            }
        }
        if( is_array($intro->input->post('arguments')) ) 
        {
            $arguments = $intro->input->post('arguments');
        }
        if( $this->type == 1 ) 
        {
            echo "\n\t\t\n\t\t<tr>\n\t\t\t<th colspan='2'>Fetching Options</th>\n\t\t</tr>\n\t\t\n\t\t<tr>\n\t\t\t<td>User Agent : </td>\n\t\t\t<td><input  type=\"text\" name=\"arguments[1]\" value=\"" . $arguments[1] . "\" class='form-control'></td>\n\t\t</tr>\n\n\t\t<tr>\n\t\t\t<td>HTTP Proxy : </td>\n\t\t\t<td><input  type=\"text\" name=\"arguments[2]\" value=\"" . $arguments[2] . "\" class='form-control' placeholder=\"example: 192.168.1.1:8080\"></td>\n\t\t</tr>\n\n\t\t<tr>\n\t\t\t<td>Cookie : </td>\n\t\t\t<td><input  type=\"text\" name=\"arguments[17]\" value=\"" . $arguments[17] . "\" class='form-control'></td>\n\t\t</tr>";
            echo "\n\t\t<tr>\n\t\t\t<th colspan='2'>STB Devices (Stalker Portal)</th>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Assign Stalker Portal Number : </td>\n\t\t\t<td><input  type=\"text\" name=\"number\" value=\"" . $number . "\"></td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Allow Recording of The Stream : </td>\n\t\t\t<td>" . _obf_0D363B0530401A36170C40253C1B2D5C2B1D340A0D3D22('allow_record', $allow_record, 'yes') . "</td>\n\t\t</tr>";
        }
        echo "\n\t\t<tr>\n\t\t\t<th colspan='2'>Transcode Stream</th>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Enable Transcode : </td>\n\t\t\t<td>" . _obf_0D363B0530401A36170C40253C1B2D5C2B1D340A0D3D22('enable_transcode', $enable_transcode, 'yes') . "</td>\n\t\t</tr>\n\t\t<tr class='transcode'>\n\t\t\t<td>" . tooltip('tooltip_streams', 'transcode_profile_id') . " Select Transcode Profile (Optional): </td>\n\t\t\t<td>" . _obf_0D311A13371B215B013B112303362D1032353D2E344022('transcode_profile_id', 'No Transcode', 'transcoding_profiles', $transcode_profile_id, 'profile_id', 'profile_name', '', ' order by profile_name ASC') . ("\n\t\t\t\t" . $this->error('transcode_profile_id') . "\n\t\t\t\t<a href=\"") . $intro->app_url('transcode_profiles', 'index') . "\" target=\"_blank\" class=\"btn btn-default\">Manage</a>\n\t\t\t</td>\n\t\t</tr>";

        // Source Options for Live TV
        if( $this->type == 1 )
        {
            echo "\n\t\t<tr>\n\t\t\t<th colspan='2'>Source Options</th>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Use Direct Source & don't restream it : </td>\n\t\t\t<td>" . _obf_0D363B0530401A36170C40253C1B2D5C2B1D340A0D3D22('direct_source', $direct_source, 'yes') . "</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>RTMP Output: </td>\n\t\t\t<td>" . _obf_0D363B0530401A36170C40253C1B2D5C2B1D340A0D3D22('rtmp_output', $rtmp_output, 'yes') . "</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Segment Time (seconds): </td>\n\t\t\t<td><input type=\"text\" name=\"segment_time\" value=\"" . $segment_time . "\" class=\"form-control\" style=\"width:100px;\"> \n\t\t\t<span style='color:#666;font-size:11px;'>(HLS segment duration in seconds, default: 10)</span>\n\t\t\t</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Generate New Timestamps: </td>\n\t\t\t<td>" . _obf_0D363B0530401A36170C40253C1B2D5C2B1D340A0D3D22('gen_timestamps', $gen_timestamps, 'yes') . "</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Read Input Source in Native Frames: </td>\n\t\t\t<td>" . _obf_0D363B0530401A36170C40253C1B2D5C2B1D340A0D3D22('read_native', $read_native, 'yes') . "</td>\n\t\t</tr>\n\t\t<tr>\n\t\t\t<td>Stream all the codecs found on the video: </td>\n\t\t\t<td>" . _obf_0D363B0530401A36170C40253C1B2D5C2B1D340A0D3D22('stream_all', $stream_all, 'yes') . "</td>\n\t\t</tr>";
        }

        echo "\n\t\t\n\t\t<tr>\n\t\t\t<td></td>\n\t\t\t<td>\n\t\t\t\t<input type=\"hidden\" name=\"app_name\"  value=\"" . $this->appname . "\">\n\t\t\t\t<input type=\"hidden\" name=\"t\"  value=\"" . $t . "\">\n\t\t\t\t<input type=\"hidden\" name=\"id\"  value=\"" . $id . "\">\n\t\t\t\t<input type=\"hidden\" name=\"IF\"  value=\"" . $IF . "\">\n\t\t\t\t<input type=\"hidden\" name=\"page\"  value=\"" . $page . "\">\n\t\t\t\t<button type=\"submit\" class=\"btn btn-success\">\n\t\t\t\t\t<i class=\"" . $btn['img_icon'] . '"></i> ' . $btn['name'] . " \n\t\t\t\t</button>\n\t\t\t</td>\n\t\t</tr>\n\t\t</table>\n\t\t</form>\n\t\t</div><!-- End form-edit-wrapper -->\n\t\t\n\t\t<script>\n\t\t// Auto-enable Redirect Stream when Append Token is enabled\n\t\tjQuery(document).ready(function($) {\n\t\t\t// Listen for changes on append_flussonic_token radio buttons\n\t\t\t$('input[name=\"append_flussonic_token\"]').on('change', function() {\n\t\t\t\tif($(this).val() == '1' && $(this).is(':checked')) {\n\t\t\t\t\t// Auto-enable Redirect Stream\n\t\t\t\t\t$('input[name=\"redirect_stream\"][value=\"1\"]').prop('checked', true).trigger('change');\n\t\t\t\t}\n\t\t\t});\n\t\t\t\n\t\t\t// Debug: Log form values before submit\n\t\t\t$('form[name=\"form_add\"]').on('submit', function() {\n\t\t\t\tvar status_val = $('input[name=\"status\"]:checked').val();\n\t\t\t\tvar direct_source_val = $('input[name=\"direct_source\"]:checked').val();\n\t\t\t\tconsole.log('Form Submit - status:', status_val, 'direct_source:', direct_source_val);\n\t\t\t\t// Don't prevent submit, just log\n\t\t\t\treturn true;\n\t\t\t});\n\t\t\t\n\t\t\t// Balancing Type selector handler (same as bulk page)\n\t\t\t$(\"#sortable\").sortable();\n\t\t\t\n\t\t\t$('select[name=\"lb_tree_type\"]').on('change', function(){\n\t\t\t\tvar type = $(this).val();\n\t\t\t\t\n\t\t\t\tif(type == 'tree'){\n\t\t\t\t\t $(\"#sortable label\").css(\"display\", \"block\");\n\t\t\t\t\t $(\"#balanceTypeTXT_form\").html(\"<span style='color:#238c00;display:block;'>✅ First server on the list will host stream source and below LBs will use it.<br/>🔄 Drag by mouse servers to sort them.</span>\");\n\t\t\t\t\t \n\t\t\t\t}else if(type == 'normal'){\n\t\t\t\t\t$(\"#sortable label\").css(\"display\", \"inline\");\n\t\t\t\t\t$(\"#balanceTypeTXT_form\").html(\"<span style='color:#d93600;display:block;'>⚠️ This will create as many connections to source as the number of LB you select.\"\n\t\t\t\t\t\t+\"<br/>ℹ️ If you choose 1 LB, it will use 1 connection on source. If you add more LB it will create more connections to source.</span>\");\n\t\t\t\t}else{\n\t\t\t\t\t$(\"#balanceTypeTXT_form\").html(\"\");\n\t\t\t\t}\n\t\t\t});\n\t\t});\n\t\t</script>\n\t\t\n\t\t</form>";
        echo _obf_0D011E16010C0A3322370E3E072C312F130B400C152411();
        $StreamsEdit->JavaScript();
    }
    public function validate_source($stream_source)
    {
        if( !is_array($stream_source) ) 
        {
            return false;
        }
        foreach( $stream_source as $link ) 
        {
            if( $link == '' ) 
            {
                return false;
            }
        }
        return true;
    }
    public function validate_servers($server)
    {
        if( is_null($server) ) 
        {
            return false;
        }
        if( !is_array($server) ) 
        {
            return false;
        }
        foreach( $server as $srv ) 
        {
            if( intval($srv) == 0 ) 
            {
                return false;
            }
        }
        return true;
    }
    public function InsertUpdateArguments($stream_id, $arguments, $type = 'insert')
    {
        global $intro;
        global $error;
        $stream_id = intval($stream_id);
        if( is_array($arguments) && count($arguments) > 0 ) 
        {
            foreach( $arguments as $argument_id => $value ) 
            {
                if( $value != '' ) 
                {
                    $arg = [];
                    $arg['stream_id'] = $stream_id;
                    $arg['argument_id'] = $argument_id;
                    $arg['value'] = $value;
                    if( $type == 'insert' ) 
                    {
                        $intro->db->insert('streams_options', $arg);
                    }
                    if( $type == 'update' ) 
                    {
                        $sql7 = $intro->db->query_fast('select id,value from `streams_options` ' . (' WHERE argument_id=' . $argument_id . ' AND stream_id=' . $stream_id));
                        if( $intro->db->returned_rows == 0 ) 
                        {
                            $intro->db->insert('streams_options', $arg);
                        }
                        else
                        {
                            $intro->db->update('streams_options', $arg, 'argument_id=' . $argument_id . ' AND stream_id=' . $stream_id);
                        }
                    }
                }
                else
                {
                    $intro->db->delete('streams_options', 'argument_id=' . $argument_id . ' AND stream_id=' . $stream_id);
                }
            }
        }
    }
    public function doAdd()
    {
        global $intro;
        global $error;
        $msg = '';
        $time = time();
        $category_id = intval($intro->input->post('category_id'));
        $stream_display_name = trim($intro->input->post('stream_display_name'));
        $stream_display_name = str_replace('\'', '\'', $stream_display_name);
        $notes = trim($intro->input->post('notes'));
        $stream_source = $intro->input->post('stream_source');
        $movie_location = $intro->input->post('movie_location');
        $server = $intro->input->post('server');
        $stream_icon = trim($intro->input->post('stream_icon'));
        $direct_source = intval($intro->input->post('direct_source'));
        $redirect_stream = intval($intro->input->post('redirect_stream'));
        $bouqet_id = intval($intro->input->post('bouqet_id'));
        $on_demand = trim($intro->input->post('on_demand'));
        $probesize_ondemand = intval($intro->input->post('probesize_ondemand'));
        $transcode_profile_id = intval($intro->input->post('transcode_profile_id'));
        $arguments = $intro->input->post('arguments');
        if( $category_id == 0 || !$this->validate_servers($server) ) 
        {
            if( $category_id == 0 ) 
            {
                $msg .= '<li>Category is required! </li>';
            }
            if( !$this->validate_servers($server) ) 
            {
                $msg .= '<li>Please Choose Server!</li>';
            }
            _obf_0D103C08311F24242D2F281F0B3E28333032320A031011($msg, 'danger');
            $this->Form('add');
            exit();
        }
        if( $this->type == 3 ) 
        {
            if( !isset($server[0]) || is_array($server) && count($server) > 1 ) 
            {
                if( !isset($server[0]) || is_array($server) && count($server) > 1 ) 
                {
                    $error['server'] = '<li class=\'error\'>Please Choose only on Server!</li>';
                }
                _obf_0D103C08311F24242D2F281F0B3E28333032320A031011('Please fix errors' . implode(' ', $error), 'danger');
                $this->Form('add');
                exit();
            }
            if( $server[0] != intval($movie_location) ) 
            {
                $error['server'] = '<li class=\'error\'>(Channel Location) and (Load Balance Servers) must be the same server.</li>';
                _obf_0D103C08311F24242D2F281F0B3E28333032320A031011('Please fix errors' . implode(' ', $error), 'danger');
                $this->Form('add');
                exit();
            }
        }
        $success_count = $duplicated_count = 0;
        if( !is_uploaded_file($_FILES['m3u_file']['tmp_name']) ) 
        {
            if( $stream_display_name == '' || !$this->validate_source($stream_source) ) 
            {
                if( $stream_display_name == '' ) 
                {
                    $msg .= '<li>Name is required </li>';
                }
                if( !$this->validate_source($stream_source) ) 
                {
                    $msg .= '<li>stream source is required </li>';
                }
                _obf_0D103C08311F24242D2F281F0B3E28333032320A031011($msg, 'danger');
                $this->Form('add');
                exit();
            }
            $data = [];
            $data['category_id'] = $category_id;
            $data['stream_display_name'] = $stream_display_name;
            $data['stream_source'] = json_encode($stream_source);
            $data['stream_icon'] = trim($intro->input->post('stream_icon'));
            $data['added'] = $time;
            $data['redirect_stream'] = $redirect_stream;
            $data['direct_source'] = $direct_source;
            $data['notes'] = $notes;
            $data['type'] = $this->type;
            $data['status'] = intval($intro->input->post('status')); // Enable/disable stream
            $data['rtmp_output'] = intval($intro->input->post('rtmp_output'));
            $data['gen_timestamps'] = intval($intro->input->post('gen_timestamps'));
            $data['read_native'] = intval($intro->input->post('read_native'));
            $data['stream_all'] = intval($intro->input->post('stream_all'));
            $data['custom_sid'] = trim($intro->input->post('custom_sid'));
            $data['custom_ffmpeg'] = trim($intro->input->post('custom_ffmpeg'));
            $data['custom_map'] = trim($intro->input->post('custom_map'));
            $data['enable_transcode'] = intval($intro->input->post('enable_transcode'));
            $data['transcode_profile_id'] = intval($intro->input->post('transcode_profile_id'));
            if( $this->type == 1 )
            {
                $data['tv_archive_duration'] = intval($intro->input->post('tv_archive_duration'));
                $data['tv_archive_server_id'] = intval($intro->input->post('tv_archive_server_id'));
                $data['number'] = intval($intro->input->post('number'));
                $data['allow_record'] = intval($intro->input->post('allow_record'));
                $data['delay_minutes'] = intval($intro->input->post('delay_minutes'));
                $data['probesize_ondemand'] = ($probesize_ondemand == 0 ? 128000 : $probesize_ondemand);
                $data['segment_time'] = intval($intro->input->post('segment_time'));
                if($data['segment_time'] == 0) $data['segment_time'] = 10; // Default to 10 if not set
            }
            if( $this->type == 3 ) 
            {
                $data['stream_source'] = json_encode(array_filter(explode("\n", $stream_source[0]), function($a)
                {
                    return trim(preg_replace("/\r|\n/", '', $a)) !== '';
                }));
                $data['created_channel_location'] = $server[0];
                $data['stream_all'] = 1;
                $data['read_native'] = 1;
                $data['transcode_profile_id'] = $transcode_profile_id;
                $on_demand = 'off';
            }
            $intro->db->insert('streams', $data);
            $stream_id = $intro->db->insert_id();
            if( $this->type == 1 ) 
            {
                $this->InsertUpdateArguments($stream_id, $arguments, 'insert');
            }
            foreach( $server as $key => $val ) 
            {
                $data2 = [];
                $data2['server_id'] = $val;
                $data2['stream_id'] = $stream_id;
                $data2['on_demand'] = ($on_demand == 'on' ? 1 : 0);
                $intro->db->insert('streams_sys', $data2);
            }
            if( $bouqet_id > 0 ) 
            {
                $sql2 = $intro->db->query('SELECT bouquet_channels FROM bouquets WHERE id=' . $bouqet_id . ';');
                $row2 = $intro->db->fetch_assoc($sql2);
                $bouquet_channels = json_decode($row2['bouquet_channels'], true);
                if( is_array($bouquet_channels) && count($bouquet_channels) > 0 ) 
                {
                    array_unshift($bouquet_channels, $stream_id);
                    $intro->db->update('bouquets', ['bouquet_channels' => json_encode($bouquet_channels)], 'id=' . $bouqet_id);
                }
            }
            // Sync to remote panel with same ID
            error_log("doAdd: Syncing stream '$stream_display_name' (ID: $stream_id) to remote panel");
            $add_result = $this->addStreamToRemote($stream_display_name, $this->type, $stream_source, $stream_id);
            error_log("doAdd: addStreamToRemote result: " . ($add_result ? 'success' : 'failed'));

            // Sync fields to remote panel including status
            // NOTE: direct_streaming_on_demand is NOT synced here - it's handled by setStreamModeOnRemote()
            // which properly sets both direct_streaming_on_demand AND on_demand_servers together
            $fields_to_sync = [
                'redirect' => $data['redirect_stream'],
                'stream_all_tracks' => isset($data['stream_all']) ? $data['stream_all'] : 0,
                'append_flussonic_token' => isset($data['append_flussonic_token']) ? $data['append_flussonic_token'] : 0,
                'disabled' => ($status == 0 ? 1 : 0),  // Sync status to remote: local status=1 → remote disabled=0
                'genpts' => isset($data['gen_timestamps']) ? $data['gen_timestamps'] : 1,  // Sync Generate PTS (default: 1)
                'segment_time' => isset($data['segment_time']) ? $data['segment_time'] : 10,  // Sync Segment Time (default: 10)
                'logo_image' => isset($data['stream_icon']) ? $data['stream_icon'] : ''  // Sync icon
            ];
            error_log("doAdd: Syncing fields: " . json_encode($fields_to_sync));
            $update_result = $this->updateStreamFieldsInRemote($stream_display_name, $fields_to_sync);
            error_log("doAdd: updateStreamFieldsInRemote result: " . ($update_result ? 'success' : 'failed'));

            // Sync category to remote
            if($category_id > 0) {
                $this->syncCategoryToRemote($stream_id, $category_id);
            }

            // Sync bouquet to remote
            if($bouqet_id > 0) {
                $this->syncBouquetToRemote($stream_id, $bouqet_id);
            }

            // Sync selected servers to remote panel
            if($on_demand == 'on' || $on_demand == 'off') {
                error_log("doAdd: Syncing servers to remote - servers: " . json_encode($server) . ", on_demand: $on_demand, probesize: $probesize_ondemand");
                $remote_on_demand = ($on_demand == 'on') ? 1 : 0;
                $this->setStreamModeOnRemote($stream_display_name, $remote_on_demand, $server, $probesize_ondemand);
                $this->ensureRemoteStreamServer($stream_id, $server, $on_demand);
            }

            $msg = '<h3>Stream (' . $stream_display_name . ') added successfully.</h3>';
        }
        else if( is_uploaded_file($_FILES['m3u_file']['tmp_name']) ) 
        {
            $strms = new M3uParse($_FILES['m3u_file']['tmp_name']);
            if( !is_array($strms->found_titles) ) 
            {
                _obf_0D103C08311F24242D2F281F0B3E28333032320A031011('<li>Error: nothing found in the uploaded m3u file!</li>', 'danger');
                $this->Form('add');
                exit();
            }
            $ar_strm_ids = [];
            foreach( $strms->found_titles as $sn => $row ) 
            {
                if( isset($row['title']) && isset($row['url']) ) 
                {
                    $stream_display_name = trim($row['title']);
                    $stream_display_name = str_replace('\'', '\'', $stream_display_name);
                    $url = trim($row['url']);
                    $intro->db->query('SELECT stream_display_name FROM streams where stream_display_name=\'' . $stream_display_name . '\'');
                    if( $intro->db->returned_rows > 0 ) 
                    {
                        $msg .= _obf_0D3D40321528110F062A0B0321102712170C15030F2232('Duplicated: ' . $stream_display_name, 'danger');
                        $duplicated_count++;
                    }
                    else
                    {
                        $data = [];
                        $data['category_id'] = $category_id;
                        $data['stream_display_name'] = $stream_display_name;
                        $data['stream_source'] = json_encode([$url]);
                        $data['stream_icon'] = trim($intro->input->post('stream_icon'));
                        $data['added'] = $time;
                        $data['redirect_stream'] = $redirect_stream;
                        $data['direct_source'] = $direct_source;
                        $data['notes'] = $notes;
                        $data['type'] = $this->type;
                        $data['status'] = intval($intro->input->post('status')); // Enable/disable stream
                        $data['rtmp_output'] = intval($intro->input->post('rtmp_output'));
                        $data['gen_timestamps'] = intval($intro->input->post('gen_timestamps'));
                        $data['read_native'] = intval($intro->input->post('read_native'));
                        $data['stream_all'] = intval($intro->input->post('stream_all'));
                        $data['custom_sid'] = trim($intro->input->post('custom_sid'));
                        $data['custom_ffmpeg'] = trim($intro->input->post('custom_ffmpeg'));
                        $data['custom_map'] = trim($intro->input->post('custom_map'));
                        if( $this->type == 1 ) 
                        {
                            $data['tv_archive_duration'] = intval($intro->input->post('tv_archive_duration'));
                            $data['tv_archive_server_id'] = intval($intro->input->post('tv_archive_server_id'));
                            $data['number'] = intval($intro->input->post('number'));
                            $data['allow_record'] = intval($intro->input->post('allow_record'));
                            $data['delay_minutes'] = intval($intro->input->post('delay_minutes'));
                            $data['probesize_ondemand'] = ($probesize_ondemand == 0 ? 128000 : $probesize_ondemand);
                        }
                        $intro->db->insert('streams', $data);
                        $stream_id = $inset_id = $intro->db->insert_id();
                        $this->InsertUpdateArguments($stream_id, $arguments, 'insert');
                        $ar_strm_ids[] = $inset_id;
                        foreach( $server as $key => $val ) 
                        {
                            $data2 = [];
                            $data2['server_id'] = $val;
                            $data2['stream_id'] = $inset_id;
                            $data2['pid'] = -1;
                            $data2['monitor_pid'] = -1;
                            $data2['on_demand'] = ($on_demand == 'on' ? 1 : 0);
                            $intro->db->insert('streams_sys', $data2);
                        }
                        // Sync to remote panel with same ID
                        $this->addStreamToRemote($stream_display_name, $this->type, [$url], $inset_id);

                        // Sync fields to remote panel including status
                        $this->updateStreamFieldsInRemote($stream_display_name, [
                            'redirect' => $data['redirect_stream'],
                            'stream_all_tracks' => isset($data['stream_all']) ? $data['stream_all'] : 0,
                            'append_flussonic_token' => 0,
                            'disabled' => ($status == 0 ? 1 : 0),  // Sync status to remote: local status=1 → remote disabled=0
                            'direct_streaming_on_demand' => ($on_demand == 'on' ? 1 : 0),  // Sync OnDemand mode
                            'logo_image' => isset($data['stream_icon']) ? $data['stream_icon'] : ''  // Sync icon
                        ]);

                        // Sync category to remote
                        if($category_id > 0) {
                            $this->syncCategoryToRemote($inset_id, $category_id);
                        }

                        $success_count++;
                        $msg .= _obf_0D3D40321528110F062A0B0321102712170C15030F2232('Success: ' . $stream_display_name, 'success');
                    }
                }
            }
            if( $bouqet_id > 0 && count($ar_strm_ids) > 0 ) 
            {
                $sql2 = $intro->db->query('SELECT bouquet_channels FROM bouquets WHERE id=' . $bouqet_id . ';');
                $row2 = $intro->db->fetch_assoc($sql2);
                $bouquet_channels = json_decode($row2['bouquet_channels'], true);
                $bouquet_channels = array_merge($ar_strm_ids, $bouquet_channels);
                if( is_array($bouquet_channels) && count($bouquet_channels) > 0 ) 
                {
                    $intro->db->update('bouquets', ['bouquet_channels' => json_encode($bouquet_channels)], 'id=' . $bouqet_id);
                }
            }
        }
        if( $msg == '' ) 
        {
            $msg = '<h3 style=\'color:red\'>Unknow error!!! nothing added.</h3>';
        }
        if( $success_count > 0 || $duplicated_count > 0 ) 
        {
            $this->nav();
            echo _obf_0D032526222A033D1A2F331C092F2C3636101E15182F22('m3u upload results');
            echo '<h1>Total added Streams = ' . $success_count . "</h1>\n\t\t\t<h1>Total duplicated Streams = " . $duplicated_count . '</h1>' . $msg;
            echo _obf_0D011E16010C0A3322370E3E072C312F130B400C152411();
            unset($msg);
            exit();
        }
        if( $this->type == 3 ) 
        {
            $x = new XtreamApi('EmptyFunc');
            $x->Created($stream_id, $data['stream_source']);
        }
        _obf_0D103C08311F24242D2F281F0B3E28333032320A031011($msg, 'success');
        $intro->redirect($this->appname, 'Form', '?t=add');
    }
    public function doEdit()
    {
        global $intro;
        global $array;

        // Log that doEdit was called
        error_log("========== doEdit() CALLED ==========");
        error_log("doEdit: POST data: " . json_encode($_POST));

        $data = [];
        $stream_display_name = trim($intro->input->post('stream_display_name'));
        $stream_source = $intro->input->post('stream_source');
        $arguments = $intro->input->post('arguments');
        $server = $intro->input->post('server');
        $page = intval($intro->input->post('page'));
        $on_demand = trim($intro->input->post('on_demand'));
        $id = $stream_id = intval($intro->input->post('id'));
        $probesize_ondemand = intval($intro->input->post('probesize_ondemand'));
        $bouqet_id = intval($intro->input->post('bouqet_id'));

        // Log on_demand value
        error_log("doEdit: Stream ID=$id, on_demand value = '$on_demand', server=" . json_encode($server));
        if( $stream_display_name == '' || !$this->validate_source($stream_source) || !$this->validate_servers($server) ) 
        {
            if( $stream_display_name == '' ) 
            {
                $msg .= '<li>Stream name is required. </li>';
            }
            if( !$this->validate_source($stream_source) ) 
            {
                $msg .= '<li>Stream source is required </li>';
            }
            if( !$this->validate_servers($server) ) 
            {
                $msg .= '<li>Please Choose Server</li>';
            }
            _obf_0D103C08311F24242D2F281F0B3E28333032320A031011($msg, 'danger');
            $this->Form('edit');
            exit();
        }
        $data = [];
        $data['category_id'] = intval($intro->input->post('category_id'));
        $data['stream_display_name'] = $stream_display_name;
        $data['stream_source'] = json_encode($stream_source);
        $data['stream_icon'] = trim($intro->input->post('stream_icon'));
        $data['redirect_stream'] = intval($intro->input->post('redirect_stream'));
        $data['direct_source'] = intval($intro->input->post('direct_source'));

        // Debug logging for direct_source
        error_log("doEdit: Stream ID=$id, direct_source POST value = " . $intro->input->post('direct_source') . ", after intval = " . $data['direct_source']);

        $data['notes'] = trim($intro->input->post('notes'));
        $data['rtmp_output'] = intval($intro->input->post('rtmp_output'));
        $data['gen_timestamps'] = intval($intro->input->post('gen_timestamps'));
        $data['read_native'] = intval($intro->input->post('read_native'));
        $data['stream_all'] = intval($intro->input->post('stream_all'));
        $data['append_flussonic_token'] = intval($intro->input->post('append_flussonic_token'));
        $data['custom_sid'] = trim($intro->input->post('custom_sid'));
        $data['custom_ffmpeg'] = trim($intro->input->post('custom_ffmpeg'));
        $data['custom_map'] = trim($intro->input->post('custom_map'));
        $data['enable_transcode'] = intval($intro->input->post('enable_transcode'));
        $data['transcode_profile_id'] = intval($intro->input->post('transcode_profile_id'));
        // Handle status field (enable/disable stream)
        $data['status'] = intval($intro->input->post('status'));
        error_log("doEdit: Stream ID=$id, status POST value = '" . $intro->input->post('status') . "', after intval = " . $data['status']);

        if( $this->type == 1 )
        {
            $data['tv_archive_duration'] = intval($intro->input->post('tv_archive_duration'));
            $data['tv_archive_server_id'] = intval($intro->input->post('tv_archive_server_id'));
            $data['number'] = intval($intro->input->post('number'));
            $data['allow_record'] = intval($intro->input->post('allow_record'));
            $data['delay_minutes'] = intval($intro->input->post('delay_minutes'));
            $data['probesize_ondemand'] = ($probesize_ondemand == 0 ? 128000 : $probesize_ondemand);
            $data['segment_time'] = intval($intro->input->post('segment_time'));
            if($data['segment_time'] == 0) $data['segment_time'] = 10; // Default to 10 if not set
        }
        if( $this->type == 3 ) 
        {
            $data['stream_source'] = json_encode(array_filter(explode("\n", $stream_source[0]), function($a)
            {
                return trim(preg_replace("/\r|\n/", '', $a)) !== '';
            }));
        }
        if( $this->admin['level'] == 5 )
        {
            // Reseller can only edit stream_icon and status
            $status_value = $data['status'];  // Save status before clearing
            $data = [];
            $data['stream_icon'] = trim($intro->input->post('stream_icon'));
            $data['status'] = $status_value;  // Restore status
        }

        // Get old stream name for remote sync
        $old_stream_sql = $intro->db->query('SELECT stream_display_name FROM streams WHERE id=' . $id);
        $old_stream_row = $intro->db->fetch_assoc($old_stream_sql);
        $old_stream_name = $old_stream_row['stream_display_name'];

        $intro->db->update('streams', $data, 'id=' . $id);
        error_log("doEdit: Database updated for stream ID=$id with data: " . json_encode($data));

        // Verify what was actually saved
        $verify = $intro->db->query("SELECT status, direct_source, redirect_stream FROM streams WHERE id=$id");
        $verify_row = $intro->db->fetch_assoc($verify);
        error_log("doEdit: Verification #1 after save - status=" . $verify_row['status'] . ", direct_source=" . $verify_row['direct_source'] . ", redirect_stream=" . $verify_row['redirect_stream']);

        // Also log if this is a reseller account
        error_log("doEdit: Admin level = " . $this->admin['level']);

        // Sync to remote panel - use type=0 (Live TV) to preserve remote type
        // Remote uses type=0 for Live TV, we don't want to change it
        $this->updateStreamInRemote($old_stream_name, $stream_display_name, 0, $stream_source);

        // Checkpoint A
        $vA = $intro->db->query("SELECT status, direct_source FROM streams WHERE id=$id");
        $vA_row = $intro->db->fetch_assoc($vA);
        error_log("doEdit: Checkpoint A after updateStreamInRemote - status=" . $vA_row['status'] . ", direct_source=" . $vA_row['direct_source']);

        // Sync additional fields to remote panel including icon and status
        // NOTE: Status is now synced TO remote when changed locally (one-way: local → remote)
        // The daemon no longer syncs status FROM remote to prevent overwriting local changes
        // NOTE: direct_streaming_on_demand is NOT synced here - it's handled by setStreamModeOnRemote()
        // in InsertUpdateStreamServers() which properly sets both direct_streaming_on_demand AND on_demand_servers together
        $this->updateStreamFieldsInRemote($stream_display_name, [
            'redirect' => $data['redirect_stream'],
            'stream_all_tracks' => $data['stream_all'],
            'append_flussonic_token' => $data['append_flussonic_token'],
            'disabled' => ($data['status'] == 0 ? 1 : 0),  // Sync status to remote: local status=1 → remote disabled=0
            'genpts' => $data['gen_timestamps'],  // Sync Generate PTS to remote
            'segment_time' => isset($data['segment_time']) ? $data['segment_time'] : 10,  // Sync Segment Time (default: 10)
            'logo_image' => $data['stream_icon']  // Sync icon to remote (column: logo_image)
        ]);

        // Checkpoint B
        $vB = $intro->db->query("SELECT status, direct_source FROM streams WHERE id=$id");
        $vB_row = $intro->db->fetch_assoc($vB);
        error_log("doEdit: Checkpoint B after updateStreamFieldsInRemote - status=" . $vB_row['status'] . ", direct_source=" . $vB_row['direct_source']);

        // Sync category to remote
        if($data['category_id'] > 0) {
            $this->syncCategoryToRemote($id, $data['category_id']);
        }

        // Checkpoint C
        $vC = $intro->db->query("SELECT status, direct_source FROM streams WHERE id=$id");
        $vC_row = $intro->db->fetch_assoc($vC);
        error_log("doEdit: Checkpoint C after syncCategoryToRemote - status=" . $vC_row['status'] . ", direct_source=" . $vC_row['direct_source']);

        // Handle bouquet changes
        if($bouqet_id > 0) {
            // Remove stream from all bouquets first
            $all_bouquets = $intro->db->query("SELECT id, bouquet_channels FROM bouquets WHERE bouquet_status=1");
            while($bouquet = $intro->db->fetch_assoc($all_bouquets)) {
                $channels = json_decode($bouquet['bouquet_channels'], true);
                if(is_array($channels) && in_array($id, $channels)) {
                    $channels = array_diff($channels, [$id]);
                    $intro->db->update('bouquets', ['bouquet_channels' => json_encode(array_values($channels))], 'id=' . $bouquet['id']);
                }
            }

            // Add to selected bouquet
            $bouquet_sql = $intro->db->query('SELECT bouquet_channels FROM bouquets WHERE id=' . $bouqet_id);
            if($intro->db->returned_rows > 0) {
                $bouquet_row = $intro->db->fetch_assoc($bouquet_sql);
                $bouquet_channels = json_decode($bouquet_row['bouquet_channels'], true);
                if(!is_array($bouquet_channels)) $bouquet_channels = [];
                if(!in_array($id, $bouquet_channels)) {
                    array_unshift($bouquet_channels, $id);
                    $intro->db->update('bouquets', ['bouquet_channels' => json_encode($bouquet_channels)], 'id=' . $bouqet_id);
                }
            }

            // Sync to remote
            $this->syncBouquetToRemote($id, $bouqet_id);
        }

        // Checkpoint D
        $vD = $intro->db->query("SELECT status, direct_source FROM streams WHERE id=$id");
        $vD_row = $intro->db->fetch_assoc($vD);
        error_log("doEdit: Checkpoint D before InsertUpdateArguments - status=" . $vD_row['status'] . ", direct_source=" . $vD_row['direct_source']);

        $this->InsertUpdateArguments($id, $arguments, 'update');

        // Checkpoint E
        $vE = $intro->db->query("SELECT status, direct_source FROM streams WHERE id=$id");
        $vE_row = $intro->db->fetch_assoc($vE);
        error_log("doEdit: Checkpoint E after InsertUpdateArguments - status=" . $vE_row['status'] . ", direct_source=" . $vE_row['direct_source']);

        // Pass probesize to InsertUpdateStreamServers
        $this->InsertUpdateStreamServers($server, $id, $on_demand, $probesize_ondemand);

        // Checkpoint F
        $vF = $intro->db->query("SELECT status, direct_source FROM streams WHERE id=$id");
        $vF_row = $intro->db->fetch_assoc($vF);
        error_log("doEdit: Checkpoint F after InsertUpdateStreamServers - status=" . $vF_row['status'] . ", direct_source=" . $vF_row['direct_source']);

        // Final verification before redirect
        $verify2 = $intro->db->query("SELECT status, direct_source, redirect_stream FROM streams WHERE id=$id");
        $verify2_row = $intro->db->fetch_assoc($verify2);
        error_log("doEdit: Verification #2 FINAL before redirect - status=" . $verify2_row['status'] . ", direct_source=" . $verify2_row['direct_source'] . ", redirect_stream=" . $verify2_row['redirect_stream'] . ", on_demand=$on_demand");

        // Restart stream if requested
        $restart = intval($intro->input->post('restart'));
        if( $restart == 1 )
        {
            error_log("doEdit: Restarting stream $id via API");
            // Stop stream first
            $this->stopStreamViaApi($id);
            sleep(2);
            // Start stream
            $this->startStreamViaApi($id);
        }

        if( $this->type == 3 )
        {
            $x = new XtreamApi('EmptyFunc');
            $x->Created($stream_id, $data['stream_source']);
            $intro->redirect('streams_created', 'index', '?page=' . $page . '&id=' . $id . '&from=edit');
            exit();
        }
        $intro->redirect($this->appname, 'index', '?page=' . $page . '&id=' . $id . '&from=edit');
    }
    public function InsertUpdateStreamServers($LB_Servers, $stream_id, $on_demand, $probesize = 512000)
    {
        global $intro;
        global $sess_admin;
        global $array;

        // Sync streams.direct_source with on_demand FIRST (before checking server changes)
        // This ensures consistency between on_demand dropdown and direct_source field
        // on_demand='on' → direct_source=0 (OnDemand mode)
        // on_demand='off' → direct_source=1 (Live mode)
        // This prevents auto_sync_daemon from overwriting the user's choice
        if($on_demand == 'on') {
            $intro->db->update('streams', ['direct_source' => 0], 'id=' . $stream_id);
            error_log("InsertUpdateStreamServers: Updated streams.direct_source=0 (OnDemand) for stream $stream_id");
        } else if($on_demand == 'off') {
            $intro->db->update('streams', ['direct_source' => 1], 'id=' . $stream_id);
            error_log("InsertUpdateStreamServers: Updated streams.direct_source=1 (Live) for stream $stream_id");
        }

        // Sync on_demand to remote server BEFORE checking server changes
        // This ensures remote is updated even when servers don't change
        if($on_demand == 'on' || $on_demand == 'off') {
            $sql_name = $intro->db->query("SELECT stream_display_name, status, direct_source FROM streams WHERE id = $stream_id");
            $row_name = $intro->db->fetch_assoc($sql_name);
            if($row_name) {
                $stream_name = $row_name['stream_display_name'];
                $remote_on_demand = ($on_demand == 'on' || $on_demand == 'live_ondemand') ? 1 : 0;

                // Pass selected servers and probesize to remote
                $this->setStreamModeOnRemote($stream_name, $remote_on_demand, $LB_Servers, $probesize);
                error_log("InsertUpdateStreamServers: Updated remote for stream $stream_id - remote_on_demand=$remote_on_demand");
            }
        }

        $dbServ = [];
        $changes = false;
        $sql = $intro->db->query('SELECT server_id FROM streams_sys WHERE stream_id=' . $stream_id . ';');
        if( $intro->db->returned_rows > 0 )
        {
            while( $row = $intro->db->fetch_assoc($sql) )
            {
                $dbServ[] = $row['server_id'];
            }
            if( serialize($LB_Servers) != serialize($dbServ) )
            {
                $changes = true;
            }
        }
        else
        {
            $changes = true;
        }
        if( !$changes )
        {
            return null;
        }
        $x = new XtreamApi('EmptyFunc');
        $x->Stop($stream_id);
        $sql = $intro->db->query('DELETE FROM streams_sys WHERE stream_id=' . $stream_id . ';');
        foreach( $LB_Servers as $key => $val )
        {
            $data2 = [];
            $data2['server_id'] = $val;
            $data2['stream_id'] = $stream_id;
            $data2['pid'] = -1;
            $data2['monitor_pid'] = -1;
            if( $on_demand == 'on' )
            {
                $data2['on_demand'] = 1;
            }
            if( $on_demand == 'off' )
            {
                $data2['on_demand'] = 0;
            }
            $intro->db->insert('streams_sys', $data2);
        }
    }
    public function Del()
    {
        global $intro;
        global $sess_admin;
        global $array;
        $id = intval($intro->input->get_post('id'));
        policy($sess_admin['adminid'], $this->appname . '.php', 'del');
        if( $this->admin['level'] == 1 )
        {
            // Get stream name for remote sync before deleting
            $sql_name = $intro->db->query('SELECT stream_display_name FROM streams WHERE id=' . $id);
            $row_name = $intro->db->fetch_assoc($sql_name);
            $stream_name = $row_name['stream_display_name'];

            $sql = $intro->db->query('DELETE FROM streams WHERE id=' . $id . ' ');
            $sql = $intro->db->query('DELETE FROM streams_sys WHERE stream_id=' . $id . ' ');

            // Sync delete to remote panel
            $this->deleteStreamFromRemote($stream_name);
        }
        $intro->redirect($this->appname);
    }
}
