<?php 
class SqlFormatter
{
    protected static $reserved = [
        'ACCESSIBLE', 
        'ACTION', 
        'AGAINST', 
        'AGGREGATE', 
        'ALGORITHM', 
        'ALL', 
        'ALTER', 
        'ANALYSE', 
        'ANALYZE', 
        'AS', 
        'ASC', 
        'AUTOCOMMIT', 
        'AUTO_INCREMENT', 
        'BACKUP', 
        'BEGIN', 
        'BETWEEN', 
        'BINLOG', 
        'BOTH', 
        'CASCADE', 
        'CASE', 
        'CHANGE', 
        'CHANGED', 
        'CHARACTER SET', 
        'CHARSET', 
        'CHECK', 
        'CHECKSUM', 
        'COLLATE', 
        'COLLATION', 
        'COLUMN', 
        'COLUMNS', 
        'COMMENT', 
        'COMMIT', 
        'COMMITTED', 
        'COMPRESSED', 
        'CONCURRENT', 
        'CONSTRAINT', 
        'CONTAINS', 
        'CONVERT', 
        'CREATE', 
        'CROSS', 
        'CURRENT_TIMESTAMP', 
        'DATABASE', 
        'DATABASES', 
        'DAY', 
        'DAY_HOUR', 
        'DAY_MINUTE', 
        'DAY_SECOND', 
        'DEFAULT', 
        'DEFINER', 
        'DELAYED', 
        'DELETE', 
        'DESC', 
        'DESCRIBE', 
        'DETERMINISTIC', 
        'DISTINCT', 
        'DISTINCTROW', 
        'DIV', 
        'DO', 
        'DUMPFILE', 
        'DUPLICATE', 
        'DYNAMIC', 
        'ELSE', 
        'ENCLOSED', 
        'END', 
        'ENGINE', 
        'ENGINE_TYPE', 
        'ENGINES', 
        'ESCAPE', 
        'ESCAPED', 
        'EVENTS', 
        'EXEC', 
        'EXECUTE', 
        'EXISTS', 
        'EXPLAIN', 
        'EXTENDED', 
        'FAST', 
        'FIELDS', 
        'FILE', 
        'FIRST', 
        'FIXED', 
        'FLUSH', 
        'FOR', 
        'FORCE', 
        'FOREIGN', 
        'FULL', 
        'FULLTEXT', 
        'FUNCTION', 
        'GLOBAL', 
        'GRANT', 
        'GRANTS', 
        'GROUP_CONCAT', 
        'HEAP', 
        'HIGH_PRIORITY', 
        'HOSTS', 
        'HOUR', 
        'HOUR_MINUTE', 
        'HOUR_SECOND', 
        'IDENTIFIED', 
        'IF', 
        'IFNULL', 
        'IGNORE', 
        'IN', 
        'INDEX', 
        'INDEXES', 
        'INFILE', 
        'INSERT', 
        'INSERT_ID', 
        'INSERT_METHOD', 
        'INTERVAL', 
        'INTO', 
        'INVOKER', 
        'IS', 
        'ISOLATION', 
        'KEY', 
        'KEYS', 
        'KILL', 
        'LAST_INSERT_ID', 
        'LEADING', 
        'LEVEL', 
        'LIKE', 
        'LINEAR', 
        'LINES', 
        'LOAD', 
        'LOCAL', 
        'LOCK', 
        'LOCKS', 
        'LOGS', 
        'LOW_PRIORITY', 
        'MARIA', 
        'MASTER', 
        'MASTER_CONNECT_RETRY', 
        'MASTER_HOST', 
        'MASTER_LOG_FILE', 
        'MATCH', 
        'MAX_CONNECTIONS_PER_HOUR', 
        'MAX_QUERIES_PER_HOUR', 
        'MAX_ROWS', 
        'MAX_UPDATES_PER_HOUR', 
        'MAX_USER_CONNECTIONS', 
        'MEDIUM', 
        'MERGE', 
        'MINUTE', 
        'MINUTE_SECOND', 
        'MIN_ROWS', 
        'MODE', 
        'MODIFY', 
        'MONTH', 
        'MRG_MYISAM', 
        'MYISAM', 
        'NAMES', 
        'NATURAL', 
        'NOT', 
        'NOW()', 
        'NULL', 
        'OFFSET', 
        'ON', 
        'OPEN', 
        'OPTIMIZE', 
        'OPTION', 
        'OPTIONALLY', 
        'ON UPDATE', 
        'ON DELETE', 
        'OUTFILE', 
        'PACK_KEYS', 
        'PAGE', 
        'PARTIAL', 
        'PARTITION', 
        'PARTITIONS', 
        'PASSWORD', 
        'PRIMARY', 
        'PRIVILEGES', 
        'PROCEDURE', 
        'PROCESS', 
        'PROCESSLIST', 
        'PURGE', 
        'QUICK', 
        'RANGE', 
        'RAID0', 
        'RAID_CHUNKS', 
        'RAID_CHUNKSIZE', 
        'RAID_TYPE', 
        'READ', 
        'READ_ONLY', 
        'READ_WRITE', 
        'REFERENCES', 
        'REGEXP', 
        'RELOAD', 
        'RENAME', 
        'REPAIR', 
        'REPEATABLE', 
        'REPLACE', 
        'REPLICATION', 
        'RESET', 
        'RESTORE', 
        'RESTRICT', 
        'RETURN', 
        'RETURNS', 
        'REVOKE', 
        'RLIKE', 
        'ROLLBACK', 
        'ROW', 
        'ROWS', 
        'ROW_FORMAT', 
        'SECOND', 
        'SECURITY', 
        'SEPARATOR', 
        'SERIALIZABLE', 
        'SESSION', 
        'SHARE', 
        'SHOW', 
        'SHUTDOWN', 
        'SLAVE', 
        'SONAME', 
        'SOUNDS', 
        'SQL', 
        'SQL_AUTO_IS_NULL', 
        'SQL_BIG_RESULT', 
        'SQL_BIG_SELECTS', 
        'SQL_BIG_TABLES', 
        'SQL_BUFFER_RESULT', 
        'SQL_CALC_FOUND_ROWS', 
        'SQL_LOG_BIN', 
        'SQL_LOG_OFF', 
        'SQL_LOG_UPDATE', 
        'SQL_LOW_PRIORITY_UPDATES', 
        'SQL_MAX_JOIN_SIZE', 
        'SQL_QUOTE_SHOW_CREATE', 
        'SQL_SAFE_UPDATES', 
        'SQL_SELECT_LIMIT', 
        'SQL_SLAVE_SKIP_COUNTER', 
        'SQL_SMALL_RESULT', 
        'SQL_WARNINGS', 
        'SQL_CACHE', 
        'SQL_NO_CACHE', 
        'START', 
        'STARTING', 
        'STATUS', 
        'STOP', 
        'STORAGE', 
        'STRAIGHT_JOIN', 
        'STRING', 
        'STRIPED', 
        'SUPER', 
        'TABLE', 
        'TABLES', 
        'TEMPORARY', 
        'TERMINATED', 
        'THEN', 
        'TO', 
        'TRAILING', 
        'TRANSACTIONAL', 
        'TRUE', 
        'TRUNCATE', 
        'TYPE', 
        'TYPES', 
        'UNCOMMITTED', 
        'UNIQUE', 
        'UNLOCK', 
        'UNSIGNED', 
        'USAGE', 
        'USE', 
        'USING', 
        'VARIABLES', 
        'VIEW', 
        'WHEN', 
        'WITH', 
        'WORK', 
        'WRITE', 
        'YEAR_MONTH'
    ];
    protected static $reserved_toplevel = [
        'SELECT', 
        'FROM', 
        'WHERE', 
        'SET', 
        'ORDER BY', 
        'GROUP BY', 
        'LIMIT', 
        'DROP', 
        'VALUES', 
        'UPDATE', 
        'HAVING', 
        'ADD', 
        'AFTER', 
        'ALTER TABLE', 
        'DELETE FROM', 
        'UNION ALL', 
        'UNION', 
        'EXCEPT', 
        'INTERSECT'
    ];
    protected static $reserved_newline = [
        'LEFT OUTER JOIN', 
        'RIGHT OUTER JOIN', 
        'LEFT JOIN', 
        'RIGHT JOIN', 
        'OUTER JOIN', 
        'INNER JOIN', 
        'JOIN', 
        'XOR', 
        'OR', 
        'AND'
    ];
    protected static $functions = [
        'ABS', 
        'ACOS', 
        'ADDDATE', 
        'ADDTIME', 
        'AES_DECRYPT', 
        'AES_ENCRYPT', 
        'AREA', 
        'ASBINARY', 
        'ASCII', 
        'ASIN', 
        'ASTEXT', 
        'ATAN', 
        'ATAN2', 
        'AVG', 
        'BDMPOLYFROMTEXT', 
        'BDMPOLYFROMWKB', 
        'BDPOLYFROMTEXT', 
        'BDPOLYFROMWKB', 
        'BENCHMARK', 
        'BIN', 
        'BIT_AND', 
        'BIT_COUNT', 
        'BIT_LENGTH', 
        'BIT_OR', 
        'BIT_XOR', 
        'BOUNDARY', 
        'BUFFER', 
        'CAST', 
        'CEIL', 
        'CEILING', 
        'CENTROID', 
        'CHAR', 
        'CHARACTER_LENGTH', 
        'CHARSET', 
        'CHAR_LENGTH', 
        'COALESCE', 
        'COERCIBILITY', 
        'COLLATION', 
        'COMPRESS', 
        'CONCAT', 
        'CONCAT_WS', 
        'CONNECTION_ID', 
        'CONTAINS', 
        'CONV', 
        'CONVERT', 
        'CONVERT_TZ', 
        'CONVEXHULL', 
        'COS', 
        'COT', 
        'COUNT', 
        'CRC32', 
        'CROSSES', 
        'CURDATE', 
        'CURRENT_DATE', 
        'CURRENT_TIME', 
        'CURRENT_TIMESTAMP', 
        'CURRENT_USER', 
        'CURTIME', 
        'DATABASE', 
        'DATE', 
        'DATEDIFF', 
        'DATE_ADD', 
        'DATE_DIFF', 
        'DATE_FORMAT', 
        'DATE_SUB', 
        'DAY', 
        'DAYNAME', 
        'DAYOFMONTH', 
        'DAYOFWEEK', 
        'DAYOFYEAR', 
        'DECODE', 
        'DEFAULT', 
        'DEGREES', 
        'DES_DECRYPT', 
        'DES_ENCRYPT', 
        'DIFFERENCE', 
        'DIMENSION', 
        'DISJOINT', 
        'DISTANCE', 
        'ELT', 
        'ENCODE', 
        'ENCRYPT', 
        'ENDPOINT', 
        'ENVELOPE', 
        'EQUALS', 
        'EXP', 
        'EXPORT_SET', 
        'EXTERIORRING', 
        'EXTRACT', 
        'EXTRACTVALUE', 
        'FIELD', 
        'FIND_IN_SET', 
        'FLOOR', 
        'FORMAT', 
        'FOUND_ROWS', 
        'FROM_DAYS', 
        'FROM_UNIXTIME', 
        'GEOMCOLLFROMTEXT', 
        'GEOMCOLLFROMWKB', 
        'GEOMETRYCOLLECTION', 
        'GEOMETRYCOLLECTIONFROMTEXT', 
        'GEOMETRYCOLLECTIONFROMWKB', 
        'GEOMETRYFROMTEXT', 
        'GEOMETRYFROMWKB', 
        'GEOMETRYN', 
        'GEOMETRYTYPE', 
        'GEOMFROMTEXT', 
        'GEOMFROMWKB', 
        'GET_FORMAT', 
        'GET_LOCK', 
        'GLENGTH', 
        'GREATEST', 
        'GROUP_CONCAT', 
        'GROUP_UNIQUE_USERS', 
        'HEX', 
        'HOUR', 
        'IF', 
        'IFNULL', 
        'INET_ATON', 
        'INET_NTOA', 
        'INSERT', 
        'INSTR', 
        'INTERIORRINGN', 
        'INTERSECTION', 
        'INTERSECTS', 
        'INTERVAL', 
        'ISCLOSED', 
        'ISEMPTY', 
        'ISNULL', 
        'ISRING', 
        'ISSIMPLE', 
        'IS_FREE_LOCK', 
        'IS_USED_LOCK', 
        'LAST_DAY', 
        'LAST_INSERT_ID', 
        'LCASE', 
        'LEAST', 
        'LEFT', 
        'LENGTH', 
        'LINEFROMTEXT', 
        'LINEFROMWKB', 
        'LINESTRING', 
        'LINESTRINGFROMTEXT', 
        'LINESTRINGFROMWKB', 
        'LN', 
        'LOAD_FILE', 
        'LOCALTIME', 
        'LOCALTIMESTAMP', 
        'LOCATE', 
        'LOG', 
        'LOG10', 
        'LOG2', 
        'LOWER', 
        'LPAD', 
        'LTRIM', 
        'MAKEDATE', 
        'MAKETIME', 
        'MAKE_SET', 
        'MASTER_POS_WAIT', 
        'MAX', 
        'MBRCONTAINS', 
        'MBRDISJOINT', 
        'MBREQUAL', 
        'MBRINTERSECTS', 
        'MBROVERLAPS', 
        'MBRTOUCHES', 
        'MBRWITHIN', 
        'MD5', 
        'MICROSECOND', 
        'MID', 
        'MIN', 
        'MINUTE', 
        'MLINEFROMTEXT', 
        'MLINEFROMWKB', 
        'MOD', 
        'MONTH', 
        'MONTHNAME', 
        'MPOINTFROMTEXT', 
        'MPOINTFROMWKB', 
        'MPOLYFROMTEXT', 
        'MPOLYFROMWKB', 
        'MULTILINESTRING', 
        'MULTILINESTRINGFROMTEXT', 
        'MULTILINESTRINGFROMWKB', 
        'MULTIPOINT', 
        'MULTIPOINTFROMTEXT', 
        'MULTIPOINTFROMWKB', 
        'MULTIPOLYGON', 
        'MULTIPOLYGONFROMTEXT', 
        'MULTIPOLYGONFROMWKB', 
        'NAME_CONST', 
        'NULLIF', 
        'NUMGEOMETRIES', 
        'NUMINTERIORRINGS', 
        'NUMPOINTS', 
        'OCT', 
        'OCTET_LENGTH', 
        'OLD_PASSWORD', 
        'ORD', 
        'OVERLAPS', 
        'PASSWORD', 
        'PERIOD_ADD', 
        'PERIOD_DIFF', 
        'PI', 
        'POINT', 
        'POINTFROMTEXT', 
        'POINTFROMWKB', 
        'POINTN', 
        'POINTONSURFACE', 
        'POLYFROMTEXT', 
        'POLYFROMWKB', 
        'POLYGON', 
        'POLYGONFROMTEXT', 
        'POLYGONFROMWKB', 
        'POSITION', 
        'POW', 
        'POWER', 
        'QUARTER', 
        'QUOTE', 
        'RADIANS', 
        'RAND', 
        'RELATED', 
        'RELEASE_LOCK', 
        'REPEAT', 
        'REPLACE', 
        'REVERSE', 
        'RIGHT', 
        'ROUND', 
        'ROW_COUNT', 
        'RPAD', 
        'RTRIM', 
        'SCHEMA', 
        'SECOND', 
        'SEC_TO_TIME', 
        'SESSION_USER', 
        'SHA', 
        'SHA1', 
        'SIGN', 
        'SIN', 
        'SLEEP', 
        'SOUNDEX', 
        'SPACE', 
        'SQRT', 
        'SRID', 
        'STARTPOINT', 
        'STD', 
        'STDDEV', 
        'STDDEV_POP', 
        'STDDEV_SAMP', 
        'STRCMP', 
        'STR_TO_DATE', 
        'SUBDATE', 
        'SUBSTR', 
        'SUBSTRING', 
        'SUBSTRING_INDEX', 
        'SUBTIME', 
        'SUM', 
        'SYMDIFFERENCE', 
        'SYSDATE', 
        'SYSTEM_USER', 
        'TAN', 
        'TIME', 
        'TIMEDIFF', 
        'TIMESTAMP', 
        'TIMESTAMPADD', 
        'TIMESTAMPDIFF', 
        'TIME_FORMAT', 
        'TIME_TO_SEC', 
        'TOUCHES', 
        'TO_DAYS', 
        'TRIM', 
        'TRUNCATE', 
        'UCASE', 
        'UNCOMPRESS', 
        'UNCOMPRESSED_LENGTH', 
        'UNHEX', 
        'UNIQUE_USERS', 
        'UNIX_TIMESTAMP', 
        'UPDATEXML', 
        'UPPER', 
        'USER', 
        'UTC_DATE', 
        'UTC_TIME', 
        'UTC_TIMESTAMP', 
        'UUID', 
        'VARIANCE', 
        'VAR_POP', 
        'VAR_SAMP', 
        'VERSION', 
        'WEEK', 
        'WEEKDAY', 
        'WEEKOFYEAR', 
        'WITHIN', 
        'X', 
        'Y', 
        'YEAR', 
        'YEARWEEK'
    ];
    protected static $boundaries = [
        ',', 
        ';', 
        ':', 
        ')', 
        '(', 
        '.', 
        '=', 
        '<', 
        '>', 
        '+', 
        '-', 
        '*', 
        '/', 
        '!', 
        '^', 
        '%', 
        '|', 
        '&', 
        '#'
    ];
    public static $quote_attributes = 'style="color: blue;"';
    public static $backtick_quote_attributes = 'style="color: purple;"';
    public static $reserved_attributes = 'style="font-weight:bold;"';
    public static $boundary_attributes = '';
    public static $number_attributes = 'style="color: green;"';
    public static $word_attributes = 'style="color: #333;"';
    public static $error_attributes = 'style="background-color: red;"';
    public static $comment_attributes = 'style="color: #aaa;"';
    public static $variable_attributes = 'style="color: orange;"';
    public static $pre_attributes = 'style="color: black; background-color: white;"';
    public static $cli = null;
    public static $cli_quote = "\e[34;1m";
    public static $cli_backtick_quote = "\e[35;1m";
    public static $cli_reserved = "\e[37m";
    public static $cli_boundary = '';
    public static $cli_number = "\e[32;1m";
    public static $cli_word = '';
    public static $cli_error = "\e[31;1;7m";
    public static $cli_comment = "\e[30;1m";
    public static $cli_functions = "\e[37m";
    public static $cli_variable = "\e[36;1m";
    public static $tab = '  ';
    public static $use_pre = true;
    protected static $init = null;
    protected static $regex_boundaries = null;
    protected static $regex_reserved = null;
    protected static $regex_reserved_newline = null;
    protected static $regex_reserved_toplevel = null;
    protected static $regex_function = null;
    public static $max_cachekey_size = 15;
    protected static $token_cache = [];
    protected static $cache_hits = 0;
    protected static $cache_misses = 0;
    const TOKEN_TYPE_WHITESPACE = 0;
    const TOKEN_TYPE_WORD = 1;
    const TOKEN_TYPE_QUOTE = 2;
    const TOKEN_TYPE_BACKTICK_QUOTE = 3;
    const TOKEN_TYPE_RESERVED = 4;
    const TOKEN_TYPE_RESERVED_TOPLEVEL = 5;
    const TOKEN_TYPE_RESERVED_NEWLINE = 6;
    const TOKEN_TYPE_BOUNDARY = 7;
    const TOKEN_TYPE_COMMENT = 8;
    const TOKEN_TYPE_BLOCK_COMMENT = 9;
    const TOKEN_TYPE_NUMBER = 10;
    const TOKEN_TYPE_ERROR = 11;
    const TOKEN_TYPE_VARIABLE = 12;
    const TOKEN_TYPE = 0;
    const TOKEN_VALUE = 1;
    public static function getCacheStats()
    {
        return [
            'hits' => self::$cache_hits, 
            'misses' => self::$cache_misses, 
            'entries' => count(self::$token_cache), 
            'size' => strlen(serialize(self::$token_cache))
        ];
    }
    protected static function init()
    {
        if( self::$init ) 
        {
            return null;
        }
        $reservedMap = array_combine(self::$reserved, array_map('strlen', self::$reserved));
        arsort($reservedMap);
        self::$reserved = array_keys($reservedMap);
        self::$regex_boundaries = '(' . implode('|', array_map([
            'SqlFormatter', 
            'quote_regex'
        ], self::$boundaries)) . ')';
        self::$regex_reserved = '(' . implode('|', array_map([
            'SqlFormatter', 
            'quote_regex'
        ], self::$reserved)) . ')';
        self::$regex_reserved_toplevel = str_replace(' ', '\s+', '(' . implode('|', array_map([
            'SqlFormatter', 
            'quote_regex'
        ], self::$reserved_toplevel)) . ')');
        self::$regex_reserved_newline = str_replace(' ', '\s+', '(' . implode('|', array_map([
            'SqlFormatter', 
            'quote_regex'
        ], self::$reserved_newline)) . ')');
        self::$regex_function = '(' . implode('|', array_map([
            'SqlFormatter', 
            'quote_regex'
        ], self::$functions)) . ')';
        self::$init = true;
    }
    protected static function getNextToken($string, $previous = null)
    {
        if( preg_match('/^\s+/', $string, $matches) ) 
        {
            return [
                1 => $matches[0], 
                0 => 0
            ];
        }
        if( $string[0] === '#' || isset($string[1]) && $string[0] === '-' && $string[1] === '-' || $string[0] === '/' && $string[1] === '*' ) 
        {
            if( $string[0] === '-' || $string[0] === '#' ) 
            {
                $last = strpos($string, "\n");
                $type = 8;
            }
            else
            {
                $last = strpos($string, '*/', 2) + 2;
                $type = 9;
            }
            if( $last === false ) 
            {
                $last = strlen($string);
            }
            return [
                1 => substr($string, 0, $last), 
                0 => $type
            ];
        }
        if( $string[0] === '"' || $string[0] === '\'' || $string[0] === '`' || $string[0] === '[' ) 
        {
            $return = [
                ($string[0] === '`' || $string[0] === '[' ? 3 : 2), 
                self::getQuotedString($string)
            ];
            return $return;
        }
        if( ($string[0] === '@' || $string[0] === ':') && isset($string[1]) ) 
        {
            $ret = [
                '1' => null, 
                '0' => 12
            ];
            if( $string[1] === '"' || $string[1] === '\'' || $string[1] === '`' ) 
            {
                $ret[1] = $string[0] . self::getQuotedString(substr($string, 1));
            }
            else
            {
                preg_match('/^(' . $string[0] . '[a-zA-Z0-9\._\$]+)/', $string, $matches);
                if( $matches ) 
                {
                    $ret[1] = $matches[1];
                }
            }
            if( $ret[1] !== null ) 
            {
                return $ret;
            }
        }
        if( preg_match('/^([0-9]+(\.[0-9]+)?|0x[0-9a-fA-F]+|0b[01]+)($|\s|"\'`|' . self::$regex_boundaries . ')/', $string, $matches) ) 
        {
            return [
                1 => $matches[1], 
                0 => 10
            ];
        }
        if( preg_match('/^(' . self::$regex_boundaries . ')/', $string, $matches) ) 
        {
            return [
                1 => $matches[1], 
                0 => 7
            ];
        }
        if( !$previous || !isset($previous[1]) || $previous[1] !== '.' ) 
        {
            $upper = strtoupper($string);
            if( preg_match('/^(' . self::$regex_reserved_toplevel . ')($|\s|' . self::$regex_boundaries . ')/', $upper, $matches) ) 
            {
                return [
                    5, 
                    substr($string, 0, strlen($matches[1]))
                ];
            }
            if( preg_match('/^(' . self::$regex_reserved_newline . ')($|\s|' . self::$regex_boundaries . ')/', $upper, $matches) ) 
            {
                return [
                    6, 
                    substr($string, 0, strlen($matches[1]))
                ];
            }
            if( preg_match('/^(' . self::$regex_reserved . ')($|\s|' . self::$regex_boundaries . ')/', $upper, $matches) ) 
            {
                return [
                    4, 
                    substr($string, 0, strlen($matches[1]))
                ];
            }
        }
        $upper = strtoupper($string);
        if( preg_match('/^(' . self::$regex_function . '[(]|\s|[)])/', $upper, $matches) ) 
        {
            return [
                4, 
                substr($string, 0, strlen($matches[1]) - 1)
            ];
        }
        preg_match('/^(.*?)($|\s|["\'`]|' . self::$regex_boundaries . ')/', $string, $matches);
        return [
            1 => $matches[1], 
            0 => 1
        ];
    }
    protected static function getQuotedString($string)
    {
        $ret = null;
        if( preg_match('/^(((`[^`]*($|`))+)|((\[[^\]]*($|\]))(\][^\]]*($|\]))*)|(("[^"\\]*(?:\\.[^"\\]*)*("|$))+)|((\'[^\'\\]*(?:\\.[^\'\\]*)*(\'|$))+))/s', $string, $matches) ) 
        {
            $ret = $matches[1];
        }
        return $ret;
    }
    protected static function tokenize($string)
    {
        self::init();
        $tokens = [];
        $original_length = strlen($string);
        $old_string_len = strlen($string) + 1;
        $token = null;
        for( $current_length = strlen($string); $current_length; $current_length -= $token_length ) 
        {
            if( $old_string_len <= $current_length ) 
            {
                $tokens[] = [
                    1 => $string, 
                    0 => 11
                ];
                return $tokens;
            }
            $old_string_len = $current_length;
            if( self::$max_cachekey_size <= $current_length ) 
            {
                $cacheKey = substr($string, 0, self::$max_cachekey_size);
            }
            else
            {
                $cacheKey = false;
            }
            if( $cacheKey && isset(self::$token_cache[$cacheKey]) ) 
            {
                $token = self::$token_cache[$cacheKey];
                $token_length = strlen($token[1]);
                self::$cache_hits++;
            }
            else
            {
                $token = self::getNextToken($string, $token);
                $token_length = strlen($token[1]);
                self::$cache_misses++;
                if( $cacheKey && $token_length < self::$max_cachekey_size ) 
                {
                    self::$token_cache[$cacheKey] = $token;
                }
            }
            $tokens[] = $token;
            $string = substr($string, $token_length);
        }
        return $tokens;
    }
    public static function format($string, $highlight = true)
    {
        $return = '';
        $tab = "\t";
        $indent_level = 0;
        $newline = false;
        $inline_parentheses = false;
        $increase_special_indent = false;
        $increase_block_indent = false;
        $indent_types = [];
        $added_newline = false;
        $inline_count = 0;
        $inline_indented = false;
        $clause_limit = false;
        $original_tokens = self::tokenize($string);
        $tokens = [];
        foreach( $original_tokens as $i => $token ) 
        {
            if( $token[0] !== 0 ) 
            {
                $token['i'] = $i;
                $tokens[] = $token;
            }
        }
        foreach( $tokens as $i => $token ) 
        {
            if( $highlight ) 
            {
                $highlighted = self::highlightToken($token);
            }
            else
            {
                $highlighted = $token[1];
            }
            if( $increase_special_indent ) 
            {
                $indent_level++;
                $increase_special_indent = false;
                array_unshift($indent_types, 'special');
            }
            if( $increase_block_indent ) 
            {
                $indent_level++;
                $increase_block_indent = false;
                array_unshift($indent_types, 'block');
            }
            if( $newline ) 
            {
                $return .= ("\n" . str_repeat($tab, $indent_level));
                $newline = false;
                $added_newline = true;
            }
            else
            {
                $added_newline = false;
            }
            if( $token[0] === 8 || $token[0] === 9 ) 
            {
                if( $token[0] === 9 ) 
                {
                    $indent = str_repeat($tab, $indent_level);
                    $return .= ("\n" . $indent);
                    $highlighted = str_replace("\n", "\n" . $indent, $highlighted);
                }
                $return .= $highlighted;
                $newline = true;
                continue;
            }
            if( $inline_parentheses ) 
            {
                if( $token[1] === ')' ) 
                {
                    $return = rtrim($return, ' ');
                    if( $inline_indented ) 
                    {
                        array_shift($indent_types);
                        $indent_level--;
                        $return .= ("\n" . str_repeat($tab, $indent_level));
                    }
                    $inline_parentheses = false;
                    $return .= ($highlighted . ' ');
                    continue;
                }
                if( $token[1] === ',' && $inline_count >= 30 ) 
                {
                    $inline_count = 0;
                    $newline = true;
                }
                $inline_count += strlen($token[1]);
            }
            if( $token[1] === '(' ) 
            {
                $length = 0;
                for( $j = 1; $j <= 250; $j++ ) 
                {
                    if( !isset($tokens[$i + $j]) ) 
                    {
                        break;
                    }
                    $next = $tokens[$i + $j];
                    if( $next[1] === ')' ) 
                    {
                        $inline_parentheses = true;
                        $inline_count = 0;
                        $inline_indented = false;
                        break;
                    }
                    if( $next[1] === ';' || $next[1] === '(' ) 
                    {
                        break;
                    }
                    if( $next[0] === 5 || $next[0] === 6 || $next[0] === 8 || $next[0] === 9 ) 
                    {
                        break;
                    }
                    $length += strlen($next[1]);
                }
                if( $inline_parentheses && $length > 30 ) 
                {
                    $increase_block_indent = true;
                    $inline_indented = true;
                    $newline = true;
                }
                if( isset($original_tokens[$token['i'] - 1]) && $original_tokens[$token['i'] - 1][0] !== 0 ) 
                {
                    $return = rtrim($return, ' ');
                }
                if( !$inline_parentheses ) 
                {
                    $increase_block_indent = true;
                    $newline = true;
                }
            }
            else if( $token[1] === ')' ) 
            {
                $return = rtrim($return, ' ');
                $indent_level--;
                while( $j = array_shift($indent_types) ) 
                {
                    if( $j === 'special' ) 
                    {
                        $indent_level--;
                        continue;
                    }
                    break;
                }
                if( $indent_level < 0 ) 
                {
                    $indent_level = 0;
                    if( $highlight ) 
                    {
                        $return .= ("\n" . self::highlightError($token[1]));
                        continue;
                    }
                }
                if( !$added_newline ) 
                {
                    $return .= ("\n" . str_repeat($tab, $indent_level));
                }
            }
            else if( $token[0] === 5 ) 
            {
                $increase_special_indent = true;
                reset($indent_types);
                if( current($indent_types) === 'special' ) 
                {
                    $indent_level--;
                    array_shift($indent_types);
                }
                $newline = true;
                if( !$added_newline ) 
                {
                    $return .= ("\n" . str_repeat($tab, $indent_level));
                }
                else
                {
                    $return = rtrim($return, $tab) . str_repeat($tab, $indent_level);
                }
                if( strpos($token[1], ' ') !== false || strpos($token[1], "\n") !== false || strpos($token[1], "\t") !== false ) 
                {
                    $highlighted = preg_replace('/\s+/', ' ', $highlighted);
                }
                if( $token[1] === 'LIMIT' && !$inline_parentheses ) 
                {
                    $clause_limit = true;
                }
            }
            else if( $clause_limit && $token[1] !== ',' && $token[0] !== 10 && $token[0] !== 0 ) 
            {
                $clause_limit = false;
            }
            else if( $token[1] === ',' && !$inline_parentheses ) 
            {
                if( $clause_limit === true ) 
                {
                    $newline = false;
                    $clause_limit = false;
                }
                else
                {
                    $newline = true;
                }
            }
            else if( $token[0] === 6 ) 
            {
                if( !$added_newline ) 
                {
                    $return .= ("\n" . str_repeat($tab, $indent_level));
                }
                if( strpos($token[1], ' ') !== false || strpos($token[1], "\n") !== false || strpos($token[1], "\t") !== false ) 
                {
                    $highlighted = preg_replace('/\s+/', ' ', $highlighted);
                }
            }
            else if( $token[0] === 7 && isset($tokens[$i - 1]) && $tokens[$i - 1][0] === 7 && isset($original_tokens[$token['i'] - 1]) && $original_tokens[$token['i'] - 1][0] !== 0 ) 
            {
                $return = rtrim($return, ' ');
            }
            if( $token[1] === '.' || $token[1] === ',' || $token[1] === ';' ) 
            {
                $return = rtrim($return, ' ');
            }
            $return .= ($highlighted . ' ');
            if( $token[1] === '(' || $token[1] === '.' ) 
            {
                $return = rtrim($return, ' ');
            }
            if( $token[1] === '-' && isset($tokens[$i + 1]) && $tokens[$i + 1][0] === 10 && isset($tokens[$i - 1]) ) 
            {
                $prev = $tokens[$i - 1][0];
                if( $prev !== 2 && $prev !== 3 && $prev !== 1 && $prev !== 10 ) 
                {
                    $return = rtrim($return, ' ');
                }
            }
        }
        if( $highlight && array_search('block', $indent_types) !== false ) 
        {
            $return .= ("\n" . self::highlightError('WARNING: unclosed parentheses or section'));
        }
        $return = trim(str_replace("\t", self::$tab, $return));
        if( $highlight ) 
        {
            $return = self::output($return);
        }
        return $return;
    }
    public static function highlight($string)
    {
        $tokens = self::tokenize($string);
        $return = '';
        foreach( $tokens as $token ) 
        {
            $return .= self::highlightToken($token);
        }
        return self::output($return);
    }
    public static function splitQuery($string)
    {
        $queries = [];
        $current_query = '';
        $empty = true;
        $tokens = self::tokenize($string);
        foreach( $tokens as $token ) 
        {
            if( $token[1] === ';' ) 
            {
                if( !$empty ) 
                {
                    $queries[] = $current_query . ';';
                }
                $current_query = '';
                $empty = true;
                continue;
            }
            if( $token[0] !== 0 && $token[0] !== 8 && $token[0] !== 9 ) 
            {
                $empty = false;
            }
            $current_query .= $token[1];
        }
        if( !$empty ) 
        {
            $queries[] = trim($current_query);
        }
        return $queries;
    }
    public static function removeComments($string)
    {
        $result = '';
        $tokens = self::tokenize($string);
        foreach( $tokens as $token ) 
        {
            if( $token[0] === 8 || $token[0] === 9 ) 
            {
                continue;
            }
            $result .= $token[1];
        }
        $result = self::format($result, false);
        return $result;
    }
    public static function compress($string)
    {
        $result = '';
        $tokens = self::tokenize($string);
        $whitespace = true;
        foreach( $tokens as $token ) 
        {
            if( $token[0] === 8 || $token[0] === 9 ) 
            {
                continue;
            }
            else if( $token[0] === 4 || $token[0] === 6 || $token[0] === 5 ) 
            {
                $token[1] = preg_replace('/\s+/', ' ', $token[1]);
            }
            if( $token[0] === 0 ) 
            {
                if( $whitespace ) 
                {
                    continue;
                }
                else
                {
                    $whitespace = true;
                    $token[1] = ' ';
                }
            }
            else
            {
                $whitespace = false;
            }
            $result .= $token[1];
        }
        return rtrim($result);
    }
    protected static function highlightToken($token)
    {
        $type = $token[0];
        if( self::is_cli() ) 
        {
            $token = $token[1];
        }
        else if( defined('ENT_IGNORE') ) 
        {
            $token = htmlentities($token[1], ENT_COMPAT | ENT_IGNORE, 'UTF-8');
        }
        else
        {
            $token = htmlentities($token[1], ENT_COMPAT, 'UTF-8');
        }
        if( $type === 7 ) 
        {
            return self::highlightBoundary($token);
        }
        else if( $type === 1 ) 
        {
            return self::highlightWord($token);
        }
        else if( $type === 3 ) 
        {
            return self::highlightBacktickQuote($token);
        }
        else if( $type === 2 ) 
        {
            return self::highlightQuote($token);
        }
        else if( $type === 4 ) 
        {
            return self::highlightReservedWord($token);
        }
        else if( $type === 5 ) 
        {
            return self::highlightReservedWord($token);
        }
        else if( $type === 6 ) 
        {
            return self::highlightReservedWord($token);
        }
        else if( $type === 10 ) 
        {
            return self::highlightNumber($token);
        }
        else if( $type === 12 ) 
        {
            return self::highlightVariable($token);
        }
        else if( $type === 8 || $type === 9 ) 
        {
            return self::highlightComment($token);
        }
        return $token;
    }
    protected static function highlightQuote($value)
    {
        if( self::is_cli() ) 
        {
            return self::$cli_quote . $value . "\e[0m";
        }
        else
        {
            return '<span ' . self::$quote_attributes . '>' . $value . '</span>';
        }
    }
    protected static function highlightBacktickQuote($value)
    {
        if( self::is_cli() ) 
        {
            return self::$cli_backtick_quote . $value . "\e[0m";
        }
        else
        {
            return '<span ' . self::$backtick_quote_attributes . '>' . $value . '</span>';
        }
    }
    protected static function highlightReservedWord($value)
    {
        if( self::is_cli() ) 
        {
            return self::$cli_reserved . $value . "\e[0m";
        }
        else
        {
            return '<span ' . self::$reserved_attributes . '>' . $value . '</span>';
        }
    }
    protected static function highlightBoundary($value)
    {
        if( $value === '(' || $value === ')' ) 
        {
            return $value;
        }
        if( self::is_cli() ) 
        {
            return self::$cli_boundary . $value . "\e[0m";
        }
        else
        {
            return '<span ' . self::$boundary_attributes . '>' . $value . '</span>';
        }
    }
    protected static function highlightNumber($value)
    {
        if( self::is_cli() ) 
        {
            return self::$cli_number . $value . "\e[0m";
        }
        else
        {
            return '<span ' . self::$number_attributes . '>' . $value . '</span>';
        }
    }
    protected static function highlightError($value)
    {
        if( self::is_cli() ) 
        {
            return self::$cli_error . $value . "\e[0m";
        }
        else
        {
            return '<span ' . self::$error_attributes . '>' . $value . '</span>';
        }
    }
    protected static function highlightComment($value)
    {
        if( self::is_cli() ) 
        {
            return self::$cli_comment . $value . "\e[0m";
        }
        else
        {
            return '<span ' . self::$comment_attributes . '>' . $value . '</span>';
        }
    }
    protected static function highlightWord($value)
    {
        if( self::is_cli() ) 
        {
            return self::$cli_word . $value . "\e[0m";
        }
        else
        {
            return '<span ' . self::$word_attributes . '>' . $value . '</span>';
        }
    }
    protected static function highlightVariable($value)
    {
        if( self::is_cli() ) 
        {
            return self::$cli_variable . $value . "\e[0m";
        }
        else
        {
            return '<span ' . self::$variable_attributes . '>' . $value . '</span>';
        }
    }
    private static function quote_regex($a)
    {
        return preg_quote($a, '/');
    }
    private static function output($string)
    {
        if( self::is_cli() ) 
        {
            return $string . "\n";
        }
        else
        {
            $string = trim($string);
            if( !self::$use_pre ) 
            {
                return $string;
            }
            return '<pre ' . self::$pre_attributes . '>' . $string . '</pre>';
        }
    }
    private static function is_cli()
    {
        if( isset(self::$cli) ) 
        {
            return self::$cli;
        }
        else
        {
            return php_sapi_name() === 'cli';
        }
    }
}
