<?php
/**
 * IPTV Live Stream Handler with Auto-Fallback
 * يتحقق من صلاحية الروابط ويختار الرابط الصحيح تلقائياً
 */

// Get URL parameters
$path_info = isset($_SERVER['PATH_INFO']) ? $_SERVER['PATH_INFO'] : '';
$parts = explode('/', trim($path_info, '/'));

if (count($parts) < 3) {
    header('HTTP/1.1 400 Bad Request');
    die('Invalid request format. Expected: /live/username/password/stream_id.ts');
}

$username = $parts[0];
$password = $parts[1];
$stream_file = $parts[2];

// Extract stream_id from filename (e.g., "123.ts" -> 123)
$stream_id = (int)preg_replace('/\.[^.]+$/', '', $stream_file);

if (!$stream_id) {
    header('HTTP/1.1 400 Bad Request');
    die('Invalid stream ID');
}

// Load config
require_once __DIR__ . '/../includes/Config.php';

try {
    $pdo = new PDO(
        "mysql:host={$config['db']['hostname']};dbname={$config['db']['database']};charset=utf8",
        $config['db']['username'],
        $config['db']['password']
    );
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Verify user
    $stmt = $pdo->prepare("SELECT id, username, exp_date FROM users WHERE username = ? AND password = ? LIMIT 1");
    $stmt->execute([$username, $password]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        header('HTTP/1.1 403 Forbidden');
        die('Authentication failed');
    }

    // Check if expired
    if ($user['exp_date'] && $user['exp_date'] < time()) {
        header('HTTP/1.1 403 Forbidden');
        die('Account expired');
    }

    // Get stream info
    $stmt = $pdo->prepare("SELECT stream_source, stream_display_name, direct_source, status FROM streams WHERE id = ? AND type = 1 LIMIT 1");
    $stmt->execute([$stream_id]);
    $stream = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$stream) {
        header('HTTP/1.1 404 Not Found');
        die('Stream not found');
    }

    if ($stream['status'] != 1) {
        header('HTTP/1.1 503 Service Unavailable');
        die('Stream is disabled');
    }

    // Parse stream sources
    $sources = json_decode($stream['stream_source'], true);
    if (!$sources || !is_array($sources)) {
        header('HTTP/1.1 500 Internal Server Error');
        die('No stream sources available');
    }

    // Function to check if URL is accessible (quick check)
    function quickCheckUrl($url, $timeout = 2) {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_NOBODY => true,  // HEAD request only
            CURLOPT_TIMEOUT => $timeout,
            CURLOPT_CONNECTTIMEOUT => $timeout,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Linux; Android 12) ExoPlayerLib/2.11.8'
        ]);

        curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        // Accept 200-399 codes (including redirects)
        return ($http_code >= 200 && $http_code < 400);
    }

    // Try each source until we find a working one
    $working_url = null;
    $tested_sources = [];

    foreach ($sources as $source_url) {
        if (empty($source_url)) continue;

        // Clean URL
        $source_url = stripslashes($source_url);

        // Skip duplicates
        if (in_array($source_url, $tested_sources)) continue;
        $tested_sources[] = $source_url;

        // Quick check if URL is accessible
        if (quickCheckUrl($source_url, 2)) {
            $working_url = $source_url;
            break;
        }
    }

    // If no working URL found after quick check, use the first one anyway
    // (it might work even if the HEAD request failed)
    if (!$working_url && !empty($sources)) {
        $working_url = stripslashes($sources[0]);
    }

    if (!$working_url) {
        header('HTTP/1.1 503 Service Unavailable');
        die('No working stream sources available');
    }

    // Redirect to the working stream
    header('Location: ' . $working_url);
    exit;

} catch (Exception $e) {
    header('HTTP/1.1 500 Internal Server Error');
    die('Error: ' . $e->getMessage());
}
