#!/usr/bin/php
<?php
// Codec and Bitrate sync daemon V2 - Runs every 3 seconds
// Fetches codec and bitrate from running ffmpeg processes via parsing command line and progress files
// Updates local streams_sys table with stream_info and bitrate

require_once('/var/www/html/iptv/includes/Config.php');

// Remote connection details
$remote_host = "204.188.233.170";
$remote_user = "root";
$remote_pass = "R7nMi4KCzZv920pG";

// Local connection
$local_host = $config['db']['hostname'];
$local_user = $config['db']['username'];
$local_pass = $config['db']['password'];
$local_db = $config['db']['database'];

// Log file
$logfile = '/var/log/sync_codec_daemon.log';

function logMsg($msg) {
    global $logfile;
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logfile, "[$timestamp] $msg\n", FILE_APPEND);
}

logMsg("=== Codec/Bitrate Sync Daemon V2 Started ===");
logMsg("Remote: $remote_host");
logMsg("Check interval: 3 seconds");

// Main loop
$iteration = 0;
while(true) {
    $iteration++;
    $start_time = microtime(true);

    try {
        // Connect to local database
        $local_conn = @new mysqli($local_host, $local_user, $local_pass, $local_db);
        if($local_conn->connect_error) {
            logMsg("ERROR: Local DB connection failed: " . $local_conn->connect_error);
            sleep(3);
            continue;
        }

        // Get full ffmpeg process list from remote
        $ffmpeg_cmd = "sshpass -p " . escapeshellarg($remote_pass) . " ssh -o StrictHostKeyChecking=no " .
                      escapeshellarg($remote_user) . "@" . escapeshellarg($remote_host) .
                      " \"ps -eo pid,args | grep 'ffmpeg.*streams/[0-9]*_' | grep -v grep | grep -v sudo\" 2>&1";

        $ffmpeg_result = shell_exec($ffmpeg_cmd);

        if(empty($ffmpeg_result)) {
            $local_conn->close();
            sleep(3);
            continue;
        }

        $ffmpeg_lines = explode("\n", trim($ffmpeg_result));
        $updated = 0;
        $errors = 0;

        // Clear codec info for redirect streams (they shouldn't have codec data)
        $local_conn->query("UPDATE streams_sys sys
                           JOIN streams s ON sys.stream_id = s.id
                           SET sys.stream_info = NULL, sys.bitrate = NULL
                           WHERE s.redirect_stream = 1 AND (sys.stream_info IS NOT NULL OR sys.bitrate IS NOT NULL)");

        foreach($ffmpeg_lines as $line) {
            if(empty($line)) continue;

            // Extract stream ID and PID
            if(!preg_match('/^\s*(\d+)\s+.*streams\/(\d+)_/', $line, $matches)) {
                continue;
            }

            $pid = intval($matches[1]);
            $stream_id = intval($matches[2]);

            // Skip redirect streams - they don't need codec info
            $redirect_check = $local_conn->query("SELECT redirect_stream FROM streams WHERE id = $stream_id");
            if($redirect_check && $redirect_check->num_rows > 0) {
                $redirect_row = $redirect_check->fetch_assoc();
                if($redirect_row['redirect_stream'] == 1) {
                    continue; // Skip this stream, it's a redirect
                }
            }

            // Parse codec info from ffmpeg command line
            $stream_info = [];

            // Check for video codec
            if(preg_match('/-vcodec\s+(\S+)/', $line, $vcodec_match)) {
                $vcodec = $vcodec_match[1];
                if($vcodec != 'copy') {
                    $stream_info['video_codec'] = $vcodec;
                    $stream_info['video_transcoded'] = true;
                } else {
                    // Try to get source codec from input
                    $stream_info['video_codec'] = 'copy';
                    $stream_info['video_transcoded'] = false;
                }
            }

            // Check for audio codec
            if(preg_match('/-acodec\s+(\S+)/', $line, $acodec_match)) {
                $acodec = $acodec_match[1];
                if($acodec != 'copy') {
                    $stream_info['audio_codec'] = $acodec;
                    $stream_info['audio_transcoded'] = true;
                } else {
                    $stream_info['audio_codec'] = 'copy';
                    $stream_info['audio_transcoded'] = false;
                }
            }

            // Get resolution from command line if present
            if(preg_match('/-s\s+(\d+)x(\d+)/', $line, $size_match)) {
                $stream_info['width'] = intval($size_match[1]);
                $stream_info['height'] = intval($size_match[2]);
            }

            // Get FPS if present
            if(preg_match('/-r\s+(\d+)/', $line, $fps_match)) {
                $stream_info['fps'] = intval($fps_match[1]);
            }

            // Read progress file for bitrate info
            $progress_file = "/home/midnightstreamer/iptv_midnight_streamer/streams/{$stream_id}_1_.progress";
            $progress_cmd = "sshpass -p " . escapeshellarg($remote_pass) . " ssh -o StrictHostKeyChecking=no " .
                           escapeshellarg($remote_user) . "@" . escapeshellarg($remote_host) .
                           " \"cat $progress_file 2>/dev/null\" 2>&1";

            $progress_result = shell_exec($progress_cmd);
            $bitrate_kbps = 0;

            if(!empty($progress_result) && strpos($progress_result, 'No such file') === false) {
                // Parse progress file
                if(preg_match('/bitrate=([\\d.]+)/', $progress_result, $br_match)) {
                    $br_value = $br_match[1];
                    if($br_value != 'N/A' && is_numeric($br_value)) {
                        $bitrate_kbps = round(floatval($br_value));
                    }
                }

                if(preg_match('/fps=([\\d.]+)/', $progress_result, $fps_match)) {
                    $stream_info['current_fps'] = round(floatval($fps_match[1]), 2);
                }

                if(preg_match('/speed=([\\d.]+)x/', $progress_result, $speed_match)) {
                    $stream_info['speed'] = $speed_match[1] . 'x';
                }

                if(preg_match('/frame=(\\d+)/', $progress_result, $frame_match)) {
                    $stream_info['frames'] = intval($frame_match[1]);
                }
            }

            // Calculate bitrate from HLS segment files (most accurate for copy streams)
            if($bitrate_kbps == 0) {
                // Get average size of recent .ts segments
                $segment_cmd = "sshpass -p " . escapeshellarg($remote_pass) . " ssh -o StrictHostKeyChecking=no " .
                              escapeshellarg($remote_user) . "@" . escapeshellarg($remote_host) .
                              " \"ls -l /home/midnightstreamer/iptv_midnight_streamer/streams/{$stream_id}_1_*.ts 2>/dev/null | tail -5 | awk '{sum+=\\\$5; count++} END {if(count>0) print sum/count}'\" 2>&1";

                $avg_segment_size = trim(shell_exec($segment_cmd));

                if(!empty($avg_segment_size) && is_numeric($avg_segment_size) && $avg_segment_size > 0) {
                    // Segment duration is 10 seconds (from hls_time 10)
                    // Bitrate (kbps) = (bytes * 8 bits/byte) / (10 seconds * 1000)
                    $bitrate_kbps = round(($avg_segment_size * 8) / (10 * 1000));
                    $stream_info['bitrate_source'] = 'calculated_from_segments';
                }
            }

            // Always probe HLS output to get actual codec names (even if we have bitrate from segments)
            $m3u8_file = "/home/midnightstreamer/iptv_midnight_streamer/streams/{$stream_id}_1_.m3u8";
            $ffprobe_cmd = "sshpass -p " . escapeshellarg($remote_pass) . " ssh -o StrictHostKeyChecking=no " .
                          escapeshellarg($remote_user) . "@" . escapeshellarg($remote_host) .
                          " \"timeout 5 /home/midnightstreamer/iptv_midnight_streamer/bin/ffprobe -v quiet -print_format json -show_format -show_streams $m3u8_file 2>/dev/null\" 2>&1";

            $ffprobe_result = shell_exec($ffprobe_cmd);

            if(!empty($ffprobe_result)) {
                $probe_data = json_decode($ffprobe_result, true);

                // Get bitrate from format if we don't have it yet
                if($bitrate_kbps == 0 && isset($probe_data['format']['bit_rate']) && intval($probe_data['format']['bit_rate']) > 0) {
                    $bitrate_kbps = round(intval($probe_data['format']['bit_rate']) / 1000);
                }

                // Try to get bitrate from individual streams if still don't have it
                if($bitrate_kbps == 0 && isset($probe_data['streams'])) {
                    $total_bitrate = 0;
                    foreach($probe_data['streams'] as $stream) {
                        if(isset($stream['bit_rate']) && intval($stream['bit_rate']) > 0) {
                            $total_bitrate += intval($stream['bit_rate']);
                        }
                    }
                    if($total_bitrate > 0) {
                        $bitrate_kbps = round($total_bitrate / 1000);
                    }
                }

                // Get codec info from output (ALWAYS do this for copy streams)
                if(isset($probe_data['streams'])) {
                        foreach($probe_data['streams'] as $stream) {
                            if($stream['codec_type'] == 'video') {
                                if($stream_info['video_codec'] == 'copy') {
                                    $stream_info['video_codec'] = $stream['codec_name'];
                                }
                                if(!isset($stream_info['width']) || $stream_info['width'] == 0) {
                                    $stream_info['width'] = isset($stream['width']) ? $stream['width'] : 0;
                                    $stream_info['height'] = isset($stream['height']) ? $stream['height'] : 0;
                                }
                                if(isset($stream['r_frame_rate'])) {
                                    $stream_info['fps'] = $stream['r_frame_rate'];
                                }
                            } elseif($stream['codec_type'] == 'audio') {
                                if($stream_info['audio_codec'] == 'copy') {
                                    $stream_info['audio_codec'] = $stream['codec_name'];
                                }
                                if(isset($stream['sample_rate'])) {
                                    $stream_info['audio_sample_rate'] = intval($stream['sample_rate']);
                                }
                                if(isset($stream['channels'])) {
                                    $stream_info['audio_channels'] = intval($stream['channels']);
                                }
                            }
                        }
                }
            }

            // If still no bitrate, probe the source URL (last resort)
            if($bitrate_kbps == 0) {
                // Get source URL from local database
                $url_result = $local_conn->query("SELECT stream_source FROM streams WHERE id = $stream_id");
                if($url_result && $url_result->num_rows > 0) {
                    $url_row = $url_result->fetch_assoc();
                    $source_urls = json_decode($url_row['stream_source'], true);

                    if(is_array($source_urls) && count($source_urls) > 0) {
                        $source_url = $source_urls[0];

                        // Probe source URL
                        $source_probe_cmd = "sshpass -p " . escapeshellarg($remote_pass) . " ssh -o StrictHostKeyChecking=no " .
                                            escapeshellarg($remote_user) . "@" . escapeshellarg($remote_host) .
                                            " \"timeout 10 /home/midnightstreamer/iptv_midnight_streamer/bin/ffprobe -v quiet -print_format json -show_format " .
                                            escapeshellarg($source_url) . " 2>/dev/null\" 2>&1";

                        $source_probe_result = shell_exec($source_probe_cmd);

                        if(!empty($source_probe_result)) {
                            $source_probe_data = json_decode($source_probe_result, true);
                            if(isset($source_probe_data['format']['bit_rate']) && intval($source_probe_data['format']['bit_rate']) > 0) {
                                $bitrate_kbps = round(intval($source_probe_data['format']['bit_rate']) / 1000);
                                $stream_info['bitrate_source'] = 'probed_from_source';
                            }
                        }
                    }
                }
            }

            // Convert to expected format for web interface
            $formatted_info = [
                'codecs' => [
                    'video' => [
                        'codec_name' => isset($stream_info['video_codec']) ? $stream_info['video_codec'] : '-',
                        'width' => isset($stream_info['width']) ? $stream_info['width'] : 0,
                        'height' => isset($stream_info['height']) ? $stream_info['height'] : 0,
                        'start_time' => 0 // Not available from our method
                    ],
                    'audio' => [
                        'codec_name' => isset($stream_info['audio_codec']) ? $stream_info['audio_codec'] : '-'
                    ]
                ],
                'speed' => isset($stream_info['speed']) ? $stream_info['speed'] : '1.0x',
                'fps' => isset($stream_info['current_fps']) ? $stream_info['current_fps'] : 0,
                'frames' => isset($stream_info['frames']) ? $stream_info['frames'] : 0
            ];

            // Store codec info as JSON
            $stream_info_json = json_encode($formatted_info);
            $stream_info_escaped = $local_conn->real_escape_string($stream_info_json);

            // Update streams_sys
            $update_sql = "UPDATE streams_sys SET
                          stream_info = '$stream_info_escaped'";

            if($bitrate_kbps > 0) {
                $update_sql .= ", bitrate = $bitrate_kbps";
            }

            $update_sql .= " WHERE stream_id = $stream_id AND server_id = 1";

            if($local_conn->query($update_sql)) {
                $updated++;
            } else {
                logMsg("ERROR updating stream $stream_id: " . $local_conn->error);
                $errors++;
            }
        }

        // Log summary every 20 iterations (every minute)
        if($iteration % 20 == 0) {
            logMsg("Iteration $iteration - Updated: $updated, Errors: $errors");
        }

        $local_conn->close();

    } catch(Exception $e) {
        logMsg("ERROR: " . $e->getMessage());
    }

    // Sleep for 3 seconds
    $elapsed = microtime(true) - $start_time;
    $sleep_time = max(0, 3 - $elapsed);
    usleep($sleep_time * 1000000);
}
?>
