<?php
/**
 * Sync EPG Data from Source Server
 * Downloads XMLTV EPG and updates mapping
 */

set_time_limit(0);
ini_set('memory_limit', '512M');

require_once __DIR__ . '/includes/Config.php';

echo "Starting EPG Sync...\n";
echo str_repeat("=", 60) . "\n";

try {
    // Connect to local database
    $pdo = new PDO(
        "mysql:host={$config['db']['hostname']};dbname={$config['db']['database']};charset=utf8",
        $config['db']['username'],
        $config['db']['password']
    );
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Get panel settings
    $stmt = $pdo->query("SELECT setting_key, setting_value FROM panel_settings");
    $settings = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }

    $panel_url = $settings['panel_url'] ?? '';
    $source_user = $settings['panel_username'] ?? '';
    $source_pass = $settings['panel_password'] ?? '';

    if (empty($panel_url)) {
        die("Error: Panel URL not configured\n");
    }

    $parsed = parse_url($panel_url);
    $host = $parsed['host'];
    $port = $parsed['port'] ?? '80';
    $scheme = $parsed['scheme'] ?? 'http';

    echo "Source Server: $scheme://$host:$port\n";
    echo str_repeat("-", 60) . "\n";

    // Step 1: Download XMLTV
    echo "\n[1/2] Downloading XMLTV EPG...\n";
    $xmltv_url = "$scheme://$host:$port/xmltv.php?username=" . urlencode($source_user)
               . "&password=" . urlencode($source_pass);

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $xmltv_url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 300,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1
    ]);

    echo "  Fetching from: $xmltv_url\n";
    $xml = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);

    if ($http_code != 200 || empty($xml)) {
        die("  Error: Failed to download XMLTV (HTTP $http_code). $error\n");
    }

    // Validate XML
    if (strpos($xml, '<?xml') === false) {
        die("  Error: Invalid XML response\n");
    }

    // Save XML
    $cache_dir = __DIR__ . '/cache';
    if (!is_dir($cache_dir)) {
        mkdir($cache_dir, 0755, true);
    }

    $xml_file = $cache_dir . '/epg.xml';
    file_put_contents($xml_file, $xml);

    $size_mb = number_format(strlen($xml) / 1024 / 1024, 2);
    echo "  ✓ Downloaded and saved: $size_mb MB\n";
    echo "  ✓ File: $xml_file\n";

    // Count channels
    preg_match_all('/<channel id="([^"]*)"/', $xml, $channels);
    $channel_count = count($channels[1]);
    echo "  ✓ Found $channel_count channels\n";

    // Count programmes
    preg_match_all('/<programme/', $xml, $programmes);
    $programme_count = count($programmes[0]);
    echo "  ✓ Found $programme_count programmes\n";

    // Step 2: Build EPG Mapping
    echo "\n[2/2] Building EPG Mapping from database...\n";

    $command = "sshpass -p 'R7nMi4KCzZv920pG' ssh -o StrictHostKeyChecking=no root@204.188.233.170 " .
               "'mysql midnight_iptv -N -e \"SELECT id, epg_channel_id FROM streams WHERE epg_channel_id IS NOT NULL AND epg_channel_id != \\\"\\\";\"'";

    $output = shell_exec($command);

    if (empty($output)) {
        die("  Error: Failed to fetch mapping from database\n");
    }

    $mapping = [];
    $lines = explode("\n", trim($output));
    foreach ($lines as $line) {
        $parts = explode("\t", $line);
        if (count($parts) == 2) {
            $mapping[$parts[0]] = $parts[1];
        }
    }

    $mapping_file = $cache_dir . '/epg_mapping.json';
    file_put_contents($mapping_file, json_encode($mapping, JSON_UNESCAPED_UNICODE));

    echo "  ✓ Built mapping: " . count($mapping) . " streams\n";
    echo "  ✓ File: $mapping_file\n";

    // Summary
    echo "\n" . str_repeat("=", 60) . "\n";
    echo "✓ EPG Sync Completed Successfully!\n";
    echo str_repeat("=", 60) . "\n";
    echo "\nStatistics:\n";
    echo "  - Channels: $channel_count\n";
    echo "  - Programmes: $programme_count\n";
    echo "  - Stream Mappings: " . count($mapping) . "\n";
    echo "  - XML Size: $size_mb MB\n";
    echo "  - Cached at: " . date('Y-m-d H:i:s') . "\n";

    // Show sample
    echo "\nSample Mappings:\n";
    $sample = array_slice($mapping, 0, 5, true);
    foreach ($sample as $stream_id => $tvg_id) {
        echo "  - Stream $stream_id => $tvg_id\n";
    }

    echo "\n✓ Ready to use!\n";

} catch (Exception $e) {
    echo "\nError: " . $e->getMessage() . "\n";
    echo $e->getTraceAsString() . "\n";
    exit(1);
}
?>
