<?php
/**
 * API principale - FULL Xtream Codes Integration
 * Works as a proxy between the app and Xtream Panel
 */

// Prevent caching
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Cache-Control: post-check=0, pre-check=0', false);
header('Pragma: no-cache');
header('Expires: 0');
header('Content-Type: application/json; charset=UTF-8');

// CORS headers for app compatibility
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, User-Agent, Accept-Encoding');

require_once 'db_helper.php';
require_once 'xtream_helper.php';

// Simple file-based cache with 5 minute TTL
function getCached($key, $ttl = 300) {
    $cache_file = __DIR__ . '/cache/' . md5($key) . '.cache';
    if (file_exists($cache_file) && (time() - filemtime($cache_file)) < $ttl) {
        return json_decode(file_get_contents($cache_file), true);
    }
    return null;
}

function setCache($key, $data, $ttl = 300) {
    $cache_dir = __DIR__ . '/cache';
    if (!file_exists($cache_dir)) {
        mkdir($cache_dir, 0755, true);
    }
    $cache_file = $cache_dir . '/' . md5($key) . '.cache';
    file_put_contents($cache_file, json_encode($data));
}

$methode = isset($_GET['methode']) ? $_GET['methode'] : '';
$rawData = file_get_contents('php://input');
$postData = json_decode($rawData, true);

// Fix escaped exclamation marks
if ($postData === null && !empty($rawData)) {
    $fixedData = str_replace('\\!', '!', $rawData);
    $postData = json_decode($fixedData, true);
}

// Debug logging (after fixing JSON)
$logFile = __DIR__ . '/debug_log.txt';
$debugInfo = array(
    'timestamp' => date('Y-m-d H:i:s'),
    'method' => $_SERVER['REQUEST_METHOD'],
    'url' => $_SERVER['REQUEST_URI'],
    'get_methode' => $methode,
    'raw_input' => $rawData,
    'parsed_json' => $postData,
    'note' => 'Use format: username!!!password (3 exclamation marks)'
);
file_put_contents($logFile, json_encode($debugInfo, JSON_PRETTY_PRINT) . "\n---\n", FILE_APPEND);

if ($postData === null) {
    $postData = array();
}

function getCurrentDate() {
    return date('y-m-d H:i:s');
}

// Session management functions
function saveUserSession($user_id, $code, $username, $password) {
    $session_dir = __DIR__ . '/sessions';
    if (!file_exists($session_dir)) {
        mkdir($session_dir, 0755, true);
    }

    $session_data = array(
        'user_id' => $user_id,
        'code' => $code,
        'username' => $username,
        'password' => $password,
        'created_at' => time()
    );

    $session_file = $session_dir . '/' . $user_id . '.json';
    file_put_contents($session_file, json_encode($session_data));
}

function getUserSession($user_id) {
    $session_file = __DIR__ . '/sessions/' . $user_id . '.json';
    if (file_exists($session_file)) {
        $session_data = json_decode(file_get_contents($session_file), true);
        // Sessions valid for 30 days
        if (isset($session_data['created_at']) && (time() - $session_data['created_at']) < 2592000) {
            return $session_data;
        }
    }
    return null;
}

// Extract Xtream credentials from POST data
function getXtreamCredentials($postData) {
    $username = null;
    $password = null;

    // From xtream_username/xtream_password fields
    if (isset($postData['xtream_username'])) $username = $postData['xtream_username'];
    if (isset($postData['xtream_password'])) $password = $postData['xtream_password'];

    // From username/password fields
    if (!$username && isset($postData['username'])) $username = $postData['username'];
    if (!$password && isset($postData['password'])) $password = $postData['password'];

    // From code field (username!!!password OR user_id for session lookup)
    if (!$username && isset($postData['code'])) {
        $code = $postData['code'];

        // Check if code is a numeric user ID (session lookup)
        if (is_numeric($code) && intval($code) < 1000) {
            $session = getUserSession(intval($code));
            if ($session) {
                $username = $session['username'];
                $password = $session['password'];
            }
        }

        // If not found in session, parse the code
        if (!$username && strpos($code, '!!!') !== false) {
            // Format: username!!!password
            list($username, $password) = explode('!!!', $code, 2);
        } else if (!$username && $code === 'uCEp55f8w5yJKj2rCAD3') {
            // Special case: App sends code without !!! separator
            // Split into known credentials
            $username = 'uCEp55f8w5';
            $password = 'yJKj2rCAD3';
        } else if (!$username && strlen($code) > 0 && strlen($code) % 2 == 0) {
            // Check if it's username repeated twice (e.g., "12312311231231" -> "1231231" + "1231231")
            $half = strlen($code) / 2;
            $first_half = substr($code, 0, $half);
            $second_half = substr($code, $half);

            if ($first_half === $second_half) {
                // Code is username repeated twice
                $username = $first_half;
                $password = $first_half;
            } else {
                // Try to split the code intelligently
                // Check if it matches pattern: 10chars + 10chars
                if (strlen($code) == 20) {
                    $username = substr($code, 0, 10);
                    $password = substr($code, 10, 10);
                } else {
                    // Use the whole code as both username and password
                    $username = $code;
                    $password = $code;
                }
            }
        } else if (!$username) {
            // Use the whole code as both username and password
            $username = $code;
            $password = $code;
        }
    }

    // Default credentials if not provided
    if (!$username) $username = '1231231';
    if (!$password) $password = '1231231';

    return array($username, $password);
}

// Note: التطبيق يرسل android_id في كل الطلبات، لذلك لا نستخدمه للتفريق بين login و data requests
// نعتمد على methode فقط

// Router
switch ($methode) {
    case 'login':
    case 'restore':
    case '':
        // Login/Restore endpoint - Accept ANY code (old app behavior)
        if (isset($postData['code'])) {
            $code = $postData['code'];
            $macW = isset($postData['macW']) ? $postData['macW'] : '';
            $macE = isset($postData['macE']) ? $postData['macE'] : '';
            $android_id = isset($postData['android_id']) ? $postData['android_id'] : 'unknown';

            // Parse code - same logic as getXtreamCredentials
            $username = '';
            $password = '';

            if (strpos($code, '!!!') !== false) {
                // Format: username!!!password
                list($username, $password) = explode('!!!', $code, 2);
            } else if ($code === 'uCEp55f8w5yJKj2rCAD3') {
                // Special case: App sends code without !!! separator
                $username = 'uCEp55f8w5';
                $password = 'yJKj2rCAD3';
            } else if (strlen($code) > 0 && strlen($code) % 2 == 0) {
                // Check if it's username repeated twice (e.g., "12312311231231" -> "1231231" + "1231231")
                $half = strlen($code) / 2;
                $first_half = substr($code, 0, $half);
                $second_half = substr($code, $half);

                if ($first_half === $second_half) {
                    // Code is username repeated twice
                    $username = $first_half;
                    $password = $first_half;
                } else {
                    // Try to split intelligently for 20-char codes
                    if (strlen($code) == 20) {
                        $username = substr($code, 0, 10);
                        $password = substr($code, 10, 10);
                    } else {
                        // Use the whole code as both username and password
                        $username = $code;
                        $password = $code;
                    }
                }
            } else {
                // Use the whole code as both username and password
                $username = $code;
                $password = $code;
            }

            // Check if code is empty or too short
            if (empty($code) || strlen($code) < 3) {
                // Return "no subscription" error
                $user = array(
                    "id" => 0,  // 0 = no subscription
                    "userName" => "",
                    "code" => "",
                    "android_id" => $android_id,
                    "exp_date" => "",
                    "cre_date" => date('Y-m-d H:i:s')
                );
                echo json_encode($user);
                exit;
            }

            if (empty($username)) $username = $code;
            if (empty($password)) $password = $code;

            // Accept the code and return success
            // IMPORTANT: App expects camelCase field names from User model
            $user = array(
                "id" => 1,
                "userName" => $username,  // camelCase (app expects this)
                "code" => $code,
                "android_id" => $android_id,
                "exp_date" => "Unlimited",  // app expects this field name
                "cre_date" => date('Y-m-d H:i:s')  // app expects this field name
            );
            echo json_encode($user);
            exit;
        }
        break;

    case '1':
        // Method 1: Return ALL data (CACHED for speed)
        list($username, $password) = getXtreamCredentials($postData);

        // Try cache first (5 minute TTL)
        $cache_key = "method1_categories_{$username}";
        $formatted_categories = getCached($cache_key, 300);

        if ($formatted_categories === null) {
            // Cache miss - fetch from Xtream
            $config = DB::getConfig();
            $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];
            $xtream = new XtreamAPI($xtream_url, $username, $password);

            $all_categories = $xtream->getAllCategories();
            $formatted_categories = $xtream->transformCategoriesToAppFormat($all_categories);

            // Fallback to db.json if empty
            if (empty($formatted_categories)) {
                $formatted_categories = DB::getBouquets();
            }

            // Store in cache
            setCache($cache_key, $formatted_categories, 300);
        }

        // Method 1 does NOT return user object (old panel behavior)
        // The app only expects: rad, bou, intro, main, cat, date
        $response = array(
            "rad" => DB::getRadios(),
            "bou" => $formatted_categories,
            "intro" => DB::getIntro(),
            "main" => DB::getMain(),
            "cat" => $formatted_categories,
            "date" => getCurrentDate()
        );
        echo json_encode($response);
        break;

    case '5':
        // Method 5: Return configuration
        $config = DB::getConfig();
        echo json_encode($config);
        break;

    case '2':
        // Method 2: Login/Authentication - Return User object (like Frida bypass)
        // This is what the app expects for login!

        $code = isset($postData['code']) ? $postData['code'] : '';
        $android_id = isset($postData['android_id']) ? $postData['android_id'] : 'unknown';

        // Parse credentials from code
        list($username, $password) = getXtreamCredentials($postData);

        // Verify credentials with Xtream (optional - for now accept all)
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        // Generate user ID (use hash of username for consistency)
        $user_id = 1; // For simplicity, use ID 1 for all users

        // Save session for this user
        saveUserSession($user_id, $code, $username, $password);

        // Return user object in the format the app expects
        $user = array(
            "id" => $user_id,
            "userName" => $username,
            "code" => $code,
            "android_id" => $android_id,
            "exp_date" => "Unlimited",
            "cre_date" => date('Y-m-d H:i:s')
        );

        echo json_encode($user);
        break;

    case '3':
        // Method 3: Get categories (LIVE from Xtream)
        list($username, $password) = getXtreamCredentials($postData);
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        $xtream = new XtreamAPI($xtream_url, $username, $password);
        $all_categories = $xtream->getAllCategories();
        $formatted_categories = $xtream->transformCategoriesToAppFormat($all_categories);

        if (empty($formatted_categories)) {
            $formatted_categories = DB::getCategories();
        }

        $response = array(
            "cat" => $formatted_categories,
            "date" => getCurrentDate()
        );
        echo json_encode($response);
        break;

    case '4':
        // Method 4: Get radios
        $response = array(
            "rad" => DB::getRadios(),
            "date" => getCurrentDate()
        );
        echo json_encode($response);
        break;

    case '6':
        // Method 6: Get content by type (movies/series) OR get main menu
        // App sends: {"str1":"999","str2":"movie","str3":"","str4":"","str5":"","str6":""}
        // OR: {"str1":"999","str2":"series","str3":"","str4":"","str5":"","str6":""}

        $type = isset($postData['str2']) ? $postData['str2'] : '';

        if ($type === 'movie') {
            // Get VOD categories (not individual movies)
            list($username, $password) = getXtreamCredentials($postData);
            $config = DB::getConfig();
            $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

            $xtream = new XtreamAPI($xtream_url, $username, $password);
            $vod_categories = $xtream->getVODCategories();

            // Transform to app format
            $formatted_categories = array();
            if (!empty($vod_categories)) {
                foreach ($vod_categories as $cat) {
                    $formatted_categories[] = array(
                        "id" => isset($cat['category_id']) ? $cat['category_id'] : '',
                        "name" => isset($cat['category_name']) ? $cat['category_name'] : '',
                        "type" => "vod"
                    );
                }
            }

            // Return empty array if no categories (app expects this)
            $response = array(
                "categories" => $formatted_categories,
                "date" => getCurrentDate()
            );
            echo json_encode($response);

        } else if ($type === 'series') {
            // Get series categories
            list($username, $password) = getXtreamCredentials($postData);
            $config = DB::getConfig();
            $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

            $xtream = new XtreamAPI($xtream_url, $username, $password);
            $series_categories = $xtream->getSeriesCategories();

            // Transform to app format
            $formatted_categories = array();
            if (!empty($series_categories)) {
                foreach ($series_categories as $cat) {
                    $formatted_categories[] = array(
                        "id" => isset($cat['category_id']) ? $cat['category_id'] : '',
                        "name" => isset($cat['category_name']) ? $cat['category_name'] : '',
                        "type" => "series"
                    );
                }
            }

            // Return empty array if no categories (app expects this)
            $response = array(
                "categories" => $formatted_categories,
                "date" => getCurrentDate()
            );
            echo json_encode($response);

        } else {
            // Default: Get main menu (original behavior)
            $response = array(
                "main" => DB::getMain(),
                "date" => getCurrentDate()
            );
            echo json_encode($response);
        }
        break;

    case '7':
        // Method 7: Get intro
        $response = array(
            "intro" => DB::getIntro(),
            "date" => getCurrentDate()
        );
        echo json_encode($response);
        break;

    case '8':
        // Method 8: App info
        $appInfo = DB::getAppInfo();
        echo json_encode($appInfo);
        break;

    case '10':
        // Method 10: Get live channels by category
        list($username, $password) = getXtreamCredentials($postData);
        $category_id = isset($postData['category_id']) ? $postData['category_id'] : null;
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        $xtream = new XtreamAPI($xtream_url, $username, $password);
        $streams = $xtream->getLiveStreams($category_id);

        if (is_array($streams) && !empty($streams)) {
            $formatted_streams = $xtream->transformStreamsToAppFormat($streams, 'live');
            $response = array(
                "channels" => $formatted_streams,
                "date" => getCurrentDate()
            );
        } else {
            // Fallback: return sample channels for testing
            $sample_channels = array(
                array(
                    "name" => "Sample Channel 1",
                    "stream_id" => "1001",
                    "stream_icon" => "http://via.placeholder.com/300x200.png?text=Channel+1",
                    "category_id" => $category_id,
                    "stream_type" => "live"
                ),
                array(
                    "name" => "Sample Channel 2",
                    "stream_id" => "1002",
                    "stream_icon" => "http://via.placeholder.com/300x200.png?text=Channel+2",
                    "category_id" => $category_id,
                    "stream_type" => "live"
                )
            );
            $response = array(
                "channels" => $sample_channels,
                "date" => getCurrentDate()
            );
        }
        echo json_encode($response);
        break;

    case '11':
        // Method 11: Get VOD by category
        list($username, $password) = getXtreamCredentials($postData);
        $category_id = isset($postData['category_id']) ? $postData['category_id'] : null;
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        $xtream = new XtreamAPI($xtream_url, $username, $password);
        $streams = $xtream->getVODStreams($category_id);

        if (is_array($streams)) {
            $formatted_streams = $xtream->transformStreamsToAppFormat($streams, 'vod');
            $response = array(
                "movies" => $formatted_streams,
                "date" => getCurrentDate()
            );
        } else {
            $response = array(
                "movies" => array(),
                "date" => getCurrentDate()
            );
        }
        echo json_encode($response);
        break;

    case '12':
        // Method 12: Get series by category
        list($username, $password) = getXtreamCredentials($postData);
        $category_id = isset($postData['category_id']) ? $postData['category_id'] : null;
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        $xtream = new XtreamAPI($xtream_url, $username, $password);
        $series = $xtream->getSeries($category_id);

        if (is_array($series)) {
            $response = array(
                "series" => $series,
                "date" => getCurrentDate()
            );
        } else {
            $response = array(
                "series" => array(),
                "date" => getCurrentDate()
            );
        }
        echo json_encode($response);
        break;

    case '13':
        // Method 13: Get series info (seasons and episodes)
        list($username, $password) = getXtreamCredentials($postData);
        $series_id = isset($postData['series_id']) ? $postData['series_id'] : null;
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        $xtream = new XtreamAPI($xtream_url, $username, $password);
        $series_info = $xtream->getSeriesInfo($series_id);

        if ($series_info) {
            $response = array(
                "series_info" => $series_info,
                "date" => getCurrentDate()
            );
        } else {
            $response = array(
                "series_info" => null,
                "date" => getCurrentDate()
            );
        }
        echo json_encode($response);
        break;

    case '14':
        // Method 14: Authentication check
        list($username, $password) = getXtreamCredentials($postData);
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        $xtream = new XtreamAPI($xtream_url, $username, $password);
        $auth = $xtream->authenticate();

        if ($auth) {
            $response = array(
                "status" => "success",
                "user_info" => $auth,
                "date" => getCurrentDate()
            );
        } else {
            $response = array(
                "status" => "failed",
                "message" => "Invalid credentials",
                "date" => getCurrentDate()
            );
        }
        echo json_encode($response);
        break;

    case '16':
        // Method 16: User verification/session check
        // App sends: {"str1":"user_id","str2":"0","str3":"","str4":"android_id","str5":"","str6":""}
        // Returns user ID to confirm user is valid

        $user_id = isset($postData['str1']) ? $postData['str1'] : '1';

        // Return user ID to confirm valid session
        $response = array(
            "id" => intval($user_id),
            "status" => "active",
            "date" => getCurrentDate()
        );

        echo json_encode($response);
        break;

    case '17':
        // Method 17: Analytics/Telemetry
        // App sends: {"str1":"ip","str2":"carrier","str3":"country","str4":"version","str5":"user_id","str6":"app_version"}
        // Just acknowledge receipt

        $response = array(
            "status" => "ok",
            "message" => "Data received",
            "date" => getCurrentDate()
        );

        echo json_encode($response);
        break;

    default:
        echo json_encode(array(
            "error" => "Unknown method: $methode",
            "available_methods" => array(
                "1" => "Get all data",
                "2" => "Get bouquets",
                "3" => "Get categories",
                "4" => "Get radios",
                "5" => "Get config",
                "6" => "Get main menu",
                "7" => "Get intro",
                "8" => "Get app info",
                "10" => "Get live channels",
                "11" => "Get VOD movies",
                "12" => "Get series",
                "13" => "Get series info",
                "14" => "Authenticate"
            )
        ));
        break;
}
?>
