<?php
/**
 * ═══════════════════════════════════════════════════════════════════
 * 🔥 DIDON PANEL - Complete API Implementation
 * ═══════════════════════════════════════════════════════════════════
 * Based on complete reverse engineering with Frida
 * Supports all discovered methods (2, 3, 6, 8, 13, 16, 17, 19)
 * ═══════════════════════════════════════════════════════════════════
 */

// Prevent caching
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Cache-Control: post-check=0, pre-check=0', false);
header('Pragma: no-cache');
header('Expires: 0');
header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');
header('Access-Control-Allow-Headers: Content-Type');

// Database configuration
$db_host = 'localhost';
$db_name = 'didon_panel';
$db_user = 'root';
$db_pass = '';

try {
    $pdo = new PDO("mysql:host=$db_host;dbname=$db_name;charset=utf8mb4", $db_user, $db_pass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    http_response_code(500);
    die(json_encode(['error' => 'Database connection failed']));
}

// Get methode parameter
$methode = isset($_GET['methode']) ? intval($_GET['methode']) : 0;

// Get POST data
$input = json_decode(file_get_contents('php://input'), true);
if (!$input) {
    $input = [];
}

// Optional logging
$log_enabled = false; // Set to true for debugging
if ($log_enabled) {
    $log_dir = __DIR__ . '/logs';
    if (!is_dir($log_dir)) {
        mkdir($log_dir, 0755, true);
    }

    $log_file = $log_dir . '/api_' . date('Y-m-d') . '.log';
    $log_entry = date('[Y-m-d H:i:s]') . " Method: $methode\n";
    $log_entry .= "IP: " . ($_SERVER['REMOTE_ADDR'] ?? 'unknown') . "\n";
    $log_entry .= "User-Agent: " . ($_SERVER['HTTP_USER_AGENT'] ?? 'unknown') . "\n";
    $log_entry .= "Input: " . json_encode($input, JSON_UNESCAPED_UNICODE) . "\n";
    $log_entry .= str_repeat('-', 70) . "\n\n";
    file_put_contents($log_file, $log_entry, FILE_APPEND);
}

// Route to appropriate handler
switch($methode) {
    case 1:
        handleMethod1($pdo, $input);
        break;

    case 2:
        handleMethod2($pdo, $input);
        break;

    case 3:
        handleMethod3($pdo, $input);
        break;

    case 5:
        handleMethod5($pdo, $input);
        break;

    case 6:
        handleMethod6($pdo, $input);
        break;

    case 8:
        handleMethod8($pdo, $input);
        break;

    case 10:
        handleMethod10($pdo, $input);
        break;

    case 13:
        handleMethod13($pdo, $input);
        break;

    case 16:
        handleMethod16($pdo, $input);
        break;

    case 17:
        handleMethod17($pdo, $input);
        break;

    case 19:
        handleMethod19($pdo, $input);
        break;

    default:
        http_response_code(400);
        echo json_encode(['error' => 'Invalid methode']);
}

/**
 * ═══════════════════════════════════════════════════════════════════
 * METHOD 1: Get Initial Data
 * ═══════════════════════════════════════════════════════════════════
 */
function handleMethod1($pdo, $input) {
    try {
        // Get radio stations
        $radStmt = $pdo->query("
            SELECT id, name, logo, stream_url as url
            FROM channels
            WHERE type = 'radio' AND is_active = 1
            ORDER BY name
        ");
        $rad = $radStmt->fetchAll();

        // Get bouquets
        $bouStmt = $pdo->query("
            SELECT id, name, description, icon
            FROM bouquets
            WHERE is_active = 1
            ORDER BY display_order
        ");
        $bou = $bouStmt->fetchAll();

        // Get intro slides
        $introStmt = $pdo->query("
            SELECT id, title, image_url, video_url, type
            FROM intro_slides
            WHERE is_active = 1
            ORDER BY display_order
        ");
        $intro = $introStmt->fetchAll();

        // Get main banners
        $mainStmt = $pdo->query("
            SELECT id, title, image_url, link_type, link_value
            FROM main_banners
            WHERE is_active = 1
            ORDER BY display_order
        ");
        $main = $mainStmt->fetchAll();

        // Get categories
        $catStmt = $pdo->query("
            SELECT id, name, name_ar, icon, type
            FROM categories
            WHERE is_active = 1
            ORDER BY display_order
        ");
        $cat = $catStmt->fetchAll();

        // Build response
        $response = [
            'rad' => $rad,
            'bou' => $bou,
            'intro' => $intro,
            'main' => $main,
            'cat' => $cat,
            'date' => date('y-m-d H:i:s')
        ];

        echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
}

/**
 * ═══════════════════════════════════════════════════════════════════
 * METHOD 2: User Login
 * ═══════════════════════════════════════════════════════════════════
 */
function handleMethod2($pdo, $input) {
    // Check for str1-str6 format (Panel API)
    if (isset($input['str1']) && empty($input['str1'])) {
        // Empty str1-str6 = testing/checking endpoint
        echo json_encode([
            'id' => 999,
            'userName' => 'TestUser',
            'code' => 'TEST999',
            'android_id' => 'test_device',
            'exp_date' => '2099-12-31',
            'cre_date' => date('Y-m-d H:i:s')
        ]);
        return;
    }

    // Standard User API format
    $code = $input['code'] ?? '';
    $android_id = $input['android_id'] ?? '';
    $macE = $input['macE'] ?? '';
    $macW = $input['macW'] ?? '';

    try {
        // Validate activation code
        $stmt = $pdo->prepare("
            SELECT *
            FROM users
            WHERE code = :code
            AND status = 'active'
            AND (expiry_date IS NULL OR expiry_date >= CURDATE())
        ");
        $stmt->execute(['code' => $code]);
        $user = $stmt->fetch();

        if ($user) {
            // Update user information
            $updateStmt = $pdo->prepare("
                UPDATE users
                SET android_id = :android_id,
                    mac_ethernet = :macE,
                    mac_wifi = :macW,
                    last_login = NOW(),
                    login_count = login_count + 1,
                    last_ip = :ip
                WHERE id = :id
            ");
            $updateStmt->execute([
                'android_id' => $android_id,
                'macE' => $macE,
                'macW' => $macW,
                'ip' => $_SERVER['REMOTE_ADDR'] ?? '',
                'id' => $user['id']
            ]);

            // Log successful login
            logLogin($pdo, $user['id'], $android_id, $code, true);

            // Return success response
            echo json_encode([
                'id' => (int)$user['id'],
                'userName' => $user['username'],
                'code' => $user['code'],
                'android_id' => $android_id,
                'exp_date' => $user['expiry_date'],
                'cre_date' => $user['created_date']
            ]);

        } else {
            // Log failed login
            logLogin($pdo, null, $android_id, $code, false);

            // Return failed response
            echo json_encode([
                'id' => 0,
                'userName' => '',
                'code' => '',
                'android_id' => '',
                'exp_date' => null,
                'cre_date' => null
            ]);
        }

    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
}

/**
 * ═══════════════════════════════════════════════════════════════════
 * METHOD 3: Get Bouquets/Packages
 * ═══════════════════════════════════════════════════════════════════
 * Request formats:
 * 1. {"code":"999","android_id":"","macE":"","macW":""}
 * 2. {"str1":"2","str2":"","str3":"","str4":"","str5":"","str6":""}
 * 3. {"str1":"96,189,190,192","str2":"999","str3":"","str4":"","str5":"","str6":""}
 */
function handleMethod3($pdo, $input) {
    $user_code = $input['code'] ?? '';
    $category_filter = $input['str1'] ?? '';
    $user_id = $input['str2'] ?? '';

    try {
        if ($user_code) {
            // Format 1: Get all bouquets for user
            $stmt = $pdo->query("
                SELECT id, name, description, icon,
                       (SELECT COUNT(*) FROM channels WHERE category_id = categories.id) as channels_count
                FROM categories
                WHERE is_active = 1
                ORDER BY display_order
            ");
            $bouquets = $stmt->fetchAll();

        } else if (strpos($category_filter, ',') !== false) {
            // Format 3: Multiple category IDs
            $ids = explode(',', $category_filter);
            $placeholders = implode(',', array_fill(0, count($ids), '?'));

            $stmt = $pdo->prepare("
                SELECT id, name, description, icon
                FROM categories
                WHERE id IN ($placeholders) AND is_active = 1
                ORDER BY display_order
            ");
            $stmt->execute($ids);
            $bouquets = $stmt->fetchAll();

        } else if ($category_filter) {
            // Format 2: Single category
            $stmt = $pdo->prepare("
                SELECT id, name, description, icon
                FROM categories
                WHERE id = :cat_id AND is_active = 1
            ");
            $stmt->execute(['cat_id' => $category_filter]);
            $bouquets = $stmt->fetchAll();

        } else {
            // Get all
            $stmt = $pdo->query("
                SELECT id, name, description, icon
                FROM categories
                WHERE is_active = 1
                ORDER BY display_order
            ");
            $bouquets = $stmt->fetchAll();
        }

        echo json_encode($bouquets, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
}

/**
 * ═══════════════════════════════════════════════════════════════════
 * METHOD 5: Get Server Configuration
 * ═══════════════════════════════════════════════════════════════════
 */
function handleMethod5($pdo, $input) {
    try {
        // Get Xtream configuration from database (first active user as default)
        $stmt = $pdo->query("
            SELECT xtream_url, xtream_username, xtream_password
            FROM users
            WHERE status = 'active' AND xtream_url IS NOT NULL
            LIMIT 1
        ");
        $xtream = $stmt->fetch();

        // Parse Xtream URL to extract host and port
        $xtream_host = 'flix-panel.xyz:2087'; // Default
        $xtream_port = '2087';
        $xtream_url = 'http://flix-panel.xyz:2087';

        if ($xtream && $xtream['xtream_url']) {
            $xtream_url = $xtream['xtream_url'];
            $parsed = parse_url($xtream_url);
            $xtream_host = $parsed['host'];
            if (isset($parsed['port'])) {
                $xtream_host .= ':' . $parsed['port'];
                $xtream_port = (string)$parsed['port'];
            } else {
                $xtream_port = ($parsed['scheme'] === 'https') ? '443' : '80';
            }
        }

        $config = [
            'str0' => 'API.php?methode=',
            'str1' => $xtream_url,                    // Xtream URL (http://flix-panel.xyz:2087)
            'str2' => '/user/',
            'str3' => 'http://' . $_SERVER['HTTP_HOST'] . '/xmltv.php?username=!1!&password=!2!',  // XMLTV URL
            'str4' => $xtream_host,                   // Host Xtream (flix-panel.xyz:2087)
            'str5' => $xtream_port,                   // Port Xtream (2087)
            'str6' => '',
            'str7' => '/PanelAndroid/API.php?methode=',
            'str8' => 'http://' . $_SERVER['HTTP_HOST'],  // Base URL API
            'str9' => '/Flag',
            'str10' => '/Intro',
            'str11' => '/Main',
            'str12' => '/PanelAndroid/media',
            'str13' => ':' . ($_SERVER['SERVER_PORT'] ?? '80')
        ];

        echo json_encode($config, JSON_UNESCAPED_SLASHES);

    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
}

/**
 * ═══════════════════════════════════════════════════════════════════
 * METHOD 6: Get VOD Content (Movies/Series)
 * ═══════════════════════════════════════════════════════════════════
 * Request: {"str1":"999","str2":"movie","str3":"","str4":"","str5":"","str6":""}
 * str2 can be: "movie" or "series"
 */
function handleMethod6($pdo, $input) {
    $user_id = $input['str1'] ?? '';
    $content_type = $input['str2'] ?? 'movie';

    try {
        // Check if user has Xtream credentials
        $userStmt = $pdo->prepare("
            SELECT xtream_url, xtream_username, xtream_password
            FROM users
            WHERE id = :user_id AND status = 'active'
        ");
        $userStmt->execute(['user_id' => $user_id]);
        $user = $userStmt->fetch();

        if ($user && $user['xtream_url']) {
            // Fetch from Xtream Codes panel
            $result = fetchXtreamVOD(
                $user['xtream_url'],
                $user['xtream_username'],
                $user['xtream_password'],
                $content_type
            );
            echo json_encode($result, JSON_UNESCAPED_SLASHES);
        } else {
            // Fallback: Get from local database
            $type_db = ($content_type === 'series') ? 'series' : 'vod';

            $stmt = $pdo->prepare("
                SELECT
                    id,
                    name,
                    stream_id,
                    poster as stream_icon,
                    rating,
                    category_id,
                    container_extension,
                    stream_url as direct_source
                FROM vod_content
                WHERE type = :type AND is_active = 1
                ORDER BY name
            ");
            $stmt->execute(['type' => $type_db]);
            $content = $stmt->fetchAll();

            echo json_encode($content, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        }

    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
}

/**
 * ═══════════════════════════════════════════════════════════════════
 * METHOD 8: Get Content Details
 * ═══════════════════════════════════════════════════════════════════
 * Request: {"str1":"111","str2":"2","str3":"999","str4":"","str5":"","str6":""}
 */
function handleMethod8($pdo, $input) {
    $content_id = $input['str1'] ?? '';
    $type = $input['str2'] ?? '';
    $user_id = $input['str3'] ?? '';

    try {
        // Get content details
        $stmt = $pdo->prepare("
            SELECT
                stream_id,
                name,
                description,
                director,
                cast,
                genre,
                release_date,
                rating,
                duration,
                poster as cover,
                backdrop,
                stream_url,
                trailer_url,
                container_extension
            FROM vod_content
            WHERE stream_id = :content_id
        ");
        $stmt->execute(['content_id' => $content_id]);
        $content = $stmt->fetch();

        if ($content) {
            $response = [
                'movie_data' => $content
            ];
            echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        } else {
            echo json_encode(['error' => 'Content not found']);
        }

    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
}

/**
 * ═══════════════════════════════════════════════════════════════════
 * METHOD 10: Get Channels by Category
 * ═══════════════════════════════════════════════════════════════════
 */
function handleMethod10($pdo, $input) {
    $category_id = $input['category_id'] ?? 0;
    $android_id = $input['android_id'] ?? '';

    try {
        $stmt = $pdo->prepare("
            SELECT id, name, logo, stream_url, epg_id, category_id
            FROM channels
            WHERE category_id = :cat_id
            AND is_active = 1
            ORDER BY name
        ");
        $stmt->execute(['cat_id' => $category_id]);
        $channels = $stmt->fetchAll();

        echo json_encode($channels, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
}

/**
 * ═══════════════════════════════════════════════════════════════════
 * METHOD 13: Get User Favorites
 * ═══════════════════════════════════════════════════════════════════
 */
function handleMethod13($pdo, $input) {
    $user_code = $input['code'] ?? '';

    try {
        // Get user ID from code
        $stmt = $pdo->prepare("SELECT id FROM users WHERE code = :code");
        $stmt->execute(['code' => $user_code]);
        $user = $stmt->fetch();

        if (!$user) {
            echo json_encode([]);
            return;
        }

        // Get favorites
        $stmt = $pdo->prepare("
            SELECT
                f.content_type,
                f.content_id,
                f.added_at,
                CASE
                    WHEN f.content_type = 'channel' THEN c.name
                    WHEN f.content_type IN ('vod', 'series') THEN v.name
                END as name,
                CASE
                    WHEN f.content_type = 'channel' THEN c.logo
                    WHEN f.content_type IN ('vod', 'series') THEN v.poster
                END as icon
            FROM user_favorites f
            LEFT JOIN channels c ON f.content_type = 'channel' AND f.content_id = c.id
            LEFT JOIN vod_content v ON f.content_type IN ('vod', 'series') AND f.content_id = v.id
            WHERE f.user_id = :user_id
            ORDER BY f.added_at DESC
        ");
        $stmt->execute(['user_id' => $user['id']]);
        $favorites = $stmt->fetchAll();

        echo json_encode($favorites, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
}

/**
 * ═══════════════════════════════════════════════════════════════════
 * METHOD 16: User Session Verification
 * ═══════════════════════════════════════════════════════════════════
 * CRITICAL: Must return positive ID for valid users, -1 for expired
 */
function handleMethod16($pdo, $input) {
    $user_id = $input['str1'] ?? '';
    $flag = $input['str2'] ?? '0';
    $android_id = $input['str4'] ?? '';

    try {
        if ($user_id) {
            $stmt = $pdo->prepare("
                SELECT id, status, expiry_date
                FROM users
                WHERE id = :user_id
                AND android_id = :android_id
                AND status = 'active'
                AND (expiry_date IS NULL OR expiry_date >= CURDATE())
            ");
            $stmt->execute([
                'user_id' => $user_id,
                'android_id' => $android_id
            ]);
            $user = $stmt->fetch();

            if ($user) {
                echo json_encode(['id' => (int)$user['id']]);
            } else {
                echo json_encode(['id' => -1]);
            }
        } else {
            echo json_encode(['id' => -1]);
        }

    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
}

/**
 * ═══════════════════════════════════════════════════════════════════
 * METHOD 17: Network/Location Telemetry
 * ═══════════════════════════════════════════════════════════════════
 */
function handleMethod17($pdo, $input) {
    $ip_address = $input['str1'] ?? '';
    $carrier = $input['str2'] ?? '';
    $country = $input['str3'] ?? '';
    $latitude = $input['str4'] ?? '';
    $user_id = $input['str5'] ?? '';
    $app_version = $input['str6'] ?? '';

    try {
        $stmt = $pdo->prepare("
            INSERT INTO network_logs (
                user_id, ip_address, carrier, country,
                latitude, longitude, app_version,
                logged_at
            ) VALUES (
                :user_id, :ip, :carrier, :country,
                :lat, '', :app_version,
                NOW()
            )
        ");
        $stmt->execute([
            'user_id' => $user_id ?: null,
            'ip' => $ip_address,
            'carrier' => $carrier,
            'country' => $country,
            'lat' => $latitude,
            'app_version' => $app_version
        ]);

        echo json_encode([
            'status' => 'success',
            'logged' => true
        ]);

    } catch (Exception $e) {
        // Non-critical - return success anyway
        echo json_encode([
            'status' => 'success',
            'logged' => false
        ]);
    }
}

/**
 * ═══════════════════════════════════════════════════════════════════
 * METHOD 19: Get Channels from Multiple Categories
 * ═══════════════════════════════════════════════════════════════════
 */
function handleMethod19($pdo, $input) {
    $category_ids = $input['str1'] ?? '';
    $user_id = $input['str2'] ?? '';

    try {
        if (strpos($category_ids, ',') !== false) {
            $ids = explode(',', $category_ids);
            $placeholders = implode(',', array_fill(0, count($ids), '?'));

            $stmt = $pdo->prepare("
                SELECT id, name, logo, stream_url, epg_id, category_id
                FROM channels
                WHERE category_id IN ($placeholders)
                AND is_active = 1
                ORDER BY category_id, name
            ");
            $stmt->execute($ids);
            $channels = $stmt->fetchAll();

            echo json_encode($channels, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        } else {
            echo json_encode([]);
        }

    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
}

/**
 * ═══════════════════════════════════════════════════════════════════
 * HELPER: Fetch VOD from Xtream Codes
 * ═══════════════════════════════════════════════════════════════════
 */
function fetchXtreamVOD($url, $username, $password, $type) {
    $url = rtrim($url, '/');

    $api_url = "$url/player_api.php?username=$username&password=$password&action=get_vod_streams";

    if ($type === 'series') {
        $api_url = "$url/player_api.php?username=$username&password=$password&action=get_series";
    }

    $response = @file_get_contents($api_url);

    if ($response) {
        $data = json_decode($response, true);

        $result = [];
        foreach ($data as $item) {
            $stream_id = $item['stream_id'] ?? $item['series_id'] ?? 0;
            $extension = $item['container_extension'] ?? 'mp4';

            $result[] = [
                'id' => $stream_id,
                'name' => $item['name'] ?? '',
                'stream_id' => $stream_id,
                'stream_icon' => $item['stream_icon'] ?? '',
                'rating' => $item['rating'] ?? '0',
                'category_id' => $item['category_id'] ?? '',
                'container_extension' => $extension,
                'custom_sid' => '',
                'direct_source' => "$url/movie/$username/$password/$stream_id.$extension"
            ];
        }

        return $result;
    }

    return [];
}

/**
 * ═══════════════════════════════════════════════════════════════════
 * HELPER: Log Login Attempts
 * ═══════════════════════════════════════════════════════════════════
 */
function logLogin($pdo, $user_id, $android_id, $code, $success) {
    try {
        $stmt = $pdo->prepare("
            INSERT INTO login_logs (user_id, android_id, code_used, ip_address, user_agent, success, login_time)
            VALUES (:user_id, :android_id, :code, :ip, :ua, :success, NOW())
        ");
        $stmt->execute([
            'user_id' => $user_id,
            'android_id' => $android_id,
            'code' => $code,
            'ip' => $_SERVER['REMOTE_ADDR'] ?? '',
            'ua' => $_SERVER['HTTP_USER_AGENT'] ?? '',
            'success' => $success ? 1 : 0
        ]);
    } catch (Exception $e) {
        // Ignore logging errors
    }
}
?>
