# 🔧 Method 5: Server Configuration - التفصيل الصحيح

## 📋 Request Format

```json
{
  "str1": "",
  "str2": "",
  "str3": "",
  "str4": "",
  "str5": "",
  "str6": ""
}
```

**Note:** Method 5 usually doesn't require input parameters (no authentication).

---

## ✅ Response Format (CORRECT)

```json
{
  "str0": "API.php?methode=",
  "str1": "http://flix-panel.xyz:2087",       // ✅ Xtream URL
  "str2": "/user/",
  "str3": "http://your-server/xmltv.php?username=!1!&password=!2!",  // ✅ XMLTV URL
  "str4": "flix-panel.xyz:2087",              // ✅ Host Xtream (مع البورت)
  "str5": "2087",                              // ✅ Port Xtream
  "str6": "",
  "str7": "/PanelAndroid/API.php?methode=",
  "str8": "http://your-server",                // ✅ Base URL API
  "str9": "/Flag",
  "str10": "/Intro",
  "str11": "/Main",
  "str12": "/PanelAndroid/media",
  "str13": ":80"
}
```

---

## 🔍 Field Mapping (CORRECTED)

| Field | Purpose | Example | Notes |
|-------|---------|---------|-------|
| **str0** | API Method Prefix | `"API.php?methode="` | Used for building API URLs |
| **str1** | **Xtream URL** | `"http://flix-panel.xyz:2087"` | ✅ Full Xtream Codes server URL |
| **str2** | User Path | `"/user/"` | Path for user-related endpoints |
| **str3** | **XMLTV URL** | `"http://server/xmltv.php?username=!1!&password=!2!"` | ✅ EPG/XMLTV endpoint |
| **str4** | **Host Xtream** | `"flix-panel.xyz:2087"` | ✅ Xtream host with port |
| **str5** | **Port Xtream** | `"2087"` | ✅ Xtream port number |
| **str6** | Reserved | `""` | Empty/unused |
| **str7** | API Path | `"/PanelAndroid/API.php?methode="` | Panel API path |
| **str8** | **Base URL API** | `"http://your-server"` | ✅ Main panel base URL |
| **str9** | Flag Path | `"/Flag"` | Country flag images path |
| **str10** | Intro Path | `"/Intro"` | Intro video/images path |
| **str11** | Main Path | `"/Main"` | Main content path |
| **str12** | Media Path | `"/PanelAndroid/media"` | Media files base path |
| **str13** | API Port | `":80"` or `":443"` | Panel API port |

---

## 🎯 Key Points

### 1. Xtream Codes Integration

The app expects **3 separate Xtream fields**:

```
str1 = Full URL    → http://flix-panel.xyz:2087
str4 = Host:Port   → flix-panel.xyz:2087
str5 = Port only   → 2087
```

**Why separate?**
- `str1` - Used for full API calls to Xtream
- `str4` - Used for building custom stream URLs
- `str5` - Used for connection validation/port checking

### 2. URL Building

The app builds different types of URLs:

**For Xtream Codes:**
```
{str1}/player_api.php?username=X&password=Y
{str1}/live/username/password/stream_id.ext
```

**For Panel API:**
```
{str8}{str7}1  → http://your-server/PanelAndroid/API.php?methode=1
```

**For Media:**
```
{str8}{str12}{str9}/usa.png  → http://your-server/PanelAndroid/media/Flag/usa.png
```

### 3. XMLTV Format

```
str3: "http://your-server/xmltv.php?username=!1!&password=!2!"
```

The `!1!` and `!2!` are **placeholders** that the app replaces with:
- `!1!` → actual username
- `!2!` → actual password

---

## 💻 Implementation (API_NEW.php)

```php
function handleMethod5($pdo, $input) {
    // Get Xtream configuration from database
    $stmt = $pdo->query("
        SELECT xtream_url, xtream_username, xtream_password
        FROM users
        WHERE status = 'active' AND xtream_url IS NOT NULL
        LIMIT 1
    ");
    $xtream = $stmt->fetch();

    // Parse Xtream URL to extract host and port
    $xtream_host = 'flix-panel.xyz:2087'; // Default
    $xtream_port = '2087';
    $xtream_url = 'http://flix-panel.xyz:2087';

    if ($xtream && $xtream['xtream_url']) {
        $xtream_url = $xtream['xtream_url'];
        $parsed = parse_url($xtream_url);

        // Extract host (without port)
        $xtream_host = $parsed['host'];

        // Extract port
        if (isset($parsed['port'])) {
            $xtream_host .= ':' . $parsed['port'];  // Add port to host
            $xtream_port = (string)$parsed['port'];
        } else {
            $xtream_port = ($parsed['scheme'] === 'https') ? '443' : '80';
        }
    }

    $config = [
        'str0' => 'API.php?methode=',
        'str1' => $xtream_url,                    // ✅ Xtream URL
        'str2' => '/user/',
        'str3' => 'http://' . $_SERVER['HTTP_HOST'] . '/xmltv.php?username=!1!&password=!2!',
        'str4' => $xtream_host,                   // ✅ Host Xtream
        'str5' => $xtream_port,                   // ✅ Port Xtream
        'str6' => '',
        'str7' => '/PanelAndroid/API.php?methode=',
        'str8' => 'http://' . $_SERVER['HTTP_HOST'],  // ✅ Base URL API
        'str9' => '/Flag',
        'str10' => '/Intro',
        'str11' => '/Main',
        'str12' => '/PanelAndroid/media',
        'str13' => ':' . ($_SERVER['SERVER_PORT'] ?? '80')
    ];

    echo json_encode($config, JSON_UNESCAPED_SLASHES);
}
```

---

## 🧪 Testing

### Test Request:

```bash
curl http://localhost/PanelAndroid/API.php?methode=5
```

### Expected Response:

```json
{
  "str0": "API.php?methode=",
  "str1": "http://flix-panel.xyz:2087",
  "str2": "/user/",
  "str3": "http://localhost/xmltv.php?username=!1!&password=!2!",
  "str4": "flix-panel.xyz:2087",
  "str5": "2087",
  "str6": "",
  "str7": "/PanelAndroid/API.php?methode=",
  "str8": "http://localhost",
  "str9": "/Flag",
  "str10": "/Intro",
  "str11": "/Main",
  "str12": "/PanelAndroid/media",
  "str13": ":80"
}
```

### Verify Xtream Fields:

```bash
curl http://localhost/PanelAndroid/API.php?methode=5 | jq '{str1, str4, str5, str8}'
```

Should show:
```json
{
  "str1": "http://flix-panel.xyz:2087",
  "str4": "flix-panel.xyz:2087",
  "str5": "2087",
  "str8": "http://localhost"
}
```

---

## 🔧 Configuration in Database

### Set Xtream URL:

```sql
UPDATE users
SET xtream_url = 'http://flix-panel.xyz:2087',
    xtream_username = '1231231',
    xtream_password = '1231231'
WHERE code = '123456';
```

### For HTTPS Xtream:

```sql
UPDATE users
SET xtream_url = 'https://flix-panel.xyz:2087'
WHERE code = '123456';
```

The implementation will automatically extract:
- `str1` = `https://flix-panel.xyz:2087`
- `str4` = `flix-panel.xyz:2087`
- `str5` = `2087`

---

## ❌ Common Mistakes (FIXED)

### ❌ WRONG:
```json
{
  "str1": "http://localhost",          // ❌ Panel URL instead of Xtream
  "str4": "http://localhost",          // ❌ Should be host:port only
  "str5": "80"                         // ❌ Panel port instead of Xtream
}
```

### ✅ CORRECT:
```json
{
  "str1": "http://flix-panel.xyz:2087",  // ✅ Xtream full URL
  "str4": "flix-panel.xyz:2087",         // ✅ Xtream host:port (no http://)
  "str5": "2087",                        // ✅ Xtream port
  "str8": "http://localhost"             // ✅ Panel base URL
}
```

---

## 📊 Flow Diagram

```
DIDON App
    │
    ├─ Calls Method 5 → Get Configuration
    │
    ├─ Receives:
    │   ├─ str1 (Xtream URL) ────────┐
    │   ├─ str4 (Xtream Host:Port) ──┼─→ For Xtream API calls
    │   ├─ str5 (Xtream Port) ───────┘
    │   │
    │   ├─ str8 (Panel Base URL) ────┬─→ For Panel API calls
    │   └─ str7 (Panel API Path) ────┘
    │
    ├─ Builds Xtream URLs:
    │   └─ {str1}/player_api.php?username=X&password=Y
    │
    └─ Builds Panel URLs:
        └─ {str8}{str7}6 → Method 6 for VOD
```

---

## ✅ Summary

**Corrected Field Mapping:**

| Old (WRONG) | New (CORRECT) | Field |
|-------------|---------------|-------|
| Panel URL | **Xtream URL** | str1 |
| XMLTV URL | **XMLTV URL** | str3 |
| Panel URL | **Host Xtream (with port)** | str4 |
| Panel Port | **Port Xtream** | str5 |
| Panel URL | **Base URL API** | str8 |

**Key Changes:**
- `str1` now returns Xtream URL, not panel URL
- `str4` now returns Xtream host:port, not panel URL
- `str5` now returns Xtream port, not panel port
- `str8` remains panel base URL

**Status:** ✅ Fixed in API_NEW.php
**Date:** 2025-11-19
**Version:** 2.1 (Corrected Xtream field mapping)
