#!/bin/bash
# ═══════════════════════════════════════════════════════════════════
# 🔥 DIDON Panel - System Status Checker
# ═══════════════════════════════════════════════════════════════════

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

echo -e "${CYAN}"
cat << "EOF"
╔═══════════════════════════════════════════════════════════════════╗
║                                                                   ║
║   🔥 DIDON PANEL - SYSTEM STATUS CHECK                           ║
║                                                                   ║
╚═══════════════════════════════════════════════════════════════════╝
EOF
echo -e "${NC}\n"

# Check files
echo -e "${BLUE}📁 Checking Installation Files...${NC}\n"

check_file() {
    local file=$1
    local description=$2

    if [ -f "$file" ]; then
        size=$(ls -lh "$file" | awk '{print $5}')
        echo -e "${GREEN}✅${NC} $description: ${CYAN}$size${NC}"
    else
        echo -e "${RED}❌${NC} $description: ${RED}NOT FOUND${NC}"
    fi
}

check_file "/var/www/html/PanelAndroid/API_NEW.php" "API_NEW.php (New implementation)"
check_file "/var/www/html/PanelAndroid/API.php" "API.php (Current)"
check_file "/var/www/html/PanelAndroid/update_database.sql" "Database update script"
check_file "/var/www/html/PanelAndroid/test_all_methods.sh" "Testing script"
check_file "/var/www/html/PanelAndroid/install.sh" "Installation script"
check_file "/var/www/html/PanelAndroid/README_FINAL.md" "Arabic documentation"
check_file "/var/www/html/PanelAndroid/COMPLETE_SETUP_GUIDE.md" "English documentation"

echo ""

# Check which API is active
echo -e "${BLUE}🔍 API Status...${NC}\n"

if [ -f "/var/www/html/PanelAndroid/API.php" ]; then
    api_size=$(ls -lh /var/www/html/PanelAndroid/API.php | awk '{print $5}')

    if grep -q "handleMethod16" /var/www/html/PanelAndroid/API.php 2>/dev/null; then
        echo -e "${GREEN}✅${NC} API.php contains NEW implementation (has handleMethod16)"
        echo -e "   Size: ${CYAN}${api_size}${NC}"
    else
        echo -e "${YELLOW}⚠️${NC}  API.php is OLD implementation (no handleMethod16)"
        echo -e "   Size: ${CYAN}${api_size}${NC}"
        echo -e "${YELLOW}   → Run install.sh or manually deploy API_NEW.php${NC}"
    fi
fi

echo ""

# Check MySQL/MariaDB
echo -e "${BLUE}🗄️  Database Status...${NC}\n"

if systemctl is-active --quiet mysql || systemctl is-active --quiet mariadb; then
    echo -e "${GREEN}✅${NC} MySQL/MariaDB is running"

    # Try to connect
    if mysql -e "SELECT 1;" 2>/dev/null >/dev/null; then
        echo -e "${GREEN}✅${NC} MySQL connection successful (no password)"

        # Check if didon_panel exists
        if mysql -e "USE didon_panel; SELECT 1;" 2>/dev/null >/dev/null; then
            echo -e "${GREEN}✅${NC} Database 'didon_panel' exists"

            # Count tables
            table_count=$(mysql didon_panel -e "SHOW TABLES;" 2>/dev/null | wc -l)
            echo -e "${GREEN}✅${NC} Tables found: ${CYAN}$((table_count-1))${NC}"
        else
            echo -e "${YELLOW}⚠️${NC}  Database 'didon_panel' not found"
            echo -e "${YELLOW}   → Run: mysql < update_database.sql${NC}"
        fi
    else
        echo -e "${YELLOW}⚠️${NC}  MySQL requires password"
        echo -e "${YELLOW}   → You'll need password to run install.sh${NC}"
    fi
else
    echo -e "${RED}❌${NC} MySQL/MariaDB is not running"
fi

echo ""

# Check Apache/Nginx
echo -e "${BLUE}🌐 Web Server Status...${NC}\n"

if systemctl is-active --quiet apache2; then
    echo -e "${GREEN}✅${NC} Apache is running"
elif systemctl is-active --quiet nginx; then
    echo -e "${GREEN}✅${NC} Nginx is running"
else
    echo -e "${RED}❌${NC} No web server detected"
fi

echo ""

# Check PHP
echo -e "${BLUE}🐘 PHP Status...${NC}\n"

if command -v php &> /dev/null; then
    php_version=$(php -v | head -n1)
    echo -e "${GREEN}✅${NC} PHP installed: ${CYAN}${php_version}${NC}"

    # Check PDO
    if php -m | grep -q PDO; then
        echo -e "${GREEN}✅${NC} PDO extension available"
    else
        echo -e "${RED}❌${NC} PDO extension missing (required for MySQL)"
    fi

    # Check pdo_mysql
    if php -m | grep -q pdo_mysql; then
        echo -e "${GREEN}✅${NC} PDO MySQL driver available"
    else
        echo -e "${RED}❌${NC} PDO MySQL driver missing"
    fi
else
    echo -e "${RED}❌${NC} PHP not found"
fi

echo ""

# Check Frida scripts
echo -e "${BLUE}📱 Frida Scripts...${NC}\n"

check_file "/var/www/html/didon_DISCOVER_FIXED.js" "Discovery script (with Method 16 bypass)"

echo ""

# Check logs directory
echo -e "${BLUE}📊 Logs...${NC}\n"

if [ -d "/var/www/html/PanelAndroid/logs" ]; then
    log_count=$(ls /var/www/html/PanelAndroid/logs/*.log 2>/dev/null | wc -l)
    echo -e "${GREEN}✅${NC} Logs directory exists"
    echo -e "   Log files: ${CYAN}${log_count}${NC}"
else
    echo -e "${YELLOW}⚠️${NC}  Logs directory not created yet"
    echo -e "${YELLOW}   → Will be created during installation${NC}"
fi

echo ""

# Test API endpoint
echo -e "${BLUE}🔌 API Endpoint Test...${NC}\n"

if command -v curl &> /dev/null; then
    response=$(curl -s -o /dev/null -w "%{http_code}" http://localhost/PanelAndroid/API.php?methode=5 2>/dev/null)

    if [ "$response" = "200" ]; then
        echo -e "${GREEN}✅${NC} API endpoint responding (HTTP 200)"
        echo -e "   Testing Method 5..."
        curl -s http://localhost/PanelAndroid/API.php?methode=5 | head -c 100
        echo -e "\n"
    elif [ "$response" = "000" ]; then
        echo -e "${RED}❌${NC} Cannot connect to API (check web server)"
    else
        echo -e "${YELLOW}⚠️${NC}  API returned HTTP ${response}"
    fi
else
    echo -e "${YELLOW}⚠️${NC}  curl not available for testing"
fi

echo ""

# Summary
echo -e "${CYAN}"
echo "╔═══════════════════════════════════════════════════════════════════╗"
echo "║                     📋 SUMMARY                                    ║"
echo "╚═══════════════════════════════════════════════════════════════════╝"
echo -e "${NC}\n"

# Count issues
issues=0

[ ! -f "/var/www/html/PanelAndroid/API_NEW.php" ] && ((issues++))
! systemctl is-active --quiet mysql && ! systemctl is-active --quiet mariadb && ((issues++))
! systemctl is-active --quiet apache2 && ! systemctl is-active --quiet nginx && ((issues++))

if [ $issues -eq 0 ]; then
    echo -e "${GREEN}✅ System is ready for installation!${NC}\n"

    echo -e "${YELLOW}📍 Next Steps:${NC}"
    echo -e "   1. Run: ${CYAN}cd /var/www/html/PanelAndroid${NC}"
    echo -e "   2. Run: ${CYAN}./install.sh${NC} (automatic)"
    echo -e "      OR follow manual steps in INSTALLATION_STATUS.md"
    echo ""
    echo -e "${YELLOW}📖 Documentation:${NC}"
    echo -e "   ${CYAN}INSTALLATION_STATUS.md${NC} - Current status & installation options"
    echo -e "   ${CYAN}README_FINAL.md${NC} - Complete Arabic guide"
    echo -e "   ${CYAN}COMPLETE_SETUP_GUIDE.md${NC} - English setup guide"
else
    echo -e "${RED}❌ Found ${issues} issue(s) - check details above${NC}\n"
fi

echo ""
echo -e "${BLUE}╔═══════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${BLUE}║  💡 Tip: Run ${CYAN}cat INSTALLATION_STATUS.md${BLUE} for detailed info     ║${NC}"
echo -e "${BLUE}╚═══════════════════════════════════════════════════════════════════╝${NC}"
echo ""
