#!/bin/bash
# ═══════════════════════════════════════════════════════════════════
# 🔥 DIDON API - Complete Method Testing Script
# ═══════════════════════════════════════════════════════════════════

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Configuration
SERVER="http://localhost"
API_URL="${SERVER}/PanelAndroid/API_NEW.php"

echo -e "${BLUE}"
echo "╔═══════════════════════════════════════════════════════════════════╗"
echo "║        🔥 DIDON API - COMPLETE METHOD TESTING                    ║"
echo "╚═══════════════════════════════════════════════════════════════════╝"
echo -e "${NC}\n"

# Test counter
PASSED=0
FAILED=0

# Function to test API endpoint
test_method() {
    local method_num=$1
    local description=$2
    local request_body=$3

    echo -e "${YELLOW}═══════════════════════════════════════════════════════════════════${NC}"
    echo -e "${BLUE}[TEST] Method ${method_num}: ${description}${NC}"
    echo -e "${YELLOW}═══════════════════════════════════════════════════════════════════${NC}"

    echo -e "📍 URL: ${API_URL}?methode=${method_num}"
    echo -e "📦 Request: ${request_body}\n"

    response=$(curl -s -w "\n%{http_code}" -X POST "${API_URL}?methode=${method_num}" \
        -H "Content-Type: application/json" \
        -d "${request_body}")

    http_code=$(echo "$response" | tail -n1)
    body=$(echo "$response" | sed '$d')

    if [ "$http_code" = "200" ]; then
        echo -e "${GREEN}✅ Status: ${http_code} OK${NC}"
        echo -e "📥 Response:\n${body}\n"

        # Pretty print JSON if possible
        if command -v python3 &> /dev/null; then
            echo -e "${GREEN}🔍 Parsed Response:${NC}"
            echo "$body" | python3 -m json.tool 2>/dev/null || echo "$body"
        fi

        PASSED=$((PASSED + 1))
    else
        echo -e "${RED}❌ Status: ${http_code} FAILED${NC}"
        echo -e "📥 Response:\n${body}\n"
        FAILED=$((FAILED + 1))
    fi

    echo ""
}

# ═══════════════════════════════════════════════════════════════════
# TEST METHOD 5: Get Server Configuration (No auth required)
# ═══════════════════════════════════════════════════════════════════
test_method "5" \
    "Get Server Configuration" \
    '{"str1":"","str2":"","str3":"","str4":"","str5":"","str6":""}'

# ═══════════════════════════════════════════════════════════════════
# TEST METHOD 2: User Login
# ═══════════════════════════════════════════════════════════════════
echo -e "${BLUE}[INFO] Testing with activation code: 123456${NC}\n"

test_method "2" \
    "User Login (Valid Code)" \
    '{"code":"123456","android_id":"test_device_123","macE":"","macW":"0"}'

test_method "2" \
    "User Login (Invalid Code)" \
    '{"code":"INVALID999","android_id":"test_device_123","macE":"","macW":"0"}'

test_method "2" \
    "User Login (Panel format - testing)" \
    '{"str1":"","str2":"","str3":"","str4":"","str5":"","str6":""}'

# ═══════════════════════════════════════════════════════════════════
# TEST METHOD 1: Get Initial Data
# ═══════════════════════════════════════════════════════════════════
test_method "1" \
    "Get Initial Data (Categories, Intro, etc.)" \
    '{"code":"123456","android_id":"test_device_123","macE":"","macW":"0"}'

# ═══════════════════════════════════════════════════════════════════
# TEST METHOD 3: Get Bouquets
# ═══════════════════════════════════════════════════════════════════
test_method "3" \
    "Get Bouquets (Format 1: User code)" \
    '{"code":"999","android_id":"","macE":"","macW":""}'

test_method "3" \
    "Get Bouquets (Format 2: Single category)" \
    '{"str1":"2","str2":"","str3":"","str4":"","str5":"","str6":""}'

test_method "3" \
    "Get Bouquets (Format 3: Multiple categories)" \
    '{"str1":"1,2,3","str2":"999","str3":"","str4":"","str5":"","str6":""}'

# ═══════════════════════════════════════════════════════════════════
# TEST METHOD 10: Get Channels by Category
# ═══════════════════════════════════════════════════════════════════
test_method "10" \
    "Get Channels for Category 1 (Live TV)" \
    '{"category_id":"1","android_id":"test_device_123","macE":"","macW":"0"}'

test_method "10" \
    "Get Channels for Category 2 (Movies)" \
    '{"category_id":"2","android_id":"test_device_123","macE":"","macW":"0"}'

# ═══════════════════════════════════════════════════════════════════
# TEST METHOD 6: Get VOD Content
# ═══════════════════════════════════════════════════════════════════
test_method "6" \
    "Get VOD Content (Movies)" \
    '{"str1":"999","str2":"movie","str3":"","str4":"","str5":"","str6":""}'

test_method "6" \
    "Get VOD Content (Series)" \
    '{"str1":"999","str2":"series","str3":"","str4":"","str5":"","str6":""}'

# ═══════════════════════════════════════════════════════════════════
# TEST METHOD 8: Get Content Details
# ═══════════════════════════════════════════════════════════════════
test_method "8" \
    "Get Content Details (Stream ID 1001)" \
    '{"str1":"1001","str2":"2","str3":"999","str4":"","str5":"","str6":""}'

# ═══════════════════════════════════════════════════════════════════
# TEST METHOD 16: User Session Verification
# ═══════════════════════════════════════════════════════════════════
test_method "16" \
    "User Verification (Valid)" \
    '{"str1":"999","str2":"0","str3":"","str4":"test_device_123","str5":"","str6":""}'

test_method "16" \
    "User Verification (Invalid)" \
    '{"str1":"9999","str2":"0","str3":"","str4":"invalid_device","str5":"","str6":""}'

# ═══════════════════════════════════════════════════════════════════
# TEST METHOD 17: Network Telemetry
# ═══════════════════════════════════════════════════════════════════
test_method "17" \
    "Submit Network/Location Info" \
    '{"str1":"192.168.1.100","str2":"WiFi","str3":"Algeria","str4":"36.7538","str5":"999","str6":"2.6"}'

# ═══════════════════════════════════════════════════════════════════
# TEST METHOD 13: Get Favorites
# ═══════════════════════════════════════════════════════════════════
test_method "13" \
    "Get User Favorites" \
    '{"code":"123456","android_id":"","macE":"","macW":""}'

# ═══════════════════════════════════════════════════════════════════
# TEST METHOD 19: Get Multiple Categories
# ═══════════════════════════════════════════════════════════════════
test_method "19" \
    "Get Channels from Multiple Categories" \
    '{"str1":"1,2,3","str2":"999","str3":"","str4":"","str5":"","str6":""}'

# ═══════════════════════════════════════════════════════════════════
# SUMMARY
# ═══════════════════════════════════════════════════════════════════
echo -e "\n${BLUE}"
echo "╔═══════════════════════════════════════════════════════════════════╗"
echo "║                     📊 TEST SUMMARY                               ║"
echo "╚═══════════════════════════════════════════════════════════════════╝"
echo -e "${NC}"

TOTAL=$((PASSED + FAILED))
echo -e "Total Tests: ${TOTAL}"
echo -e "${GREEN}Passed: ${PASSED}${NC}"
echo -e "${RED}Failed: ${FAILED}${NC}"

if [ $FAILED -eq 0 ]; then
    echo -e "\n${GREEN}✅ ALL TESTS PASSED!${NC}\n"
    exit 0
else
    echo -e "\n${RED}❌ SOME TESTS FAILED!${NC}\n"
    exit 1
fi
