-- ═══════════════════════════════════════════════════════════════════
-- 🔥 DIDON PANEL - Database Update for New Methods
-- ═══════════════════════════════════════════════════════════════════
-- Run this to add support for Methods 3, 6, 8, 13, 17, 19
-- ═══════════════════════════════════════════════════════════════════

USE didon_panel;

-- ═══════════════════════════════════════════════════════════════════
-- Add Xtream Codes credentials to users table
-- ═══════════════════════════════════════════════════════════════════
ALTER TABLE users
ADD COLUMN IF NOT EXISTS xtream_url VARCHAR(255) DEFAULT NULL AFTER last_ip,
ADD COLUMN IF NOT EXISTS xtream_username VARCHAR(100) DEFAULT NULL AFTER xtream_url,
ADD COLUMN IF NOT EXISTS xtream_password VARCHAR(100) DEFAULT NULL AFTER xtream_username;

-- Update test user with Xtream credentials
UPDATE users
SET xtream_url = 'https://flix-panel.xyz:2087',
    xtream_username = '1231231',
    xtream_password = '1231231'
WHERE code = '123456';

-- ═══════════════════════════════════════════════════════════════════
-- VOD Content Table
-- ═══════════════════════════════════════════════════════════════════
CREATE TABLE IF NOT EXISTS vod_content (
    id INT PRIMARY KEY AUTO_INCREMENT,
    stream_id INT UNIQUE NOT NULL,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    type ENUM('vod', 'series') NOT NULL,
    category_id INT,
    poster VARCHAR(500),
    backdrop VARCHAR(500),
    stream_url VARCHAR(1000),
    trailer_url VARCHAR(500),
    director VARCHAR(255),
    cast TEXT,
    genre VARCHAR(255),
    release_date DATE,
    rating VARCHAR(10),
    duration VARCHAR(50),
    container_extension VARCHAR(10) DEFAULT 'mp4',
    is_active BOOLEAN DEFAULT TRUE,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,

    INDEX idx_stream_id (stream_id),
    INDEX idx_type (type),
    INDEX idx_category (category_id),
    INDEX idx_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ═══════════════════════════════════════════════════════════════════
-- User Favorites Table (for Method 13)
-- ═══════════════════════════════════════════════════════════════════
CREATE TABLE IF NOT EXISTS user_favorites (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    content_type ENUM('channel', 'vod', 'series') NOT NULL,
    content_id INT NOT NULL,
    added_at DATETIME DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_favorite (user_id, content_type, content_id),
    INDEX idx_user (user_id),
    INDEX idx_content (content_type, content_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ═══════════════════════════════════════════════════════════════════
-- Network Logs Table (for Method 17) - Already exists, verify structure
-- ═══════════════════════════════════════════════════════════════════
CREATE TABLE IF NOT EXISTS network_logs (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id INT,
    ip_address VARCHAR(45),
    carrier VARCHAR(255),
    country VARCHAR(100),
    latitude VARCHAR(50),
    longitude VARCHAR(50),
    app_version VARCHAR(20),
    logged_at DATETIME DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user (user_id),
    INDEX idx_time (logged_at),
    INDEX idx_country (country)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ═══════════════════════════════════════════════════════════════════
-- Add bouquet relationship to channels
-- ═══════════════════════════════════════════════════════════════════
ALTER TABLE channels
ADD COLUMN IF NOT EXISTS bouquet_id INT DEFAULT NULL AFTER category_id,
ADD INDEX IF NOT EXISTS idx_bouquet (bouquet_id);

-- ═══════════════════════════════════════════════════════════════════
-- Sample VOD Data (for testing)
-- ═══════════════════════════════════════════════════════════════════
INSERT IGNORE INTO vod_content (stream_id, name, description, type, category_id, poster, rating, duration, container_extension) VALUES
(1001, 'Test Movie 1', 'Sample movie for testing', 'vod', 2, 'http://via.placeholder.com/300x400.png?text=Movie1', '7.5', '120 min', 'mp4'),
(1002, 'Test Movie 2', 'Another sample movie', 'vod', 2, 'http://via.placeholder.com/300x400.png?text=Movie2', '8.0', '135 min', 'mp4'),
(2001, 'Test Series 1', 'Sample series for testing', 'series', 3, 'http://via.placeholder.com/300x400.png?text=Series1', '8.5', '45 min/ep', 'mp4');

-- ═══════════════════════════════════════════════════════════════════
-- Verify Tables
-- ═══════════════════════════════════════════════════════════════════
SELECT '✅ Database updated successfully!' AS status;

SELECT 'Tables verification:' AS info;
SELECT
    TABLE_NAME,
    TABLE_ROWS
FROM information_schema.TABLES
WHERE TABLE_SCHEMA = 'didon_panel'
AND TABLE_NAME IN ('users', 'categories', 'channels', 'vod_content', 'user_favorites', 'network_logs')
ORDER BY TABLE_NAME;

SELECT 'Test user Xtream credentials:' AS info;
SELECT id, username, code, xtream_url, xtream_username
FROM users
WHERE code = '123456';
