<?php
/**
 * Xtream Codes API Helper
 * Handles all communication with Xtream Panel
 */

class XtreamAPI {
    private $base_url;
    private $username;
    private $password;

    public function __construct($base_url = null, $username = null, $password = null) {
        $config = DB::getConfig();
        $this->base_url = $base_url ?: (isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4']);
        $this->username = $username ?: (isset($config['xtream_username']) ? $config['xtream_username'] : '1231231');
        $this->password = $password ?: (isset($config['xtream_password']) ? $config['xtream_password'] : '1231231');
    }

    /**
     * Make HTTP request to Xtream API
     */
    private function makeRequest($endpoint) {
        $url = $this->base_url . $endpoint;

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);

        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($http_code != 200 || empty($response)) {
            error_log("Xtream API Error: HTTP $http_code - $error - URL: $url");
            return null;
        }

        $data = json_decode($response, true);
        return $data ?: null;
    }

    /**
     * Authenticate user
     */
    public function authenticate() {
        $endpoint = "/player_api.php?username={$this->username}&password={$this->password}";
        return $this->makeRequest($endpoint);
    }

    /**
     * Get live categories
     */
    public function getLiveCategories() {
        $endpoint = "/player_api.php?username={$this->username}&password={$this->password}&action=get_live_categories";
        return $this->makeRequest($endpoint);
    }

    /**
     * Get VOD categories
     */
    public function getVODCategories() {
        $endpoint = "/player_api.php?username={$this->username}&password={$this->password}&action=get_vod_categories";
        return $this->makeRequest($endpoint);
    }

    /**
     * Get series categories
     */
    public function getSeriesCategories() {
        $endpoint = "/player_api.php?username={$this->username}&password={$this->password}&action=get_series_categories";
        return $this->makeRequest($endpoint);
    }

    /**
     * Get all categories (live + vod + series)
     */
    public function getAllCategories() {
        $all_categories = array();

        $live = $this->getLiveCategories();
        if (is_array($live)) {
            foreach ($live as $cat) {
                $all_categories[] = array(
                    'id' => isset($cat['id']) ? $cat['id'] : (isset($cat['category_id']) ? $cat['category_id'] : ''),
                    'name' => isset($cat['name']) ? $cat['name'] : (isset($cat['category_name']) ? $cat['category_name'] : ''),
                    'type' => 'live'
                );
            }
        }

        $vod = $this->getVODCategories();
        if (is_array($vod)) {
            foreach ($vod as $cat) {
                $all_categories[] = array(
                    'id' => isset($cat['id']) ? $cat['id'] : (isset($cat['category_id']) ? $cat['category_id'] : ''),
                    'name' => isset($cat['name']) ? $cat['name'] : (isset($cat['category_name']) ? $cat['category_name'] : ''),
                    'type' => 'vod'
                );
            }
        }

        $series = $this->getSeriesCategories();
        if (is_array($series)) {
            foreach ($series as $cat) {
                $all_categories[] = array(
                    'id' => isset($cat['id']) ? $cat['id'] : (isset($cat['category_id']) ? $cat['category_id'] : ''),
                    'name' => isset($cat['name']) ? $cat['name'] : (isset($cat['category_name']) ? $cat['category_name'] : ''),
                    'type' => 'series'
                );
            }
        }

        return $all_categories;
    }

    /**
     * Get live streams
     */
    public function getLiveStreams($category_id = null) {
        if ($category_id) {
            $endpoint = "/player_api.php?username={$this->username}&password={$this->password}&action=get_live_streams&category_id={$category_id}";
        } else {
            $endpoint = "/player_api.php?username={$this->username}&password={$this->password}&action=get_live_streams";
        }
        return $this->makeRequest($endpoint);
    }

    /**
     * Get VOD streams
     */
    public function getVODStreams($category_id = null) {
        if ($category_id) {
            $endpoint = "/player_api.php?username={$this->username}&password={$this->password}&action=get_vod_streams&category_id={$category_id}";
        } else {
            $endpoint = "/player_api.php?username={$this->username}&password={$this->password}&action=get_vod_streams";
        }
        return $this->makeRequest($endpoint);
    }

    /**
     * Get series
     */
    public function getSeries($category_id = null) {
        if ($category_id) {
            $endpoint = "/api/player/{$this->username}/{$this->password}/series_cat/{$category_id}";
        } else {
            $endpoint = "/api/player/{$this->username}/{$this->password}/series_cat";
        }
        return $this->makeRequest($endpoint);
    }

    /**
     * Get series info (seasons and episodes)
     */
    public function getSeriesInfo($series_id) {
        $endpoint = "/api/player/{$this->username}/{$this->password}/series/seasons/{$series_id}";
        return $this->makeRequest($endpoint);
    }

    /**
     * Get VOD info
     */
    public function getVODInfo($vod_id) {
        $endpoint = "/api/player/{$this->username}/{$this->password}/vod/info/{$vod_id}";
        return $this->makeRequest($endpoint);
    }

    /**
     * Transform categories to app format
     */
    public function transformCategoriesToAppFormat($categories) {
        $result = array();

        foreach ($categories as $cat) {
            // Support both formats: Xtream API (category_name/category_id) and db.json (name/id)
            $cat_name = isset($cat['category_name']) ? $cat['category_name'] : (isset($cat['name']) ? $cat['name'] : '');
            $cat_id = isset($cat['category_id']) ? $cat['category_id'] : (isset($cat['id']) ? $cat['id'] : '');

            // Clean category name for image filename
            $safe_name = strtolower($cat_name);
            $safe_name = str_replace(array(' ', '/', '\\', '+', '&'), '_', $safe_name);
            $safe_name = preg_replace('/_+/', '_', $safe_name);
            $safe_name = preg_replace('/[^a-z0-9_-]/', '', $safe_name);
            $safe_name = trim($safe_name, '_');

            $result[] = array(
                "name" => $cat_id,
                "path" => $safe_name . ".png"
            );
        }

        return $result;
    }

    /**
     * Transform streams to app format (for channels list)
     */
    public function transformStreamsToAppFormat($streams, $type = 'live') {
        $result = array();

        foreach ($streams as $stream) {
            $stream_id = isset($stream['id']) ? $stream['id'] : (isset($stream['stream_id']) ? $stream['stream_id'] : '');
            $stream_name = isset($stream['name']) ? $stream['name'] : '';
            $category_id = isset($stream['category_id']) ? $stream['category_id'] : '';

            // Build stream URL
            $stream_url = '';
            if ($type == 'live') {
                $stream_url = "{$this->base_url}/live/{$this->username}/{$this->password}/{$stream_id}.ts";
            } elseif ($type == 'vod') {
                $container = isset($stream['container_extension']) ? $stream['container_extension'] : 'mp4';
                $stream_url = "{$this->base_url}/movie/{$this->username}/{$this->password}/{$stream_id}.{$container}";
            }

            // Get logo/poster
            $logo = isset($stream['logo']) ? $stream['logo'] : '';
            $poster = isset($stream['stream_icon']) ? $stream['stream_icon'] : $logo;

            $result[] = array(
                "id" => $stream_id,
                "name" => $stream_name,
                "category_id" => $category_id,
                "url" => $stream_url,
                "logo" => $logo,
                "poster" => $poster,
                "type" => $type
            );
        }

        return $result;
    }
}
?>
