#!/bin/bash

#############################################
# VOD Panel - Complete Installation Script
# Ubuntu/Debian compatible
#############################################

set -e  # Exit on error

echo "============================================"
echo "   VOD Panel Installation Script"
echo "============================================"
echo ""

# Check if running as root
if [ "$EUID" -ne 0 ]; then
    echo "❌ Please run as root (sudo bash INSTALL.sh)"
    exit 1
fi

echo "✓ Running as root"
echo ""

# Variables
MYSQL_ROOT_PASSWORD="Admin@123"
ADMIN_USERNAME="admin"
ADMIN_PASSWORD="admin"
DB_NAME="vod_import"
PANEL_PATH="/var/www/html/VOD"
SERVER_IP=$(hostname -I | awk '{print $1}')

echo "📋 Installation Configuration:"
echo "   MySQL Root Password: $MYSQL_ROOT_PASSWORD"
echo "   Admin Username: $ADMIN_USERNAME"
echo "   Admin Password: $ADMIN_PASSWORD"
echo "   Database Name: $DB_NAME"
echo "   Panel Path: $PANEL_PATH"
echo "   Server IP: $SERVER_IP"
echo ""
read -p "Press Enter to continue or Ctrl+C to cancel..."
echo ""

#############################################
# Step 1: Update System
#############################################
echo "📦 Step 1/8: Updating system packages..."
apt-get update -qq
apt-get upgrade -y -qq
echo "✅ System updated"
echo ""

#############################################
# Step 2: Install Apache
#############################################
echo "🌐 Step 2/8: Installing Apache web server..."
apt-get install -y apache2
systemctl enable apache2
systemctl start apache2
echo "✅ Apache installed and started"
echo ""

#############################################
# Step 3: Install MySQL
#############################################
echo "🗄️ Step 3/8: Installing MySQL server..."
export DEBIAN_FRONTEND=noninteractive
apt-get install -y mysql-server

# Set MySQL root password
mysql -e "ALTER USER 'root'@'localhost' IDENTIFIED WITH mysql_native_password BY '$MYSQL_ROOT_PASSWORD';" 2>/dev/null || \
mysql -e "UPDATE mysql.user SET authentication_string=PASSWORD('$MYSQL_ROOT_PASSWORD') WHERE User='root';" 2>/dev/null

mysql -e "FLUSH PRIVILEGES;"
systemctl enable mysql
systemctl restart mysql
echo "✅ MySQL installed and configured"
echo ""

#############################################
# Step 4: Install PHP 7.4
#############################################
echo "🐘 Step 4/8: Installing PHP 7.4 and extensions..."
apt-get install -y software-properties-common
add-apt-repository -y ppa:ondrej/php
apt-get update -qq

apt-get install -y \
    php7.4 \
    php7.4-cli \
    php7.4-common \
    php7.4-mysql \
    php7.4-curl \
    php7.4-json \
    php7.4-mbstring \
    php7.4-xml \
    php7.4-zip \
    libapache2-mod-php7.4

# Set PHP 7.4 as default
update-alternatives --set php /usr/bin/php7.4

# Configure PHP
sed -i 's/upload_max_filesize = .*/upload_max_filesize = 2048M/' /etc/php/7.4/apache2/php.ini
sed -i 's/post_max_size = .*/post_max_size = 2048M/' /etc/php/7.4/apache2/php.ini
sed -i 's/memory_limit = .*/memory_limit = 512M/' /etc/php/7.4/apache2/php.ini
sed -i 's/max_execution_time = .*/max_execution_time = 300/' /etc/php/7.4/apache2/php.ini

systemctl restart apache2
echo "✅ PHP 7.4 installed and configured"
echo ""

#############################################
# Step 5: Install Additional Tools
#############################################
echo "🔧 Step 5/8: Installing additional tools..."
apt-get install -y \
    curl \
    wget \
    unzip \
    git \
    ffmpeg \
    python3 \
    python3-pip \
    jq
echo "✅ Additional tools installed"
echo ""

#############################################
# Step 6: Create Database and Tables
#############################################
echo "💾 Step 6/8: Creating database and tables..."

mysql -u root -p"$MYSQL_ROOT_PASSWORD" <<EOF
-- Create database
CREATE DATABASE IF NOT EXISTS $DB_NAME CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE $DB_NAME;

-- Users table
CREATE TABLE IF NOT EXISTS users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    username VARCHAR(50) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100),
    role VARCHAR(20) DEFAULT 'user',
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Series table
CREATE TABLE IF NOT EXISTS series (
    id INT PRIMARY KEY AUTO_INCREMENT,
    tmdb_id INT,
    imdb_id VARCHAR(20),
    title VARCHAR(255) NOT NULL,
    original_title VARCHAR(255),
    year INT,
    clean_name VARCHAR(255),
    description TEXT,
    plot TEXT,
    genre VARCHAR(255),
    cast TEXT,
    rating VARCHAR(10),
    poster VARCHAR(500),
    backdrop VARCHAR(500),
    seasons_count INT DEFAULT 0,
    episodes_count INT DEFAULT 0,
    status VARCHAR(50),
    last_episode_date DATE,
    is_adult TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_title (title),
    INDEX idx_tmdb (tmdb_id),
    INDEX idx_episodes (episodes_count)
);

-- Episodes table
CREATE TABLE IF NOT EXISTS episodes (
    id INT PRIMARY KEY AUTO_INCREMENT,
    series_id INT NOT NULL,
    season_number INT NOT NULL,
    episode_number INT NOT NULL,
    title VARCHAR(255),
    plot TEXT,
    duration INT,
    air_date DATE,
    stream_url VARCHAR(1000),
    container_extension VARCHAR(10),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (series_id) REFERENCES series(id) ON DELETE CASCADE,
    INDEX idx_series (series_id),
    INDEX idx_season (season_number),
    INDEX idx_episode (episode_number)
);

-- Movies table
CREATE TABLE IF NOT EXISTS movies (
    id INT PRIMARY KEY AUTO_INCREMENT,
    tmdb_id INT,
    imdb_id VARCHAR(20),
    title VARCHAR(255) NOT NULL,
    original_title VARCHAR(255),
    year INT,
    clean_name VARCHAR(255),
    description TEXT,
    plot TEXT,
    genre VARCHAR(255),
    director VARCHAR(255),
    cast TEXT,
    rating VARCHAR(10),
    rating_5based VARCHAR(10),
    duration INT,
    poster VARCHAR(500),
    backdrop VARCHAR(500),
    trailer_url VARCHAR(500),
    stream_url VARCHAR(1000),
    container_extension VARCHAR(10),
    is_adult TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_title (title),
    INDEX idx_tmdb (tmdb_id)
);

-- Categories table
CREATE TABLE IF NOT EXISTS categories (
    id INT PRIMARY KEY AUTO_INCREMENT,
    external_id INT,
    name VARCHAR(100) NOT NULL,
    type ENUM('movie', 'series') NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Remote servers table
CREATE TABLE IF NOT EXISTS remote_servers (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    host VARCHAR(255) NOT NULL,
    port INT DEFAULT 8080,
    username VARCHAR(100) NOT NULL,
    password VARCHAR(255) NOT NULL,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Servers table (Xtream codes servers)
CREATE TABLE IF NOT EXISTS servers (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    url VARCHAR(255) NOT NULL,
    username VARCHAR(100) NOT NULL,
    password VARCHAR(255) NOT NULL,
    is_active TINYINT(1) DEFAULT 1,
    last_check TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Download jobs table
CREATE TABLE IF NOT EXISTS download_jobs (
    id INT PRIMARY KEY AUTO_INCREMENT,
    remote_server_id INT,
    source_url VARCHAR(1000) NOT NULL,
    destination_path VARCHAR(500),
    filename VARCHAR(255) NOT NULL,
    movie_id INT,
    episode_id INT,
    status ENUM('pending','downloading','completed','failed','cancelled') DEFAULT 'pending',
    progress INT DEFAULT 0,
    downloaded_bytes BIGINT DEFAULT 0,
    total_bytes BIGINT,
    download_speed INT,
    error_message TEXT,
    started_at DATETIME,
    completed_at DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_status (status)
);

-- Import logs table
CREATE TABLE IF NOT EXISTS import_logs (
    id INT PRIMARY KEY AUTO_INCREMENT,
    server_id INT,
    type VARCHAR(50),
    status VARCHAR(50),
    message TEXT,
    details TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Settings table
CREATE TABLE IF NOT EXISTS settings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(100) UNIQUE NOT NULL,
    setting_value TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Insert default admin user
INSERT INTO users (username, password, email, role, is_active)
VALUES ('$ADMIN_USERNAME', '\$2y\$10\$Cg4AfsMGK6xh2ulO24T3xePw1PpZLZm06KILFQ3bX6Jq7XClnSTOC', 'admin@localhost', 'admin', 1)
ON DUPLICATE KEY UPDATE password='\$2y\$10\$Cg4AfsMGK6xh2ulO24T3xePw1PpZLZm06KILFQ3bX6Jq7XClnSTOC';

EOF

echo "✅ Database and tables created"
echo "✅ Admin user created (username: $ADMIN_USERNAME, password: $ADMIN_PASSWORD)"
echo ""

#############################################
# Step 7: Configure Apache
#############################################
echo "⚙️ Step 7/8: Configuring Apache..."

# Enable required modules
a2enmod rewrite
a2enmod headers
a2enmod ssl

# Set permissions
chown -R www-data:www-data $PANEL_PATH
chmod -R 755 $PANEL_PATH

# Restart Apache
systemctl restart apache2
echo "✅ Apache configured"
echo ""

#############################################
# Step 8: Configure Firewall (if UFW installed)
#############################################
echo "🔥 Step 8/8: Configuring firewall..."
if command -v ufw &> /dev/null; then
    ufw allow 80/tcp
    ufw allow 443/tcp
    echo "✅ Firewall configured (ports 80, 443 opened)"
else
    echo "⚠️ UFW not installed, skipping firewall configuration"
fi
echo ""

#############################################
# Installation Complete
#############################################
echo "============================================"
echo "   ✅ Installation Complete!"
echo "============================================"
echo ""
echo "📋 Access Information:"
echo "   Panel URL: http://$SERVER_IP/VOD/"
echo "   Login URL: http://$SERVER_IP/VOD/login.php"
echo ""
echo "🔐 Login Credentials:"
echo "   Username: $ADMIN_USERNAME"
echo "   Password: $ADMIN_PASSWORD"
echo ""
echo "💾 Database Information:"
echo "   Database: $DB_NAME"
echo "   MySQL Root Password: $MYSQL_ROOT_PASSWORD"
echo "   Host: localhost"
echo ""
echo "📁 Panel Location:"
echo "   $PANEL_PATH"
echo ""
echo "🔧 Useful Commands:"
echo "   Check Apache status: systemctl status apache2"
echo "   Check MySQL status: systemctl status mysql"
echo "   View Apache logs: tail -f /var/log/apache2/error.log"
echo "   Restart services: systemctl restart apache2 mysql"
echo ""
echo "📝 Next Steps:"
echo "   1. Open: http://$SERVER_IP/VOD/"
echo "   2. Login with credentials above"
echo "   3. Add your Xtream Codes servers"
echo "   4. Import series and movies"
echo ""
echo "============================================"
