/**
 * VOD Import System - Frontend JavaScript
 */

const API_BASE = '/VOD/api';
let currentPage = 'dashboard';
let moviesOffset = 0;
const moviesLimit = 24;
let selectedMovies = new Map();
let movieViewMode = 'grid';
let currentFilters = { category: '', year: '', search: '' };
let allMoviesData = [];

// Series variables
let seriesOffset = 0;
const seriesLimit = 24;
let selectedSeries = new Map();
let seriesViewMode = 'grid';
let seriesFilters = { category: '', year: '', search: '' };
let allSeriesData = [];
let seriesCategoriesMap = {}; // Store category ID to name mapping
let currentSeriesDetail = null;
let selectedEpisodes = new Map();
let downloadType = 'movies'; // 'movies' or 'series'

// ==================== Theme Management ====================
let currentThemeMode = 'dark';

function initTheme() {
    const savedMode = localStorage.getItem('vod-theme-mode') || 'dark';
    applyThemeMode(savedMode);
}

function setThemeMode(mode) {
    console.log('setThemeMode called with:', mode);
    localStorage.setItem('vod-theme-mode', mode);
    applyThemeMode(mode);
    // Show notification if function is available
    if (typeof showAlert === 'function') {
        showAlert(`Theme: ${mode.charAt(0).toUpperCase() + mode.slice(1)}`, 'success');
    }
    console.log('Theme mode set successfully');
}

function applyThemeMode(mode) {
    currentThemeMode = mode;
    let theme = mode;

    // If auto, detect system preference
    if (mode === 'auto') {
        theme = window.matchMedia('(prefers-color-scheme: dark)').matches ? 'dark' : 'light';
    }

    document.documentElement.setAttribute('data-theme', theme);
    updateThemeButton(theme);
    updateThemeCards(mode);
}

function toggleTheme() {
    // Cycle through: dark -> light -> auto -> dark
    const modes = ['dark', 'light', 'auto'];
    const currentIndex = modes.indexOf(currentThemeMode);
    const nextMode = modes[(currentIndex + 1) % modes.length];
    setThemeMode(nextMode);
}

function updateThemeButton(theme) {
    const icon = document.getElementById('theme-icon');
    const text = document.getElementById('theme-text');
    if (icon && text) {
        if (currentThemeMode === 'auto') {
            icon.className = 'bi bi-circle-half';
            text.textContent = 'Auto';
        } else if (theme === 'dark') {
            icon.className = 'bi bi-moon-fill';
            text.textContent = 'Dark';
        } else {
            icon.className = 'bi bi-sun-fill';
            text.textContent = 'Light';
        }
    }
}

function updateThemeCards(mode) {
    // Update theme card selection in settings
    document.querySelectorAll('.theme-card').forEach(card => card.classList.remove('active'));
    const activeCard = document.getElementById(`theme-card-${mode}`);
    if (activeCard) activeCard.classList.add('active');
}

// Listen for system theme changes (for auto mode)
window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', (e) => {
    if (currentThemeMode === 'auto') {
        const theme = e.matches ? 'dark' : 'light';
        document.documentElement.setAttribute('data-theme', theme);
        updateThemeButton(theme);
    }
});

// Initialize theme on page load
initTheme();

// Initialize
document.addEventListener('DOMContentLoaded', function() {
    // Theme is already initialized, just update UI
    updateThemeCards(currentThemeMode);
    // Navigation
    document.querySelectorAll('.nav-link[data-page]').forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            navigateTo(this.dataset.page);
        });
    });

    // Server type toggle
    document.querySelector('select[name="type"]')?.addEventListener('change', function() {
        const xtreamFields = document.querySelector('.xtream-fields');
        if (xtreamFields) {
            xtreamFields.style.display = this.value === 'xtream' ? 'flex' : 'none';
        }
    });

    // Settings form
    document.getElementById('settings-form')?.addEventListener('submit', function(e) {
        e.preventDefault();
        saveSettings();
    });

    // Import server change
    document.getElementById('import-server')?.addEventListener('change', function() {
        if (this.value) {
            loadCategories(this.value, 'import-category');
        }
    });

    // Load initial page
    loadDashboard();
});

// Navigation
function navigateTo(page) {
    // Stop any running auto-refresh when leaving a page
    if (currentPage === 'download-jobs' && page !== 'download-jobs') {
        stopDownloadJobsAutoRefresh();
    }

    // Update nav
    document.querySelectorAll('.nav-link').forEach(link => {
        link.classList.remove('active');
        if (link.dataset.page === page) {
            link.classList.add('active');
        }
    });

    // Show page
    document.querySelectorAll('.page').forEach(p => {
        p.style.display = 'none';
    });
    document.getElementById(`page-${page}`).style.display = 'block';

    currentPage = page;

    // Load page data
    switch (page) {
        case 'dashboard':
            loadDashboard();
            break;
        case 'servers':
            loadServers();
            break;
        case 'remote-servers':
            loadRemoteServers();
            break;
        case 'movies':
            loadMovies();
            break;
        case 'series':
            loadSeries();
            break;
        case 'import':
            loadServersSelect();
            break;
        case 'download-jobs':
            loadDownloadJobs();
            startDownloadJobsAutoRefresh(); // Auto-refresh for real-time progress
            break;
        case 'logs':
            loadLogs();
            break;
        case 'users':
            loadUsers();
            break;
        case 'settings':
            loadSettings();
            break;
    }
}

// API Helper
async function apiCall(endpoint, method = 'GET', data = null) {
    const options = {
        method,
        headers: {
            'Content-Type': 'application/json'
        }
    };

    if (data && method !== 'GET') {
        options.body = JSON.stringify(data);
    }

    try {
        // Validate endpoint
        if (!endpoint || endpoint === 'undefined') {
            console.error('Invalid API endpoint:', endpoint);
            return { error: 'Invalid API endpoint' };
        }
        const url = endpoint.includes('?') ? `${API_BASE}/?endpoint=${endpoint.split('?')[0]}&${endpoint.split('?')[1]}` : `${API_BASE}/?endpoint=${endpoint}`;
        const response = await fetch(url, options);
        const text = await response.text();
        if (!text || text.trim() === '') {
            return { error: 'Empty response from server' };
        }
        return JSON.parse(text);
    } catch (error) {
        console.error('API Error:', error);
        return { error: error.message };
    }
}

// Dashboard
async function loadDashboard() {
    const stats = await apiCall('stats');

    if (stats && !stats.error) {
        const setEl = (id, val) => { const el = document.getElementById(id); if (el) el.textContent = val || 0; };
        setEl('stat-movies', stats.content?.movies);
        setEl('stat-series', stats.content?.series);
        setEl('stat-episodes', stats.content?.episodes);
        setEl('stat-servers-online', stats.servers?.online);
        setEl('stat-links', stats.content?.links);
        setEl('stat-queue', stats.downloads?.pending);
    }

    loadNewestMovies();
    loadNewestSeries();
    loadServersStatus();
}

async function loadNewestMovies() {
    const container = document.getElementById('newest-movies');
    container.innerHTML = '<div class="loading"><div class="spinner-border"></div></div>';

    const movies = await apiCall('movies/newest?limit=16');

    if (Array.isArray(movies) && movies.length > 0) {
        container.innerHTML = movies.map(movie => {
            const rating = movie.rating ? parseFloat(movie.rating).toFixed(1) : null;
            const ratingHtml = rating ? `<div class="movie-rating">${rating}</div>` : '';
            return `
            <div class="col-md-3 col-6">
                <div class="movie-card" onclick="navigateTo('movies'); showMovieDetails(${movie.id})" style="cursor: pointer;">
                    <img src="${movie.poster || 'data:image/svg+xml,%3Csvg xmlns=%22http://www.w3.org/2000/svg%22 width=%22200%22 height=%22300%22%3E%3Crect fill=%22%231e293b%22 width=%22200%22 height=%22300%22/%3E%3Ctext fill=%22%2394a3b8%22 x=%2250%25%22 y=%2250%25%22 text-anchor=%22middle%22%3ENo Image%3C/text%3E%3C/svg%3E'}"
                        class="movie-poster" alt="${movie.title}">
                    <div class="movie-info">
                        ${ratingHtml}
                        <div class="movie-title" title="${movie.title}">${movie.title}</div>
                        <div class="movie-year">${movie.year || ''}</div>
                    </div>
                </div>
            </div>
        `}).join('');
    } else {
        container.innerHTML = '<div class="col-12 text-center text-secondary">No movies found</div>';
    }
}

async function loadNewestSeries() {
    const container = document.getElementById('newest-series');
    if (!container) return;

    container.innerHTML = '<div class="loading"><div class="spinner-border"></div></div>';

    const series = await apiCall('series/newest?limit=16');

    if (Array.isArray(series) && series.length > 0) {
        container.innerHTML = series.map(s => {
            const rating = s.rating ? parseFloat(s.rating).toFixed(1) : null;
            const ratingHtml = rating ? `<div class="movie-rating">${rating}</div>` : '';
            return `
            <div class="col-md-3 col-6">
                <div class="movie-card" onclick="navigateTo('series'); showSeriesDetails(${s.id})" style="cursor: pointer;">
                    <img src="${s.poster || 'data:image/svg+xml,%3Csvg xmlns=%22http://www.w3.org/2000/svg%22 width=%22200%22 height=%22300%22%3E%3Crect fill=%22%231e293b%22 width=%22200%22 height=%22300%22/%3E%3Ctext fill=%22%2394a3b8%22 x=%2250%25%22 y=%2250%25%22 text-anchor=%22middle%22%3ENo Image%3C/text%3E%3C/svg%3E'}"
                        class="movie-poster" alt="${s.title}">
                    <div class="movie-info">
                        ${ratingHtml}
                        <div class="movie-title" title="${s.title}">${s.title}</div>
                        <div class="movie-year">${s.year || ''} ${s.episode_count ? '• ' + s.episode_count + ' eps' : ''}</div>
                    </div>
                </div>
            </div>
        `}).join('');
    } else {
        container.innerHTML = '<div class="col-12 text-center text-secondary">No series found. Import series from Source Servers.</div>';
    }
}

async function loadServersStatus() {
    const container = document.getElementById('servers-status');
    const servers = await apiCall('servers');

    if (Array.isArray(servers) && servers.length > 0) {
        container.innerHTML = servers.map(server => {
            const statusColor = server.status === 'online' ? '#22c55e' :
                               server.status === 'error' ? '#ef4444' :
                               server.status === 'offline' ? '#64748b' : '#eab308';
            const statusBg = server.status === 'online' ? 'rgba(34, 197, 94, 0.15)' :
                            server.status === 'error' ? 'rgba(239, 68, 68, 0.15)' :
                            server.status === 'offline' ? 'rgba(100, 116, 139, 0.15)' : 'rgba(234, 179, 8, 0.15)';
            const statusIcon = server.status === 'online' ? 'bi-check-circle-fill' :
                              server.status === 'error' ? 'bi-x-circle-fill' :
                              server.status === 'offline' ? 'bi-dash-circle-fill' : 'bi-exclamation-circle-fill';
            return `
            <div class="d-flex justify-content-between align-items-center mb-2 p-3 rounded"
                style="background: ${statusBg}; border-left: 3px solid ${statusColor};">
                <div>
                    <strong style="color: ${statusColor};">${server.name}</strong>
                    <div class="small text-secondary">${server.type === 'xtream' ? 'Xtream Codes' : server.type}</div>
                </div>
                <span style="color: ${statusColor}; font-weight: 600;">
                    <i class="bi ${statusIcon} me-1"></i>
                    ${server.status ? server.status.charAt(0).toUpperCase() + server.status.slice(1) : 'Unknown'}
                </span>
            </div>
        `}).join('');
    } else {
        container.innerHTML = '<p class="text-secondary text-center py-3"><i class="bi bi-hdd-network fs-3 d-block mb-2"></i>No servers configured</p>';
    }
}

// Servers
let currentServerInfoId = null;
let currentServerInfoType = 'movie';

async function loadServers() {
    const tbody = document.getElementById('servers-table');
    tbody.innerHTML = '<tr><td colspan="9" class="text-center"><div class="spinner-border spinner-border-sm"></div></td></tr>';

    const servers = await apiCall('servers');

    if (Array.isArray(servers) && servers.length > 0) {
        tbody.innerHTML = servers.map(server => `
            <tr>
                <td>
                    <strong>${server.name}</strong>
                    ${server.is_active ? '' : '<span class="badge bg-secondary ms-2">Disabled</span>'}
                </td>
                <td><span class="badge bg-info">${server.type}</span></td>
                <td class="small text-truncate" style="max-width: 200px;" title="${server.url}">${server.url}</td>
                <td>
                    <span class="badge bg-${server.status === 'online' ? 'success' : server.status === 'error' ? 'danger' : 'secondary'}">
                        ${server.status || 'Unknown'}
                    </span>
                </td>
                <td class="small">${server.last_check || '-'}</td>
                <td><span class="badge bg-primary">${server.movies_count || 0}</span></td>
                <td><span class="badge bg-info">${server.series_count || 0}</span></td>
                <td><span class="badge bg-success">${server.live_count || 0}</span></td>
                <td>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-info" onclick="showServerInfo(${server.id})" title="Server Info">
                            <i class="bi bi-info-circle"></i>
                        </button>
                        <button class="btn btn-outline-secondary" onclick="editServer(${server.id})" title="Edit Server">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-outline-primary" onclick="checkServer(${server.id})" title="Check Status">
                            <i class="bi bi-arrow-repeat"></i>
                        </button>
                        <button class="btn btn-outline-success" onclick="syncCategories(${server.id})" title="Sync Categories">
                            <i class="bi bi-folder-symlink"></i>
                        </button>
                        <button class="btn btn-outline-warning" onclick="toggleServer(${server.id})" title="Enable/Disable">
                            <i class="bi bi-power"></i>
                        </button>
                        <button class="btn btn-outline-danger" onclick="deleteServer(${server.id})" title="Delete">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `).join('');
    } else {
        tbody.innerHTML = '<tr><td colspan="9" class="text-center text-secondary">No servers found</td></tr>';
    }
}

async function showServerInfo(serverId) {
    currentServerInfoId = serverId;
    currentServerInfoType = 'movie';

    // Get server details
    const server = await apiCall(`servers/${serverId}`);
    if (server.error) {
        showAlert('Failed to load server info', 'danger');
        return;
    }

    document.getElementById('server-info-name').textContent = server.name;
    document.getElementById('server-info-movies').textContent = server.movies_count || 0;
    document.getElementById('server-info-series').textContent = server.series_count || 0;
    document.getElementById('server-info-live').textContent = server.live_count || 0;

    // Count categories
    const categories = await apiCall(`categories?server_id=${serverId}`);
    document.getElementById('server-info-categories').textContent = Array.isArray(categories) ? categories.length : 0;

    // Reset tabs
    document.querySelectorAll('#serverInfoTabs .nav-link').forEach(t => t.classList.remove('active'));
    document.querySelector('#serverInfoTabs .nav-link').classList.add('active');

    // Load movie categories by default
    await loadServerCategories('movie');

    new bootstrap.Modal(document.getElementById('serverInfoModal')).show();
}

async function loadServerCategories(type) {
    currentServerInfoType = type;

    // Update tabs
    document.querySelectorAll('#serverInfoTabs .nav-link').forEach(t => t.classList.remove('active'));
    event?.target?.classList.add('active');

    const titles = { movie: 'Movie Categories', series: 'Series Categories', live: 'Live Categories' };
    document.getElementById('server-categories-title').textContent = titles[type] || 'Categories';

    const container = document.getElementById('server-categories-list');
    container.innerHTML = '<div class="text-center"><div class="spinner-border spinner-border-sm"></div> Loading...</div>';

    const categories = await apiCall(`categories?server_id=${currentServerInfoId}&type=${type}`);

    if (Array.isArray(categories) && categories.length > 0) {
        container.innerHTML = `
            <div class="list-group">
                ${categories.map(cat => `
                    <label class="list-group-item d-flex align-items-center">
                        <input type="checkbox" class="form-check-input me-3 server-cat-checkbox"
                            value="${cat.external_id}" data-name="${cat.name}">
                        <div class="flex-grow-1">
                            <strong>${cat.name}</strong>
                        </div>
                        <span class="badge bg-secondary">${cat.external_id}</span>
                    </label>
                `).join('')}
            </div>
        `;
    } else {
        container.innerHTML = '<p class="text-secondary text-center">No categories found. Try syncing categories first.</p>';
    }
}

function selectAllServerCategories() {
    const checkboxes = document.querySelectorAll('.server-cat-checkbox');
    const allChecked = Array.from(checkboxes).every(cb => cb.checked);
    checkboxes.forEach(cb => cb.checked = !allChecked);
}

async function importSelectedCategories() {
    const selected = Array.from(document.querySelectorAll('.server-cat-checkbox:checked'))
        .map(cb => ({ id: cb.value, name: cb.dataset.name }));

    if (selected.length === 0) {
        showAlert('Please select at least one category', 'warning');
        return;
    }

    showAlert(`Importing ${selected.length} categories...`, 'info');

    const action = currentServerInfoType === 'series' ? 'import/series' : 'import/server';
    let totalImported = 0;
    let totalErrors = 0;

    for (const cat of selected) {
        const result = await apiCall(action, 'POST', {
            server_id: currentServerInfoId,
            category_id: parseInt(cat.id),
            force: false
        });

        if (result.imported !== undefined) {
            totalImported += result.imported;
        }
        if (result.errors) {
            totalErrors += result.errors;
        }
    }

    showAlert(`Import complete: ${totalImported} items imported, ${totalErrors} errors`, totalErrors > 0 ? 'warning' : 'success');
}

async function syncServerCategories() {
    showAlert('Syncing all categories...', 'info');
    const result = await apiCall(`servers/sync-categories/${currentServerInfoId}`, 'POST');

    if (result.success) {
        showAlert(`Synced ${result.synced || 0} categories`, 'success');
        loadServerCategories(currentServerInfoType);
    } else {
        showAlert(result.error || 'Sync failed', 'danger');
    }
}

async function addServer() {
    const form = document.getElementById('add-server-form');
    const formData = new FormData(form);
    const data = Object.fromEntries(formData);

    const result = await apiCall('servers/add', 'POST', data);

    if (result.success) {
        bootstrap.Modal.getInstance(document.getElementById('addServerModal')).hide();
        form.reset();
        loadServers();
        showAlert('Server added successfully', 'success');
    } else {
        showAlert(result.error || 'Failed to add server', 'danger');
    }
}

async function checkServer(id) {
    showAlert('Checking server...', 'info');
    const result = await apiCall(`servers/check/${id}`);

    if (result.status) {
        showAlert(`Server status: ${result.status}`, result.status === 'online' ? 'success' : 'warning');
        loadServers();
    } else {
        showAlert(result.error || 'Check failed', 'danger');
    }
}

async function syncCategories(id) {
    showAlert('Syncing categories...', 'info');
    const result = await apiCall(`servers/sync-categories/${id}`, 'POST');

    if (result.success) {
        showAlert(`Synced ${result.added} categories`, 'success');
    } else {
        showAlert(result.error || 'Sync failed', 'danger');
    }
}

async function toggleServer(id) {
    const result = await apiCall(`servers/toggle/${id}`, 'POST');

    if (result.success) {
        loadServers();
    }
}

async function deleteServer(id) {
    if (!confirm('Are you sure you want to delete this server?')) return;

    const result = await apiCall(`servers/${id}`, 'DELETE');

    if (result.success) {
        loadServers();
        showAlert('Server deleted', 'success');
    } else {
        showAlert(result.error || 'Delete failed', 'danger');
    }
}

// Edit Server
let editingServerId = null;

async function editServer(id) {
    editingServerId = id;
    const server = await apiCall(`servers/${id}`);

    if (server.error) {
        showAlert('Failed to load server details', 'danger');
        return;
    }

    // Fill the edit form
    document.getElementById('edit-server-name').value = server.name || '';
    document.getElementById('edit-server-url').value = server.url || '';
    document.getElementById('edit-server-port').value = server.port || 80;
    document.getElementById('edit-server-username').value = server.username || '';
    document.getElementById('edit-server-password').value = server.password || '';
    document.getElementById('edit-server-type').value = server.type || 'xtream';

    new bootstrap.Modal(document.getElementById('editServerModal')).show();
}

async function saveServerEdit() {
    if (!editingServerId) return;

    const data = {
        name: document.getElementById('edit-server-name').value.trim(),
        url: document.getElementById('edit-server-url').value.trim(),
        port: parseInt(document.getElementById('edit-server-port').value) || 80,
        username: document.getElementById('edit-server-username').value.trim(),
        password: document.getElementById('edit-server-password').value.trim(),
        type: document.getElementById('edit-server-type').value
    };

    if (!data.name || !data.url) {
        showAlert('Name and URL are required', 'warning');
        return;
    }

    const result = await apiCall(`servers/${editingServerId}`, 'PUT', data);

    if (result.success) {
        bootstrap.Modal.getInstance(document.getElementById('editServerModal')).hide();
        loadServers();
        showAlert('Server updated successfully', 'success');
        editingServerId = null;
    } else {
        showAlert(result.error || 'Failed to update server', 'danger');
    }
}

// Movies
async function loadMovies(reset = true) {
    if (reset) {
        moviesOffset = 0;
        allMoviesData = [];
        loadMovieFilters();
    }

    const container = document.getElementById('movies-grid');

    if (moviesOffset === 0) {
        container.innerHTML = '<div class="col-12 text-center"><div class="spinner-border"></div></div>';
    }

    let endpoint = `movies?limit=${moviesLimit}&offset=${moviesOffset}`;
    if (currentFilters.category) endpoint += `&category_id=${currentFilters.category}`;
    if (currentFilters.year) endpoint += `&year=${currentFilters.year}`;
    if (currentFilters.search) endpoint += `&search=${encodeURIComponent(currentFilters.search)}`;

    const movies = await apiCall(endpoint);

    if (Array.isArray(movies) && movies.length > 0) {
        allMoviesData = reset ? movies : [...allMoviesData, ...movies];
        const html = movies.map(movie => renderMovieCard(movie)).join('');

        if (moviesOffset === 0) {
            container.innerHTML = html;
        } else {
            container.innerHTML += html;
        }

        moviesOffset += moviesLimit;
        updateMoviesStats();

        document.getElementById('load-more-movies').style.display =
            movies.length < moviesLimit ? 'none' : 'block';
    } else if (moviesOffset === 0) {
        container.innerHTML = '<div class="col-12 text-center text-secondary">No movies found</div>';
        document.getElementById('load-more-movies').style.display = 'none';
    }
}

function renderMovieCard(movie) {
    const isSelected = selectedMovies.has(movie.id);
    const posterUrl = movie.poster || 'data:image/svg+xml,%3Csvg xmlns=%22http://www.w3.org/2000/svg%22 width=%22200%22 height=%22300%22%3E%3Crect fill=%22%231e293b%22 width=%22200%22 height=%22300%22/%3E%3Ctext fill=%22%2394a3b8%22 x=%2250%25%22 y=%2250%25%22 text-anchor=%22middle%22%3ENo Poster%3C/text%3E%3C/svg%3E';
    const safeTitle = (movie.title || '').replace(/'/g, "\\'").replace(/"/g, '&quot;');
    const safeClean = (movie.clean_name || '').replace(/'/g, "\\'").replace(/"/g, '&quot;');

    if (movieViewMode === 'grid') {
        return `
            <div class="col-xl-2 col-lg-3 col-md-4 col-6">
                <div class="movie-card ${isSelected ? 'border border-success' : ''}" data-movie-id="${movie.id}">
                    <div class="position-relative">
                        <img src="${posterUrl}" class="movie-poster" alt="${safeTitle}" loading="lazy"
                            onclick="showMovieDetails(${movie.id})" style="cursor:pointer;">
                        <div style="position:absolute;top:5px;left:5px;">
                            <input type="checkbox" class="form-check-input movie-checkbox" style="transform:scale(1.3);"
                                ${isSelected ? 'checked' : ''}
                                onchange="toggleMovieSelection(${movie.id}, '${safeTitle}', '${safeClean}', event)">
                        </div>
                    </div>
                    <div class="movie-info">
                        <div class="movie-title" title="${safeTitle}">${movie.title}</div>
                        <div class="d-flex justify-content-between align-items-center">
                            <span class="movie-year">${movie.year || ''}</span>
                            <span class="badge bg-primary">${movie.link_count || 0}</span>
                        </div>
                    </div>
                </div>
            </div>
        `;
    } else {
        return `
            <div class="col-12">
                <div class="card mb-2 ${isSelected ? 'border-success' : ''}" data-movie-id="${movie.id}">
                    <div class="card-body p-2">
                        <div class="d-flex align-items-center">
                            <input type="checkbox" class="form-check-input me-3 movie-checkbox"
                                ${isSelected ? 'checked' : ''}
                                onchange="toggleMovieSelection(${movie.id}, '${safeTitle}', '${safeClean}', event)">
                            <img src="${posterUrl}" style="width:50px;height:75px;object-fit:cover;cursor:pointer;"
                                class="rounded me-3" onclick="showMovieDetails(${movie.id})">
                            <div class="flex-grow-1">
                                <strong>${movie.title}</strong>
                                <div class="small text-secondary">${movie.year || ''}</div>
                            </div>
                            <span class="badge bg-primary">${movie.link_count || 0} links</span>
                            <button class="btn btn-sm btn-outline-success ms-2" onclick="quickDownload(${movie.id})">
                                <i class="bi bi-download"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;
    }
}

async function loadMovieFilters() {
    const categories = await apiCall('categories?type=movie');
    const categorySelect = document.getElementById('movie-category-filter');
    if (categorySelect && Array.isArray(categories)) {
        categorySelect.innerHTML = '<option value="">All Categories</option>' +
            categories.map(cat => `<option value="${cat.external_id}">${cat.name}</option>`).join('');
    }

    const years = await apiCall('movies/years');
    const yearSelect = document.getElementById('movie-year-filter');
    if (yearSelect && Array.isArray(years)) {
        yearSelect.innerHTML = '<option value="">All Years</option>' +
            years.map(y => `<option value="${y}">${y}</option>`).join('');
    }
}

function filterMovies() {
    currentFilters.category = document.getElementById('movie-category-filter')?.value || '';
    currentFilters.year = document.getElementById('movie-year-filter')?.value || '';
    loadMovies(true);
}

function loadMoreMovies() {
    loadMovies(false);
}

async function searchMovies() {
    currentFilters.search = document.getElementById('movie-search').value.trim();
    loadMovies(true);
}

function setMovieView(mode) {
    movieViewMode = mode;
    document.getElementById('view-grid')?.classList.toggle('active', mode === 'grid');
    document.getElementById('view-list')?.classList.toggle('active', mode === 'list');
    const container = document.getElementById('movies-grid');
    container.innerHTML = allMoviesData.map(movie => renderMovieCard(movie)).join('');
}

function toggleMovieSelection(id, title, cleanName, event) {
    event?.stopPropagation();
    if (selectedMovies.has(id)) {
        selectedMovies.delete(id);
    } else {
        selectedMovies.set(id, { id, title, cleanName });
    }
    updateSelectionUI();
}

function toggleSelectAll() {
    const selectAll = document.getElementById('select-all-movies')?.checked;
    if (selectAll) {
        allMoviesData.forEach(movie => {
            selectedMovies.set(movie.id, { id: movie.id, title: movie.title, cleanName: movie.clean_name || '' });
        });
    } else {
        selectedMovies.clear();
    }
    updateSelectionUI();
    document.querySelectorAll('.movie-checkbox').forEach(cb => cb.checked = selectAll);
}

function updateSelectionUI() {
    const count = selectedMovies.size;
    const countEl = document.getElementById('selected-count');
    const btnEl = document.getElementById('download-selected-btn');
    if (countEl) countEl.textContent = count;
    if (btnEl) btnEl.style.display = count > 0 ? 'inline-block' : 'none';
    updateMoviesStats();
}

function updateMoviesStats() {
    const stats = document.getElementById('movies-stats');
    if (stats) {
        const selectedCount = selectedMovies.size;
        stats.innerHTML = `<span class="text-info">Showing ${allMoviesData.length} movies</span> | <span class="${selectedCount > 0 ? 'text-success fw-bold' : 'text-secondary'}">${selectedCount} selected</span>`;
    }
}

async function showDownloadModal() {
    if (selectedMovies.size === 0) { showAlert('Please select movies', 'warning'); return; }
    const servers = await apiCall('remote-servers');
    const serverSelect = document.getElementById('download-remote-server');
    if (serverSelect && Array.isArray(servers)) {
        serverSelect.innerHTML = '<option value="">-- Select Server --</option>' +
            servers.filter(s => s.is_active).map(s => `<option value="${s.id}">${s.name} (${s.type} - ${s.host})</option>`).join('');
    }
    document.getElementById('download-movie-count').textContent = selectedMovies.size;
    document.getElementById('selected-movies-list').innerHTML = Array.from(selectedMovies.values())
        .map(m => `<div class="small py-1 border-bottom">${m.title}</div>`).join('');
    new bootstrap.Modal(document.getElementById('downloadModal')).show();
}

async function startRemoteDownload() {
    const serverId = document.getElementById('download-remote-server').value;
    if (!serverId) { showAlert('Please select a server', 'warning'); return; }
    showAlert('Creating download jobs...', 'info');

    const items = [];
    for (const [movieId, movie] of selectedMovies) {
        const movieData = await apiCall(`movies/${movieId}`);
        if (movieData?.links?.length > 0) {
            const link = movieData.links[0];
            items.push({
                movie_id: movieId,
                url: link.stream_url || link.url,
                filename: `${movie.title.replace(/[^a-zA-Z0-9 ]/g, '').replace(/ /g, '.')}.${link.container_extension || 'mp4'}`
            });
        }
    }

    if (items.length === 0) { showAlert('No valid links found', 'danger'); return; }

    const result = await apiCall('remote-servers/bulk-download', 'POST', {
        remote_server_id: parseInt(serverId), items
    });

    if (result.queued > 0) {
        showAlert(`${result.queued} downloads queued!`, 'success');
        bootstrap.Modal.getInstance(document.getElementById('downloadModal')).hide();
        selectedMovies.clear();
        updateSelectionUI();
        // Redirect to Download Manager page
        navigateTo('download-jobs');
    } else {
        showAlert(result.error || 'Failed to queue', 'danger');
    }
}

function quickDownload(movieId) {
    selectedMovies.clear();
    const movie = allMoviesData.find(m => m.id === movieId);
    if (movie) selectedMovies.set(movieId, { id: movieId, title: movie.title, cleanName: movie.clean_name || '' });
    showDownloadModal();
}

// Import
let currentImportType = 'movie'; // 'movie' or 'series'

function setImportType(type) {
    currentImportType = type;

    // Update tabs
    document.querySelectorAll('#import-tabs .nav-link').forEach(tab => {
        tab.classList.remove('active');
    });
    event.target.closest('.nav-link').classList.add('active');

    // Update card title
    const cardTitle = document.getElementById('import-card-title');
    if (cardTitle) {
        cardTitle.textContent = type === 'movie' ? 'Import Movies from Server' : 'Import Series from Server';
    }

    // Update check new title
    const checkTitle = document.getElementById('check-new-title');
    const checkDesc = document.getElementById('check-new-desc');
    const checkBtn = document.getElementById('check-new-btn-text');
    if (checkTitle) checkTitle.textContent = type === 'movie' ? 'Check New Movies' : 'Check New Series';
    if (checkDesc) checkDesc.textContent = type === 'movie' ? 'Scans all servers for new movies.' : 'Scans all servers for new series.';
    if (checkBtn) checkBtn.textContent = type === 'movie' ? 'Check for New Movies' : 'Check for New Series';

    // Reload categories for selected server
    loadImportCategories();
}

async function loadServersSelect() {
    const servers = await apiCall('servers');

    const selects = ['import-server', 'sync-server'];

    selects.forEach(selectId => {
        const select = document.getElementById(selectId);
        if (select && Array.isArray(servers)) {
            select.innerHTML = '<option value="">-- Select Server --</option>' +
                servers.filter(s => s.is_active).map(s =>
                    `<option value="${s.id}">${s.name}</option>`
                ).join('');
        }
    });
}

async function loadImportCategories() {
    const serverId = document.getElementById('import-server')?.value;
    const categorySelect = document.getElementById('import-category');

    if (!categorySelect) return;

    if (!serverId) {
        categorySelect.innerHTML = '<option value="">All Categories</option>';
        return;
    }

    const categories = await apiCall(`categories?server_id=${serverId}&type=${currentImportType}`);

    if (Array.isArray(categories)) {
        categorySelect.innerHTML = '<option value="">All Categories</option>' +
            categories.map(c => `<option value="${c.external_id}">${c.name}</option>`).join('');
    }
}

async function loadCategories(serverId, selectId) {
    const select = document.getElementById(selectId);
    select.innerHTML = '<option value="">Loading...</option>';

    const categories = await apiCall(`categories?server_id=${serverId}&type=${currentImportType}`);

    select.innerHTML = '<option value="">All Categories</option>';

    if (Array.isArray(categories)) {
        select.innerHTML += categories.map(cat =>
            `<option value="${cat.external_id}">${cat.name}</option>`
        ).join('');
    }
}

async function startImport() {
    const serverId = document.getElementById('import-server').value;
    const categoryId = document.getElementById('import-category').value;
    const force = document.getElementById('import-force').checked;

    if (!serverId) {
        showAlert('Please select a server', 'warning');
        return;
    }

    // Determine which import action to use
    const action = currentImportType === 'series' ? 'import/series' : 'import/server';

    addImportLog(`Starting ${currentImportType} import...`, 'info');

    const result = await apiCall(action, 'POST', {
        server_id: parseInt(serverId),
        category_id: categoryId ? parseInt(categoryId) : null,
        force
    });

    if (result.success !== false) {
        if (currentImportType === 'series') {
            addImportLog(`Imported: ${result.imported} new series, ${result.updated} updated`, 'success');
        } else {
            addImportLog(`Imported: ${result.imported} new, ${result.updated} updated, ${result.skipped || 0} skipped`, 'success');
        }
        if (result.errors > 0) {
            addImportLog(`Errors: ${result.errors}`, 'warning');
        }
    } else {
        addImportLog(`Error: ${result.error}`, 'error');
    }
}

async function startSmartSync() {
    const serverId = document.getElementById('sync-server').value;

    if (!serverId) {
        showAlert('Please select a server', 'warning');
        return;
    }

    addImportLog('Starting Smart Sync...', 'info');

    const result = await apiCall('import/smart-sync', 'POST', {
        server_id: parseInt(serverId)
    });

    if (result.success) {
        addImportLog(`Total on server: ${result.total_on_server}`, 'info');
        addImportLog(`Already exists: ${result.existing}`, 'info');
        addImportLog(`New found: ${result.new_found}`, 'success');
        addImportLog(`Imported: ${result.imported}`, 'success');
    } else {
        addImportLog(`Error: ${result.error}`, 'error');
    }
}

async function checkNewContent() {
    const resultDiv = document.getElementById('new-movies-result');
    resultDiv.style.display = 'block';
    resultDiv.innerHTML = '<div class="spinner-border spinner-border-sm"></div> Checking...';

    const action = currentImportType === 'series' ? 'import/check-new-series' : 'import/check-new';
    const result = await apiCall(action, 'POST');

    if (result.success) {
        const total = result.total_new || 0;
        const dataKey = currentImportType === 'series' ? 'new_series_by_server' : 'new_movies_by_server';
        const label = currentImportType === 'series' ? 'series' : 'movies';

        let html = `<div class="alert alert-success">Found ${total} new ${label}</div>`;

        if (result[dataKey] && Object.keys(result[dataKey]).length > 0) {
            html += '<ul class="list-group">';
            for (const [server, count] of Object.entries(result[dataKey])) {
                html += `<li class="list-group-item d-flex justify-content-between">
                    <span>${server}</span>
                    <span class="badge bg-primary">${count}</span>
                </li>`;
            }
            html += '</ul>';
        }

        resultDiv.innerHTML = html;
    } else {
        resultDiv.innerHTML = `<div class="alert alert-danger">${result.error}</div>`;
    }
}

// Legacy alias
async function checkNewMovies() {
    return checkNewContent();
}

// ==================== Import Series from URL ====================

let importSeriesUrlMode = 'url';

function setImportSeriesUrlMode(mode) {
    importSeriesUrlMode = mode;

    document.querySelectorAll('#importSeriesUrlTabs .nav-link').forEach(t => t.classList.remove('active'));
    event.target.classList.add('active');

    document.getElementById('import-series-url-mode').style.display = mode === 'url' ? 'block' : 'none';
    document.getElementById('import-series-api-mode').style.display = mode === 'api' ? 'block' : 'none';
}

function parseM3uUrl(url) {
    try {
        const urlObj = new URL(url);
        const params = new URLSearchParams(urlObj.search);
        return {
            server: `${urlObj.protocol}//${urlObj.host}`,
            username: params.get('username'),
            password: params.get('password')
        };
    } catch (e) {
        return null;
    }
}

function getSeriesUrlCredentials() {
    if (importSeriesUrlMode === 'url') {
        const m3uUrl = document.getElementById('import-series-m3u-url').value.trim();
        if (!m3uUrl) return null;
        return parseM3uUrl(m3uUrl);
    } else {
        const serverUrl = document.getElementById('import-series-server-url').value.trim();
        const username = document.getElementById('import-series-username').value.trim();
        const password = document.getElementById('import-series-password').value.trim();
        if (!serverUrl || !username || !password) return null;
        return { server: serverUrl, username, password };
    }
}

async function loadSeriesUrlCategories() {
    const creds = getSeriesUrlCredentials();
    if (!creds) {
        showAlert('Please enter valid URL or credentials', 'warning');
        return;
    }

    showAlert('Loading categories...', 'info');

    const result = await apiCall('import/url-categories', 'POST', {
        server: creds.server,
        username: creds.username,
        password: creds.password,
        type: 'series'
    });

    const select = document.getElementById('import-series-category');

    if (result.categories && result.categories.length > 0) {
        select.innerHTML = '<option value="">All Series Categories</option>' +
            result.categories.map(c => `<option value="${c.category_id}">${c.category_name}</option>`).join('');
        showAlert(`Loaded ${result.categories.length} categories`, 'success');
    } else {
        showAlert(result.error || 'No categories found', 'danger');
    }
}

async function previewSeriesUrl() {
    const creds = getSeriesUrlCredentials();
    if (!creds) {
        showAlert('Please enter valid URL or credentials', 'warning');
        return;
    }

    const categoryId = document.getElementById('import-series-category').value;

    showAlert('Loading preview...', 'info');

    const result = await apiCall('import/url-preview', 'POST', {
        server: creds.server,
        username: creds.username,
        password: creds.password,
        type: 'series',
        category_id: categoryId || null,
        limit: 10
    });

    const previewDiv = document.getElementById('import-series-url-preview');
    const contentDiv = document.getElementById('import-series-url-preview-content');

    if (result.items && result.items.length > 0) {
        previewDiv.style.display = 'block';
        contentDiv.innerHTML = `
            <div class="alert alert-success mb-2">Found ${result.total} series</div>
            <div class="list-group" style="max-height: 200px; overflow-y: auto;">
                ${result.items.map(s => `
                    <div class="list-group-item d-flex align-items-center">
                        <img src="${s.cover || ''}" style="width: 40px; height: 60px; object-fit: cover;" class="me-2 rounded">
                        <div>
                            <strong>${s.name}</strong>
                            <div class="small text-secondary">${s.year || ''}</div>
                        </div>
                    </div>
                `).join('')}
            </div>
            <small class="text-secondary">Showing first 10 of ${result.total} series</small>
        `;
    } else {
        previewDiv.style.display = 'block';
        contentDiv.innerHTML = `<div class="alert alert-warning">${result.error || 'No series found'}</div>`;
    }
}

async function importSeriesFromUrl() {
    const creds = getSeriesUrlCredentials();
    if (!creds) {
        showAlert('Please enter valid URL or credentials', 'warning');
        return;
    }

    const serverName = document.getElementById('import-series-server-name').value.trim() || 'URL Import';
    const categoryId = document.getElementById('import-series-category').value;

    showAlert('Starting import... This may take a while.', 'info');

    const result = await apiCall('import/url-series', 'POST', {
        server: creds.server,
        username: creds.username,
        password: creds.password,
        server_name: serverName,
        category_id: categoryId || null
    });

    if (result.success) {
        showAlert(`Import complete: ${result.imported} series imported, ${result.episodes || 0} episodes`, 'success');
        bootstrap.Modal.getInstance(document.getElementById('importSeriesUrlModal')).hide();
        loadSeries(true);
    } else {
        showAlert(result.error || 'Import failed', 'danger');
    }
}

function addImportLog(message, type = 'info') {
    const container = document.getElementById('import-log');
    const timestamp = new Date().toLocaleTimeString('ar-EG');

    container.innerHTML = `
        <div class="log-entry ${type}">
            <span class="text-secondary">[${timestamp}]</span> ${message}
        </div>
    ` + container.innerHTML;
}

// FolderWatch
async function scanWatchFolder() {
    const result = await apiCall('folderwatch/scan', 'POST');

    document.getElementById('scan-count').textContent = `${result.total || 0} files`;
    document.getElementById('valid-count').textContent = result.valid?.length || 0;
    document.getElementById('invalid-count').textContent = result.invalid?.length || 0;

    // Valid files
    const validTable = document.getElementById('valid-files-table');
    if (result.valid && result.valid.length > 0) {
        validTable.innerHTML = result.valid.map(file => `
            <tr>
                <td class="small">${file.filename}</td>
                <td><span class="badge bg-${file.info.type === 'movie' ? 'primary' : 'info'}">${file.info.type}</span></td>
                <td>${file.info.title}</td>
                <td>${file.info.year || '-'}</td>
            </tr>
        `).join('');
    } else {
        validTable.innerHTML = '<tr><td colspan="4" class="text-center">No valid files found</td></tr>';
    }

    // Invalid files
    const invalidTable = document.getElementById('invalid-files-table');
    if (result.invalid && result.invalid.length > 0) {
        invalidTable.innerHTML = result.invalid.map(file => `
            <tr>
                <td class="small">${file.filename}</td>
                <td class="small text-danger">${file.errors?.join(', ') || ''}</td>
                <td class="small text-success">${file.suggestion || '-'}</td>
            </tr>
        `).join('');
    } else {
        invalidTable.innerHTML = '<tr><td colspan="3" class="text-center">No invalid files found</td></tr>';
    }
}

async function processWatchFolder() {
    const result = await apiCall('folderwatch/process', 'POST');

    if (result.processed) {
        showAlert(`Processed ${result.processed.length} files`, 'success');
        scanWatchFolder();
    }
}

async function validateFilename() {
    const filename = document.getElementById('validate-filename').value.trim();
    if (!filename) return;

    const result = await apiCall('folderwatch/validate', 'POST', { filename });

    const resultDiv = document.getElementById('validate-result');
    resultDiv.style.display = 'block';

    if (result.valid) {
        resultDiv.innerHTML = `
            <div class="alert alert-success">
                <strong>Valid!</strong><br>
                Type: ${result.type}<br>
                Title: ${result.title}<br>
                Year: ${result.year || '-'}
                ${result.season ? `<br>Season: ${result.season}, Episode: ${result.episode}` : ''}
            </div>
        `;
    } else {
        resultDiv.innerHTML = `
            <div class="alert alert-danger">
                <strong>Invalid!</strong><br>
                ${result.errors?.join('<br>') || ''}
                ${result.suggestion ? `<br><br><strong>Suggestion:</strong><br><code>${result.suggestion}</code>` : ''}
            </div>
        `;
    }
}

// Logs
async function loadLogs() {
    const container = document.getElementById('logs-container');
    container.innerHTML = '<div class="loading"><div class="spinner-border"></div></div>';

    const status = document.getElementById('log-status').value;
    const type = document.getElementById('log-type').value;
    const limitEl = document.getElementById('log-limit');
    const limit = limitEl ? limitEl.value : 100;

    let endpoint = `logs?limit=${limit}`;
    if (status) endpoint += `&status=${status}`;
    if (type) endpoint += `&type=${type}`;

    // Load logs and stats in parallel
    const [logs, stats] = await Promise.all([
        apiCall(endpoint),
        apiCall('logs/stats')
    ]);

    // Update stats cards
    if (stats && !stats.error) {
        document.getElementById('log-stat-total').textContent = stats.total || 0;
        document.getElementById('log-stat-success').textContent = stats.success || 0;
        document.getElementById('log-stat-warning').textContent = stats.warning || 0;
        document.getElementById('log-stat-error').textContent = stats.error || 0;
    }

    if (Array.isArray(logs) && logs.length > 0) {
        container.innerHTML = logs.map(log => `
            <div class="log-entry ${log.status}" style="padding: 8px 12px; margin-bottom: 4px; border-radius: 6px; background: rgba(255,255,255,0.03); border-left: 3px solid ${log.status === 'error' ? '#ef4444' : log.status === 'warning' ? '#eab308' : log.status === 'success' ? '#22c55e' : '#3b82f6'};">
                <span class="text-secondary small">[${log.created_at}]</span>
                <span class="badge bg-${log.status === 'error' ? 'danger' : log.status === 'warning' ? 'warning' : log.status === 'success' ? 'success' : 'info'} ms-2 me-2">
                    ${log.status}
                </span>
                <strong class="text-light">${log.action}</strong>: <span class="text-secondary">${log.message}</span>
            </div>
        `).join('');
    } else {
        container.innerHTML = '<p class="text-secondary text-center py-4"><i class="bi bi-inbox fs-1 d-block mb-2"></i>No logs found</p>';
    }
}

async function clearLogs(action) {
    console.log('clearLogs called with action:', action);

    const confirmMessages = {
        'clear': 'Are you sure you want to delete ALL logs? This cannot be undone.',
        'success': 'Delete all success logs?',
        'warning': 'Delete all warning logs?',
        'error': 'Delete all error logs?',
        'info': 'Delete all info logs?',
        'old': 'Delete all logs older than 7 days?'
    };

    if (!confirm(confirmMessages[action] || 'Are you sure?')) {
        console.log('User cancelled');
        return;
    }

    try {
        console.log('Making API call to:', `logs/${action}`);
        const result = await apiCall(`logs/${action}`, 'DELETE');
        console.log('API result:', result);

        if (result && result.success) {
            showAlert(result.message || 'Logs cleared successfully', 'success');
            // Force reload logs and clear container first
            const container = document.getElementById('logs-container');
            if (container) {
                container.innerHTML = '<div class="loading"><div class="spinner-border"></div></div>';
            }
            // Update stats to 0
            document.getElementById('log-stat-total').textContent = '0';
            document.getElementById('log-stat-success').textContent = '0';
            document.getElementById('log-stat-warning').textContent = '0';
            document.getElementById('log-stat-error').textContent = '0';
            // Reload logs
            await loadLogs();
        } else {
            showAlert(result.error || 'Failed to clear logs', 'danger');
        }
    } catch (err) {
        console.error('clearLogs error:', err);
        showAlert('Error clearing logs: ' + err.message, 'danger');
    }
}

async function exportLogs() {
    const status = document.getElementById('log-status').value;
    const type = document.getElementById('log-type').value;

    let endpoint = 'logs?limit=10000';
    if (status) endpoint += `&status=${status}`;
    if (type) endpoint += `&type=${type}`;

    const logs = await apiCall(endpoint);

    if (!Array.isArray(logs) || logs.length === 0) {
        showAlert('No logs to export', 'warning');
        return;
    }

    // Create CSV content
    const headers = ['Date', 'Status', 'Action', 'Message'];
    const csvContent = [
        headers.join(','),
        ...logs.map(log => [
            `"${log.created_at}"`,
            `"${log.status}"`,
            `"${log.action}"`,
            `"${(log.message || '').replace(/"/g, '""')}"`
        ].join(','))
    ].join('\n');

    // Download file
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = `logs_${new Date().toISOString().split('T')[0]}.csv`;
    link.click();

    showAlert(`Exported ${logs.length} logs`, 'success');
}

// Settings
async function loadSettings() {
    const settings = await apiCall('settings');

    const form = document.getElementById('settings-form');

    for (const [key, data] of Object.entries(settings)) {
        const input = form.querySelector(`[name="${key}"]`);
        if (input) {
            if (input.type === 'checkbox') {
                input.checked = data.value;
            } else {
                input.value = data.value || '';
            }
        }
    }

    // Update appearance settings selections
    const savedMode = localStorage.getItem('vod-theme-mode') || 'dark';
    // Highlight active theme button
    document.querySelectorAll('.theme-btn').forEach(b => b.classList.remove('active'));
    const activeBtn = document.getElementById(`theme-btn-${savedMode}`);
    if (activeBtn) activeBtn.classList.add('active');

    // Load auto-download settings
    loadAutoDownloadSettings();
}

// Auto Download Functions
async function loadAutoDownloadSettings() {
    // Load remote servers for dropdown
    const servers = await apiCall('remote-servers');
    const serverSelect = document.getElementById('auto-download-server');
    if (serverSelect && Array.isArray(servers)) {
        serverSelect.innerHTML = '<option value="">Select server...</option>' +
            servers.filter(s => s.is_active).map(s =>
                `<option value="${s.id}">${s.name} (${s.host})</option>`
            ).join('');
    }

    // Load saved settings
    const settings = await apiCall('settings');
    if (settings.auto_download_movies) {
        document.getElementById('auto-download-movies').checked = settings.auto_download_movies.value === '1';
    }
    if (settings.auto_download_series) {
        document.getElementById('auto-download-series').checked = settings.auto_download_series.value === '1';
    }
    if (settings.auto_download_server && serverSelect) {
        serverSelect.value = settings.auto_download_server.value || '';
    }
    if (settings.auto_download_schedule) {
        document.getElementById('auto-download-schedule').value = settings.auto_download_schedule.value || 'manual';
    }
    if (settings.auto_download_movies_folder) {
        document.getElementById('auto-download-movies-folder').value = settings.auto_download_movies_folder.value || '';
    }
    if (settings.auto_download_series_folder) {
        document.getElementById('auto-download-series-folder').value = settings.auto_download_series_folder.value || '';
    }
}

async function saveAutoDownloadSettings() {
    const data = {
        auto_download_movies: document.getElementById('auto-download-movies').checked ? '1' : '0',
        auto_download_series: document.getElementById('auto-download-series').checked ? '1' : '0',
        auto_download_server: document.getElementById('auto-download-server').value,
        auto_download_schedule: document.getElementById('auto-download-schedule').value,
        auto_download_movies_folder: document.getElementById('auto-download-movies-folder').value,
        auto_download_series_folder: document.getElementById('auto-download-series-folder').value
    };

    const result = await apiCall('settings', 'POST', data);
    if (result.success) {
        showAlert('Auto download settings saved', 'success');
    } else {
        showAlert('Failed to save settings', 'danger');
    }
}

// Browse folder for auto-download settings
let currentBrowseTarget = null;

async function browseFolder(target) {
    const serverId = document.getElementById('auto-download-server').value;
    if (!serverId) {
        showAlert('Please select a remote server first', 'warning');
        return;
    }

    currentBrowseTarget = target;

    // Open folder browser modal
    const modal = new bootstrap.Modal(document.getElementById('folderBrowserModal'));
    modal.show();

    // Load root folders
    loadFolderBrowser(serverId, '/');
}

async function loadFolderBrowser(serverId, path) {
    const container = document.getElementById('folder-browser-list');
    container.innerHTML = '<div class="text-center py-3"><div class="spinner-border spinner-border-sm"></div> Loading...</div>';

    document.getElementById('folder-browser-path').textContent = path;
    document.getElementById('folder-browser-path').dataset.path = path;

    const result = await apiCall(`remote-servers/browse-folders/${serverId}`, 'POST', { path: path });

    if (result.error) {
        container.innerHTML = `<div class="text-danger p-3">${result.error}</div>`;
        return;
    }

    let html = '';

    // Add parent folder option if not at root
    if (path !== '/') {
        const parentPath = path.split('/').slice(0, -1).join('/') || '/';
        html += `<a href="#" class="list-group-item list-group-item-action" onclick="loadFolderBrowser(${serverId}, '${parentPath}'); return false;">
            <i class="bi bi-arrow-up me-2"></i>.. (Parent Folder)
        </a>`;
    }

    if (result.folders && result.folders.length > 0) {
        html += result.folders.map(folder => `
            <a href="#" class="list-group-item list-group-item-action d-flex justify-content-between align-items-center"
               onclick="loadFolderBrowser(${serverId}, '${folder.path}'); return false;">
                <span><i class="bi bi-folder-fill text-warning me-2"></i>${folder.name}</span>
                <button class="btn btn-sm btn-primary" onclick="selectFolder('${folder.path}'); event.stopPropagation(); return false;">
                    Select
                </button>
            </a>
        `).join('');
    } else {
        html += '<div class="text-muted p-3 text-center">No subfolders found</div>';
    }

    container.innerHTML = html;
}

function selectFolder(path) {
    if (currentBrowseTarget === 'movies') {
        document.getElementById('auto-download-movies-folder').value = path;
    } else if (currentBrowseTarget === 'series') {
        document.getElementById('auto-download-series-folder').value = path;
    }

    bootstrap.Modal.getInstance(document.getElementById('folderBrowserModal')).hide();
    showAlert(`Folder selected: ${path}`, 'success');
}

function selectCurrentFolder() {
    const path = document.getElementById('folder-browser-path').dataset.path;
    selectFolder(path);
}

async function runAutoDownload() {
    const serverId = document.getElementById('auto-download-server').value;
    if (!serverId) {
        showAlert('Please select a remote server first', 'warning');
        return;
    }

    showAlert('Starting auto download...', 'info');

    const result = await apiCall('auto-download/run', 'POST', { server_id: serverId });

    if (result.success) {
        showAlert(`Auto download started: ${result.movies_queued || 0} movies, ${result.episodes_queued || 0} episodes queued`, 'success');
        navigateTo('download-jobs');
    } else {
        showAlert(result.error || 'Failed to start auto download', 'danger');
    }
}

async function saveSettings() {
    const form = document.getElementById('settings-form');
    const formData = new FormData(form);

    const data = {};
    form.querySelectorAll('input, select').forEach(input => {
        if (input.type === 'checkbox') {
            data[input.name] = { value: input.checked, type: 'bool' };
        } else if (input.type === 'number') {
            data[input.name] = { value: parseInt(input.value) || 0, type: 'int' };
        } else {
            data[input.name] = { value: input.value, type: 'string' };
        }
    });

    const result = await apiCall('settings', 'POST', data);

    if (result.success) {
        showAlert('Settings saved', 'success');
    } else {
        showAlert('Failed to save settings', 'danger');
    }
}

// Helpers
function showAlert(message, type = 'info') {
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    alertDiv.style.cssText = 'top: 20px; left: 50%; transform: translateX(-50%); z-index: 9999; min-width: 300px;';
    alertDiv.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;

    document.body.appendChild(alertDiv);

    setTimeout(() => {
        alertDiv.remove();
    }, 5000);
}

async function showMovieDetails(id) {
    const movie = await apiCall(`movies/${id}`);
    if (movie.error) {
        showAlert('Failed to load movie details', 'danger');
        return;
    }

    document.getElementById('movie-detail-title').textContent = movie.title || 'Movie Details';
    document.getElementById('movie-detail-name').textContent = movie.title || '';
    document.getElementById('movie-detail-year').textContent = movie.year || '';
    document.getElementById('movie-detail-plot').textContent = movie.plot || 'No description available';
    document.getElementById('movie-detail-genre').textContent = movie.genre || '-';
    document.getElementById('movie-detail-rating').textContent = movie.rating ? `${movie.rating}/10` : '-';
    document.getElementById('movie-detail-duration').textContent = movie.duration ? `${movie.duration} min` : '-';

    const posterUrl = movie.poster || 'data:image/svg+xml,%3Csvg xmlns=%22http://www.w3.org/2000/svg%22 width=%22200%22 height=%22300%22%3E%3Crect fill=%22%231e293b%22 width=%22200%22 height=%22300%22/%3E%3Ctext fill=%22%2394a3b8%22 x=%2250%25%22 y=%2250%25%22 text-anchor=%22middle%22%3ENo Poster%3C/text%3E%3C/svg%3E';
    document.getElementById('movie-detail-poster').src = posterUrl;

    const links = movie.links || [];
    document.getElementById('movie-detail-links-count').textContent = links.length;

    if (links.length > 0) {
        document.getElementById('movie-detail-links').innerHTML = links.map((link, idx) => `
            <div class="d-flex justify-content-between align-items-center py-1 border-bottom">
                <small class="text-secondary">${link.container_extension?.toUpperCase() || 'MP4'} - Link ${idx + 1}</small>
                <button class="btn btn-sm btn-outline-success" onclick="downloadSingleLink('${link.stream_url || link.url}', '${(movie.title || 'movie').replace(/'/g, "\\'")}')">
                    <i class="bi bi-download"></i>
                </button>
            </div>
        `).join('');
    } else {
        document.getElementById('movie-detail-links').innerHTML = '<p class="text-secondary small">No links available</p>';
    }

    // Setup download button
    document.getElementById('movie-detail-download-btn').onclick = () => {
        selectedMovies.clear();
        selectedMovies.set(movie.id, { id: movie.id, title: movie.title, cleanName: movie.clean_name || '' });
        bootstrap.Modal.getInstance(document.getElementById('movieDetailsModal')).hide();
        showDownloadModal();
    };

    new bootstrap.Modal(document.getElementById('movieDetailsModal')).show();
}

function downloadSingleLink(url, title) {
    window.open(url, '_blank');
}

// ==================== Remote Servers ====================

async function loadRemoteServers() {
    const tbody = document.getElementById('remote-servers-table');
    tbody.innerHTML = '<tr><td colspan="5" class="text-center"><div class="spinner-border spinner-border-sm"></div></td></tr>';

    const servers = await apiCall('remote-servers');

    if (Array.isArray(servers) && servers.length > 0) {
        tbody.innerHTML = servers.map(server => `
            <tr>
                <td>
                    <strong>${server.name}</strong>
                    <span class="badge bg-${server.type === 'ssh' ? 'success' : server.type === 'sftp' ? 'info' : 'warning'} ms-1">${server.type.toUpperCase()}</span>
                    ${server.is_active ? '' : '<span class="badge bg-secondary ms-1">Disabled</span>'}
                    <div class="small text-muted">${server.remote_path || '-'}</div>
                </td>
                <td class="small">${server.host}:${server.port}</td>
                <td>
                    <div id="stats-${server.id}" class="server-stats">
                        <div class="text-muted small"><i class="bi bi-hourglass-split me-1"></i>Loading stats...</div>
                    </div>
                </td>
                <td>
                    <span class="badge bg-${server.status === 'online' ? 'success' : server.status === 'error' ? 'danger' : 'secondary'}">
                        ${server.status || 'Unknown'}
                    </span>
                    <div class="small text-muted">${server.last_check || '-'}</div>
                </td>
                <td>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-primary" onclick="loadServerStats(${server.id})" title="Refresh Stats">
                            <i class="bi bi-arrow-repeat"></i>
                        </button>
                        <button class="btn btn-outline-info" onclick="editRemoteServer(${server.id})" title="Edit">
                            <i class="bi bi-pencil"></i>
                        </button>
                        <button class="btn btn-outline-warning" onclick="toggleRemoteServer(${server.id})" title="Enable/Disable">
                            <i class="bi bi-power"></i>
                        </button>
                        <button class="btn btn-outline-danger" onclick="deleteRemoteServer(${server.id})" title="Delete">
                            <i class="bi bi-trash"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `).join('');

        // Auto-load stats for all active servers
        servers.forEach(server => {
            if (server.is_active && (server.type === 'ssh' || server.type === 'sftp')) {
                loadServerStats(server.id);
            } else {
                const statsDiv = document.getElementById(`stats-${server.id}`);
                if (statsDiv) {
                    statsDiv.innerHTML = '<span class="text-muted small">N/A</span>';
                }
            }
        });
    } else {
        tbody.innerHTML = '<tr><td colspan="5" class="text-center text-secondary">No remote servers configured</td></tr>';
    }
}

async function addRemoteServer() {
    const form = document.getElementById('add-remote-server-form');
    const formData = new FormData(form);
    const data = Object.fromEntries(formData);

    const result = await apiCall('remote-servers/add', 'POST', data);

    if (result.success) {
        bootstrap.Modal.getInstance(document.getElementById('addRemoteServerModal')).hide();
        form.reset();
        loadRemoteServers();
        showAlert('Remote server added successfully', 'success');
    } else {
        showAlert(result.error || 'Failed to add remote server', 'danger');
    }
}

async function editRemoteServer(id) {
    const server = await apiCall(`remote-servers/${id}`);

    if (server.error) {
        showAlert('Failed to load server details', 'danger');
        return;
    }

    // Populate form
    document.getElementById('edit-remote-server-id').value = server.id;
    document.getElementById('edit-rs-name').value = server.name || '';
    document.getElementById('edit-rs-type').value = server.type || 'ssh';
    document.getElementById('edit-rs-host').value = server.host || '';
    document.getElementById('edit-rs-port').value = server.port || 22;
    document.getElementById('edit-rs-username').value = server.username || '';
    document.getElementById('edit-rs-password').value = ''; // Don't show password
    document.getElementById('edit-rs-private-key').value = ''; // Don't show key
    document.getElementById('edit-rs-remote-path').value = server.remote_path || '/var/www/html/media';
    document.getElementById('edit-rs-priority').value = server.priority || 1;
    document.getElementById('edit-rs-active').checked = server.is_active == 1;

    new bootstrap.Modal(document.getElementById('editRemoteServerModal')).show();
}

async function updateRemoteServer() {
    const form = document.getElementById('edit-remote-server-form');
    const formData = new FormData(form);
    const data = {};

    // Only include non-empty values
    for (const [key, value] of formData.entries()) {
        if (key === 'is_active') {
            data[key] = document.getElementById('edit-rs-active').checked ? 1 : 0;
        } else if (value !== '' || key === 'id') {
            data[key] = value;
        }
    }

    // Ensure is_active is set
    if (!data.hasOwnProperty('is_active')) {
        data['is_active'] = document.getElementById('edit-rs-active').checked ? 1 : 0;
    }

    const id = data.id;
    delete data.id;

    const result = await apiCall(`remote-servers/${id}`, 'PUT', data);

    if (result.success) {
        bootstrap.Modal.getInstance(document.getElementById('editRemoteServerModal')).hide();
        loadRemoteServers();
        showAlert('Remote server updated successfully', 'success');
    } else {
        showAlert(result.error || 'Failed to update remote server', 'danger');
    }
}

async function testRemoteServer(id) {
    showAlert('Testing connection...', 'info');
    const result = await apiCall(`remote-servers/test/${id}`, 'POST');

    if (result.success) {
        showAlert(`Connection successful! Response time: ${result.response_time}ms`, 'success');
        loadRemoteServers();
    } else {
        showAlert(`Connection failed: ${result.error}`, 'danger');
        loadRemoteServers();
    }
}

async function toggleRemoteServer(id) {
    const server = await apiCall(`remote-servers/${id}`);
    if (server.error) return;

    const newStatus = server.is_active == 1 ? 0 : 1;
    const result = await apiCall(`remote-servers/${id}`, 'PUT', { is_active: newStatus });

    if (result.success) {
        loadRemoteServers();
        showAlert(`Server ${newStatus ? 'enabled' : 'disabled'}`, 'success');
    } else {
        showAlert('Failed to toggle server status', 'danger');
    }
}

async function deleteRemoteServer(id) {
    if (!confirm('Are you sure you want to delete this remote server?')) return;

    const result = await apiCall(`remote-servers/${id}`, 'DELETE');

    if (result.success) {
        loadRemoteServers();
        showAlert('Remote server deleted', 'success');
    } else {
        showAlert(result.error || 'Failed to delete', 'danger');
    }
}

async function loadServerStats(serverId) {
    console.log('loadServerStats called for server:', serverId);
    const statsDiv = document.getElementById(`stats-${serverId}`);
    if (statsDiv) {
        statsDiv.innerHTML = '<div class="text-muted small"><i class="bi bi-hourglass-split me-1"></i>Loading...</div>';
    }

    const result = await apiCall(`remote-servers/stats/${serverId}`, 'GET');
    console.log('Stats result:', result);

    if (statsDiv) {
        if (result && result.success) {
            let html = '<div class="d-flex flex-wrap gap-2 small">';

            // CPU
            if (result.cpu) {
                const cpuColor = result.cpu.used > 80 ? 'danger' : result.cpu.used > 50 ? 'warning' : 'success';
                html += `<span class="badge bg-${cpuColor}" title="CPU Usage"><i class="bi bi-cpu me-1"></i>${result.cpu.used}%</span>`;
            }

            // RAM
            if (result.ram) {
                const ramColor = result.ram.percent > 80 ? 'danger' : result.ram.percent > 60 ? 'warning' : 'info';
                html += `<span class="badge bg-${ramColor}" title="RAM: ${result.ram.used} / ${result.ram.total}"><i class="bi bi-memory me-1"></i>${result.ram.percent}%</span>`;
            }

            // Disk
            if (result.disk) {
                const diskFree = 100 - result.disk.percent;
                const diskColor = diskFree < 20 ? 'danger' : diskFree < 40 ? 'warning' : 'success';
                html += `<span class="badge bg-${diskColor}" title="Disk: ${result.disk.free} free / ${result.disk.total}"><i class="bi bi-hdd me-1"></i>${result.disk.free}</span>`;
            }

            // Network
            if (result.network) {
                html += `<span class="badge bg-secondary" title="Network RX/TX"><i class="bi bi-arrow-down-up me-1"></i>${result.network.rx}/${result.network.tx}</span>`;
            }

            html += '</div>';
            statsDiv.innerHTML = html;
        } else {
            statsDiv.innerHTML = '<span class="text-danger small"><i class="bi bi-exclamation-circle me-1"></i>Error</span>';
        }
    }
}

async function checkServerStorage(serverId) {
    // Legacy function - redirect to loadServerStats
    loadServerStats(serverId);
}

// ==================== Download Jobs ====================
let downloadJobsRefreshInterval = null;

function startDownloadJobsAutoRefresh() {
    // Clear any existing interval
    stopDownloadJobsAutoRefresh();
    // Refresh every 3 seconds while on download-jobs page
    downloadJobsRefreshInterval = setInterval(() => {
        if (currentPage === 'download-jobs') {
            loadDownloadJobs(true); // silent refresh (no loading spinner)
        } else {
            stopDownloadJobsAutoRefresh();
        }
    }, 3000);
}

function stopDownloadJobsAutoRefresh() {
    if (downloadJobsRefreshInterval) {
        clearInterval(downloadJobsRefreshInterval);
        downloadJobsRefreshInterval = null;
    }
}

async function loadDownloadJobs(silentRefresh = false) {
    const tbody = document.getElementById('download-jobs-table');

    // Only show loading spinner on initial load, not silent refresh
    if (!silentRefresh) {
        tbody.innerHTML = '<tr><td colspan="6" class="text-center"><div class="spinner-border spinner-border-sm"></div></td></tr>';
    }

    // Load stats
    const stats = await apiCall('download-jobs/stats');
    if (!stats.error) {
        document.getElementById('jobs-pending').textContent = stats.pending || 0;
        document.getElementById('jobs-downloading').textContent = stats.downloading || 0;
        document.getElementById('jobs-completed').textContent = stats.completed || 0;
        document.getElementById('jobs-failed').textContent = stats.failed || 0;
    }

    // Load jobs
    const status = document.getElementById('jobs-status-filter')?.value || '';
    let endpoint = 'download-jobs?limit=50';
    if (status) endpoint += `&status=${status}`;

    const jobs = await apiCall(endpoint);

    if (Array.isArray(jobs) && jobs.length > 0) {
        tbody.innerHTML = jobs.map(job => {
            const statusConfig = {
                'completed': { color: 'success', icon: 'check-circle-fill', progress: 'bg-success' },
                'failed': { color: 'danger', icon: 'x-circle-fill', progress: 'bg-danger' },
                'downloading': { color: 'info', icon: 'arrow-down-circle-fill', progress: 'bg-info progress-bar-striped progress-bar-animated' },
                'pending': { color: 'secondary', icon: 'hourglass-split', progress: 'bg-secondary' }
            };
            const config = statusConfig[job.status] || statusConfig.pending;
            const progress = job.status === 'completed' ? 100 : (job.progress || 0);
            const filename = job.destination_path?.split('/').pop() || '-';

            // Format file size
            const formatBytes = (bytes) => {
                if (!bytes || bytes === 0) return '';
                const sizes = ['B', 'KB', 'MB', 'GB'];
                const i = Math.floor(Math.log(bytes) / Math.log(1024));
                return (bytes / Math.pow(1024, i)).toFixed(1) + ' ' + sizes[i];
            };
            const downloaded = formatBytes(job.downloaded_bytes);
            const total = formatBytes(job.total_bytes);
            const sizeInfo = total ? `${downloaded} / ${total}` : '';

            // Calculate speed and time remaining
            const speed = job.download_speed || 0; // bytes/sec
            const speedKBs = speed > 0 ? (speed / 1024).toFixed(1) : '0';
            const speedInfo = speed > 0 ? `${speedKBs} KB/s` : '-';

            let timeRemaining = '-';
            if (speed > 0 && job.total_bytes && job.downloaded_bytes) {
                const remainingBytes = job.total_bytes - job.downloaded_bytes;
                const secondsRemaining = remainingBytes / speed;
                if (secondsRemaining < 60) {
                    timeRemaining = `${Math.ceil(secondsRemaining)}s`;
                } else if (secondsRemaining < 3600) {
                    timeRemaining = `${Math.ceil(secondsRemaining / 60)}m`;
                } else {
                    const hours = Math.floor(secondsRemaining / 3600);
                    const mins = Math.ceil((secondsRemaining % 3600) / 60);
                    timeRemaining = `${hours}h ${mins}m`;
                }
            }

            return `
            <tr style="border-bottom: 1px solid var(--border-color);">
                <td class="text-light align-middle">${job.id}</td>
                <td class="align-middle">
                    <span class="badge bg-primary bg-opacity-25 text-primary">${job.server_name || '-'}</span>
                </td>
                <td class="align-middle">
                    <div class="d-flex align-items-center">
                        <i class="bi bi-file-earmark-play text-info me-2"></i>
                        <span class="text-light small text-truncate" style="max-width: 250px;" title="${job.destination_path}">${filename}</span>
                    </div>
                </td>
                <td class="align-middle">
                    <div class="d-flex flex-column gap-1">
                        <div class="d-flex align-items-center gap-2">
                            <div class="progress flex-grow-1" style="height: 10px; background: rgba(255,255,255,0.1); border-radius: 5px;">
                                <div class="progress-bar ${config.progress}" style="width: ${progress}%; border-radius: 5px; transition: width 0.3s;"></div>
                            </div>
                            <span class="text-light fw-bold" style="width: 45px;">${progress}%</span>
                        </div>
                        <div class="d-flex justify-content-between">
                            ${sizeInfo ? `<small class="text-secondary">${sizeInfo}</small>` : ''}
                            ${job.status === 'downloading' ? `<small class="text-info">${speedInfo}</small>` : ''}
                        </div>
                    </div>
                </td>
                <td class="align-middle">
                    <span class="badge bg-${config.color} d-flex align-items-center gap-1" style="width: fit-content;">
                        <i class="bi bi-${config.icon}"></i>
                        ${job.status}
                    </span>
                </td>
                <td class="align-middle">
                    <div class="d-flex flex-column">
                        <small class="text-secondary">${job.created_at?.substring(5, 16) || '-'}</small>
                        ${job.status === 'downloading' ? `<small class="text-info">⏱ ${timeRemaining}</small>` : ''}
                    </div>
                </td>
                <td class="align-middle">
                    <div class="btn-group btn-group-sm">
                        ${job.status === 'failed' ? `<button class="btn btn-outline-warning" onclick="retryJob(${job.id})" title="Retry"><i class="bi bi-arrow-repeat"></i></button>` : ''}
                        ${job.status === 'downloading' ? `<button class="btn btn-outline-danger" onclick="cancelJob(${job.id})" title="Cancel"><i class="bi bi-stop-fill"></i></button>` : ''}
                        <button class="btn btn-outline-secondary" onclick="deleteJob(${job.id})" title="Delete"><i class="bi bi-trash"></i></button>
                    </div>
                </td>
            </tr>
        `}).join('');
    } else {
        tbody.innerHTML = `<tr><td colspan="7" class="text-center py-5">
            <i class="bi bi-inbox fs-1 text-secondary d-block mb-2"></i>
            <span class="text-secondary">No download jobs</span>
        </td></tr>`;
    }
}

// ==================== Users Management ====================

async function loadUsers() {
    const tbody = document.getElementById('users-table');
    if (!tbody) return;

    tbody.innerHTML = '<tr><td colspan="7" class="text-center"><div class="spinner-border spinner-border-sm"></div></td></tr>';

    const users = await apiCall('users');

    if (Array.isArray(users) && users.length > 0) {
        // Update stats
        const totalEl = document.getElementById('user-stat-total');
        const adminEl = document.getElementById('user-stat-admin');
        const activeEl = document.getElementById('user-stat-active');
        const inactiveEl = document.getElementById('user-stat-inactive');

        if (totalEl) totalEl.textContent = users.length;
        if (adminEl) adminEl.textContent = users.filter(u => u.role === 'admin').length;
        if (activeEl) activeEl.textContent = users.filter(u => u.is_active).length;
        if (inactiveEl) inactiveEl.textContent = users.filter(u => !u.is_active).length;

        tbody.innerHTML = users.map(user => `
            <tr>
                <td>
                    <div class="d-flex align-items-center">
                        <div class="avatar-sm rounded-circle bg-${user.role === 'admin' ? 'danger' : 'primary'} text-white d-flex align-items-center justify-content-center me-2" style="width: 32px; height: 32px; font-size: 14px;">
                            ${user.username.charAt(0).toUpperCase()}
                        </div>
                        <strong class="text-light">${user.username}</strong>
                    </div>
                </td>
                <td class="text-secondary">${user.email || '-'}</td>
                <td>
                    <span class="badge bg-${user.role === 'admin' ? 'danger' : user.role === 'user' ? 'primary' : 'secondary'}">
                        <i class="bi bi-${user.role === 'admin' ? 'shield-check' : 'person'} me-1"></i>${user.role}
                    </span>
                </td>
                <td>
                    <span class="badge bg-${user.is_active ? 'success' : 'secondary'}">
                        <i class="bi bi-${user.is_active ? 'check-circle' : 'x-circle'} me-1"></i>
                        ${user.is_active ? 'Active' : 'Disabled'}
                    </span>
                </td>
                <td class="small text-secondary">${user.last_login || 'Never'}</td>
                <td class="small text-secondary">${user.login_ip || '-'}</td>
                <td>
                    <div class="dropdown">
                        <button class="btn btn-sm btn-outline-secondary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                            <i class="bi bi-three-dots"></i>
                        </button>
                        <ul class="dropdown-menu dropdown-menu-end dropdown-menu-dark">
                            <li><a class="dropdown-item" href="#" onclick="showEditUser(${user.id}, '${user.username}', '${user.email || ''}', '${user.role}')"><i class="bi bi-pencil me-2"></i>Edit User</a></li>
                            <li><a class="dropdown-item" href="#" onclick="showResetPassword(${user.id}, '${user.username}')"><i class="bi bi-key me-2"></i>Reset Password</a></li>
                            <li><a class="dropdown-item" href="#" onclick="generateApiToken(${user.id}, '${user.username}')"><i class="bi bi-code-slash me-2"></i>Generate API Token</a></li>
                            <li><hr class="dropdown-divider"></li>
                            <li><a class="dropdown-item ${user.is_active ? 'text-warning' : 'text-success'}" href="#" onclick="toggleUser(${user.id})">
                                <i class="bi bi-power me-2"></i>${user.is_active ? 'Disable User' : 'Enable User'}
                            </a></li>
                            <li><hr class="dropdown-divider"></li>
                            <li><a class="dropdown-item text-danger" href="#" onclick="deleteUser(${user.id})"><i class="bi bi-trash me-2"></i>Delete User</a></li>
                        </ul>
                    </div>
                </td>
            </tr>
        `).join('');
    } else {
        tbody.innerHTML = '<tr><td colspan="7" class="text-center text-secondary py-4"><i class="bi bi-people fs-1 d-block mb-2"></i>No users found</td></tr>';
    }
}

async function addUser() {
    const form = document.getElementById('add-user-form');
    const formData = new FormData(form);
    const data = Object.fromEntries(formData);

    const result = await apiCall('users/add', 'POST', data);

    if (result.success) {
        bootstrap.Modal.getInstance(document.getElementById('addUserModal')).hide();
        form.reset();
        loadUsers();
        showAlert('User added successfully', 'success');
    } else {
        showAlert(result.error || 'Failed to add user', 'danger');
    }
}

async function toggleUser(id) {
    const result = await apiCall(`users/toggle/${id}`, 'POST');
    if (result.success) {
        loadUsers();
    }
}

async function deleteUser(id) {
    if (!confirm('Are you sure you want to delete this user?')) return;

    const result = await apiCall(`users/${id}`, 'DELETE');

    if (result.success) {
        loadUsers();
        showAlert('User deleted', 'success');
    } else {
        showAlert(result.error || 'Failed to delete user', 'danger');
    }
}

function showEditUser(id, username, email, role) {
    document.getElementById('edit-user-id').value = id;
    document.getElementById('edit-user-username').value = username;
    document.getElementById('edit-user-email').value = email;
    document.getElementById('edit-user-role').value = role;
    new bootstrap.Modal(document.getElementById('editUserModal')).show();
}

async function saveEditUser() {
    const form = document.getElementById('edit-user-form');
    const formData = new FormData(form);
    const data = Object.fromEntries(formData);
    const id = data.id;
    delete data.id;

    const result = await apiCall(`users/${id}`, 'PUT', data);

    if (result.success) {
        bootstrap.Modal.getInstance(document.getElementById('editUserModal')).hide();
        loadUsers();
        showAlert('User updated successfully', 'success');
    } else {
        showAlert(result.error || 'Failed to update user', 'danger');
    }
}

function showResetPassword(id, username) {
    document.getElementById('reset-password-user-id').value = id;
    document.getElementById('reset-password-username').textContent = username;
    document.getElementById('reset-password-input').value = '';
    document.getElementById('reset-password-confirm').value = '';
    new bootstrap.Modal(document.getElementById('resetPasswordModal')).show();
}

async function saveResetPassword() {
    const password = document.getElementById('reset-password-input').value;
    const confirm = document.getElementById('reset-password-confirm').value;
    const id = document.getElementById('reset-password-user-id').value;

    if (password !== confirm) {
        showAlert('Passwords do not match', 'danger');
        return;
    }

    if (password.length < 6) {
        showAlert('Password must be at least 6 characters', 'danger');
        return;
    }

    const result = await apiCall(`users/${id}/password`, 'PUT', { password });

    if (result.success) {
        bootstrap.Modal.getInstance(document.getElementById('resetPasswordModal')).hide();
        showAlert('Password reset successfully', 'success');
    } else {
        showAlert(result.error || 'Failed to reset password', 'danger');
    }
}

async function generateApiToken(id, username) {
    if (!confirm(`Generate new API token for ${username}? This will invalidate any existing token.`)) return;

    const result = await apiCall(`users/${id}/token`, 'POST');

    if (result.success && result.token) {
        document.getElementById('api-token-username').textContent = username;
        document.getElementById('api-token-value').value = result.token;
        new bootstrap.Modal(document.getElementById('apiTokenModal')).show();
    } else {
        showAlert(result.error || 'Failed to generate token', 'danger');
    }
}

function copyApiToken() {
    const tokenInput = document.getElementById('api-token-value');
    tokenInput.select();
    document.execCommand('copy');
    showAlert('Token copied to clipboard', 'success');
}

// ==================== Series Functions ====================

async function loadSeries(reset = true) {
    if (reset) {
        seriesOffset = 0;
        allSeriesData = [];
        loadSeriesFilters();
    }

    const container = document.getElementById('series-grid');
    if (!container) return;

    if (seriesOffset === 0) {
        container.innerHTML = '<div class="col-12 text-center"><div class="spinner-border"></div></div>';
    }

    let endpoint = `series?limit=${seriesLimit}&offset=${seriesOffset}`;
    // Category filter disabled - old category_id system doesn't match imported data
    // if (seriesFilters.category) endpoint += `&category_id=${seriesFilters.category}`;
    if (seriesFilters.year) endpoint += `&year=${seriesFilters.year}`;
    if (seriesFilters.search) endpoint += `&search=${encodeURIComponent(seriesFilters.search)}`;

    const series = await apiCall(endpoint);

    if (Array.isArray(series) && series.length > 0) {
        allSeriesData = reset ? series : [...allSeriesData, ...series];

        // Client-side category filtering by genre
        let filteredSeries = allSeriesData;
        if (seriesFilters.category) {
            const categoryName = getCategoryNameById(seriesFilters.category);
            console.log('Category filter:', seriesFilters.category, '→', categoryName);
            if (categoryName) {
                const keywords = extractCategoryKeywords(categoryName);
                console.log('Keywords:', keywords);
                filteredSeries = allSeriesData.filter(s => {
                    if (!s.genre) return false;
                    const genreLower = s.genre.toLowerCase();
                    // Match if any keyword is found in the genre
                    return keywords.some(keyword => genreLower.includes(keyword));
                });
                console.log('Filtered from', allSeriesData.length, 'to', filteredSeries.length, 'series');
            }
        }

        const html = filteredSeries.map(s => {
            try {
                return renderSeriesCard(s);
            } catch (e) {
                console.error('Error rendering series card:', s, e);
                return '';
            }
        }).join('');

        if (!html || html.trim() === '') {
            if (seriesOffset === 0) {
                if (seriesFilters.category && filteredSeries.length === 0) {
                    container.innerHTML = `
                        <div class="col-12 text-center text-secondary">
                            <div class="py-5">
                                <i class="bi bi-funnel" style="font-size: 4rem; opacity: 0.3;"></i>
                                <h5 class="mt-3">No Series Found</h5>
                                <p>No series match the selected category filter.</p>
                                <button onclick="clearSeriesFilters()" class="btn btn-primary mt-2">
                                    <i class="bi bi-x-circle"></i> Clear Filters
                                </button>
                            </div>
                        </div>`;
                } else {
                    container.innerHTML = `<div class="col-12 alert alert-warning">Error rendering series. Check console for details.</div>`;
                }
            }
            return;
        }

        if (seriesOffset === 0) {
            container.innerHTML = html;
        } else {
            container.innerHTML += html;
        }

        seriesOffset += seriesLimit;
        updateSeriesStats();

        document.getElementById('load-more-series').style.display =
            series.length < seriesLimit ? 'none' : 'block';
    } else if (seriesOffset === 0) {
        // Check if there's a category filter causing no results
        if (seriesFilters.category) {
            container.innerHTML = `
                <div class="col-12 text-center text-secondary">
                    <div class="py-5">
                        <i class="bi bi-funnel" style="font-size: 4rem; opacity: 0.3;"></i>
                        <h5 class="mt-3">No Series Found</h5>
                        <p>No series match the selected category filter.</p>
                        <button onclick="clearSeriesFilters()" class="btn btn-primary mt-2">
                            <i class="bi bi-x-circle"></i> Clear Filters
                        </button>
                    </div>
                </div>`;
        } else {
            container.innerHTML = `
                <div class="col-12 text-center text-secondary">
                    <div class="py-5">
                        <i class="bi bi-collection-play" style="font-size: 4rem; opacity: 0.3;"></i>
                        <h5 class="mt-3">No Series Found</h5>
                        <p>Import series from your source servers to see them here.</p>
                        <a href="#" onclick="navigateTo('import')" class="btn btn-primary mt-2">
                            <i class="bi bi-cloud-download"></i> Go to Import
                        </a>
                    </div>
                </div>`;
        }
        document.getElementById('load-more-series').style.display = 'none';
    }
}

function renderSeriesCard(series) {
    const isSelected = selectedSeries.has(series.id);
    const posterUrl = series.poster || 'data:image/svg+xml,%3Csvg xmlns=%22http://www.w3.org/2000/svg%22 width=%22200%22 height=%22300%22%3E%3Crect fill=%22%231e293b%22 width=%22200%22 height=%22300%22/%3E%3Ctext fill=%22%2394a3b8%22 x=%2250%25%22 y=%2250%25%22 text-anchor=%22middle%22%3ENo Poster%3C/text%3E%3C/svg%3E';
    const safeTitle = (series.title || '').replace(/'/g, "\\'").replace(/"/g, '&quot;');

    if (seriesViewMode === 'grid') {
        return `
            <div class="col-xl-2 col-lg-3 col-md-4 col-6">
                <div class="movie-card ${isSelected ? 'border border-success' : ''}" data-series-id="${series.id}">
                    <div class="position-relative">
                        <img src="${posterUrl}" class="movie-poster" alt="${safeTitle}" loading="lazy"
                            onclick="showSeriesDetails(${series.id})" style="cursor:pointer;">
                        <div style="position:absolute;top:5px;left:5px;">
                            <input type="checkbox" class="form-check-input series-checkbox" style="transform:scale(1.3);"
                                ${isSelected ? 'checked' : ''}
                                onchange="toggleSeriesSelection(${series.id}, '${safeTitle}', event)">
                        </div>
                        <span class="badge bg-info position-absolute" style="top:5px;right:5px;">
                            ${series.episode_count || 0} eps
                        </span>
                    </div>
                    <div class="movie-info">
                        <div class="movie-title" title="${safeTitle}">${series.title}</div>
                        <div class="d-flex justify-content-between align-items-center">
                            <span class="movie-year">${series.year || ''}</span>
                            <span class="badge bg-${series.status === 'ongoing' ? 'success' : 'secondary'}">${series.status || 'N/A'}</span>
                        </div>
                    </div>
                </div>
            </div>
        `;
    } else {
        return `
            <div class="col-12">
                <div class="card mb-2 ${isSelected ? 'border-success' : ''}" data-series-id="${series.id}">
                    <div class="card-body p-2">
                        <div class="d-flex align-items-center">
                            <input type="checkbox" class="form-check-input me-3 series-checkbox"
                                ${isSelected ? 'checked' : ''}
                                onchange="toggleSeriesSelection(${series.id}, '${safeTitle}', event)">
                            <img src="${posterUrl}" style="width:50px;height:75px;object-fit:cover;cursor:pointer;"
                                class="rounded me-3" onclick="showSeriesDetails(${series.id})">
                            <div class="flex-grow-1">
                                <strong>${series.title}</strong>
                                <div class="small text-secondary">${series.year || ''} | ${series.seasons_count || 0} seasons</div>
                            </div>
                            <span class="badge bg-info">${series.episode_count || 0} episodes</span>
                            <button class="btn btn-sm btn-outline-success ms-2" onclick="showSeriesDetails(${series.id})">
                                <i class="bi bi-eye"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;
    }
}

async function loadSeriesFilters() {
    const categories = await apiCall('categories?type=series');
    const categorySelect = document.getElementById('series-category-filter');
    if (categorySelect && Array.isArray(categories)) {
        // Store category mapping for client-side filtering
        seriesCategoriesMap = {};
        categories.forEach(cat => {
            seriesCategoriesMap[cat.external_id] = cat.name;
        });

        categorySelect.innerHTML = '<option value="">All Categories</option>' +
            categories.map(cat => `<option value="${cat.external_id}">${cat.name}</option>`).join('');
    }

    const years = await apiCall('series/years');
    const yearSelect = document.getElementById('series-year-filter');
    if (yearSelect && Array.isArray(years)) {
        yearSelect.innerHTML = '<option value="">All Years</option>' +
            years.map(y => `<option value="${y}">${y}</option>`).join('');
    }
}

// Helper function to get category name by ID
function getCategoryNameById(categoryId) {
    return seriesCategoriesMap[categoryId] || null;
}

// Extract keywords from category name for matching
function extractCategoryKeywords(categoryName) {
    if (!categoryName) return [];

    // Remove special characters and symbols
    const cleaned = categoryName.replace(/[‖|]/g, '').trim();

    // Split by spaces and common separators
    const words = cleaned.split(/[\s,]+/).filter(w => w.length > 2);

    // Add keyword mappings for better matching
    const keywords = words.map(w => w.toLowerCase());
    const extraKeywords = [];

    // Anime/Animation mapping
    if (keywords.some(k => k.includes('anime') || k.includes('انمي'))) {
        extraKeywords.push('رسوم متحركة', 'animation', 'أنمي', 'انمي', 'anime');
    }

    // Arabic series mapping
    if (keywords.some(k => k.includes('arabic') || k.includes('عربي'))) {
        extraKeywords.push('دراما', 'عربي', 'عائلي', 'drama', 'arabic');
    }

    // Netflix mapping
    if (keywords.some(k => k.includes('netflix') || k.includes('نتفليكس'))) {
        extraKeywords.push('netflix', 'نتفليكس');
    }

    // HBO mapping
    if (keywords.some(k => k.includes('hbo'))) {
        extraKeywords.push('hbo');
    }

    // Combine original keywords with extra mappings
    return [...new Set([...keywords, ...extraKeywords])];
}

function filterSeries() {
    seriesFilters.category = document.getElementById('series-category-filter')?.value || '';
    seriesFilters.year = document.getElementById('series-year-filter')?.value || '';
    loadSeries(true);
}

// NEW: Filter by actual genre data (WORKING!)
function filterSeriesByGenre() {
    const genreValue = document.getElementById('series-genre-filter')?.value || '';
    console.log('Genre filter selected:', genreValue);

    if (!genreValue) {
        // Show all series
        displayFilteredSeries(allSeriesData);
        return;
    }

    // Genre mapping for matching
    const genreMap = {
        'animation': ['animation', 'رسوم متحركة', 'أنمي', 'anime'],
        'drama': ['drama', 'دراما'],
        'comedy': ['comedy', 'كوميديا'],
        'action': ['action', 'أكشن', 'حركة'],
        'crime': ['crime', 'جريمة'],
        'mystery': ['mystery', 'غموض'],
        'adventure': ['adventure', 'مغامرات', 'مغامرة'],
        'sci-fi': ['sci-fi', 'خيال علمي', 'fantasy', 'فانتازيا'],
        'family': ['family', 'عائلي']
    };

    const keywords = genreMap[genreValue] || [genreValue];
    console.log('Matching keywords:', keywords);

    const filtered = allSeriesData.filter(s => {
        if (!s.genre) return false;
        const genreLower = s.genre.toLowerCase();
        return keywords.some(keyword => genreLower.includes(keyword.toLowerCase()));
    });

    console.log('Filtered from', allSeriesData.length, 'to', filtered.length, 'series');
    displayFilteredSeries(filtered);
}

// Display filtered series
function displayFilteredSeries(series) {
    const container = document.getElementById('series-grid');
    if (!container) return;

    if (series.length === 0) {
        container.innerHTML = `
            <div class="col-12 text-center text-secondary">
                <div class="py-5">
                    <i class="bi bi-funnel" style="font-size: 4rem; opacity: 0.3;"></i>
                    <h5 class="mt-3">No Series Found</h5>
                    <p>No series match the selected genre filter.</p>
                    <button onclick="clearGenreFilter()" class="btn btn-primary mt-2">
                        <i class="bi bi-x-circle"></i> Clear Filter
                    </button>
                </div>
            </div>`;
        return;
    }

    const html = series.map(s => {
        try {
            return renderSeriesCard(s);
        } catch (e) {
            console.error('Error rendering series card:', s, e);
            return '';
        }
    }).join('');

    container.innerHTML = html;
    updateSeriesStats();
}

function clearGenreFilter() {
    const genreSelect = document.getElementById('series-genre-filter');
    if (genreSelect) genreSelect.value = '';
    displayFilteredSeries(allSeriesData);
}

function clearSeriesFilters() {
    seriesFilters.category = '';
    seriesFilters.year = '';
    seriesFilters.search = '';
    const categorySelect = document.getElementById('series-category-filter');
    const yearSelect = document.getElementById('series-year-filter');
    const searchInput = document.getElementById('series-search');
    if (categorySelect) categorySelect.value = '';
    if (yearSelect) yearSelect.value = '';
    if (searchInput) searchInput.value = '';
    loadSeries(true);
}

function loadMoreSeries() {
    loadSeries(false);
}

async function searchSeries() {
    seriesFilters.search = document.getElementById('series-search').value.trim();
    loadSeries(true);
}

function setSeriesView(mode) {
    seriesViewMode = mode;
    document.getElementById('series-view-grid')?.classList.toggle('active', mode === 'grid');
    document.getElementById('series-view-list')?.classList.toggle('active', mode === 'list');
    const container = document.getElementById('series-grid');
    container.innerHTML = allSeriesData.map(series => renderSeriesCard(series)).join('');
}

function toggleSeriesSelection(id, title, event) {
    event?.stopPropagation();
    if (selectedSeries.has(id)) {
        selectedSeries.delete(id);
    } else {
        selectedSeries.set(id, { id, title });
    }
    updateSeriesSelectionUI();
}

function toggleSelectAllSeries() {
    const selectAll = document.getElementById('select-all-series')?.checked;
    if (selectAll) {
        allSeriesData.forEach(series => {
            selectedSeries.set(series.id, { id: series.id, title: series.title });
        });
    } else {
        selectedSeries.clear();
    }
    updateSeriesSelectionUI();
    document.querySelectorAll('.series-checkbox').forEach(cb => cb.checked = selectAll);
}

function updateSeriesSelectionUI() {
    const count = selectedSeries.size;
    const countEl = document.getElementById('selected-series-count');
    const btnEl = document.getElementById('download-selected-series-btn');
    if (countEl) countEl.textContent = count;
    if (btnEl) btnEl.style.display = count > 0 ? 'inline-block' : 'none';
    updateSeriesStats();
}

function updateSeriesStats() {
    const stats = document.getElementById('series-stats');
    if (stats) {
        const selectedCount = selectedSeries.size;
        stats.innerHTML = `<span class="text-info">Showing ${allSeriesData.length} series</span> | <span class="${selectedCount > 0 ? 'text-success fw-bold' : 'text-secondary'}">${selectedCount} selected</span>`;
    }
}

async function showSeriesDetails(id) {
    console.log('Loading series details for ID:', id);
    const series = await apiCall(`series/${id}`);
    console.log('Series API response:', series);
    console.log('Episodes count:', series.episodes?.length);
    console.log('Seasons:', series.seasons);

    if (series.error) {
        showAlert('Failed to load series details', 'danger');
        return;
    }

    currentSeriesDetail = series;
    selectedEpisodes.clear();

    document.getElementById('series-detail-title').textContent = series.title || 'Series Details';
    document.getElementById('series-detail-name').textContent = series.title || '';
    document.getElementById('series-detail-year').textContent = series.year || '';
    document.getElementById('series-detail-plot').textContent = series.plot || series.description || 'No description available';
    document.getElementById('series-detail-genre').textContent = series.genre || '-';
    document.getElementById('series-detail-rating').textContent = series.rating ? `${series.rating}/10` : '-';
    document.getElementById('series-detail-seasons').textContent = series.seasons_count || Object.keys(series.seasons || {}).length;

    const posterUrl = series.poster || 'data:image/svg+xml,%3Csvg xmlns=%22http://www.w3.org/2000/svg%22 width=%22200%22 height=%22300%22%3E%3Crect fill=%22%231e293b%22 width=%22200%22 height=%22300%22/%3E%3Ctext fill=%22%2394a3b8%22 x=%2250%25%22 y=%2250%25%22 text-anchor=%22middle%22%3ENo Poster%3C/text%3E%3C/svg%3E';
    document.getElementById('series-detail-poster').src = posterUrl;

    // Populate season dropdown
    const seasonSelect = document.getElementById('series-season-select');
    const seasons = Object.keys(series.seasons || {}).sort((a, b) => a - b);
    seasonSelect.innerHTML = '<option value="">All Seasons</option>' +
        seasons.map(s => `<option value="${s}">Season ${s}</option>`).join('');

    // Show episodes
    renderEpisodesList(series.episodes || []);

    updateSelectedEpisodesCount();

    new bootstrap.Modal(document.getElementById('seriesDetailsModal')).show();
}

function renderEpisodesList(episodes) {
    const container = document.getElementById('series-episodes-list');

    if (!episodes || episodes.length === 0) {
        container.innerHTML = '<p class="text-secondary">No episodes available</p>';
        return;
    }

    container.innerHTML = episodes.map(ep => `
        <div class="d-flex align-items-center py-2 border-bottom episode-item" data-episode-id="${ep.id}">
            <input type="checkbox" class="form-check-input me-3 episode-checkbox"
                ${selectedEpisodes.has(ep.id) ? 'checked' : ''}
                onchange="toggleEpisodeSelection(${ep.id}, '${(ep.title || '').replace(/'/g, "\\'")}', ${ep.season_number}, ${ep.episode_number})">
            <div class="flex-grow-1">
                <strong>S${String(ep.season_number).padStart(2, '0')}E${String(ep.episode_number).padStart(2, '0')}</strong>
                <span class="ms-2">${ep.title || 'Episode ' + ep.episode_number}</span>
                <div class="small text-secondary">${ep.duration ? ep.duration + ' min' : ''}</div>
            </div>
            <span class="badge bg-${ep.status === 'downloaded' ? 'success' : 'secondary'}">${ep.container_extension || 'mp4'}</span>
        </div>
    `).join('');
}

function loadSeasonEpisodes() {
    const season = document.getElementById('series-season-select').value;
    if (!currentSeriesDetail) return;

    let episodes = currentSeriesDetail.episodes || [];
    if (season) {
        episodes = episodes.filter(ep => ep.season_number == season);
    }
    renderEpisodesList(episodes);
}

function toggleEpisodeSelection(id, title, season, episode) {
    if (selectedEpisodes.has(id)) {
        selectedEpisodes.delete(id);
    } else {
        selectedEpisodes.set(id, { id, title, season, episode });
    }
    updateSelectedEpisodesCount();
}

function selectAllEpisodes() {
    const season = document.getElementById('series-season-select').value;
    let episodes = currentSeriesDetail?.episodes || [];
    if (season) {
        episodes = episodes.filter(ep => ep.season_number == season);
    }

    const allSelected = episodes.every(ep => selectedEpisodes.has(ep.id));

    if (allSelected) {
        episodes.forEach(ep => selectedEpisodes.delete(ep.id));
    } else {
        episodes.forEach(ep => {
            selectedEpisodes.set(ep.id, {
                id: ep.id,
                title: ep.title,
                season: ep.season_number,
                episode: ep.episode_number,
                url: ep.stream_url,
                extension: ep.container_extension || 'mp4'
            });
        });
    }

    document.querySelectorAll('.episode-checkbox').forEach(cb => cb.checked = !allSelected);
    updateSelectedEpisodesCount();
}

function updateSelectedEpisodesCount() {
    const countEl = document.getElementById('selected-episodes-count');
    if (countEl) countEl.textContent = selectedEpisodes.size;
}

async function downloadSelectedEpisodes() {
    if (selectedEpisodes.size === 0) {
        showAlert('Please select episodes to download', 'warning');
        return;
    }

    // Close series modal
    bootstrap.Modal.getInstance(document.getElementById('seriesDetailsModal'))?.hide();

    // Set download type
    downloadType = 'episodes';

    // Show download modal with episodes
    await showDownloadModalForEpisodes();
}

async function showDownloadModalForEpisodes() {
    const servers = await apiCall('remote-servers');
    const serverSelect = document.getElementById('download-remote-server');
    if (serverSelect && Array.isArray(servers)) {
        serverSelect.innerHTML = '<option value="">-- Select Server --</option>' +
            servers.filter(s => s.is_active).map(s => `<option value="${s.id}" data-path="${s.remote_path}">${s.name} (${s.type} - ${s.host})</option>`).join('');
    }

    document.getElementById('download-movie-count').textContent = selectedEpisodes.size;
    document.getElementById('download-type-label').textContent = 'episodes';

    const seriesName = currentSeriesDetail?.title || 'Unknown Series';
    document.getElementById('selected-movies-list').innerHTML =
        `<div class="small py-1 border-bottom"><strong class="text-info">${seriesName}</strong></div>` +
        Array.from(selectedEpisodes.entries())
            .map(([id, ep]) => `<div class="small py-1 border-bottom d-flex justify-content-between align-items-center">
                <span>S${String(ep.season).padStart(2, '0')}E${String(ep.episode).padStart(2, '0')} - ${ep.title || 'Episode'}</span>
                <button type="button" class="btn btn-sm btn-outline-danger py-0 px-1" onclick="removeFromDownload(${id}, 'episode')"><i class="bi bi-x"></i></button>
            </div>`).join('');

    // Load categories
    await loadCategoryFolders();

    new bootstrap.Modal(document.getElementById('downloadModal')).show();
}

async function showSeriesDownloadModal() {
    if (selectedSeries.size === 0) {
        showAlert('Please select series', 'warning');
        return;
    }

    downloadType = 'series';

    const servers = await apiCall('remote-servers');
    const serverSelect = document.getElementById('download-remote-server');
    if (serverSelect && Array.isArray(servers)) {
        serverSelect.innerHTML = '<option value="">-- Select Server --</option>' +
            servers.filter(s => s.is_active).map(s => `<option value="${s.id}" data-path="${s.remote_path}">${s.name} (${s.type} - ${s.host})</option>`).join('');
    }

    document.getElementById('download-movie-count').textContent = selectedSeries.size;
    document.getElementById('download-type-label').textContent = 'series';
    document.getElementById('selected-movies-list').innerHTML = Array.from(selectedSeries.entries())
        .map(([id, s]) => `<div class="small py-1 border-bottom d-flex justify-content-between align-items-center">
            <span>${s.title}</span>
            <button type="button" class="btn btn-sm btn-outline-danger py-0 px-1" onclick="removeFromDownload(${id}, 'series')"><i class="bi bi-x"></i></button>
        </div>`).join('');

    await loadCategoryFolders();

    new bootstrap.Modal(document.getElementById('downloadModal')).show();
}

// ==================== Folder Selection Functions ====================

function toggleCustomFolder() {
    const folderType = document.getElementById('download-folder-type').value;
    document.getElementById('category-folder-group').style.display = folderType === 'category' ? 'block' : 'none';
    document.getElementById('custom-folder-group').style.display = folderType === 'custom' ? 'block' : 'none';
    document.getElementById('browse-folder-group').style.display = folderType === 'browse' ? 'block' : 'none';

    // Load folders when browse is selected
    if (folderType === 'browse') {
        const serverId = document.getElementById('download-remote-server').value;
        if (serverId) {
            browseFolders(serverId);
        } else {
            showAlert('Please select a server first', 'warning');
            document.getElementById('download-folder-type').value = 'none';
            document.getElementById('browse-folder-group').style.display = 'none';
        }
    }
}

async function loadServerFolders() {
    const serverId = document.getElementById('download-remote-server').value;
    if (!serverId) return;

    const serverSelect = document.getElementById('download-remote-server');
    const selectedOption = serverSelect.options[serverSelect.selectedIndex];
    const remotePath = selectedOption.dataset.path || '/var/www/html/media';

    document.getElementById('base-path-label').textContent = remotePath + '/';

    await loadCategoryFolders();
}

async function loadCategoryFolders() {
    const type = downloadType === 'movies' ? 'movie' : 'series';
    const categories = await apiCall(`categories?type=${type}`);

    const categorySelect = document.getElementById('download-category-folder');
    if (categorySelect && Array.isArray(categories)) {
        categorySelect.innerHTML = '<option value="">-- Select Category --</option>' +
            categories.map(cat => `<option value="${cat.name}">${cat.name}</option>`).join('');
    }
}

// ==================== Remote Folder Browsing ====================

let currentBrowsePath = '';
let currentBrowseServerId = null;

async function browseFolders(serverId, path = '') {
    currentBrowseServerId = serverId;
    currentBrowsePath = path;

    const folderList = document.getElementById('remote-folders-list');
    const currentPathEl = document.getElementById('current-browse-path');

    if (folderList) folderList.innerHTML = '<p class="text-secondary p-2">Loading folders...</p>';

    try {
        const result = await apiCall(`remote-servers/${serverId}/browse-folders`, 'POST', { path });

        if (result.success) {
            currentBrowsePath = result.current_path;
            if (currentPathEl) currentPathEl.value = result.current_path;

            // Render folder list
            let html = '';

            // Parent directory link (if not at root)
            if (result.parent_path !== null) {
                html += `<a href="#" class="list-group-item list-group-item-action" onclick="browseFolders(${serverId}, '${escapeHtml(result.parent_path)}'); return false;">
                    <i class="bi bi-arrow-up-circle text-secondary me-2"></i>
                    <span class="text-muted">.. (Parent Directory)</span>
                </a>`;
            }

            // Folders
            for (const folder of result.folders) {
                html += `<a href="#" class="list-group-item list-group-item-action" onclick="browseFolders(${serverId}, '${escapeHtml(folder.path)}'); return false;">
                    <i class="bi bi-folder-fill text-warning me-2"></i>
                    <span>${escapeHtml(folder.name)}</span>
                </a>`;
            }

            if (result.folders.length === 0 && result.parent_path === null) {
                html += '<div class="text-muted text-center py-3">No folders found</div>';
            }

            if (folderList) folderList.innerHTML = html || '<div class="text-muted text-center py-3">Empty directory</div>';
        } else {
            if (folderList) folderList.innerHTML = `<div class="text-danger py-3">${result.error || 'Failed to browse folders'}</div>`;
        }
    } catch (error) {
        if (folderList) folderList.innerHTML = `<div class="text-danger py-3">Error: ${error.message}</div>`;
    }
}

function browseParentFolder() {
    if (currentBrowseServerId && currentBrowsePath) {
        const parentPath = currentBrowsePath.split('/').slice(0, -1).join('/') || '/';
        browseFolders(currentBrowseServerId, parentPath);
    }
}

function refreshBrowseFolders() {
    if (currentBrowseServerId) {
        browseFolders(currentBrowseServerId, currentBrowsePath);
    }
}

function removeFromDownload(id, type) {
    if (type === 'movie') {
        selectedMovies.delete(id);
        document.getElementById('download-movie-count').textContent = selectedMovies.size;
        document.getElementById('selected-movies-list').innerHTML = Array.from(selectedMovies.entries())
            .map(([mid, m]) => `<div class="small py-1 border-bottom d-flex justify-content-between align-items-center">
                <span>${m.title}</span>
                <button type="button" class="btn btn-sm btn-outline-danger py-0 px-1" onclick="removeFromDownload(${mid}, 'movie')"><i class="bi bi-x"></i></button>
            </div>`).join('');
    } else if (type === 'episode') {
        selectedEpisodes.delete(id);
        document.getElementById('download-movie-count').textContent = selectedEpisodes.size;
        const seriesName = currentSeriesDetail?.title || 'Unknown Series';
        document.getElementById('selected-movies-list').innerHTML =
            `<div class="small py-1 border-bottom"><strong class="text-info">${seriesName}</strong></div>` +
            Array.from(selectedEpisodes.entries())
                .map(([eid, ep]) => `<div class="small py-1 border-bottom d-flex justify-content-between align-items-center">
                    <span>S${String(ep.season).padStart(2, '0')}E${String(ep.episode).padStart(2, '0')} - ${ep.title || 'Episode'}</span>
                    <button type="button" class="btn btn-sm btn-outline-danger py-0 px-1" onclick="removeFromDownload(${eid}, 'episode')"><i class="bi bi-x"></i></button>
                </div>`).join('');
    } else if (type === 'series') {
        selectedSeries.delete(id);
        document.getElementById('download-movie-count').textContent = selectedSeries.size;
        document.getElementById('selected-movies-list').innerHTML = Array.from(selectedSeries.entries())
            .map(([sid, s]) => `<div class="small py-1 border-bottom d-flex justify-content-between align-items-center">
                <span>${s.title}</span>
                <button type="button" class="btn btn-sm btn-outline-danger py-0 px-1" onclick="removeFromDownload(${sid}, 'series')"><i class="bi bi-x"></i></button>
            </div>`).join('');
    }
    if (selectedMovies.size === 0 && selectedEpisodes.size === 0 && selectedSeries.size === 0) {
        bootstrap.Modal.getInstance(document.getElementById('downloadModal'))?.hide();
    }
}

function selectBrowsedFolder() {
    const selectedPath = document.getElementById('browse-selected-path').value;
    if (selectedPath) {
        // Use the browsed folder path for downloads
        document.getElementById('download-custom-folder').value = selectedPath;
        showAlert(`Selected folder: ${selectedPath}`, 'success');
    }
}

async function createRemoteFolder() {
    if (!currentBrowseServerId) {
        showAlert('No server selected', 'warning');
        return;
    }

    const folderNameInput = document.getElementById('new-folder-name');
    const folderName = folderNameInput?.value?.trim();

    if (!folderName) {
        showAlert('Please enter a folder name', 'warning');
        return;
    }

    const newPath = currentBrowsePath ? `${currentBrowsePath}/${folderName}` : `/${folderName}`;

    try {
        const result = await apiCall(`remote-servers/${currentBrowseServerId}/create-folder`, 'POST', { path: newPath });

        if (result.success) {
            showAlert(`Folder created: ${folderName}`, 'success');
            folderNameInput.value = '';
            refreshBrowseFolders();
        } else {
            showAlert(result.error || 'Failed to create folder', 'danger');
        }
    } catch (error) {
        showAlert('Error creating folder: ' + error.message, 'danger');
    }
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// ==================== Download Jobs Management ====================

async function clearCompletedJobs() {
    if (!confirm('Clear all completed download jobs?')) return;

    try {
        const result = await apiCall('download-jobs/clear-completed', 'POST');
        if (result.success) {
            showAlert(`Cleared ${result.cleared} completed jobs`, 'success');
            loadDownloadJobs();
        } else {
            showAlert(result.error || 'Failed to clear jobs', 'danger');
        }
    } catch (error) {
        showAlert('Error: ' + error.message, 'danger');
    }
}

async function clearFailedJobs() {
    if (!confirm('Clear all failed download jobs?')) return;

    try {
        const result = await apiCall('download-jobs/clear-failed', 'POST');
        if (result.success) {
            showAlert(`Cleared ${result.cleared} failed jobs`, 'success');
            loadDownloadJobs();
        } else {
            showAlert(result.error || 'Failed to clear jobs', 'danger');
        }
    } catch (error) {
        showAlert('Error: ' + error.message, 'danger');
    }
}

async function clearAllJobs() {
    if (!confirm('Clear ALL download jobs? This cannot be undone!')) return;

    try {
        const result = await apiCall('download-jobs/clear-all', 'POST');
        if (result.success) {
            showAlert(`Cleared ${result.cleared} jobs`, 'success');
            loadDownloadJobs();
        } else {
            showAlert(result.error || 'Failed to clear jobs', 'danger');
        }
    } catch (error) {
        showAlert('Error: ' + error.message, 'danger');
    }
}

async function retryJob(jobId) {
    try {
        const result = await apiCall(`download-jobs/${jobId}/retry`, 'POST');
        if (result.success) {
            showAlert('Job queued for retry', 'success');
            loadDownloadJobs();
        } else {
            showAlert(result.error || 'Failed to retry job', 'danger');
        }
    } catch (error) {
        showAlert('Error: ' + error.message, 'danger');
    }
}

async function cancelJob(jobId) {
    if (!confirm('Cancel this download job?')) return;

    try {
        const result = await apiCall(`download-jobs/${jobId}/cancel`, 'POST');
        if (result.success) {
            showAlert('Job cancelled', 'success');
            loadDownloadJobs();
        } else {
            showAlert(result.error || 'Failed to cancel job', 'danger');
        }
    } catch (error) {
        showAlert('Error: ' + error.message, 'danger');
    }
}

async function deleteJob(jobId) {
    if (!confirm('Delete this download job?')) return;

    try {
        const result = await apiCall(`download-jobs/${jobId}`, 'DELETE');
        if (result.success) {
            showAlert('Job deleted', 'success');
            loadDownloadJobs();
        } else {
            showAlert(result.error || 'Failed to delete job', 'danger');
        }
    } catch (error) {
        showAlert('Error: ' + error.message, 'danger');
    }
}

// Override the original showDownloadModal to support folder selection
async function showDownloadModal() {
    if (selectedMovies.size === 0) { showAlert('Please select movies', 'warning'); return; }

    downloadType = 'movies';

    const servers = await apiCall('remote-servers');
    const serverSelect = document.getElementById('download-remote-server');
    if (serverSelect && Array.isArray(servers)) {
        serverSelect.innerHTML = '<option value="">-- Select Server --</option>' +
            servers.filter(s => s.is_active).map(s => `<option value="${s.id}" data-path="${s.remote_path}">${s.name} (${s.type} - ${s.host})</option>`).join('');
    }
    document.getElementById('download-movie-count').textContent = selectedMovies.size;
    document.getElementById('download-type-label').textContent = 'movies';
    document.getElementById('selected-movies-list').innerHTML = Array.from(selectedMovies.entries())
        .map(([id, m]) => `<div class="small py-1 border-bottom d-flex justify-content-between align-items-center">
            <span>${m.title}</span>
            <button type="button" class="btn btn-sm btn-outline-danger py-0 px-1" onclick="removeFromDownload(${id}, 'movie')"><i class="bi bi-x"></i></button>
        </div>`).join('');

    await loadCategoryFolders();

    new bootstrap.Modal(document.getElementById('downloadModal')).show();
}

// Override startRemoteDownload to include folder path
async function startRemoteDownload() {
    const serverId = document.getElementById('download-remote-server').value;
    if (!serverId) { showAlert('Please select a server', 'warning'); return; }

    // Get folder configuration
    const folderType = document.getElementById('download-folder-type').value;
    let subFolder = '';

    if (folderType === 'category') {
        subFolder = document.getElementById('download-category-folder').value;
    } else if (folderType === 'custom') {
        subFolder = document.getElementById('download-custom-folder').value;
    } else if (folderType === 'browse') {
        subFolder = document.getElementById('current-browse-path').value;
    }

    const organize = document.getElementById('download-organize').checked;
    const cleanName = document.getElementById('download-rename').checked;

    showAlert('Creating download jobs...', 'info');

    const items = [];

    if (downloadType === 'movies') {
        // Fetch all movie data in parallel for speed
        const movieIds = Array.from(selectedMovies.keys());
        const moviePromises = movieIds.map(id => apiCall(`movies/${id}`));
        const moviesData = await Promise.all(moviePromises);

        moviesData.forEach((movieData, idx) => {
            if (movieData?.links?.length > 0) {
                const movie = selectedMovies.get(movieIds[idx]);
                const link = movieData.links[0];
                const streamUrl = link.stream_url || link.url;
                const ext = link.container_extension || 'mp4';
                const year = movieData.year || new Date().getFullYear();
                // Format: movie-name (2025).mp4 - Always use clean format for Folder Watch compatibility
                const movieName = movie.title.replace(/[^a-zA-Z0-9 ]/g, '').replace(/ +/g, '-').toLowerCase();
                const filename = `${movieName} (${year}).${ext}`;

                items.push({
                    movie_id: movieIds[idx],
                    url: streamUrl,
                    filename: filename,
                    subfolder: subFolder,
                    organize_by_name: false
                });
            }
        });
    } else if (downloadType === 'episodes') {
        const seriesTitle = currentSeriesDetail?.title || 'Series';
        const seriesYear = currentSeriesDetail?.year || new Date().getFullYear();
        // Format series name: series-name (2025)
        const seriesName = `${seriesTitle.replace(/[^a-zA-Z0-9 ]/g, '').replace(/ +/g, '-').toLowerCase()} (${seriesYear})`;

        for (const [epId, ep] of selectedEpisodes) {
            const episode = currentSeriesDetail?.episodes?.find(e => e.id === epId);
            if (episode?.stream_url) {
                const ext = ep.extension || episode.container_extension || 'mp4';
                // Format: series-name (2025).s02.e03.mp4 - Always use clean format
                const filename = `${seriesName}.s${String(ep.season).padStart(2, '0')}.e${String(ep.episode).padStart(2, '0')}.${ext}`;

                items.push({
                    episode_id: epId,
                    url: episode.stream_url,
                    filename: filename,
                    subfolder: subFolder,
                    organize_by_name: false
                });
            }
        }
    } else if (downloadType === 'series') {
        // Download all episodes from selected series - fetch in parallel
        const seriesIds = Array.from(selectedSeries.keys());
        const seriesPromises = seriesIds.map(id => apiCall(`series/${id}`));
        const allSeriesData = await Promise.all(seriesPromises);

        allSeriesData.forEach((seriesData, idx) => {
            const series = selectedSeries.get(seriesIds[idx]);
            const seriesTitle = series.title;
            const seriesYear = seriesData?.year || new Date().getFullYear();
            // Format: series-name (2025)
            const seriesName = `${seriesTitle.replace(/[^a-zA-Z0-9 ]/g, '').replace(/ +/g, '-').toLowerCase()} (${seriesYear})`;

            for (const ep of (seriesData?.episodes || [])) {
                const ext = ep.container_extension || 'mp4';
                // Format: series-name (2025).s02.e03.mp4 - Always use clean format
                const filename = `${seriesName}.s${String(ep.season_number).padStart(2, '0')}.e${String(ep.episode_number).padStart(2, '0')}.${ext}`;

                items.push({
                    episode_id: ep.id,
                    url: ep.stream_url,
                    filename: filename,
                    subfolder: subFolder,
                    organize_by_name: false
                });
            }
        });
    }

    if (items.length === 0) { showAlert('No valid links found', 'danger'); return; }

    const result = await apiCall('remote-servers/bulk-download', 'POST', {
        remote_server_id: parseInt(serverId),
        items: items
    });

    if (result.queued > 0) {
        showAlert(`${result.queued} downloads queued!`, 'success');
        bootstrap.Modal.getInstance(document.getElementById('downloadModal')).hide();
        selectedMovies.clear();
        selectedSeries.clear();
        selectedEpisodes.clear();
        updateSelectionUI();
        updateSeriesSelectionUI();
        // Redirect to Download Manager page automatically
        navigateTo('download-jobs');
    } else {
        showAlert(result.error || 'Failed to queue', 'danger');
    }
}
