-- VOD Import System Database Schema
-- Version: 1.0
-- Date: 2025-11-21

SET NAMES utf8mb4;
SET FOREIGN_KEY_CHECKS = 0;

-- ----------------------------
-- Database
-- ----------------------------
CREATE DATABASE IF NOT EXISTS `vod_import` DEFAULT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE `vod_import`;

-- ----------------------------
-- Table: servers (Multi-Source Servers)
-- ----------------------------
DROP TABLE IF EXISTS `servers`;
CREATE TABLE `servers` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `name` VARCHAR(100) NOT NULL,
    `type` ENUM('m3u', 'xtream') NOT NULL DEFAULT 'xtream',
    `url` VARCHAR(500) NOT NULL,
    `username` VARCHAR(100) DEFAULT NULL,
    `password` VARCHAR(100) DEFAULT NULL,
    `port` INT DEFAULT 80,
    `use_https` TINYINT(1) DEFAULT 0,
    `is_active` TINYINT(1) DEFAULT 1,
    `priority` INT DEFAULT 1 COMMENT 'Higher = preferred source',
    `last_check` DATETIME DEFAULT NULL,
    `last_sync` DATETIME DEFAULT NULL,
    `status` ENUM('online', 'offline', 'error', 'unknown') DEFAULT 'unknown',
    `movies_count` INT DEFAULT 0,
    `series_count` INT DEFAULT 0,
    `live_count` INT DEFAULT 0,
    `error_message` TEXT DEFAULT NULL,
    `config` JSON DEFAULT NULL COMMENT 'Additional server config',
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX `idx_active` (`is_active`),
    INDEX `idx_type` (`type`),
    INDEX `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ----------------------------
-- Table: categories
-- ----------------------------
DROP TABLE IF EXISTS `categories`;
CREATE TABLE `categories` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `server_id` INT UNSIGNED DEFAULT NULL,
    `external_id` VARCHAR(50) NOT NULL COMMENT 'Category ID from source',
    `name` VARCHAR(200) NOT NULL,
    `type` ENUM('live', 'movie', 'series') NOT NULL DEFAULT 'movie',
    `parent_id` INT UNSIGNED DEFAULT NULL,
    `icon` VARCHAR(500) DEFAULT NULL,
    `movies_count` INT DEFAULT 0,
    `is_active` TINYINT(1) DEFAULT 1,
    `sort_order` INT DEFAULT 0,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX `idx_server` (`server_id`),
    INDEX `idx_type` (`type`),
    INDEX `idx_external` (`external_id`),
    UNIQUE KEY `uk_server_external` (`server_id`, `external_id`, `type`),
    FOREIGN KEY (`server_id`) REFERENCES `servers`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ----------------------------
-- Table: movies
-- ----------------------------
DROP TABLE IF EXISTS `movies`;
CREATE TABLE `movies` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `tmdb_id` INT DEFAULT NULL,
    `imdb_id` VARCHAR(20) DEFAULT NULL,
    `title` VARCHAR(300) NOT NULL,
    `original_title` VARCHAR(300) DEFAULT NULL,
    `year` YEAR DEFAULT NULL,
    `clean_name` VARCHAR(300) NOT NULL COMMENT 'FolderWatch compatible name',
    `description` TEXT DEFAULT NULL,
    `plot` TEXT DEFAULT NULL,
    `genre` VARCHAR(200) DEFAULT NULL,
    `director` VARCHAR(200) DEFAULT NULL,
    `cast` TEXT DEFAULT NULL,
    `rating` DECIMAL(3,1) DEFAULT NULL,
    `rating_5based` DECIMAL(2,1) DEFAULT NULL,
    `duration` INT DEFAULT NULL COMMENT 'Duration in minutes',
    `poster` VARCHAR(500) DEFAULT NULL,
    `backdrop` VARCHAR(500) DEFAULT NULL,
    `trailer_url` VARCHAR(500) DEFAULT NULL,
    `release_date` DATE DEFAULT NULL,
    `container_extension` VARCHAR(10) DEFAULT 'mp4',
    `video_codec` VARCHAR(50) DEFAULT NULL,
    `audio_codec` VARCHAR(50) DEFAULT NULL,
    `resolution` VARCHAR(20) DEFAULT NULL COMMENT '4K, 1080p, 720p, etc',
    `file_size` BIGINT DEFAULT NULL COMMENT 'Size in bytes',
    `is_adult` TINYINT(1) DEFAULT 0,
    `is_downloaded` TINYINT(1) DEFAULT 0,
    `download_path` VARCHAR(500) DEFAULT NULL,
    `download_date` DATETIME DEFAULT NULL,
    `watch_count` INT DEFAULT 0,
    `status` ENUM('available', 'downloading', 'downloaded', 'error', 'disabled') DEFAULT 'available',
    `added_timestamp` INT DEFAULT NULL COMMENT 'Original timestamp from source',
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX `idx_tmdb` (`tmdb_id`),
    INDEX `idx_imdb` (`imdb_id`),
    INDEX `idx_year` (`year`),
    INDEX `idx_title` (`title`(100)),
    INDEX `idx_clean_name` (`clean_name`(100)),
    INDEX `idx_status` (`status`),
    INDEX `idx_downloaded` (`is_downloaded`),
    FULLTEXT KEY `ft_search` (`title`, `original_title`, `description`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ----------------------------
-- Table: movie_links (Multiple links per movie)
-- ----------------------------
DROP TABLE IF EXISTS `movie_links`;
CREATE TABLE `movie_links` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `movie_id` INT UNSIGNED NOT NULL,
    `server_id` INT UNSIGNED NOT NULL,
    `stream_id` INT NOT NULL COMMENT 'Stream ID from source',
    `category_id` INT UNSIGNED DEFAULT NULL,
    `stream_url` VARCHAR(1000) NOT NULL,
    `container_extension` VARCHAR(10) DEFAULT 'mp4',
    `quality` VARCHAR(20) DEFAULT NULL COMMENT '4K, 1080p, 720p, CAM',
    `file_size` BIGINT DEFAULT NULL,
    `bitrate` INT DEFAULT NULL,
    `is_working` TINYINT(1) DEFAULT 1,
    `is_preferred` TINYINT(1) DEFAULT 0,
    `last_check` DATETIME DEFAULT NULL,
    `check_response_time` INT DEFAULT NULL COMMENT 'Response time in ms',
    `error_count` INT DEFAULT 0,
    `priority` INT DEFAULT 1,
    `added_timestamp` INT DEFAULT NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX `idx_movie` (`movie_id`),
    INDEX `idx_server` (`server_id`),
    INDEX `idx_stream` (`stream_id`),
    INDEX `idx_working` (`is_working`),
    INDEX `idx_preferred` (`is_preferred`),
    UNIQUE KEY `uk_server_stream` (`server_id`, `stream_id`),
    FOREIGN KEY (`movie_id`) REFERENCES `movies`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`server_id`) REFERENCES `servers`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`category_id`) REFERENCES `categories`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ----------------------------
-- Table: series
-- ----------------------------
DROP TABLE IF EXISTS `series`;
CREATE TABLE `series` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `tmdb_id` INT DEFAULT NULL,
    `imdb_id` VARCHAR(20) DEFAULT NULL,
    `title` VARCHAR(300) NOT NULL,
    `original_title` VARCHAR(300) DEFAULT NULL,
    `year` YEAR DEFAULT NULL,
    `clean_name` VARCHAR(300) NOT NULL,
    `description` TEXT DEFAULT NULL,
    `plot` TEXT DEFAULT NULL,
    `genre` VARCHAR(200) DEFAULT NULL,
    `cast` TEXT DEFAULT NULL,
    `rating` DECIMAL(3,1) DEFAULT NULL,
    `poster` VARCHAR(500) DEFAULT NULL,
    `backdrop` VARCHAR(500) DEFAULT NULL,
    `seasons_count` INT DEFAULT 0,
    `episodes_count` INT DEFAULT 0,
    `status` ENUM('ongoing', 'ended', 'cancelled') DEFAULT 'ongoing',
    `last_episode_date` DATE DEFAULT NULL,
    `is_adult` TINYINT(1) DEFAULT 0,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX `idx_tmdb` (`tmdb_id`),
    INDEX `idx_title` (`title`(100)),
    INDEX `idx_clean_name` (`clean_name`(100)),
    FULLTEXT KEY `ft_search` (`title`, `original_title`, `description`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ----------------------------
-- Table: episodes
-- ----------------------------
DROP TABLE IF EXISTS `episodes`;
CREATE TABLE `episodes` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `series_id` INT UNSIGNED DEFAULT NULL,
    `server_id` INT UNSIGNED NOT NULL,
    `stream_id` INT NOT NULL,
    `season_number` INT NOT NULL,
    `episode_number` INT NOT NULL,
    `title` VARCHAR(300) DEFAULT NULL,
    `clean_name` VARCHAR(300) NOT NULL COMMENT 'series-name (year).s01.e01.ext',
    `description` TEXT DEFAULT NULL,
    `duration` INT DEFAULT NULL,
    `stream_url` VARCHAR(1000) NOT NULL,
    `container_extension` VARCHAR(10) DEFAULT 'mp4',
    `poster` VARCHAR(500) DEFAULT NULL,
    `air_date` DATE DEFAULT NULL,
    `rating` DECIMAL(3,1) DEFAULT NULL,
    `is_downloaded` TINYINT(1) DEFAULT 0,
    `download_path` VARCHAR(500) DEFAULT NULL,
    `is_orphan` TINYINT(1) DEFAULT 0 COMMENT 'Episode without linked series',
    `status` ENUM('available', 'downloading', 'downloaded', 'error') DEFAULT 'available',
    `added_timestamp` INT DEFAULT NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX `idx_series` (`series_id`),
    INDEX `idx_server` (`server_id`),
    INDEX `idx_season_episode` (`season_number`, `episode_number`),
    INDEX `idx_orphan` (`is_orphan`),
    INDEX `idx_clean_name` (`clean_name`(100)),
    FOREIGN KEY (`series_id`) REFERENCES `series`(`id`) ON DELETE SET NULL,
    FOREIGN KEY (`server_id`) REFERENCES `servers`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ----------------------------
-- Table: orphan_episodes (Unlinked episodes tracking)
-- ----------------------------
DROP TABLE IF EXISTS `orphan_episodes`;
CREATE TABLE `orphan_episodes` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `episode_id` INT UNSIGNED NOT NULL,
    `original_name` VARCHAR(500) NOT NULL,
    `parsed_series_name` VARCHAR(300) DEFAULT NULL,
    `parsed_year` YEAR DEFAULT NULL,
    `parsed_season` INT DEFAULT NULL,
    `parsed_episode` INT DEFAULT NULL,
    `suggested_series_id` INT UNSIGNED DEFAULT NULL,
    `resolution_status` ENUM('pending', 'linked', 'deleted', 'ignored') DEFAULT 'pending',
    `resolved_at` DATETIME DEFAULT NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX `idx_episode` (`episode_id`),
    INDEX `idx_status` (`resolution_status`),
    FOREIGN KEY (`episode_id`) REFERENCES `episodes`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`suggested_series_id`) REFERENCES `series`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ----------------------------
-- Table: import_logs
-- ----------------------------
DROP TABLE IF EXISTS `import_logs`;
CREATE TABLE `import_logs` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `server_id` INT UNSIGNED DEFAULT NULL,
    `action` VARCHAR(50) NOT NULL,
    `type` ENUM('movie', 'series', 'episode', 'category', 'server', 'system') NOT NULL,
    `target_id` INT DEFAULT NULL,
    `target_name` VARCHAR(300) DEFAULT NULL,
    `status` ENUM('success', 'warning', 'error', 'info') DEFAULT 'info',
    `message` TEXT DEFAULT NULL,
    `details` JSON DEFAULT NULL,
    `ip_address` VARCHAR(45) DEFAULT NULL,
    `user_agent` VARCHAR(500) DEFAULT NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX `idx_server` (`server_id`),
    INDEX `idx_action` (`action`),
    INDEX `idx_type` (`type`),
    INDEX `idx_status` (`status`),
    INDEX `idx_date` (`created_at`),
    FOREIGN KEY (`server_id`) REFERENCES `servers`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ----------------------------
-- Table: import_queue (Download queue)
-- ----------------------------
DROP TABLE IF EXISTS `import_queue`;
CREATE TABLE `import_queue` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `type` ENUM('movie', 'episode') NOT NULL,
    `target_id` INT UNSIGNED NOT NULL,
    `link_id` INT UNSIGNED DEFAULT NULL,
    `priority` INT DEFAULT 5,
    `status` ENUM('pending', 'processing', 'completed', 'failed', 'cancelled') DEFAULT 'pending',
    `progress` INT DEFAULT 0 COMMENT 'Percentage 0-100',
    `download_speed` INT DEFAULT NULL COMMENT 'bytes/second',
    `downloaded_bytes` BIGINT DEFAULT 0,
    `total_bytes` BIGINT DEFAULT NULL,
    `attempts` INT DEFAULT 0,
    `max_attempts` INT DEFAULT 3,
    `error_message` TEXT DEFAULT NULL,
    `started_at` DATETIME DEFAULT NULL,
    `completed_at` DATETIME DEFAULT NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX `idx_status` (`status`),
    INDEX `idx_type` (`type`),
    INDEX `idx_priority` (`priority`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ----------------------------
-- Table: settings
-- ----------------------------
DROP TABLE IF EXISTS `settings`;
CREATE TABLE `settings` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `key` VARCHAR(100) NOT NULL UNIQUE,
    `value` TEXT DEFAULT NULL,
    `type` ENUM('string', 'int', 'bool', 'json') DEFAULT 'string',
    `description` VARCHAR(500) DEFAULT NULL,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ----------------------------
-- Default Settings
-- ----------------------------
INSERT INTO `settings` (`key`, `value`, `type`, `description`) VALUES
('download_path', '/var/www/html/VOD/media', 'string', 'Default download path for media files'),
('folderwatch_path', '/var/www/html/VOD/watch', 'string', 'FolderWatch monitoring directory'),
('max_concurrent_downloads', '3', 'int', 'Maximum concurrent downloads'),
('auto_sync_interval', '3600', 'int', 'Auto sync interval in seconds'),
('auto_cleanup_orphans', '0', 'bool', 'Automatically cleanup orphan episodes'),
('check_links_interval', '86400', 'int', 'Link check interval in seconds'),
('tmdb_api_key', '', 'string', 'TMDB API key for metadata'),
('debug_mode', '0', 'bool', 'Enable debug logging'),
('name_format_movie', '{title} ({year})', 'string', 'Movie filename format'),
('name_format_episode', '{series} ({year}).s{season}.e{episode}', 'string', 'Episode filename format');

-- ----------------------------
-- Insert test server
-- ----------------------------
INSERT INTO `servers` (`name`, `type`, `url`, `username`, `password`, `port`, `is_active`, `priority`) VALUES
('Flix Panel', 'xtream', 'http://flix-panel.xyz', '1231231', '1231231', 80, 1, 10);

SET FOREIGN_KEY_CHECKS = 1;
