<?php
/**
 * Import episodes for multiple series in batches
 * Usage: php import_episodes_batch.php [limit] [offset]
 * Example: php import_episodes_batch.php 100 0
 */
error_reporting(E_ALL);
ini_set('display_errors', '1');
set_time_limit(0); // No time limit

require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/classes/ServerManager.php';
require_once __DIR__ . '/classes/Logger.php';

$limit = isset($argv[1]) ? (int)$argv[1] : 100;
$offset = isset($argv[2]) ? (int)$argv[2] : 0;

$db = getDB();
$logger = new Logger('episode_import');

// Get series that don't have episodes yet
$stmt = $db->prepare("
    SELECT s.id, s.tmdb_id, s.title, s.episodes_count
    FROM series s
    WHERE s.episodes_count = 0
    ORDER BY s.id DESC
    LIMIT ? OFFSET ?
");
$stmt->execute([$limit, $offset]);
$seriesList = $stmt->fetchAll();

echo "Starting episode import for " . count($seriesList) . " series (limit: $limit, offset: $offset)...\n";
echo "Start time: " . date('Y-m-d H:i:s') . "\n\n";

$totalImported = 0;
$totalErrors = 0;
$processed = 0;

foreach ($seriesList as $series) {
    $processed++;
    echo "[$processed/" . count($seriesList) . "] Processing: {$series['title']} (ID: {$series['id']})\n";

    // Get server info (server 2 = hplayer)
    $serverStmt = $db->query("SELECT * FROM servers WHERE id = 2");
    $server = $serverStmt->fetch();

    if (!$server) {
        echo "  ERROR: Server not found\n";
        continue;
    }

    // Fetch series info with episodes from API
    $baseUrl = rtrim($server['url'], '/');
    $port = $server['port'] ?? 80;
    if ($port && $port != 80 && !preg_match('/:\d+$/', $baseUrl)) {
        $baseUrl = preg_replace('/^(https?:\/\/[^\/]+)/', '$1:' . $port, $baseUrl);
    }

    $apiUrl = "{$baseUrl}/player_api.php?username={$server['username']}&password={$server['password']}&action=get_series_info&series_id={$series['tmdb_id']}";

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_USERAGENT => 'TiviMate/5.1.6 (Android 7.1.2)'
    ]);
    $response = curl_exec($ch);
    curl_close($ch);

    if (!$response) {
        echo "  ERROR: Failed to fetch episodes\n";
        $totalErrors++;
        continue;
    }

    $seriesInfo = json_decode($response, true);
    if (!is_array($seriesInfo) || !isset($seriesInfo['episodes'])) {
        echo "  SKIP: No episodes found\n";
        $totalErrors++;
        continue;
    }

    $imported = 0;
    $seasonsCount = 0;

    foreach ($seriesInfo['episodes'] as $seasonNum => $episodes) {
        $seasonsCount++;
        foreach ($episodes as $ep) {
            try {
                $streamId = (int) ($ep['id'] ?? 0);
                $episodeNum = (int) ($ep['episode_num'] ?? 0);

                // Check if episode exists
                $checkStmt = $db->prepare("SELECT id FROM episodes WHERE series_id = ? AND stream_id = ?");
                $checkStmt->execute([$series['id'], $streamId]);

                if ($checkStmt->fetch()) {
                    continue; // Already exists
                }

                $title = $ep['title'] ?? "Episode {$episodeNum}";
                $cleanName = preg_replace('/[^a-zA-Z0-9\s]/', '', strtolower($title));

                // Build stream URL
                $extension = $ep['container_extension'] ?? 'mp4';
                $streamUrl = "{$baseUrl}/series/{$server['username']}/{$server['password']}/{$streamId}.{$extension}";

                $insertStmt = $db->prepare("
                    INSERT INTO episodes (series_id, server_id, stream_id, season_number, episode_number,
                        title, clean_name, description, duration, stream_url, container_extension, poster, air_date)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");

                $insertStmt->execute([
                    $series['id'],
                    2, // server_id
                    $streamId,
                    $seasonNum,
                    $episodeNum,
                    $title,
                    $cleanName,
                    $ep['plot'] ?? null,
                    isset($ep['duration']) ? (int) $ep['duration'] : null,
                    $streamUrl,
                    $extension,
                    $ep['info']['movie_image'] ?? null,
                    $ep['info']['releasedate'] ?? null
                ]);

                $imported++;

            } catch (Exception $e) {
                // Silent fail
            }
        }
    }

    // Update series with episode count
    if ($imported > 0) {
        $updateStmt = $db->prepare("UPDATE series SET seasons_count = ?, episodes_count = ? WHERE id = ?");
        $updateStmt->execute([$seasonsCount, $imported, $series['id']]);
        echo "  ✓ Imported: {$imported} episodes from {$seasonsCount} seasons\n";
    } else {
        echo "  - No episodes imported\n";
    }

    $totalImported += $imported;

    // Show progress every 10 series
    if ($processed % 10 == 0) {
        echo "\n--- Progress: $processed/" . count($seriesList) . " series processed, $totalImported episodes imported ---\n\n";
    }
}

echo "\n=== Final Summary ===\n";
echo "End time: " . date('Y-m-d H:i:s') . "\n";
echo "Series processed: " . count($seriesList) . "\n";
echo "Total episodes imported: {$totalImported}\n";
echo "Total errors: {$totalErrors}\n";
?>
