<?php
/**
 * Stream episodes directly from source server without storing in database
 * This fetches episodes on-demand from the Xtream API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

require_once __DIR__ . '/config/config.php';

$action = $_GET['action'] ?? '';
$seriesId = $_GET['series_id'] ?? 0;

try {
    $db = getDB();

    switch ($action) {
        case 'get_episodes':
            // Get series info from database
            $stmt = $db->prepare("SELECT s.*, srv.url, srv.username, srv.password
                                   FROM series s
                                   JOIN servers srv ON srv.id = 2
                                   WHERE s.id = ?");
            $stmt->execute([$seriesId]);
            $series = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$series) {
                echo json_encode(['error' => 'Series not found']);
                exit;
            }

            // First, try to get episodes from database if they exist
            $stmt = $db->prepare("SELECT * FROM episodes WHERE series_id = ? ORDER BY season_number, episode_number");
            $stmt->execute([$seriesId]);
            $dbEpisodes = $stmt->fetchAll(PDO::FETCH_ASSOC);

            if (!empty($dbEpisodes)) {
                // Episodes found in database - use them
                echo json_encode([
                    'success' => true,
                    'series' => [
                        'id' => $series['id'],
                        'title' => $series['title'],
                        'plot' => $series['plot'],
                        'year' => $series['year'],
                        'rating' => $series['rating'],
                        'poster' => $series['poster'],
                        'seasons_count' => $series['seasons_count'],
                        'episodes_count' => $series['episodes_count']
                    ],
                    'episodes' => $dbEpisodes,
                    'source' => 'database'
                ]);
                exit;
            }

            // If no episodes in database, fetch from Xtream API
            $apiUrl = $series['url'] . '/player_api.php?username=' . urlencode($series['username']) .
                      '&password=' . urlencode($series['password']) .
                      '&action=get_series_info&series_id=' . $series['tmdb_id'];

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $apiUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($httpCode !== 200 || !$response) {
                echo json_encode(['error' => 'Failed to fetch episodes from source server']);
                exit;
            }

            $data = json_decode($response, true);

            if (!$data || !isset($data['episodes']) || empty($data['episodes'])) {
                echo json_encode(['error' => 'No episodes available on source server', 'series' => $series]);
                exit;
            }

            // Format episodes for frontend
            $episodes = [];
            foreach ($data['episodes'] as $seasonNum => $seasonEpisodes) {
                foreach ($seasonEpisodes as $episode) {
                    $episodeNum = isset($episode['episode_num']) ? (int)$episode['episode_num'] : 0;
                    $streamUrl = isset($episode['id']) ?
                        $series['url'] . '/series/' . $series['username'] . '/' . $series['password'] . '/' . $episode['id'] . '.mp4' :
                        null;

                    $episodes[] = [
                        'season_number' => (int)$seasonNum,
                        'episode_number' => $episodeNum,
                        'title' => $episode['title'] ?? "Episode $episodeNum",
                        'plot' => $episode['info']['plot'] ?? null,
                        'duration' => $episode['info']['duration'] ?? null,
                        'air_date' => $episode['info']['releasedate'] ?? null,
                        'stream_url' => $streamUrl,
                        'container_extension' => $episode['container_extension'] ?? 'mp4',
                        'rating' => $episode['info']['rating'] ?? null
                    ];
                }
            }

            // Sort episodes by season and episode number
            usort($episodes, function($a, $b) {
                if ($a['season_number'] != $b['season_number']) {
                    return $a['season_number'] - $b['season_number'];
                }
                return $a['episode_number'] - $b['episode_number'];
            });

            echo json_encode([
                'success' => true,
                'series' => [
                    'id' => $series['id'],
                    'title' => $series['title'],
                    'plot' => $series['plot'],
                    'year' => $series['year'],
                    'rating' => $series['rating'],
                    'poster' => $series['poster'],
                    'seasons_count' => count($data['episodes']),
                    'episodes_count' => count($episodes)
                ],
                'episodes' => $episodes,
                'source' => 'direct_stream'
            ]);
            break;

        case 'list_series':
            // List all series (without loading episodes)
            $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
            $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
            $search = $_GET['search'] ?? '';

            $sql = "SELECT id, title, year, plot, rating, poster, seasons_count, episodes_count
                    FROM series
                    WHERE 1=1";
            $params = [];

            if ($search) {
                $sql .= " AND title LIKE ?";
                $params[] = "%$search%";
            }

            $sql .= " ORDER BY created_at DESC LIMIT ? OFFSET ?";
            $params[] = $limit;
            $params[] = $offset;

            $stmt = $db->prepare($sql);
            $stmt->execute($params);
            $series = $stmt->fetchAll(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'series' => $series,
                'count' => count($series)
            ]);
            break;

        default:
            echo json_encode(['error' => 'Invalid action']);
    }

} catch (Exception $e) {
    echo json_encode(['error' => $e->getMessage()]);
}
