#!/bin/bash

#############################################
# Fix Script for Old VOD Panel Installation
# سكربت إصلاح التثبيت القديم
#############################################

echo "=========================================="
echo "  VOD Panel Fix Script"
echo "  سكربت إصلاح بانل VOD"
echo "=========================================="
echo ""

# Check if running as root
if [ "$EUID" -ne 0 ]; then
    echo "❌ Please run as root (sudo bash FIX_OLD_INSTALL.sh)"
    echo "❌ يرجى التشغيل كمدير (sudo bash FIX_OLD_INSTALL.sh)"
    exit 1
fi

echo "✓ Running as root"
echo "✓ التشغيل كمدير"
echo ""

# Variables
MYSQL_ROOT_PASSWORD="Admin@123"
DB_NAME="vod_import"
PANEL_PATH="/var/www/html/VOD"

echo "📋 Configuration:"
echo "   MySQL Password: $MYSQL_ROOT_PASSWORD"
echo "   Database: $DB_NAME"
echo "   Panel Path: $PANEL_PATH"
echo ""
read -p "Press Enter to continue... | اضغط Enter للمتابعة..."
echo ""

#############################################
# Fix 1: MySQL Root Password
#############################################
echo "🔧 Fix 1/3: Setting MySQL root password..."
echo "🔧 إصلاح 1/3: تعيين كلمة مرور MySQL..."
echo ""

# Try with sudo mysql (for auth_socket)
echo "   Trying with sudo mysql..."
sudo mysql -e "ALTER USER 'root'@'localhost' IDENTIFIED WITH mysql_native_password BY '$MYSQL_ROOT_PASSWORD';" 2>/dev/null
if [ $? -eq 0 ]; then
    sudo mysql -e "FLUSH PRIVILEGES;"
    echo "   ✅ Success with sudo mysql"
else
    # Try without password
    echo "   Trying without password..."
    mysql -e "ALTER USER 'root'@'localhost' IDENTIFIED WITH mysql_native_password BY '$MYSQL_ROOT_PASSWORD';" 2>/dev/null
    if [ $? -eq 0 ]; then
        mysql -e "FLUSH PRIVILEGES;"
        echo "   ✅ Success without password"
    else
        # Try with current password
        echo "   Trying with current password..."
        mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "SELECT 1;" 2>/dev/null
        if [ $? -eq 0 ]; then
            echo "   ✅ Password already set correctly"
        else
            echo "   ⚠️ Could not set password automatically"
            echo "   ⚠️ Run manually: sudo bash FIX_MYSQL.sh"
        fi
    fi
fi

# Test connection
mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "SELECT VERSION();" 2>/dev/null
if [ $? -eq 0 ]; then
    echo "   ✅ MySQL connection test: SUCCESS"
    echo ""
else
    echo "   ❌ MySQL connection test: FAILED"
    echo "   Please run: sudo bash FIX_MYSQL.sh"
    echo ""
    exit 1
fi

#############################################
# Fix 2: Database Structure
#############################################
echo "🔧 Fix 2/3: Updating database structure..."
echo "🔧 إصلاح 2/3: تحديث بنية قاعدة البيانات..."
echo ""

# Try with password first, then with sudo
mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "SELECT 1;" 2>/dev/null
if [ $? -eq 0 ]; then
    MYSQL_CMD="mysql -u root -p$MYSQL_ROOT_PASSWORD"
else
    MYSQL_CMD="sudo mysql"
fi

# Add missing columns to users table
echo "   Checking users table..."
$MYSQL_CMD $DB_NAME -e "SHOW COLUMNS FROM users LIKE 'last_login';" 2>/dev/null | grep -q "last_login"
if [ $? -ne 0 ]; then
    echo "   Adding last_login column..."
    $MYSQL_CMD $DB_NAME -e "ALTER TABLE users ADD COLUMN last_login DATETIME NULL AFTER updated_at;" 2>/dev/null
    echo "   ✅ Added last_login column"
else
    echo "   ✅ last_login column exists"
fi

$MYSQL_CMD $DB_NAME -e "SHOW COLUMNS FROM users LIKE 'login_ip';" 2>/dev/null | grep -q "login_ip"
if [ $? -ne 0 ]; then
    echo "   Adding login_ip column..."
    $MYSQL_CMD $DB_NAME -e "ALTER TABLE users ADD COLUMN login_ip VARCHAR(45) NULL AFTER last_login;" 2>/dev/null
    echo "   ✅ Added login_ip column"
else
    echo "   ✅ login_ip column exists"
fi

echo "   ✅ Database structure updated"
echo ""

#############################################
# Fix 3: Update Panel Files
#############################################
echo "🔧 Fix 3/3: Updating panel files..."
echo "🔧 إصلاح 3/3: تحديث ملفات البانل..."
echo ""

CURRENT_DIR=$(pwd)

# Update login.php if exists in current directory
if [ -f "$CURRENT_DIR/login.php" ]; then
    echo "   Updating login.php..."
    cp "$CURRENT_DIR/login.php" "$PANEL_PATH/" 2>/dev/null
    echo "   ✅ login.php updated"
fi

# Update config.php if exists
if [ -f "$CURRENT_DIR/config/config.php" ]; then
    echo "   Updating config.php..."
    cp "$CURRENT_DIR/config/config.php" "$PANEL_PATH/config/" 2>/dev/null
    echo "   ✅ config.php updated"
fi

# Set proper permissions
echo "   Setting permissions..."
chown -R www-data:www-data $PANEL_PATH
chmod -R 755 $PANEL_PATH
echo "   ✅ Permissions set"
echo ""

#############################################
# Verification
#############################################
echo "🔍 Verification | التحقق"
echo "=========================================="

# Test MySQL
echo "1. MySQL Connection:"
mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "SELECT VERSION();" 2>/dev/null
if [ $? -eq 0 ]; then
    echo "   ✅ MySQL: OK"
else
    echo "   ❌ MySQL: FAILED"
fi

# Test database
echo ""
echo "2. Database:"
$MYSQL_CMD -e "USE $DB_NAME; SELECT COUNT(*) as users FROM users;" 2>/dev/null
if [ $? -eq 0 ]; then
    echo "   ✅ Database: OK"
else
    echo "   ❌ Database: FAILED"
fi

# Test users table structure
echo ""
echo "3. Users Table Structure:"
COLUMNS=$($MYSQL_CMD $DB_NAME -e "SHOW COLUMNS FROM users;" 2>/dev/null | wc -l)
if [ $COLUMNS -ge 10 ]; then
    echo "   ✅ Users Table: OK ($COLUMNS columns)"
else
    echo "   ⚠️ Users Table: $COLUMNS columns (expected 10+)"
fi

# Test Apache
echo ""
echo "4. Apache:"
systemctl is-active --quiet apache2
if [ $? -eq 0 ]; then
    echo "   ✅ Apache: Running"
else
    echo "   ❌ Apache: Not running"
    systemctl start apache2 2>/dev/null
fi

echo ""
echo "=========================================="
echo "   ✅ Fix Complete!"
echo "   ✅ الإصلاح مكتمل!"
echo "=========================================="
echo ""
echo "📋 Access Information | معلومات الدخول:"
echo "   Panel URL: http://YOUR_IP/VOD/"
echo "   Username: admin"
echo "   Password: admin"
echo ""
echo "💾 Database:"
echo "   Database: $DB_NAME"
echo "   User: root"
echo "   Password: $MYSQL_ROOT_PASSWORD"
echo ""
echo "🔧 If login still fails | إذا فشل تسجيل الدخول:"
echo "   1. Check Apache logs: tail -f /var/log/apache2/error.log"
echo "   2. Restart Apache: systemctl restart apache2"
echo "   3. Test PHP: php -v"
echo ""
echo "=========================================="
