#!/bin/bash

#############################################
# VOD Panel - Complete Installation Script
# Ubuntu/Debian compatible
# Auto-installs everything + copies files
#############################################

set -e  # Exit on error

echo "============================================"
echo "   VOD Panel Installation Script"
echo "   سكربت تثبيت بانل VOD"
echo "============================================"
echo ""

# Check if running as root
if [ "$EUID" -ne 0 ]; then
    echo "❌ Please run as root (sudo bash INSTALL.sh)"
    echo "❌ يرجى التشغيل كمدير (sudo bash INSTALL.sh)"
    exit 1
fi

echo "✓ Running as root"
echo "✓ التشغيل كمدير"
echo ""

# Variables
MYSQL_ROOT_PASSWORD="Admin@123"
ADMIN_USERNAME="admin"
ADMIN_PASSWORD="admin"
DB_NAME="vod_import"
PANEL_PATH="/var/www/html/VOD"
CURRENT_DIR=$(pwd)
SERVER_IP=$(hostname -I | awk '{print $1}')

echo "📋 Installation Configuration:"
echo "📋 إعدادات التثبيت:"
echo "   MySQL Root Password: $MYSQL_ROOT_PASSWORD"
echo "   Admin Username: $ADMIN_USERNAME"
echo "   Admin Password: $ADMIN_PASSWORD"
echo "   Database Name: $DB_NAME"
echo "   Panel Path: $PANEL_PATH"
echo "   Current Directory: $CURRENT_DIR"
echo "   Server IP: $SERVER_IP"
echo ""
read -p "Press Enter to continue or Ctrl+C to cancel... | اضغط Enter للمتابعة أو Ctrl+C للإلغاء..."
echo ""

#############################################
# Step 1: Update System
#############################################
echo "📦 Step 1/9: Updating system packages..."
echo "📦 الخطوة 1/9: تحديث النظام..."
apt-get update -qq
apt-get upgrade -y -qq
echo "✅ System updated"
echo "✅ تم تحديث النظام"
echo ""

#############################################
# Step 2: Install Apache
#############################################
echo "🌐 Step 2/9: Installing Apache web server..."
echo "🌐 الخطوة 2/9: تثبيت Apache..."
apt-get install -y apache2
systemctl enable apache2
systemctl start apache2
echo "✅ Apache installed and started"
echo "✅ تم تثبيت Apache"
echo ""

#############################################
# Step 3: Install MySQL
#############################################
echo "🗄️ Step 3/9: Installing MySQL server..."
echo "🗄️ الخطوة 3/9: تثبيت MySQL..."
export DEBIAN_FRONTEND=noninteractive

# Pre-configure MySQL password
echo "mysql-server mysql-server/root_password password $MYSQL_ROOT_PASSWORD" | debconf-set-selections
echo "mysql-server mysql-server/root_password_again password $MYSQL_ROOT_PASSWORD" | debconf-set-selections

apt-get install -y mysql-server

# Set MySQL root password (for different MySQL versions)
mysql -e "ALTER USER 'root'@'localhost' IDENTIFIED WITH mysql_native_password BY '$MYSQL_ROOT_PASSWORD';" 2>/dev/null || \
mysql -e "SET PASSWORD FOR 'root'@'localhost' = PASSWORD('$MYSQL_ROOT_PASSWORD');" 2>/dev/null || \
mysqladmin -u root password "$MYSQL_ROOT_PASSWORD" 2>/dev/null || true

# Try with sudo for auth_socket
sudo mysql -e "ALTER USER 'root'@'localhost' IDENTIFIED WITH mysql_native_password BY '$MYSQL_ROOT_PASSWORD';" 2>/dev/null || true
sudo mysql -e "FLUSH PRIVILEGES;" 2>/dev/null || true

systemctl enable mysql
systemctl restart mysql

# Wait for MySQL to be ready
sleep 3

echo "✅ MySQL installed and configured"
echo "✅ تم تثبيت MySQL"
echo ""

#############################################
# Step 4: Install PHP 7.4
#############################################
echo "🐘 Step 4/9: Installing PHP 7.4 and extensions..."
echo "🐘 الخطوة 4/9: تثبيت PHP 7.4..."
apt-get install -y software-properties-common
add-apt-repository -y ppa:ondrej/php
apt-get update -qq

apt-get install -y \
    php7.4 \
    php7.4-cli \
    php7.4-common \
    php7.4-mysql \
    php7.4-curl \
    php7.4-json \
    php7.4-mbstring \
    php7.4-xml \
    php7.4-zip \
    libapache2-mod-php7.4

# Set PHP 7.4 as default
update-alternatives --set php /usr/bin/php7.4

# Configure PHP
sed -i 's/upload_max_filesize = .*/upload_max_filesize = 2048M/' /etc/php/7.4/apache2/php.ini
sed -i 's/post_max_size = .*/post_max_size = 2048M/' /etc/php/7.4/apache2/php.ini
sed -i 's/memory_limit = .*/memory_limit = 512M/' /etc/php/7.4/apache2/php.ini
sed -i 's/max_execution_time = .*/max_execution_time = 300/' /etc/php/7.4/apache2/php.ini

systemctl restart apache2
echo "✅ PHP 7.4 installed and configured"
echo "✅ تم تثبيت PHP 7.4"
echo ""

#############################################
# Step 5: Install Additional Tools
#############################################
echo "🔧 Step 5/9: Installing additional tools..."
echo "🔧 الخطوة 5/9: تثبيت الأدوات الإضافية..."
apt-get install -y \
    curl \
    wget \
    unzip \
    git \
    ffmpeg \
    python3 \
    python3-pip \
    sshpass \
    jq
echo "✅ Additional tools installed"
echo "✅ تم تثبيت الأدوات الإضافية"
echo ""

#############################################
# Step 6: Copy Panel Files
#############################################
echo "📁 Step 6/9: Copying panel files to $PANEL_PATH..."
echo "📁 الخطوة 6/9: نسخ ملفات البانل..."

# Create panel directory
mkdir -p $PANEL_PATH

# Copy all files from current directory to panel path
echo "   Copying from: $CURRENT_DIR"
echo "   Copying to: $PANEL_PATH"

# Copy directories
cp -r "$CURRENT_DIR/api" "$PANEL_PATH/" 2>/dev/null || true
cp -r "$CURRENT_DIR/classes" "$PANEL_PATH/" 2>/dev/null || true
cp -r "$CURRENT_DIR/config" "$PANEL_PATH/" 2>/dev/null || true
cp -r "$CURRENT_DIR/assets" "$PANEL_PATH/" 2>/dev/null || true
cp -r "$CURRENT_DIR/scripts" "$PANEL_PATH/" 2>/dev/null || true
cp -r "$CURRENT_DIR/workers" "$PANEL_PATH/" 2>/dev/null || true

# Copy main files
cp "$CURRENT_DIR/index.php" "$PANEL_PATH/" 2>/dev/null || true
cp "$CURRENT_DIR/login.php" "$PANEL_PATH/" 2>/dev/null || true
cp "$CURRENT_DIR/logout.php" "$PANEL_PATH/" 2>/dev/null || true
cp "$CURRENT_DIR/vod.html" "$PANEL_PATH/" 2>/dev/null || true
cp "$CURRENT_DIR/README.txt" "$PANEL_PATH/" 2>/dev/null || true

# Create necessary directories
mkdir -p $PANEL_PATH/movies
mkdir -p $PANEL_PATH/series

echo "✅ Panel files copied successfully"
echo "✅ تم نسخ ملفات البانل بنجاح"
echo ""

#############################################
# Step 7: Create Database and Tables
#############################################
echo "💾 Step 7/9: Creating database and tables..."
echo "💾 الخطوة 7/9: إنشاء قاعدة البيانات..."

# Try with password first, then with sudo
mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "SELECT 1;" 2>/dev/null
if [ $? -eq 0 ]; then
    echo "   Using: mysql -u root -p$MYSQL_ROOT_PASSWORD"
    MYSQL_CMD="mysql -u root -p$MYSQL_ROOT_PASSWORD"
else
    echo "   Using: sudo mysql"
    MYSQL_CMD="sudo mysql"
fi

$MYSQL_CMD <<EOF
-- Create database
CREATE DATABASE IF NOT EXISTS $DB_NAME CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE $DB_NAME;

-- Users table
CREATE TABLE IF NOT EXISTS users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    username VARCHAR(50) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100),
    role VARCHAR(20) DEFAULT 'user',
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    last_login DATETIME NULL,
    login_ip VARCHAR(45) NULL
);

-- Series table
CREATE TABLE IF NOT EXISTS series (
    id INT PRIMARY KEY AUTO_INCREMENT,
    tmdb_id INT,
    imdb_id VARCHAR(20),
    title VARCHAR(255) NOT NULL,
    original_title VARCHAR(255),
    year INT,
    clean_name VARCHAR(255),
    description TEXT,
    plot TEXT,
    genre VARCHAR(255),
    cast TEXT,
    rating VARCHAR(10),
    poster VARCHAR(500),
    backdrop VARCHAR(500),
    seasons_count INT DEFAULT 0,
    episodes_count INT DEFAULT 0,
    status VARCHAR(50),
    last_episode_date DATE,
    is_adult TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_title (title),
    INDEX idx_tmdb (tmdb_id),
    INDEX idx_episodes (episodes_count)
);

-- Episodes table
CREATE TABLE IF NOT EXISTS episodes (
    id INT PRIMARY KEY AUTO_INCREMENT,
    series_id INT NOT NULL,
    season_number INT NOT NULL,
    episode_number INT NOT NULL,
    title VARCHAR(255),
    plot TEXT,
    duration INT,
    air_date DATE,
    stream_url VARCHAR(1000),
    container_extension VARCHAR(10),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (series_id) REFERENCES series(id) ON DELETE CASCADE,
    INDEX idx_series (series_id),
    INDEX idx_season (season_number),
    INDEX idx_episode (episode_number)
);

-- Movies table
CREATE TABLE IF NOT EXISTS movies (
    id INT PRIMARY KEY AUTO_INCREMENT,
    tmdb_id INT,
    imdb_id VARCHAR(20),
    title VARCHAR(255) NOT NULL,
    original_title VARCHAR(255),
    year INT,
    clean_name VARCHAR(255),
    description TEXT,
    plot TEXT,
    genre VARCHAR(255),
    director VARCHAR(255),
    cast TEXT,
    rating VARCHAR(10),
    rating_5based VARCHAR(10),
    duration INT,
    poster VARCHAR(500),
    backdrop VARCHAR(500),
    trailer_url VARCHAR(500),
    stream_url VARCHAR(1000),
    container_extension VARCHAR(10),
    is_adult TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_title (title),
    INDEX idx_tmdb (tmdb_id)
);

-- Categories table
CREATE TABLE IF NOT EXISTS categories (
    id INT PRIMARY KEY AUTO_INCREMENT,
    external_id INT,
    name VARCHAR(100) NOT NULL,
    type ENUM('movie', 'series') NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Remote servers table
CREATE TABLE IF NOT EXISTS remote_servers (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    host VARCHAR(255) NOT NULL,
    port INT DEFAULT 8080,
    username VARCHAR(100) NOT NULL,
    password VARCHAR(255) NOT NULL,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Servers table (Xtream codes servers)
CREATE TABLE IF NOT EXISTS servers (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    url VARCHAR(255) NOT NULL,
    username VARCHAR(100) NOT NULL,
    password VARCHAR(255) NOT NULL,
    is_active TINYINT(1) DEFAULT 1,
    last_check TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Download jobs table
CREATE TABLE IF NOT EXISTS download_jobs (
    id INT PRIMARY KEY AUTO_INCREMENT,
    remote_server_id INT,
    source_url VARCHAR(1000) NOT NULL,
    destination_path VARCHAR(500),
    filename VARCHAR(255) NOT NULL,
    movie_id INT,
    episode_id INT,
    status ENUM('pending','downloading','completed','failed','cancelled') DEFAULT 'pending',
    progress INT DEFAULT 0,
    downloaded_bytes BIGINT DEFAULT 0,
    total_bytes BIGINT,
    download_speed INT,
    error_message TEXT,
    started_at DATETIME,
    completed_at DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_status (status)
);

-- Import logs table
CREATE TABLE IF NOT EXISTS import_logs (
    id INT PRIMARY KEY AUTO_INCREMENT,
    server_id INT,
    type VARCHAR(50),
    status VARCHAR(50),
    message TEXT,
    details TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Settings table
CREATE TABLE IF NOT EXISTS settings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(100) UNIQUE NOT NULL,
    setting_value TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Insert default admin user
INSERT INTO users (username, password, email, role, is_active)
VALUES ('$ADMIN_USERNAME', '\$2y\$10\$Cg4AfsMGK6xh2ulO24T3xePw1PpZLZm06KILFQ3bX6Jq7XClnSTOC', 'admin@localhost', 'admin', 1)
ON DUPLICATE KEY UPDATE password='\$2y\$10\$Cg4AfsMGK6xh2ulO24T3xePw1PpZLZm06KILFQ3bX6Jq7XClnSTOC';

EOF

echo "✅ Database and tables created"
echo "✅ Admin user created (username: $ADMIN_USERNAME, password: $ADMIN_PASSWORD)"
echo "✅ تم إنشاء قاعدة البيانات"
echo "✅ تم إنشاء حساب المدير"
echo ""

#############################################
# Step 8: Configure Apache
#############################################
echo "⚙️ Step 8/9: Configuring Apache..."
echo "⚙️ الخطوة 8/9: إعداد Apache..."

# Enable required modules
a2enmod rewrite
a2enmod headers
a2enmod ssl

# Set permissions
chown -R www-data:www-data $PANEL_PATH
chmod -R 755 $PANEL_PATH

# Create Apache virtual host (optional)
cat > /etc/apache2/sites-available/vod.conf <<VHOST
<VirtualHost *:80>
    ServerAdmin admin@localhost
    DocumentRoot $PANEL_PATH

    <Directory $PANEL_PATH>
        Options Indexes FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>

    ErrorLog \${APACHE_LOG_DIR}/vod_error.log
    CustomLog \${APACHE_LOG_DIR}/vod_access.log combined
</VirtualHost>
VHOST

# Enable site (optional - comment out if you want to use default)
# a2ensite vod.conf

# Restart Apache
systemctl restart apache2
echo "✅ Apache configured"
echo "✅ تم إعداد Apache"
echo ""

#############################################
# Step 9: Configure Firewall (if UFW installed)
#############################################
echo "🔥 Step 9/9: Configuring firewall..."
echo "🔥 الخطوة 9/9: إعداد الجدار الناري..."
if command -v ufw &> /dev/null; then
    ufw allow 80/tcp
    ufw allow 443/tcp
    echo "✅ Firewall configured (ports 80, 443 opened)"
    echo "✅ تم إعداد الجدار الناري"
else
    echo "⚠️ UFW not installed, skipping firewall configuration"
    echo "⚠️ الجدار الناري غير مثبت"
fi
echo ""

#############################################
# Installation Complete
#############################################
echo "============================================"
echo "   ✅ Installation Complete!"
echo "   ✅ اكتمل التثبيت بنجاح!"
echo "============================================"
echo ""
echo "📋 Access Information | معلومات الدخول:"
echo "   Panel URL: http://$SERVER_IP/VOD/"
echo "   Login URL: http://$SERVER_IP/VOD/login.php"
echo ""
echo "🔐 Login Credentials | بيانات الدخول:"
echo "   Username | اسم المستخدم: $ADMIN_USERNAME"
echo "   Password | كلمة المرور: $ADMIN_PASSWORD"
echo ""
echo "💾 Database Information | معلومات قاعدة البيانات:"
echo "   Database | القاعدة: $DB_NAME"
echo "   MySQL Root Password | كلمة مرور MySQL: $MYSQL_ROOT_PASSWORD"
echo "   Host | المضيف: localhost"
echo ""
echo "📁 Panel Location | موقع البانل:"
echo "   $PANEL_PATH"
echo ""
echo "🔧 Useful Commands | أوامر مفيدة:"
echo "   Check Apache: systemctl status apache2"
echo "   Check MySQL: systemctl status mysql"
echo "   View Logs: tail -f /var/log/apache2/error.log"
echo "   Restart: systemctl restart apache2 mysql"
echo ""
echo "📝 Next Steps | الخطوات التالية:"
echo "   1. Open | افتح: http://$SERVER_IP/VOD/"
echo "   2. Login | سجل الدخول: admin/admin"
echo "   3. Add Xtream server | أضف سيرفر Xtream"
echo "   4. Import content | استورد المحتوى"
echo ""
echo "============================================"
echo ""
