# Channels Mode Documentation

## Overview

The `channels` mode returns live TV channels from the MidnightStreamer panel in the correct format for the V6APK app.

## Request Format

```json
{
  "code": "236825115603",
  "mac": "8c:fd:f0:00:7f:0b",
  "sn": "8c:fd:f0:00:7f:0b",
  "model": "SM-N975F",
  "group": 1,
  "token": "c3a6dff366031159691293fecaf39679vyVNqLQB...",
  "mode": "channels"
}
```

## Response Format

The API returns an array of channel objects with the following structure:

```json
[
  {
    "id": "1",
    "type": "1",
    "stream_display_name": "LIVE-TV",
    "category_id": 1,
    "catid": "1",
    "catid_new": "x",
    "stream_icon": "https://cdn-icons-png.flaticon.com/512/2523/2523641.png",
    "view_order": 1,
    "tv_archive": "0",
    "has_epg": 0,
    "stream_url": "http://45.154.207.117:8000/live/236825115603/1475899348690bd9e230799/1.ts"
  },
  {
    "id": "2",
    "type": "1",
    "stream_display_name": "AlKuwait",
    "category_id": 1,
    "catid": "1",
    "catid_new": "x",
    "stream_icon": "https://cdn-icons-png.flaticon.com/512/2523/2523641.png",
    "view_order": 2,
    "tv_archive": "0",
    "has_epg": 0,
    "stream_url": "http://45.154.207.117:8000/live/236825115603/1475899348690bd9e230799/2.ts"
  }
]
```

## Field Descriptions

| Field | Type | Description |
|-------|------|-------------|
| `id` | string | Unique channel ID |
| `type` | string | Channel type (always "1" for live) |
| `stream_display_name` | string | Channel display name |
| `category_id` | integer | Category ID (numeric) |
| `catid` | string | Category ID (string format) |
| `catid_new` | string | New category system ID (always "x") |
| `stream_icon` | string | Channel logo/icon URL |
| `view_order` | integer | Display order in list |
| `tv_archive` | string | TV archive duration ("0" = no archive) |
| `has_epg` | integer | EPG availability (0 = no, 1 = yes) |
| `stream_url` | string | Full stream playback URL |

## Data Source

Channels are fetched from MidnightStreamer API:
```
http://45.154.207.117:8000/api/player/{username}/{password}/live/streams
```

### MidnightStreamer Response

The adapter transforms MidnightStreamer's format:

```json
{
  "id": "1",
  "name": "LIVE-TV",
  "category_id": "1",
  "logo": "",
  "poster": "",
  "cmd": "http://45.154.207.117:8000/live/236825115603/1475899348690bd9e230799/1.ts",
  "tv_archive_duration": "0"
}
```

Into the app's expected format with all required fields.

## Implementation

### File Location
`/var/www/html/iptv/V6APK/API-V6APK.php` (Lines 1234-1313)

### Key Features

1. **Token Authentication**: Requires valid token from `active` mode
2. **Auto Icon**: Uses default icon if MidnightStreamer provides empty logo
3. **Stream URL**: Full playback URL with embedded credentials
4. **Validation**: Skips channels with missing ID or name
5. **Fallback**: Returns empty array on error

### Code Flow

```
1. Validate user token
2. Fetch from MidnightStreamer: /live/streams
3. For each channel:
   - Extract: id, name, category_id, icon, cmd, tv_archive
   - Validate: must have id and name
   - Set default icon if missing
   - Build formatted object
4. Return XOR encrypted array
```

## Configuration

### Token Authentication

The `channels` mode is in the token-authenticated modes list:

```php
$token_auth_modes = [
    'movies_cat', 'movies_latest', 'movies_list', 'movies_netflix',
    'series_cat', 'series_latest', 'series_list', 'packages', 'channels'
];
```

### Panel URL

Set in database or config:
```sql
INSERT INTO solus_options (name, val) VALUES ('panel_url', 'http://45.154.207.117:8000/');
```

Or in `api_cfg_v6.php`:
```php
$_CFG['panel_url'] = 'http://45.154.207.117:8000/';
```

## Testing

### PHP Test Script

```php
<?php
$XOR_KEY = "KvuZRm365wk5Sz&Xy.Zx";

function runXOR($InputString, $KeyPhrase) {
    $KeyPhraseLength = strlen($KeyPhrase);
    $output = '';
    for ($i = 0; $i < strlen($InputString); $i++) {
        $rPos = $i % $KeyPhraseLength;
        $r = ord($InputString[$i]) ^ ord($KeyPhrase[$rPos]);
        $output .= chr($r);
    }
    return $output;
}

// 1. Get token
$json1 = json_encode([
    'code' => '236825115603',
    'mac' => '8c:fd:f0:00:7f:0b',
    'sn' => '8c:fd:f0:00:7f:0b',
    'model' => 'SM-N975F',
    'group' => 1,
    'mode' => 'active'
]);

$encrypted1 = runXOR($json1, $XOR_KEY);
$ch1 = curl_init();
curl_setopt($ch1, CURLOPT_URL, 'http://45.154.207.45/iptv/V6APK/API-V6APK.php');
curl_setopt($ch1, CURLOPT_POST, 1);
curl_setopt($ch1, CURLOPT_POSTFIELDS, 'json=' . urlencode($encrypted1));
curl_setopt($ch1, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch1, CURLOPT_HTTPHEADER, ['Content-Type: application/x-www-form-urlencoded']);
curl_setopt($ch1, CURLOPT_TIMEOUT, 10);

$response1 = curl_exec($ch1);
curl_close($ch1);

$decrypted1 = runXOR($response1, $XOR_KEY);
$auth = json_decode($decrypted1, true);
$token = $auth['token'];

// 2. Get channels
$json2 = json_encode([
    'code' => '236825115603',
    'mac' => '8c:fd:f0:00:7f:0b',
    'sn' => '8c:fd:f0:00:7f:0b',
    'model' => 'SM-N975F',
    'group' => 1,
    'token' => $token,
    'mode' => 'channels'
]);

$encrypted2 = runXOR($json2, $XOR_KEY);
$ch2 = curl_init();
curl_setopt($ch2, CURLOPT_URL, 'http://45.154.207.45/iptv/V6APK/API-V6APK.php');
curl_setopt($ch2, CURLOPT_POST, 1);
curl_setopt($ch2, CURLOPT_POSTFIELDS, 'json=' . urlencode($encrypted2));
curl_setopt($ch2, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch2, CURLOPT_HTTPHEADER, ['Content-Type: application/x-www-form-urlencoded']);
curl_setopt($ch2, CURLOPT_TIMEOUT, 10);

$response2 = curl_exec($ch2);
curl_close($ch2);

$decrypted2 = runXOR($response2, $XOR_KEY);
$channels = json_decode($decrypted2, true);

echo "Channels: " . count($channels) . "\n";
echo json_encode($channels[0], JSON_PRETTY_PRINT);
```

### Expected Output

```
Channels: 9
{
    "id": "1",
    "type": "1",
    "stream_display_name": "LIVE-TV",
    "category_id": 1,
    "catid": "1",
    "catid_new": "x",
    "stream_icon": "https://cdn-icons-png.flaticon.com/512/2523/2523641.png",
    "view_order": 1,
    "tv_archive": "0",
    "has_epg": 0,
    "stream_url": "http://45.154.207.117:8000/live/236825115603/1475899348690bd9e230799/1.ts"
}
```

## Error Handling

### Invalid Token
```json
{
  "error": "error: invalid or expired token",
  "status": 999
}
```

### MidnightStreamer Unavailable
Returns empty array:
```json
[]
```

### No Channels Available
Returns empty array:
```json
[]
```

## Stream Playback

The `stream_url` field contains the full URL with embedded credentials:
```
http://45.154.207.117:8000/live/{username}/{password}/{stream_id}.ts
```

The app can play this URL directly in its video player.

## Related Modes

- `packages` - Returns live channel categories
- `active` - Required for token generation
- `movies_latest` - Similar structure for movies
- `series_latest` - Similar structure for series

## Status

✅ **Fully Implemented and Working**

- Fetches from MidnightStreamer
- Transforms to correct format
- XOR encryption working
- Token authentication working
- All required fields present
- Returns 9 live channels

## Date Implemented

2025-11-07
