<?php
/**
 * TMDb Movie Classifier
 * Automatically classifies movies based on TMDb genres
 */

class TMDbClassifier {

    private $api_key;
    private $base_url = 'https://api.themoviedb.org/3';
    private $cache_dir;
    private $cache_time = 86400; // 24 hours

    // TMDb Genre IDs mapping to app categories
    private $genre_mapping = [
        // Kids/Family
        16 => 'kids',      // Animation
        10751 => 'kids',   // Family

        // Music
        10402 => 'music',  // Music

        // Action
        28 => 'action',    // Action
        12 => 'action',    // Adventure

        // Comedy
        35 => 'comedy',    // Comedy

        // Drama
        18 => 'drama',     // Drama
        10749 => 'drama',  // Romance

        // Horror/Thriller
        27 => 'horror',    // Horror
        53 => 'horror',    // Thriller

        // Sci-Fi/Fantasy
        878 => 'scifi',    // Science Fiction
        14 => 'scifi',     // Fantasy

        // Crime/Mystery
        80 => 'crime',     // Crime
        9648 => 'crime',   // Mystery

        // Documentary
        99 => 'documentary', // Documentary

        // War/History
        10752 => 'war',    // War
        36 => 'war',       // History

        // Western
        37 => 'western',   // Western

        // TV Movie
        10770 => 'tv_movie' // TV Movie
    ];

    // App category IDs from XC Panel - ONLY these sections are enabled
    private $app_categories = [
        'kids' => ['vod' => '46,51', 'series' => '62,59'],
        'trends' => ['vod' => '85,84,83,82,81,80'],  // Trending/New Movies
        'netflix' => ['vod' => '56'],
        'shahid' => ['vod' => '57'],  // Shahid content
        'top_movies' => ['vod' => '53,52,50,47,43'],  // Top rated movies
        'top_series' => ['series' => '1,58,60,61,63,64,65,66,67,68,69']  // Top series
    ];

    // Categories to EXCLUDE from classification (general movies/series folders)
    private $excluded_categories = [
        'vod' => [],  // Generic VOD categories to skip
        'series' => []  // Generic Series categories to skip
    ];

    // Enabled sections only
    private $enabled_sections = ['kids', 'trends', 'netflix', 'shahid', 'top_movies', 'top_series'];

    public function __construct($api_key = null) {
        global $_CFG;

        // Use provided API key or get from config
        $this->api_key = $api_key ?? ($_CFG['tmdb_api_key'] ?? '');
        $this->cache_dir = __DIR__ . '/cache/tmdb/';

        if (!is_dir($this->cache_dir)) {
            mkdir($this->cache_dir, 0755, true);
        }

        // Load enabled sections from config if available
        if (isset($_CFG['enabled_sections']) && is_array($_CFG['enabled_sections'])) {
            $this->enabled_sections = $_CFG['enabled_sections'];
        }

        // Update category mappings from config
        $this->loadCategoriesFromConfig();
    }

    /**
     * Load category IDs from global config
     */
    private function loadCategoriesFromConfig() {
        global $_CFG;

        if (isset($_CFG['kids_category_ids'])) {
            $this->app_categories['kids']['vod'] = $_CFG['kids_category_ids'];
        }
        if (isset($_CFG['kids_series_ids'])) {
            $this->app_categories['kids']['series'] = $_CFG['kids_series_ids'];
        }
        if (isset($_CFG['trends_category_ids'])) {
            $this->app_categories['trends']['vod'] = $_CFG['trends_category_ids'];
        }
        if (isset($_CFG['netflix_category_ids'])) {
            $this->app_categories['netflix']['vod'] = $_CFG['netflix_category_ids'];
        }
        if (isset($_CFG['shahid_category_ids'])) {
            $this->app_categories['shahid']['vod'] = $_CFG['shahid_category_ids'];
        }
        if (isset($_CFG['top_movies_category_ids'])) {
            $this->app_categories['top_movies']['vod'] = $_CFG['top_movies_category_ids'];
        }
        if (isset($_CFG['top_series_category_ids'])) {
            $this->app_categories['top_series']['series'] = $_CFG['top_series_category_ids'];
        }

        // Load excluded categories
        if (isset($_CFG['excluded_vod_categories']) && !empty($_CFG['excluded_vod_categories'])) {
            $this->excluded_categories['vod'] = explode(',', $_CFG['excluded_vod_categories']);
        }
        if (isset($_CFG['excluded_series_categories']) && !empty($_CFG['excluded_series_categories'])) {
            $this->excluded_categories['series'] = explode(',', $_CFG['excluded_series_categories']);
        }
    }

    /**
     * Search for a movie by title and get its details
     */
    public function searchMovie($title, $year = null) {
        $cache_key = md5($title . $year);
        $cache_file = $this->cache_dir . $cache_key . '.json';

        // Check cache
        if (file_exists($cache_file) && (time() - filemtime($cache_file)) < $this->cache_time) {
            return json_decode(file_get_contents($cache_file), true);
        }

        // Clean title for search
        $clean_title = $this->cleanTitle($title);

        $url = $this->base_url . "/search/movie?api_key={$this->api_key}&query=" . urlencode($clean_title);
        if ($year) {
            $url .= "&year=" . $year;
        }

        $response = $this->makeRequest($url);

        if ($response && isset($response['results']) && !empty($response['results'])) {
            $movie = $response['results'][0];

            // Get full movie details
            $details = $this->getMovieDetails($movie['id']);

            if ($details) {
                // Cache the result
                file_put_contents($cache_file, json_encode($details));
                return $details;
            }
        }

        return null;
    }

    /**
     * Get movie details by TMDb ID
     */
    public function getMovieDetails($tmdb_id) {
        $url = $this->base_url . "/movie/{$tmdb_id}?api_key={$this->api_key}&append_to_response=credits,videos";
        return $this->makeRequest($url);
    }

    /**
     * Classify a movie and return the appropriate app category
     */
    public function classifyMovie($movie_data) {
        if (!$movie_data || !isset($movie_data['genres'])) {
            return 'uncategorized';
        }

        $genres = $movie_data['genres'];

        // Check for Kids/Family first (priority)
        foreach ($genres as $genre) {
            if (in_array($genre['id'], [16, 10751])) {
                return 'kids';
            }
        }

        // Check for Music
        foreach ($genres as $genre) {
            if ($genre['id'] == 10402) {
                return 'music';
            }
        }

        // Check release year for "New Movies"
        if (isset($movie_data['release_date'])) {
            $year = (int)substr($movie_data['release_date'], 0, 4);
            if ($year >= 2020) {
                return 'new_movies';
            }
        }

        // Default classification based on primary genre
        if (!empty($genres)) {
            $primary_genre_id = $genres[0]['id'];
            if (isset($this->genre_mapping[$primary_genre_id])) {
                return $this->genre_mapping[$primary_genre_id];
            }
        }

        return 'uncategorized';
    }

    /**
     * Get app category IDs for a classification
     */
    public function getAppCategoryIds($classification, $type = 'vod') {
        if (isset($this->app_categories[$classification][$type])) {
            return $this->app_categories[$classification][$type];
        }
        return '';
    }

    /**
     * Classify and enhance movie data with TMDb info
     */
    public function enhanceMovieData($movie_title, $movie_year = null) {
        $tmdb_data = $this->searchMovie($movie_title, $movie_year);

        if (!$tmdb_data) {
            return [
                'classification' => 'uncategorized',
                'tmdb_data' => null
            ];
        }

        $classification = $this->classifyMovie($tmdb_data);

        return [
            'classification' => $classification,
            'category_ids' => $this->getAppCategoryIds($classification),
            'tmdb_data' => [
                'id' => $tmdb_data['id'],
                'title' => $tmdb_data['title'],
                'original_title' => $tmdb_data['original_title'] ?? '',
                'overview' => $tmdb_data['overview'] ?? '',
                'poster_path' => $tmdb_data['poster_path'] ? 'https://image.tmdb.org/t/p/w500' . $tmdb_data['poster_path'] : '',
                'backdrop_path' => $tmdb_data['backdrop_path'] ? 'https://image.tmdb.org/t/p/w1280' . $tmdb_data['backdrop_path'] : '',
                'release_date' => $tmdb_data['release_date'] ?? '',
                'vote_average' => $tmdb_data['vote_average'] ?? 0,
                'genres' => array_map(function($g) { return $g['name']; }, $tmdb_data['genres'] ?? []),
                'runtime' => $tmdb_data['runtime'] ?? 0,
                'tagline' => $tmdb_data['tagline'] ?? ''
            ]
        ];
    }

    /**
     * Get movies by classification from XC Panel
     */
    public function getMoviesByClassification($classification, $panel_url, $username, $password, $limit = 50) {
        $category_ids = $this->getAppCategoryIds($classification);

        if (empty($category_ids)) {
            return [];
        }

        $cat_array = explode(',', $category_ids);
        $all_movies = [];

        foreach ($cat_array as $cat_id) {
            $url = $panel_url . "player_api.php?username={$username}&password={$password}&action=get_vod_streams&category_id=" . trim($cat_id);

            $response = $this->makeRequest($url);

            if ($response && is_array($response)) {
                foreach ($response as $movie) {
                    $container = $movie['container_extension'] ?? 'mp4';
                    $all_movies[] = [
                        'id' => $movie['stream_id'],
                        'title' => $movie['name'],
                        'icon' => $movie['stream_icon'] ?? '',
                        'catid' => $movie['category_id'],
                        'rating' => $movie['rating'] ?? '0',
                        'stream_url' => $panel_url . "movie/{$username}/{$password}/{$movie['stream_id']}.{$container}",
                        'classification' => $classification
                    ];

                    if (count($all_movies) >= $limit) {
                        break 2;
                    }
                }
            }
        }

        return $all_movies;
    }

    /**
     * Get all genre mappings for the app
     */
    public function getGenreMappings() {
        return $this->app_categories;
    }

    /**
     * Get only enabled sections
     */
    public function getEnabledSections() {
        return $this->enabled_sections;
    }

    /**
     * Check if a section is enabled
     */
    public function isSectionEnabled($section) {
        return in_array($section, $this->enabled_sections);
    }

    /**
     * Get content for enabled sections only (Kids, Trends, Netflix, Shahid, Top Movies, Top Series)
     */
    public function getEnabledContent($panel_url, $username, $password, $limit = 50) {
        $content = [];

        foreach ($this->enabled_sections as $section) {
            if (!isset($this->app_categories[$section])) {
                continue;
            }

            $section_data = [];

            // Get VOD content
            if (isset($this->app_categories[$section]['vod'])) {
                $vod_cats = explode(',', $this->app_categories[$section]['vod']);
                foreach ($vod_cats as $cat_id) {
                    $url = $panel_url . "player_api.php?username={$username}&password={$password}&action=get_vod_streams&category_id=" . trim($cat_id);
                    $response = $this->makeRequest($url);

                    if ($response && is_array($response)) {
                        foreach ($response as $movie) {
                            $container = $movie['container_extension'] ?? 'mp4';
                            $section_data[] = [
                                'id' => $movie['stream_id'],
                                'title' => $movie['name'],
                                'icon' => $movie['stream_icon'] ?? '',
                                'catid' => $movie['category_id'],
                                'rating' => $movie['rating'] ?? '0',
                                'stream_url' => $panel_url . "movie/{$username}/{$password}/{$movie['stream_id']}.{$container}",
                                'type' => 'vod',
                                'section' => $section
                            ];

                            if (count($section_data) >= $limit) break 2;
                        }
                    }
                }
            }

            // Get Series content
            if (isset($this->app_categories[$section]['series'])) {
                $series_cats = explode(',', $this->app_categories[$section]['series']);
                foreach ($series_cats as $cat_id) {
                    $url = $panel_url . "player_api.php?username={$username}&password={$password}&action=get_series&category_id=" . trim($cat_id);
                    $response = $this->makeRequest($url);

                    if ($response && is_array($response)) {
                        foreach ($response as $series) {
                            $section_data[] = [
                                'id' => $series['series_id'],
                                'title' => $series['name'],
                                'icon' => $series['cover'] ?? '',
                                'catid' => $series['category_id'],
                                'rating' => $series['rating'] ?? '0',
                                'type' => 'series',
                                'section' => $section
                            ];

                            if (count($section_data) >= $limit) break 2;
                        }
                    }
                }
            }

            $content[$section] = $section_data;
        }

        return $content;
    }

    /**
     * Get Kids content only (movies + series)
     */
    public function getKidsContent($panel_url, $username, $password, $limit = 50) {
        return $this->getSectionContent('kids', $panel_url, $username, $password, $limit);
    }

    /**
     * Get Trends content (new/trending movies)
     */
    public function getTrendsContent($panel_url, $username, $password, $limit = 50) {
        return $this->getSectionContent('trends', $panel_url, $username, $password, $limit);
    }

    /**
     * Get Netflix content
     */
    public function getNetflixContent($panel_url, $username, $password, $limit = 50) {
        return $this->getSectionContent('netflix', $panel_url, $username, $password, $limit);
    }

    /**
     * Get Shahid content
     */
    public function getShahidContent($panel_url, $username, $password, $limit = 50) {
        return $this->getSectionContent('shahid', $panel_url, $username, $password, $limit);
    }

    /**
     * Get Top Movies
     */
    public function getTopMovies($panel_url, $username, $password, $limit = 50) {
        return $this->getSectionContent('top_movies', $panel_url, $username, $password, $limit);
    }

    /**
     * Get Top Series
     */
    public function getTopSeries($panel_url, $username, $password, $limit = 50) {
        return $this->getSectionContent('top_series', $panel_url, $username, $password, $limit);
    }

    /**
     * Get content for a specific section
     */
    private function getSectionContent($section, $panel_url, $username, $password, $limit = 50) {
        if (!$this->isSectionEnabled($section)) {
            return [];
        }

        $content = $this->getEnabledContent($panel_url, $username, $password, $limit);
        return $content[$section] ?? [];
    }

    /**
     * Clean movie title for better search results
     */
    private function cleanTitle($title) {
        // Remove year in parentheses
        $title = preg_replace('/\s*\(\d{4}\)\s*/', '', $title);

        // Remove common suffixes
        $title = preg_replace('/\s*-\s*(HD|4K|1080p|720p|BluRay|WEB-DL|HDRip).*$/i', '', $title);

        // Remove special characters
        $title = preg_replace('/[^\w\s\'-]/u', ' ', $title);

        // Clean whitespace
        $title = trim(preg_replace('/\s+/', ' ', $title));

        return $title;
    }

    /**
     * Make HTTP request
     */
    private function makeRequest($url) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_USERAGENT, 'IPTV-Panel/1.0');

        $response = curl_exec($ch);
        curl_close($ch);

        if ($response === false) {
            return null;
        }

        return json_decode($response, true);
    }
}
