# 📘 دليل استخدام التوكن (Token Usage Guide)

## 1️⃣ إضافة التوكن في Android App

### الطريقة الصحيحة لحفظ التوكن

```java
// بعد التفعيل الناجح
public void onActivationSuccess(JSONObject response) {
    try {
        // استخرج التوكن من الاستجابة
        String token = response.getString("token");
        String username = response.getString("username");
        String password = response.getString("password");
        String expire = response.getString("expire");

        // احفظ في SharedPreferences
        SharedPreferences prefs = getSharedPreferences("app_data", MODE_PRIVATE);
        SharedPreferences.Editor editor = prefs.edit();

        editor.putString("token", token);
        editor.putString("username", username);
        editor.putString("password", password);
        editor.putString("expire", expire);
        editor.putBoolean("is_activated", true);

        editor.apply(); // استخدم apply() بدلاً من commit()

        Log.d("TOKEN", "Token saved: " + token.substring(0, 32) + "...");

    } catch (JSONException e) {
        e.printStackTrace();
    }
}
```

### استخدام التوكن في الطلبات

```java
// إرسال طلب مع التوكن
public void fetchMovies() {
    SharedPreferences prefs = getSharedPreferences("app_data", MODE_PRIVATE);
    String token = prefs.getString("token", "");

    if (token.isEmpty()) {
        Log.e("TOKEN", "No token found! User must activate first.");
        return;
    }

    try {
        // أنشئ JSON للطلب
        JSONObject request = new JSONObject();
        request.put("mode", "movies_latest");
        request.put("token", token); // أضف التوكن مباشرة

        // قم بالتشفير والإرسال
        String encrypted = XOREncryption.encrypt(request.toString(), XOR_KEY);
        sendRequest(encrypted);

    } catch (JSONException e) {
        e.printStackTrace();
    }
}
```

### ⚠️ الأخطاء الشائعة

```java
// ❌ خطأ: استخدام Gson قد يُضيف escape characters
Gson gson = new Gson();
String json = gson.toJson(request); // قد يحول / إلى \/

// ✅ صحيح: استخدم JSONObject مباشرة
JSONObject request = new JSONObject();
request.put("token", token);
String json = request.toString();
```

```java
// ❌ خطأ: تعديل التوكن
String token = response.getString("token").replace("/", "\\/");

// ✅ صحيح: احفظه كما هو
String token = response.getString("token");
```

## 2️⃣ إضافة التوكن يدوياً للقاعدة

### الطريقة 1: عبر PHP Script

```bash
php /var/www/html/iptv/add_token_manually.php 426687097695 "f3ed71d56579442cbdc114e8f77e57b4..."
```

### الطريقة 2: عبر Shell Script

```bash
chmod +x /var/www/html/iptv/add_token_sql.sh
./add_token_sql.sh 426687097695 "f3ed71d56579442cbdc114e8f77e57b4..."
```

### الطريقة 3: عبر MySQL مباشرة

```sql
-- احصل على user_id
SELECT id FROM users WHERE username = '426687097695';

-- أضف التوكن (استبدل USER_ID بالرقم الفعلي)
INSERT INTO user_tokens (user_id, token, created, expires, ip, user_agent)
VALUES (
    41,  -- user_id
    'f3ed71d56579442cbdc114e8f77e57b4rueo5uPmlFslxvKta6fG9lMBRXMKLU0VDwvqXuJXnhJs/PpqTO7+kK0mtjIDjNwj1ZVOAER8JPeqKd5QljjBHQ==',
    UNIX_TIMESTAMP(),
    UNIX_TIMESTAMP() + 604800,  -- 7 days
    '127.0.0.1',
    'Manual Entry'
);
```

## 3️⃣ التحقق من التوكن

### اختبار التوكن

```bash
# اختبر أي توكن
php /var/www/html/iptv/test_token.php "your_token_here"
```

### التحقق من قاعدة البيانات

```bash
mysql -u root -p1231231 xtream_iptvpro -e "
SELECT
    t.token,
    u.username,
    FROM_UNIXTIME(t.created) as created,
    FROM_UNIXTIME(t.expires) as expires,
    (t.expires > UNIX_TIMESTAMP()) as is_valid
FROM user_tokens t
JOIN users u ON t.user_id = u.id
WHERE u.username = '426687097695'
ORDER BY t.created DESC
LIMIT 1;
"
```

## 4️⃣ مثال كامل - Android Code

```java
public class TokenManager {
    private static final String PREFS_NAME = "app_data";
    private static final String KEY_TOKEN = "token";
    private static final String XOR_KEY = "KvuZRm365wk5Sz&Xy.Zx";

    private Context context;
    private SharedPreferences prefs;

    public TokenManager(Context context) {
        this.context = context;
        this.prefs = context.getSharedPreferences(PREFS_NAME, Context.MODE_PRIVATE);
    }

    // حفظ التوكن
    public void saveToken(String token) {
        prefs.edit().putString(KEY_TOKEN, token).apply();
        Log.d("TokenManager", "Token saved successfully");
    }

    // قراءة التوكن
    public String getToken() {
        return prefs.getString(KEY_TOKEN, "");
    }

    // التحقق من وجود توكن
    public boolean hasToken() {
        return !getToken().isEmpty();
    }

    // حذف التوكن
    public void clearToken() {
        prefs.edit().remove(KEY_TOKEN).apply();
    }

    // إرسال طلب مع التوكن
    public void makeRequest(String mode, ApiCallback callback) {
        String token = getToken();

        if (token.isEmpty()) {
            callback.onError("No token available");
            return;
        }

        try {
            JSONObject request = new JSONObject();
            request.put("mode", mode);
            request.put("token", token);

            String encrypted = XOREncryption.encrypt(request.toString(), XOR_KEY);

            // أرسل الطلب للسيرفر
            sendToServer(encrypted, callback);

        } catch (JSONException e) {
            callback.onError(e.getMessage());
        }
    }

    private void sendToServer(String data, ApiCallback callback) {
        // كود الإرسال للسيرفر
        OkHttpClient client = new OkHttpClient();

        RequestBody body = new FormBody.Builder()
            .add("json", data)
            .build();

        Request request = new Request.Builder()
            .url("http://45.154.207.45/iptv/V6APK/API-V6APK.php")
            .post(body)
            .addHeader("User-Agent", "okhttp/3.10.0")
            .build();

        client.newCall(request).enqueue(new Callback() {
            @Override
            public void onResponse(Call call, Response response) throws IOException {
                String encrypted = response.body().string();
                String decrypted = XOREncryption.decrypt(encrypted, XOR_KEY);
                callback.onSuccess(decrypted);
            }

            @Override
            public void onFailure(Call call, IOException e) {
                callback.onError(e.getMessage());
            }
        });
    }
}

interface ApiCallback {
    void onSuccess(String response);
    void onError(String error);
}
```

## 5️⃣ استكشاف الأخطاء

### الخطأ: "invalid or expired token"

**الأسباب:**
1. التوكن غير موجود في قاعدة البيانات
2. التوكن منتهي الصلاحية
3. التوكن تم تعديله (escape characters)

**الحل:**
```bash
# تحقق من التوكن
php /var/www/html/iptv/test_token.php "your_token"

# أو احصل على توكن جديد
php /var/www/html/iptv/get_valid_token.php 426687097695
```

### الخطأ: التوكن مختلف عن المحفوظ

**السبب:** التطبيق يعدل التوكن أو يخزن توكن خاطئ

**الحل:**
```java
// أضف logging في التطبيق
Log.d("TOKEN_DEBUG", "Received: " + receivedToken);
Log.d("TOKEN_DEBUG", "Saved: " + savedToken);
Log.d("TOKEN_DEBUG", "Sending: " + tokenToSend);
// قارن القيم في logcat
```

## 6️⃣ أدوات مفيدة

```bash
# تفعيل كود جديد
php /var/www/html/iptv/get_valid_token.php <code> <mac>

# اختبار توكن
php /var/www/html/iptv/test_token.php "<token>"

# تشخيص المشكلة
php /var/www/html/iptv/diagnose_app_issue.php

# إضافة توكن يدوياً
php /var/www/html/iptv/add_token_manually.php <username> "<token>"
```

---

## 🔐 معلومات مهمة

- **صلاحية التوكن:** 7 أيام
- **XOR Key:** `KvuZRm365wk5Sz&Xy.Zx`
- **طول التوكن:** 120 حرف
- **تنسيق التوكن:** `[32 chars MD5][88 chars base64]`

---

**آخر تحديث:** 2025-11-07
