# 🔧 Fix: Save Token on First Activation

## ❌ المشكلة (Problem)

عند التفعيل لأول مرة، يتم إنشاء التوكن لكن **لا يتم حفظه** في قاعدة البيانات `user_tokens`.

```php
// الكود الحالي يولد التوكن فقط
$token = md5(uniqid() . $username . time()) . base64_encode(random_bytes(64));

// لكن لا يحفظه! ❌
```

---

## ✅ الحل (Solution)

أضف الكود التالي **بعد** توليد التوكن مباشرة:

### 📍 الموقع في الكود:

ابحث عن هذا السطر في `activate_new_code_enhanced`:

```php
// Generate token
$token = md5(uniqid() . $username . time()) . base64_encode(random_bytes(64));
```

### ➕ أضف هذا الكود بعده:

```php
// ═══════════════════════════════════════════════════════════════════════
// ⭐ FIX: Save token to database
// ═══════════════════════════════════════════════════════════════════════

// Get token expiry days from config (default 7 days)
$token_expiry_days = isset($_CFG['token_expiry_days']) ? intval($_CFG['token_expiry_days']) : 7;
$token_created = time();
$token_expires = $token_created + (86400 * $token_expiry_days);

// Prepare token data
$token_data = [
    'user_id' => $xtID,
    'token' => $token,
    'created' => $token_created,
    'expires' => $token_expires,
    'ip' => $this->ip(),
    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown'
];

// Insert token into user_tokens table
try {
    $token_insert_result = $this->insert('user_tokens', $token_data);

    if($token_insert_result) {
        debug_log("✅ Token saved successfully for user: " . $username);
        debug_log("Token: " . substr($token, 0, 32) . "...");
        debug_log("Expires: " . date('Y-m-d H:i:s', $token_expires));
    } else {
        debug_log("⚠️ Warning: Failed to save token for user: " . $username);
        $this->log("Failed to save token - Error: " . $this->link->error);
    }
} catch (Exception $e) {
    debug_log("❌ Error saving token: " . $e->getMessage());
    $this->log("Error saving token: " . $e->getMessage());
}

// ═══════════════════════════════════════════════════════════════════════
```

---

## 📋 الكود الكامل (Complete Code Section)

```php
// Generate token
$token = md5(uniqid() . $username . time()) . base64_encode(random_bytes(64));

// ⭐ SAVE TOKEN TO DATABASE
$token_expiry_days = isset($_CFG['token_expiry_days']) ? intval($_CFG['token_expiry_days']) : 7;
$token_created = time();
$token_expires = $token_created + (86400 * $token_expiry_days);

$token_data = [
    'user_id' => $xtID,
    'token' => $token,
    'created' => $token_created,
    'expires' => $token_expires,
    'ip' => $this->ip(),
    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown'
];

try {
    $this->insert('user_tokens', $token_data);
    debug_log("Token saved for user: " . $username);
} catch (Exception $e) {
    debug_log("Error saving token: " . $e->getMessage());
}

// Default allowed formats (continue with existing code)
$allowed_formats = ["m3u8", "ts"];
```

---

## 🧪 اختبار التعديل (Testing)

بعد إضافة الكود، اختبر التفعيل:

```bash
# 1. فعّل كود جديد
php /var/www/html/iptv/activate_and_get_token.php 225666426500 8c:fd:f0:00:7f:13

# 2. تحقق من حفظ التوكن في قاعدة البيانات
mysql -u root -p1231231 xtream_iptvpro -e "
SELECT
    u.username,
    t.token,
    FROM_UNIXTIME(t.created) as created,
    FROM_UNIXTIME(t.expires) as expires
FROM user_tokens t
JOIN users u ON t.user_id = u.id
WHERE u.username = '225666426500'
ORDER BY t.created DESC
LIMIT 1;
"

# 3. اختبر التوكن
php /var/www/html/iptv/test_token.php "token_from_database"
```

---

## ✅ ما الذي يفعله التعديل؟

### قبل التعديل:
```
1. إنشاء حساب المستخدم ✅
2. توليد التوكن ✅
3. إرجاع التوكن في الاستجابة ✅
4. حفظ التوكن في قاعدة البيانات ❌ (مفقود!)
```

### بعد التعديل:
```
1. إنشاء حساب المستخدم ✅
2. توليد التوكن ✅
3. حفظ التوكن في قاعدة البيانات ✅ (مُضاف!)
4. إرجاع التوكن في الاستجابة ✅
```

---

## 📊 بيانات التوكن المحفوظة

الكود يحفظ:

| الحقل | القيمة | الوصف |
|-------|--------|-------|
| `user_id` | `$xtID` | معرف المستخدم |
| `token` | `$token` | التوكن الكامل (120 حرف) |
| `created` | `time()` | وقت الإنشاء |
| `expires` | `time() + 7 days` | وقت الانتهاء (7 أيام) |
| `ip` | `$this->ip()` | عنوان IP |
| `user_agent` | `$_SERVER['HTTP_USER_AGENT']` | User Agent |

---

## 🔍 التحقق من نجاح التعديل

### 1. تحقق من السجلات (Logs):

```bash
tail -f /var/www/html/iptv/V6APK/logs/debug_$(date +%Y-%m-%d).log
```

يجب أن ترى:
```
✅ Token saved successfully for user: 225666426500
Token: 7682126a6f006d9aafeebdb81a192cc5...
Expires: 2025-11-14 16:33:38
```

### 2. تحقق من قاعدة البيانات:

```sql
SELECT COUNT(*) FROM user_tokens WHERE user_id = [xtID];
-- يجب أن يكون > 0
```

### 3. اختبر التوكن:

```bash
php /var/www/html/iptv/test_token.php "the_token_from_response"
# يجب أن يكون: ✅ SUCCESS! Token is valid and working.
```

---

## ⚠️ ملاحظات مهمة

1. **موقع التعديل:** أضف الكود **بعد** `$token = ...` وقبل `$allowed_formats = ...`

2. **المتغير `$xtID`:** تأكد من أنه موجود (يتم إنشاؤه عند إنشاء المستخدم)

3. **الدالة `$this->insert()`:** تأكد من أنها تعمل بشكل صحيح

4. **الجدول `user_tokens`:** تأكد من وجوده بالحقول الصحيحة:
   - `id` (INT, AUTO_INCREMENT)
   - `user_id` (INT)
   - `token` (VARCHAR(255))
   - `created` (INT)
   - `expires` (INT)
   - `ip` (VARCHAR(45))
   - `user_agent` (TEXT)

---

## 📁 الملفات المرجعية

- ✅ **activate_new_code_enhanced_FIXED.php** - الكود الكامل المُصلح
- ✅ **TOKEN_SAVE_FIX.md** - هذا الملف (دليل التعديل)
- ✅ **activate_and_get_token.php** - سكريبت اختبار
- ✅ **test_token.php** - اختبار التوكن

---

## 🎯 الخلاصة

**التعديل المطلوب:** أضف **8 أسطر فقط** بعد توليد التوكن لحفظه في قاعدة البيانات.

**النتيجة:**
- ✅ التوكن يُحفظ تلقائياً عند التفعيل
- ✅ يمكن استخدام التوكن في الطلبات اللاحقة
- ✅ التحقق من صلاحية التوكن يعمل بشكل صحيح

---

**تاريخ التحديث:** 2025-11-07
**الإصدار:** 1.0
