#!/usr/bin/php
<?php
// Automatic sync daemon - Syncs streams from remote panel every 30 seconds
// Remote DB: 204.188.233.170:7999
// Runs continuously as a systemd service

require_once('/var/www/html/iptv/includes/Config.php');

// Remote connection details
require_once('/var/www/html/iptv/includes/RemotePanelConfig.php'); $remote_host = RemotePanelConfig::get('host');
$remote_port = RemotePanelConfig::get('port');
$remote_user = RemotePanelConfig::get('user');
$remote_pass = RemotePanelConfig::get('pass');
$remote_db = RemotePanelConfig::get('database');

// Local connection
$local_host = $config['db']['hostname'];
$local_user = $config['db']['username'];
$local_pass = $config['db']['password'];
$local_db = $config['db']['database'];

// Log file
$logfile = '/var/log/auto_sync_daemon.log';

function logMsg($msg) {
    global $logfile;
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logfile, "[$timestamp] $msg\n", FILE_APPEND);
}

logMsg("=== Auto-Sync Daemon Started ===");
logMsg("Remote: $remote_host:$remote_port");
logMsg("Sync interval: 30 seconds");

// Main loop
$iteration = 0;
while(true) {
    $iteration++;
    $start_time = microtime(true);

    try {
        // Connect to local database
        $local_conn = @new mysqli($local_host, $local_user, $local_pass, $local_db);
        if($local_conn->connect_error) {
            logMsg("ERROR: Local DB connection failed: " . $local_conn->connect_error);
            sleep(30);
            continue;
        }

        // Connect to remote database
        $remote_conn = @new mysqli($remote_host, $remote_user, $remote_pass, $remote_db, $remote_port);
        if($remote_conn->connect_error) {
            logMsg("ERROR: Remote DB connection failed: " . $remote_conn->connect_error);
            $local_conn->close();
            sleep(30);
            continue;
        }

        // Get running stream PIDs and uptime from remote server
        $pid_cmd = "sshpass -p " . escapeshellarg($remote_pass) . " ssh -o StrictHostKeyChecking=no " .
                   escapeshellarg($remote_user) . "@" . escapeshellarg($remote_host) .
                   " \"ps -eo pid,etime,args | grep -E 'ffmpeg.*streams/[0-9]+' | grep -v grep\" 2>&1";

        $pid_result = shell_exec($pid_cmd);

        $running_streams = [];
        if(!empty($pid_result)) {
            $pid_lines = explode("\n", trim($pid_result));
            foreach($pid_lines as $pid_line) {
                if(preg_match('/streams\/(\d+)_/', $pid_line, $matches)) {
                    $stream_id = intval($matches[1]);
                    // Extract PID and elapsed time
                    if(preg_match('/^\s*(\d+)\s+([\d:-]+)\s+/', $pid_line, $pid_match)) {
                        $pid = intval($pid_match[1]);
                        $etime = trim($pid_match[2]);

                        // Convert etime to seconds
                        $seconds = 0;
                        $parts = explode('-', $etime);
                        if(count($parts) == 2) {
                            // Format: days-HH:MM:SS
                            $days = intval($parts[0]);
                            $time_parts = explode(':', $parts[1]);
                            $seconds = $days * 86400;
                            if(count($time_parts) == 3) {
                                $seconds += intval($time_parts[0]) * 3600 + intval($time_parts[1]) * 60 + intval($time_parts[2]);
                            }
                        } else {
                            // Format: HH:MM:SS or MM:SS
                            $time_parts = explode(':', $etime);
                            if(count($time_parts) == 3) {
                                $seconds = intval($time_parts[0]) * 3600 + intval($time_parts[1]) * 60 + intval($time_parts[2]);
                            } elseif(count($time_parts) == 2) {
                                $seconds = intval($time_parts[0]) * 60 + intval($time_parts[1]);
                            }
                        }

                        $stream_started = time() - $seconds;

                        if(!isset($running_streams[$stream_id])) {
                            $running_streams[$stream_id] = [
                                'pid' => $pid,
                                'stream_started' => $stream_started,
                                'uptime' => $seconds
                            ];
                        }
                    }
                }
            }
        }

        // Fetch streams from remote (type=0 = live channels) with category info and stream mode
        $query = "SELECT s.id, s.name, s.logo_image, s.disabled, s.urls, s.redirect, s.direct_streaming_on_demand, cs.cid as category_id
                  FROM streams s
                  LEFT JOIN categories_sids cs ON s.id = cs.sid
                  WHERE s.type=0
                  ORDER BY s.id";
        $result = $remote_conn->query($query);

        if(!$result) {
            logMsg("ERROR: Query failed: " . $remote_conn->error);
            $remote_conn->close();
            $local_conn->close();
            sleep(30);
            continue;
        }

        $updated = 0;
        $inserted = 0;
        $deleted = 0;

        // Collect all remote stream IDs
        $remote_stream_ids = [];

        while($stream = $result->fetch_assoc()) {
            $remote_stream_ids[] = intval($stream['id']);
            $id = intval($stream['id']);
            $name = $local_conn->real_escape_string($stream['name']);
            $logo_image = $stream['logo_image'];
            $disabled = intval($stream['disabled']);
            $urls = $stream['urls']; // Remote URLs field (JSON array)
            $category_id = isset($stream['category_id']) && $stream['category_id'] ? intval($stream['category_id']) : 0;
            $redirect = isset($stream['redirect']) ? intval($stream['redirect']) : 0;
            $on_demand = isset($stream['direct_streaming_on_demand']) ? intval($stream['direct_streaming_on_demand']) : 0;

            // NOTE: We do NOT sync the disabled status from remote panel
            // Local status is managed independently by SetLive/SetOnDemand functions
            // This allows streams to remain enabled locally even if disabled remotely

            // Check if exists locally (any type)
            $check = $local_conn->query("SELECT id, type, stream_display_name, stream_icon, status, stream_source, category_id, redirect_stream, direct_source FROM streams WHERE id = $id");

            if($check && $check->num_rows > 0) {
                // Update existing (ensure type is correct for live streams)
                $local_row = $check->fetch_assoc();
                $update_fields = [];

                // Always set type to 1 for live streams
                if($local_row['type'] != 1) {
                    $update_fields[] = "type = 1";
                }

                if($local_row['stream_display_name'] != $stream['name']) {
                    $update_fields[] = "stream_display_name = '$name'";
                }

                if(!empty($logo_image) && $logo_image != 'NULL' && $local_row['stream_icon'] != $logo_image) {
                    $logo_escaped = $local_conn->real_escape_string($logo_image);
                    $update_fields[] = "stream_icon = '$logo_escaped'";
                }

                // Status is NOT synced from remote - managed locally only

                // Sync URLs from remote (urls -> stream_source)
                if(!empty($urls) && $urls != 'NULL' && $local_row['stream_source'] != $urls) {
                    $urls_escaped = $local_conn->real_escape_string($urls);
                    $update_fields[] = "stream_source = '$urls_escaped'";
                }

                // Sync category_id from remote
                if($category_id > 0 && $local_row['category_id'] != $category_id) {
                    $update_fields[] = "category_id = $category_id";
                }

                // Sync redirect_stream from remote
                if($local_row['redirect_stream'] != $redirect) {
                    $update_fields[] = "redirect_stream = $redirect";
                }

                // Mode sync: Pull from remote to local (Remote is master)
                // Local Form uses InsertUpdateStreamServers which updates BOTH local and remote
                // So we sync FROM remote TO local to keep them in sync
                $local_direct_source = intval($local_row['direct_source']);
                $remote_on_demand = intval($on_demand);

                // Convert remote mode to local mode
                // remote: direct_streaming_on_demand (1=OnDemand, 0=Live)
                // local: direct_source (0=OnDemand, 1=Live)
                $expected_local_from_remote = ($remote_on_demand == 0) ? 1 : 0;

                // Sync local to match remote
                if($local_direct_source != $expected_local_from_remote) {
                    $update_fields[] = "direct_source = $expected_local_from_remote";
                }

                if(count($update_fields) > 0) {
                    $local_conn->query("UPDATE streams SET " . implode(', ', $update_fields) . " WHERE id = $id");
                    $updated++;
                }

                // Sync streams_sys.on_demand to match streams.direct_source
                // This ensures consistency after sync
                $expected_sys_on_demand = ($local_direct_source == 0) ? 1 : 0;

                $check_sys = $local_conn->query("SELECT on_demand FROM streams_sys WHERE stream_id = $id");
                if($check_sys && $check_sys->num_rows > 0) {
                    $sys_row = $check_sys->fetch_assoc();
                    if($sys_row['on_demand'] != $expected_sys_on_demand) {
                        $local_conn->query("UPDATE streams_sys SET on_demand = $expected_sys_on_demand WHERE stream_id = $id");
                    }
                } else {
                    // Insert new entry
                    $local_conn->query("INSERT INTO streams_sys (stream_id, server_id, on_demand) VALUES ($id, 1, $expected_sys_on_demand)");
                }
            } else {
                // Insert new stream - default to enabled (status=1)
                // Set direct_source based on remote on_demand (inverted)
                // remote on_demand=1 (OnDemand) -> local direct_source=0 (OnDemand)
                // remote on_demand=0 (Live) -> local direct_source=1 (Live)
                $new_direct_source = ($on_demand == 1) ? 0 : 1;

                $logo_val = empty($logo_image) || $logo_image == 'NULL' ? "''" : "'" . $local_conn->real_escape_string($logo_image) . "'";
                $urls_val = empty($urls) || $urls == 'NULL' ? "'[]'" : "'" . $local_conn->real_escape_string($urls) . "'";
                $insert_sql = "INSERT INTO streams (id, type, stream_display_name, stream_icon, stream_source, category_id, redirect_stream, direct_source, status, added)
                               VALUES ($id, 1, '$name', $logo_val, $urls_val, $category_id, $redirect, $new_direct_source, 1, " . time() . ")";

                if($local_conn->query($insert_sql)) {
                    $inserted++;
                    logMsg("Inserted new stream: $name (ID: $id) with direct_source=$new_direct_source");
                }
            }

            // Update stream running status in streams_sys
            $is_running = isset($running_streams[$id]);
            $pid_value = $is_running ? $running_streams[$id]['pid'] : 'NULL';
            $stream_status = $is_running ? 0 : 1; // 0=running, 1=stopped
            $stream_started = $is_running ? $running_streams[$id]['stream_started'] : 'NULL';
            // current_source: 0 when running (first URL), NULL when stopped
            $current_source = $is_running ? 0 : 'NULL';

            $check_sys = $local_conn->query("SELECT stream_id FROM streams_sys WHERE stream_id = $id");
            if($check_sys && $check_sys->num_rows > 0) {
                $local_conn->query("UPDATE streams_sys SET
                                   pid = $pid_value,
                                   stream_status = $stream_status,
                                   stream_started = $stream_started,
                                   current_source = $current_source
                                   WHERE stream_id = $id");
            } else {
                $local_conn->query("INSERT INTO streams_sys (stream_id, server_id, pid, stream_status, stream_started, current_source)
                                   VALUES ($id, 1, $pid_value, $stream_status, $stream_started, $current_source)");
            }
        }

        // Delete streams that no longer exist on remote panel
        // Only delete streams with type=1 (live channels synced from remote)
        if(count($remote_stream_ids) > 0) {
            $remote_ids_str = implode(',', $remote_stream_ids);
            // Find local streams that don't exist on remote anymore
            $delete_query = "SELECT id, stream_display_name FROM streams WHERE type = 1 AND id NOT IN ($remote_ids_str)";
            $delete_result = $local_conn->query($delete_query);

            if($delete_result && $delete_result->num_rows > 0) {
                while($del_row = $delete_result->fetch_assoc()) {
                    $del_id = $del_row['id'];
                    $del_name = $del_row['stream_display_name'];

                    // Delete from streams_sys first
                    $local_conn->query("DELETE FROM streams_sys WHERE stream_id = $del_id");

                    // Delete from streams
                    $local_conn->query("DELETE FROM streams WHERE id = $del_id");

                    $deleted++;
                    logMsg("Deleted stream (removed from remote): $del_name (ID: $del_id)");
                }
            }
        }

        // ========================================
        // SYNC MOVIES/VOD from remote panel
        // Remote type=1 (Movies) → Local type=2 (Movies)
        // ========================================
        $movies_query = "SELECT s.id, s.name, s.logo_image, s.disabled, s.urls, s.files, s.redirect, s.tmdb, s.year, s.container_format, s.server_id, s.servers, cs.cid as category_id
                         FROM streams s
                         LEFT JOIN categories_sids cs ON s.id = cs.sid
                         WHERE s.type=1
                         ORDER BY s.id";
        $movies_result = $remote_conn->query($movies_query);

        $movies_updated = 0;
        $movies_inserted = 0;
        $movies_deleted = 0;
        $remote_movie_ids = [];

        if($movies_result) {
            while($movie = $movies_result->fetch_assoc()) {
                $remote_movie_ids[] = intval($movie['id']);
                $id = intval($movie['id']);
                $name = $local_conn->real_escape_string($movie['name']);
                $logo_image = $movie['logo_image'];
                $disabled = intval($movie['disabled']);
                $urls = $movie['urls']; // Remote URLs field (JSON array) - for live channels
                $files = $movie['files']; // Remote files field (JSON array) - for VOD/Movies
                $category_id = isset($movie['category_id']) && $movie['category_id'] ? intval($movie['category_id']) : 0;
                $redirect = isset($movie['redirect']) ? intval($movie['redirect']) : 0;
                $tmdb_data = $movie['tmdb']; // TMDB metadata (JSON)
                $year = isset($movie['year']) ? intval($movie['year']) : 0;
                $container = isset($movie['container_format']) ? $movie['container_format'] : 'mp4';
                $server_id = $movie['server_id'] ?? 0;
                $servers_json = $movie['servers'];

                // Parse servers JSON
                $server_ids = [];
                if(!empty($servers_json)) {
                    $servers_array = json_decode($servers_json, true);
                    if(is_array($servers_array)) {
                        $server_ids = array_map('intval', $servers_array);
                    }
                }
                // If no servers in JSON, use server_id
                if(empty($server_ids) && $server_id > 0) {
                    $server_ids = [$server_id];
                }

                // Extract poster URL from TMDB data (prefer TMDB poster over local logo)
                $poster_url = $logo_image;
                if(!empty($tmdb_data)) {
                    $tmdb_array = json_decode($tmdb_data, true);
                    if(is_array($tmdb_array) && isset($tmdb_array['poster_image']) && !empty($tmdb_array['poster_image'])) {
                        $poster_url = $tmdb_array['poster_image'];
                    }
                }

                // Parse URLs from remote (JSON array format)
                // For movies (type=1), use 'files' field instead of 'urls'
                $stream_source = '';
                $source_data = !empty($files) ? $files : $urls;
                if(!empty($source_data)) {
                    $source_array = json_decode($source_data, true);
                    if(is_array($source_array) && count($source_array) > 0) {
                        $stream_source = json_encode($source_array);
                    }
                }

                // Convert disabled status (remote: 0=enabled, 1=disabled → local: 1=enabled, 0=disabled)
                $local_status = ($disabled == 0) ? 1 : 0;

                // Check if movie exists in local database
                $check_query = "SELECT id, stream_display_name, stream_source, stream_icon, status, category_id, redirect_stream, movie_propeties, target_container
                                FROM streams WHERE id = $id AND type = 2";
                $check_result = $local_conn->query($check_query);

                if($check_result && $check_result->num_rows > 0) {
                    // Movie exists - update if changed
                    $local_row = $check_result->fetch_assoc();
                    $update_fields = [];

                    if($local_row['stream_display_name'] != $name) {
                        $update_fields[] = "stream_display_name = '$name'";
                    }

                    if($local_row['stream_source'] != $stream_source && !empty($stream_source)) {
                        $stream_source_esc = $local_conn->real_escape_string($stream_source);
                        $update_fields[] = "stream_source = '$stream_source_esc'";
                    }

                    if($local_row['stream_icon'] != $poster_url) {
                        $poster_esc = $local_conn->real_escape_string($poster_url);
                        $update_fields[] = "stream_icon = '$poster_esc'";
                    }

                    if($local_row['category_id'] != $category_id && $category_id > 0) {
                        $update_fields[] = "category_id = $category_id";
                    }

                    if($local_row['redirect_stream'] != $redirect) {
                        $update_fields[] = "redirect_stream = $redirect";
                    }

                    // Sync TMDB data
                    if(!empty($tmdb_data) && $local_row['movie_propeties'] != $tmdb_data) {
                        $tmdb_esc = $local_conn->real_escape_string($tmdb_data);
                        $update_fields[] = "movie_propeties = '$tmdb_esc'";
                    }

                    // Sync container format
                    if(!empty($container) && $local_row['target_container'] != $container) {
                        $update_fields[] = "target_container = '$container'";
                    }

                    if(count($update_fields) > 0) {
                        $update_sql = "UPDATE streams SET " . implode(', ', $update_fields) . " WHERE id = $id";
                        $local_conn->query($update_sql);
                        $movies_updated++;
                    }
                } else {
                    // Movie doesn't exist - insert it
                    $stream_source_esc = $local_conn->real_escape_string($stream_source);
                    $poster_esc = $local_conn->real_escape_string($poster_url);
                    $tmdb_esc = !empty($tmdb_data) ? $local_conn->real_escape_string($tmdb_data) : '';

                    $insert_sql = "INSERT INTO streams (id, stream_display_name, stream_source, stream_icon, type, status, category_id, redirect_stream, direct_source, movie_propeties, target_container, added)
                                   VALUES ($id, '$name', '$stream_source_esc', '$poster_esc', 2, $local_status, $category_id, $redirect, 0, '$tmdb_esc', '$container', $year)";
                    $local_conn->query($insert_sql);
                    $movies_inserted++;
                    logMsg("Inserted movie from remote: $name (ID: $id)");
                }

                // Sync servers for movie (both new and existing)
                if(count($server_ids) > 0) {
                    $first_server_id = $server_ids[0];
                    // Check if streams_sys entry exists
                    $check_sys = $local_conn->query("SELECT stream_id FROM streams_sys WHERE stream_id = $id");
                    if($check_sys && $check_sys->num_rows > 0) {
                        // Update server if changed
                        $local_conn->query("UPDATE streams_sys SET server_id = $first_server_id WHERE stream_id = $id");
                    } else {
                        // Insert if not exists
                        $local_conn->query("INSERT INTO streams_sys (stream_id, server_id, on_demand, stream_status, pid)
                                          VALUES ($id, $first_server_id, 0, 1, NULL)");
                    }
                }
            }

            // Delete movies that no longer exist on remote panel
            if(count($remote_movie_ids) > 0) {
                $remote_movie_ids_str = implode(',', $remote_movie_ids);
                $delete_movies_query = "SELECT id, stream_display_name FROM streams WHERE type = 2 AND id NOT IN ($remote_movie_ids_str)";
                $delete_movies_result = $local_conn->query($delete_movies_query);

                if($delete_movies_result && $delete_movies_result->num_rows > 0) {
                    while($del_movie = $delete_movies_result->fetch_assoc()) {
                        $del_id = $del_movie['id'];
                        $del_name = $del_movie['stream_display_name'];

                        // Delete from streams
                        $local_conn->query("DELETE FROM streams WHERE id = $del_id");

                        $movies_deleted++;
                        logMsg("Deleted movie (removed from remote): $del_name (ID: $del_id)");
                    }
                }
            }

            $movies_result->free();
        }

        // ========================================
        // SYNC SERIES from remote panel
        // ========================================
        $series_updated = 0;
        $series_inserted = 0;
        $series_deleted = 0;

        if($remote_conn) {
            // First sync series categories to stream_categories table
            $cat_query = "SELECT id, name, type FROM categories WHERE type=2 ORDER BY id";
            $cat_result = $remote_conn->query($cat_query);

            while($cat = $cat_result->fetch_assoc()) {
                $cat_id = intval($cat['id']);
                $cat_name = $cat['name'];
                $cat_name_esc = $local_conn->real_escape_string($cat_name);

                // Check if category exists in stream_categories
                $check_cat = $local_conn->query("SELECT id FROM stream_categories WHERE id='$cat_id' AND category_type='series'");

                if($check_cat && $check_cat->num_rows == 0) {
                    // Insert new category with same ID
                    $local_conn->query("INSERT INTO stream_categories (id, category_name, category_type, category_icon, parent_id, cat_order, isLocked)
                                       VALUES ($cat_id, '$cat_name_esc', 'series', '', 0, 0, 0)");
                    logMsg("Inserted series category: $cat_name (ID: $cat_id)");
                }
            }
            $cat_result->free();

            // Get all series from remote panel with their categories
            $series_query = "SELECT s.id, s.name, s.logo_image, s.tmdb_id, s.tmdb, s.year, s.added_time,
                                   cs.cid as category_id
                            FROM series s
                            LEFT JOIN categories_series cs ON s.id = cs.series_id
                            ORDER BY s.id";
            $series_result = $remote_conn->query($series_query);

            $remote_series_ids = [];

            while($series = $series_result->fetch_assoc()) {
                $id = intval($series['id']);
                $name = $series['name'];
                $logo_image = $series['logo_image'];
                $tmdb_id = $series['tmdb_id'];
                $tmdb_data = $series['tmdb'];
                $year = $series['year'] ?? 0;
                $added_time = $series['added_time'] ?? time();
                $category_id = $series['category_id'] ?? 0;

                $remote_series_ids[] = $id;

                // Extract poster URL from TMDB data (prefer TMDB poster over local logo)
                $poster_url = $logo_image;
                if(!empty($tmdb_data)) {
                    $tmdb_array = json_decode($tmdb_data, true);
                    if(is_array($tmdb_array) && isset($tmdb_array['poster_image']) && !empty($tmdb_array['poster_image'])) {
                        $poster_url = $tmdb_array['poster_image'];
                    }
                }

                // Extract other TMDB fields for series table
                $plot = '';
                $cast = '';
                $rating = 0;
                $director = '';
                $release_date = '';
                $genre = '';
                $backdrop_path = '';
                $youtube_trailer = '';
                $first_air_year = '';
                $seasons_json = '[]';

                if(!empty($tmdb_data)) {
                    $tmdb_array = json_decode($tmdb_data, true);
                    if(is_array($tmdb_array)) {
                        $plot = $tmdb_array['plot'] ?? '';
                        $cast = $tmdb_array['cast'] ?? '';
                        $rating = intval($tmdb_array['rating'] ?? 0);
                        $director = $tmdb_array['director'] ?? '';
                        $release_date = $tmdb_array['release_date'] ?? '';
                        $genre = $tmdb_array['genre'] ?? '';
                        $backdrop_path = $tmdb_array['backdrop_path'] ?? '';
                        $youtube_trailer = $tmdb_array['video_trailer'] ?? '';
                        $first_air_year = $tmdb_array['first_air_year'] ?? '';
                        if(isset($tmdb_array['seasons']) && is_array($tmdb_array['seasons'])) {
                            $seasons_json = json_encode($tmdb_array['seasons']);
                        }
                    }
                }

                // Escape strings
                $name_esc = $local_conn->real_escape_string($name);
                $poster_esc = $local_conn->real_escape_string($poster_url);
                $plot_esc = $local_conn->real_escape_string($plot);
                $cast_esc = $local_conn->real_escape_string($cast);
                $director_esc = $local_conn->real_escape_string($director);
                $release_date_esc = $local_conn->real_escape_string($release_date);
                $genre_esc = $local_conn->real_escape_string($genre);
                $backdrop_esc = $local_conn->real_escape_string($backdrop_path);
                $trailer_esc = $local_conn->real_escape_string($youtube_trailer);
                $seasons_esc = $local_conn->real_escape_string($seasons_json);

                // Check if series exists in local
                $check_query = "SELECT id, title, cover, plot, cast, rating, director, releaseDate, genre, backdrop_path, youtube_trailer, seasons, tmdb_id, category_id
                               FROM series WHERE id = $id";
                $check_result = $local_conn->query($check_query);

                if($check_result && $check_result->num_rows > 0) {
                    // Series exists - update if changed
                    $local_row = $check_result->fetch_assoc();
                    $update_fields = [];

                    if($local_row['title'] != $name) {
                        $update_fields[] = "title = '$name_esc'";
                    }
                    if($local_row['cover'] != $poster_url) {
                        $update_fields[] = "cover = '$poster_esc'";
                        $update_fields[] = "cover_big = '$poster_esc'";
                    }
                    if($local_row['plot'] != $plot) {
                        $update_fields[] = "plot = '$plot_esc'";
                    }
                    if($local_row['cast'] != $cast) {
                        $update_fields[] = "cast = '$cast_esc'";
                    }
                    if($local_row['rating'] != $rating) {
                        $update_fields[] = "rating = $rating";
                    }
                    if($local_row['director'] != $director) {
                        $update_fields[] = "director = '$director_esc'";
                    }
                    if($local_row['releaseDate'] != $release_date) {
                        $update_fields[] = "releaseDate = '$release_date_esc'";
                    }
                    if($local_row['genre'] != $genre) {
                        $update_fields[] = "genre = '$genre_esc'";
                    }
                    if($local_row['backdrop_path'] != $backdrop_path) {
                        $update_fields[] = "backdrop_path = '$backdrop_esc'";
                    }
                    if($local_row['youtube_trailer'] != $youtube_trailer) {
                        $update_fields[] = "youtube_trailer = '$trailer_esc'";
                    }
                    if($local_row['seasons'] != $seasons_json) {
                        $update_fields[] = "seasons = '$seasons_esc'";
                    }
                    if($local_row['tmdb_id'] != $tmdb_id) {
                        $update_fields[] = "tmdb_id = " . ($tmdb_id ? intval($tmdb_id) : 0);
                    }
                    if($local_row['category_id'] != $category_id) {
                        $update_fields[] = "category_id = $category_id";
                    }

                    if(count($update_fields) > 0) {
                        $update_query = "UPDATE series SET " . implode(', ', $update_fields) . " WHERE id = $id";
                        $local_conn->query($update_query);
                        $series_updated++;
                        logMsg("Updated series: $name (ID: $id) - " . count($update_fields) . " fields");
                    }

                } else {
                    // Series doesn't exist - insert new
                    $insert_sql = "INSERT INTO series (id, title, cover, cover_big, plot, cast, rating, director, releaseDate, genre, backdrop_path, youtube_trailer, seasons, tmdb_id, category_id, last_modified, episode_run_time)
                                  VALUES ($id, '$name_esc', '$poster_esc', '$poster_esc', '$plot_esc', '$cast_esc', $rating, '$director_esc', '$release_date_esc', '$genre_esc', '$backdrop_esc', '$trailer_esc', '$seasons_esc', " . ($tmdb_id ? intval($tmdb_id) : 0) . ", $category_id, " . time() . ", 0)";

                    if($local_conn->query($insert_sql)) {
                        $series_inserted++;
                        logMsg("Inserted series from remote: $name (ID: $id) - Category: $category_id");
                    }
                }
            }

            // Delete series that exist locally but not on remote
            if(count($remote_series_ids) > 0) {
                $remote_ids_str = implode(',', $remote_series_ids);
                $delete_series_query = "SELECT id, title FROM series WHERE id NOT IN ($remote_ids_str)";
                $delete_series_result = $local_conn->query($delete_series_query);

                if($delete_series_result && $delete_series_result->num_rows > 0) {
                    while($del_series = $delete_series_result->fetch_assoc()) {
                        $del_id = $del_series['id'];
                        $del_name = $del_series['title'];

                        // Delete from series table
                        $local_conn->query("DELETE FROM series WHERE id = $del_id");

                        // Delete related episodes
                        $local_conn->query("DELETE FROM series_episodes WHERE series_id = $del_id");

                        $series_deleted++;
                        logMsg("Deleted series (removed from remote): $del_name (ID: $del_id)");
                    }
                }
            }

            $series_result->free();
        }

        // ========================================
        // SYNC EPISODES from remote panel
        // ========================================
        $episodes_updated = 0;
        $episodes_inserted = 0;
        $episodes_deleted = 0;

        if($remote_conn) {
            // Get all episodes from remote streams table (where series_id is not null)
            $episodes_query = "SELECT id, name, series_id, season_number, episode_number, files, urls,
                                     container_format, logo_image, year, duration, server_id, servers
                              FROM streams
                              WHERE series_id IS NOT NULL AND series_id > 0
                              ORDER BY series_id, season_number, episode_number";
            $episodes_result = $remote_conn->query($episodes_query);

            $remote_episode_ids = [];

            while($episode = $episodes_result->fetch_assoc()) {
                $episode_id = intval($episode['id']);
                $episode_name = $episode['name'];
                $series_id = intval($episode['series_id']);
                $season_num = intval($episode['season_number']);
                $episode_num = intval($episode['episode_number']);
                $files = $episode['files'];
                $urls = $episode['urls'];
                $container = $episode['container_format'] ?? 'mp4';
                $logo_image = $episode['logo_image'];
                $year = $episode['year'] ?? 0;
                $duration = $episode['duration'] ?? 0;
                $server_id = $episode['server_id'] ?? 0;
                $servers_json = $episode['servers'];

                $remote_episode_ids[] = $episode_id;

                // Parse servers JSON
                $server_ids = [];
                if(!empty($servers_json)) {
                    $servers_array = json_decode($servers_json, true);
                    if(is_array($servers_array)) {
                        $server_ids = array_map('intval', $servers_array);
                    }
                }
                // If no servers in JSON, use server_id
                if(empty($server_ids) && $server_id > 0) {
                    $server_ids = [$server_id];
                }

                // Get stream source from files or urls
                $source_data = !empty($files) ? $files : $urls;
                $stream_source = '';
                if(!empty($source_data)) {
                    $source_array = json_decode($source_data, true);
                    if(is_array($source_array) && count($source_array) > 0) {
                        $stream_source = json_encode($source_array);
                    }
                }

                // Escape strings
                $episode_name_esc = $local_conn->real_escape_string($episode_name);
                $stream_source_esc = $local_conn->real_escape_string($stream_source);
                $logo_esc = $local_conn->real_escape_string($logo_image);
                $container_esc = $local_conn->real_escape_string($container);

                // Check if episode stream exists in local streams table
                $check_stream = $local_conn->query("SELECT id, stream_display_name, stream_source FROM streams WHERE id = $episode_id AND type = 2");

                if($check_stream && $check_stream->num_rows > 0) {
                    // Episode stream exists - update if changed
                    $local_stream = $check_stream->fetch_assoc();
                    $update_fields = [];

                    if($local_stream['stream_display_name'] != $episode_name) {
                        $update_fields[] = "stream_display_name = '$episode_name_esc'";
                    }
                    if($local_stream['stream_source'] != $stream_source) {
                        $update_fields[] = "stream_source = '$stream_source_esc'";
                    }

                    if(count($update_fields) > 0) {
                        $update_query = "UPDATE streams SET " . implode(', ', $update_fields) . " WHERE id = $episode_id";
                        $local_conn->query($update_query);
                        $episodes_updated++;
                    }

                    // Check if episode exists in series_episodes
                    $check_series_ep = $local_conn->query("SELECT id FROM series_episodes WHERE stream_id = $episode_id AND series_id = $series_id");
                    if($check_series_ep && $check_series_ep->num_rows == 0) {
                        // Insert into series_episodes
                        $local_conn->query("INSERT INTO series_episodes (season_num, series_id, stream_id, sort)
                                          VALUES ($season_num, $series_id, $episode_id, $episode_num)");
                        logMsg("Linked episode to series: $episode_name (Episode ID: $episode_id, Series ID: $series_id)");
                    }

                } else {
                    // Episode doesn't exist - insert new
                    $insert_stream = "INSERT INTO streams (id, stream_display_name, stream_source, stream_icon, type, status, category_id, direct_source, target_container, added, series_no)
                                     VALUES ($episode_id, '$episode_name_esc', '$stream_source_esc', '$logo_esc', 2, 1, 0, 0, '$container_esc', $year, $episode_num)";

                    if($local_conn->query($insert_stream)) {
                        // Insert into series_episodes to link episode with series
                        $local_conn->query("INSERT INTO series_episodes (season_num, series_id, stream_id, sort)
                                          VALUES ($season_num, $series_id, $episode_id, $episode_num)");

                        // Insert into streams_sys for server assignment
                        if(count($server_ids) > 0) {
                            $first_server_id = $server_ids[0];
                            $local_conn->query("INSERT INTO streams_sys (stream_id, server_id, on_demand, stream_status, pid)
                                              VALUES ($episode_id, $first_server_id, 0, 1, NULL)");
                        }

                        $episodes_inserted++;
                        logMsg("Inserted episode: $episode_name (ID: $episode_id, Series: $series_id, S{$season_num}E{$episode_num}, Server: " . implode(',', $server_ids) . ")");
                    }
                }

                // Update/sync servers for episode (both new and existing)
                if(count($server_ids) > 0) {
                    $first_server_id = $server_ids[0];
                    // Check if streams_sys entry exists
                    $check_sys = $local_conn->query("SELECT stream_id FROM streams_sys WHERE stream_id = $episode_id");
                    if($check_sys && $check_sys->num_rows > 0) {
                        // Update server if changed
                        $local_conn->query("UPDATE streams_sys SET server_id = $first_server_id WHERE stream_id = $episode_id");
                    } else {
                        // Insert if not exists
                        $local_conn->query("INSERT INTO streams_sys (stream_id, server_id, on_demand, stream_status, pid)
                                          VALUES ($episode_id, $first_server_id, 0, 1, NULL)");
                    }
                }
            }

            // Delete episodes that exist locally but not on remote
            if(count($remote_episode_ids) > 0) {
                $remote_ids_str = implode(',', $remote_episode_ids);
                // Find streams that are episodes (type=2 and in series_episodes table)
                $delete_episodes_query = "SELECT s.id, s.stream_display_name
                                         FROM streams s
                                         INNER JOIN series_episodes se ON s.id = se.stream_id
                                         WHERE s.id NOT IN ($remote_ids_str) AND s.type = 2";
                $delete_episodes_result = $local_conn->query($delete_episodes_query);

                if($delete_episodes_result && $delete_episodes_result->num_rows > 0) {
                    while($del_ep = $delete_episodes_result->fetch_assoc()) {
                        $del_id = $del_ep['id'];
                        $del_name = $del_ep['stream_display_name'];

                        // Delete from series_episodes
                        $local_conn->query("DELETE FROM series_episodes WHERE stream_id = $del_id");

                        // Delete from streams
                        $local_conn->query("DELETE FROM streams WHERE id = $del_id");

                        $episodes_deleted++;
                        logMsg("Deleted episode (removed from remote): $del_name (ID: $del_id)");
                    }
                }
            }

            $episodes_result->free();
        }

        // ========================================
        // SYNC STREAMING SERVERS from remote panel
        // Remote: servers → Local: streaming_servers
        // ========================================
        $servers_inserted = 0;
        $servers_updated = 0;

        $servers_query = "SELECT id, name, server_ip, ssh_port, ssh_password, http_port, rtmp_port, disabled
                         FROM servers
                         ORDER BY id";

        $servers_result = $remote_conn->query($servers_query);

        if($servers_result && $servers_result->num_rows > 0) {
            while($server = $servers_result->fetch_assoc()) {
                $server_id = intval($server['id']);
                $server_name = $local_conn->real_escape_string($server['name']);
                $server_ip = $local_conn->real_escape_string($server['server_ip']);
                $ssh_port = intval($server['ssh_port']);
                $ssh_pass = $local_conn->real_escape_string($server['ssh_password']);
                $http_port = intval($server['http_port']);
                $rtmp_port = intval($server['rtmp_port']);
                $server_status = ($server['disabled'] == 0) ? 1 : 0; // Invert: disabled=0 → status=1 (active)

                // Check if server exists locally
                $check_server = $local_conn->query("SELECT id FROM streaming_servers WHERE id = $server_id");

                if($check_server && $check_server->num_rows > 0) {
                    // Update existing server - DO NOT update ports to prevent breaking client configurations
                    $local_conn->query("UPDATE streaming_servers SET
                                      server_name = '$server_name',
                                      server_ip = '$server_ip',
                                      ssh_port = $ssh_port,
                                      ssh_password = '$ssh_pass',
                                      status = $server_status
                                      WHERE id = $server_id");
                    $servers_updated++;
                } else {
                    // Insert new server
                    $local_conn->query("INSERT INTO streaming_servers (id, server_name, server_ip, vpn_ip, ssh_port, ssh_password, http_broadcast_port, rtmp_port, status, network_interface, can_delete)
                                      VALUES ($server_id, '$server_name', '$server_ip', '$server_ip', $ssh_port, '$ssh_pass', $http_port, $rtmp_port, $server_status, 'eth0', 1)");
                    $servers_inserted++;
                    logMsg("Inserted streaming server: $server_name (ID: $server_id, IP: $server_ip)");
                }
            }
            $servers_result->free();
        }

        // ========================================
        // SYNC BOUQUETS from remote panel
        // ========================================
        $bouquets_inserted = 0;
        $bouquets_updated = 0;

        $bouquets_query = "SELECT id, name, ordering FROM bouquets ORDER BY id";
        $bouquets_result = $remote_conn->query($bouquets_query);

        if($bouquets_result && $bouquets_result->num_rows > 0) {
            while($bouquet = $bouquets_result->fetch_assoc()) {
                $bouquet_id = intval($bouquet['id']);
                $bouquet_name = $local_conn->real_escape_string($bouquet['name']);
                $ordering = intval($bouquet['ordering']);

                // Check if bouquet exists locally
                $check_bouquet = $local_conn->query("SELECT id FROM bouquets WHERE id = $bouquet_id");

                if($check_bouquet && $check_bouquet->num_rows > 0) {
                    // Update existing bouquet
                    $local_conn->query("UPDATE bouquets SET
                                      bouquet_name = '$bouquet_name',
                                      bouquet_order = $ordering,
                                      view_order = $ordering
                                      WHERE id = $bouquet_id");
                    $bouquets_updated++;
                } else {
                    // Insert new bouquet
                    $local_conn->query("INSERT INTO bouquets (id, bouquet_name, bouquet_channels, bouquet_series, bouquet_order, view_order, bouquet_status, isLocked, bouquet_for)
                                      VALUES ($bouquet_id, '$bouquet_name', '', '', $ordering, $ordering, 1, 0, 0)");
                    $bouquets_inserted++;
                    logMsg("Inserted bouquet: $bouquet_name (ID: $bouquet_id)");
                }
            }
            $bouquets_result->free();
        }

        // ========================================
        // SYNC USERS (LINES) from remote panel
        // Remote: lines table → Local: users table
        // ========================================
        $users_inserted = 0;
        $users_updated = 0;

        $users_query = 'SELECT id, username, password, expire_date, max_allowed_connections, disabled, created_time, bouquets FROM `lines` WHERE mag = 0 ORDER BY id';

        $users_result = $remote_conn->query($users_query);

        if($users_result && $users_result->num_rows > 0) {
            while($user = $users_result->fetch_assoc()) {
                $user_id = intval($user['id']);
                $username = $local_conn->real_escape_string($user['username']);
                $password = $local_conn->real_escape_string($user['password']);
                $expire_date = intval($user['expire_date']);
                $max_connections = intval($user['max_allowed_connections']);
                $user_status = ($user['disabled'] == 0) ? 1 : 0; // Invert logic
                $created_time = intval($user['created_time']);
                $bouquets_json = $local_conn->real_escape_string($user['bouquets']);

                // Check if user exists locally
                $check_user = $local_conn->query("SELECT id FROM users WHERE id = $user_id");

                if($check_user && $check_user->num_rows > 0) {
                    // Update existing user
                    $local_conn->query("UPDATE users SET
                                      username = '$username',
                                      password = '$password',
                                      exp_date = $expire_date,
                                      max_connections = $max_connections,
                                      enabled = $user_status,
                                      bouquet = '$bouquets_json'
                                      WHERE id = $user_id");
                    $users_updated++;
                } else {
                    // Insert new user
                    $local_conn->query("INSERT INTO users (id, username, password, exp_date, max_connections, enabled, created_at, bouquet, admin_enabled, is_trial, created_by)
                                      VALUES ($user_id, '$username', '$password', $expire_date, $max_connections, $user_status, $created_time, '$bouquets_json', 1, 0, 1)");
                    $users_inserted++;
                    logMsg("Inserted user: $username (ID: $user_id, Expires: " . date('Y-m-d', $expire_date) . ")");
                }
            }
            $users_result->free();
        }

        // ========================================
        // SYNC LIVE STREAMS CATEGORIES from remote panel
        // Remote type=0 (Live TV) categories
        // ========================================
        $live_categories_inserted = 0;
        $live_categories_updated = 0;

        $live_cat_query = "SELECT id, name, parent, ordering FROM categories WHERE type=0 ORDER BY id";
        $live_cat_result = $remote_conn->query($live_cat_query);

        if($live_cat_result && $live_cat_result->num_rows > 0) {
            while($cat = $live_cat_result->fetch_assoc()) {
                $cat_id = intval($cat['id']);
                $cat_name = $local_conn->real_escape_string($cat['name']);
                $cat_order = intval($cat['ordering']);

                // Check if category exists locally
                $check_cat = $local_conn->query("SELECT id FROM categories WHERE id = $cat_id");

                if($check_cat && $check_cat->num_rows > 0) {
                    // Update existing category
                    $local_conn->query("UPDATE categories SET
                                      name = '$cat_name',
                                      sort_order = $cat_order
                                      WHERE id = $cat_id");
                    $live_categories_updated++;
                } else {
                    // Insert new category
                    $local_conn->query("INSERT INTO categories (id, name, sort_order, parent_id)
                                      VALUES ($cat_id, '$cat_name', $cat_order, 0)");
                    $live_categories_inserted++;
                    logMsg("Inserted live category: $cat_name (ID: $cat_id)");
                }
            }
            $live_cat_result->free();
        }

        // ========================================
        // SYNC LIVE STREAMS from remote panel
        // Remote type=0 (Live TV) → Local type=1 (Live)
        // ========================================
        $live_streams_inserted = 0;
        $live_streams_updated = 0;

        $live_query = "SELECT s.id, s.name, s.stream_source, s.disabled, s.direct_streaming_on_demand,
                             cs.cid as category_id
                      FROM streams s
                      LEFT JOIN categories_sids cs ON s.id = cs.sid
                      WHERE s.type=0
                      ORDER BY s.id";

        $live_result = $remote_conn->query($live_query);

        if($live_result && $live_result->num_rows > 0) {
            while($live = $live_result->fetch_assoc()) {
                $stream_id = intval($live['id']);
                $stream_name = $local_conn->real_escape_string($live['name']);
                $category_id = intval($live['category_id']);

                // Field mapping with inverted logic
                $remote_on_demand = intval($live['direct_streaming_on_demand']);
                $remote_disabled = intval($live['disabled']);

                $local_direct_source = ($remote_on_demand == 1) ? 0 : 1; // Invert
                $local_status = ($remote_disabled == 0) ? 1 : 0; // Invert
                $local_on_demand = $remote_on_demand; // Same

                $stream_source_str = $local_conn->real_escape_string($live['stream_source']);

                // Check if stream exists locally
                $check_stream = $local_conn->query("SELECT id FROM streams WHERE id = $stream_id");

                if($check_stream && $check_stream->num_rows > 0) {
                    // Update existing stream
                    $local_conn->query("UPDATE streams SET
                                      stream_display_name = '$stream_name',
                                      stream_source = '$stream_source_str',
                                      direct_source = $local_direct_source,
                                      status = $local_status,
                                      category_id = $category_id
                                      WHERE id = $stream_id");

                    // Update streams_sys
                    $local_conn->query("UPDATE streams_sys SET on_demand = $local_on_demand WHERE stream_id = $stream_id");

                    $live_streams_updated++;
                } else {
                    // Insert new stream
                    $local_conn->query("INSERT INTO streams (id, stream_display_name, stream_source, type, status, category_id, direct_source, added)
                                      VALUES ($stream_id, '$stream_name', '$stream_source_str', 1, $local_status, $category_id, $local_direct_source, UNIX_TIMESTAMP())");

                    // Insert into streams_sys
                    $local_conn->query("INSERT INTO streams_sys (stream_id, on_demand, stream_status, pid)
                                      VALUES ($stream_id, $local_on_demand, 1, NULL)");

                    $live_streams_inserted++;
                    logMsg("Inserted live stream: $stream_name (ID: $stream_id, Category: $category_id, Mode: " . ($local_on_demand ? "OnDemand" : "Live") . ")");
                }
            }
            $live_result->free();
        }

        // Log summary every 10 iterations (every 5 minutes)
        if($iteration % 10 == 0) {
            logMsg("Iteration $iteration - Channels: Updated: $updated, Inserted: $inserted, Deleted: $deleted");
            logMsg("Iteration $iteration - Movies: Updated: $movies_updated, Inserted: $movies_inserted, Deleted: $movies_deleted");
            logMsg("Iteration $iteration - Series: Updated: $series_updated, Inserted: $series_inserted, Deleted: $series_deleted");
            logMsg("Iteration $iteration - Episodes: Updated: $episodes_updated, Inserted: $episodes_inserted, Deleted: $episodes_deleted");
            logMsg("Iteration $iteration - Servers: Updated: $servers_updated, Inserted: $servers_inserted");
            logMsg("Iteration $iteration - Bouquets: Updated: $bouquets_updated, Inserted: $bouquets_inserted");
            logMsg("Iteration $iteration - Users: Updated: $users_updated, Inserted: $users_inserted");
            logMsg("Iteration $iteration - Live Categories: Updated: $live_categories_updated, Inserted: $live_categories_inserted");
            logMsg("Iteration $iteration - Live Streams: Updated: $live_streams_updated, Inserted: $live_streams_inserted");
        }

        $result->free();
        $remote_conn->close();
        $local_conn->close();

    } catch(Exception $e) {
        logMsg("ERROR: " . $e->getMessage());
    }

    // Sleep for 30 seconds
    $elapsed = microtime(true) - $start_time;
    $sleep_time = max(0, 30 - $elapsed);
    usleep($sleep_time * 1000000);
}
?>
