<?php
/**
 * Flix Panel Auto Sync Cron Job
 *
 * Add to crontab:
 * 0 * * * * php /var/www/html/iptv/cron/sync_flix.php >> /var/log/flix_sync.log 2>&1
 */

// Configuration
$panel_url = 'https://flix-panel.xyz:2087';
$api_key = 'eJIdy5sAgD';
$backup_dir = '/var/www/html/iptv/backups';
$sha1_file = $backup_dir . '/last_backup.sha1';
$log_file = $backup_dir . '/sync.log';

// Database connection
$db_host = 'localhost';
$db_user = 'root';
$db_pass = '';
$db_name = 'iptv';

// Ensure backup directory exists
if(!is_dir($backup_dir)) {
    mkdir($backup_dir, 0755, true);
}

function logMsg($msg) {
    global $log_file;
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($log_file, "[$timestamp] $msg\n", FILE_APPEND);
    echo "[$timestamp] $msg\n";
}

// Connect to database
$db = new mysqli($db_host, $db_user, $db_pass, $db_name);
if($db->connect_error) {
    logMsg("Database connection failed: " . $db->connect_error);
    exit(1);
}

// Check if sync is enabled
$result = $db->query("SELECT setting_value FROM panel_settings WHERE setting_key='flix_sync_enabled'");
if($result && $row = $result->fetch_assoc()) {
    if($row['setting_value'] != '1') {
        logMsg("Auto sync is disabled. Exiting.");
        exit(0);
    }
}

logMsg("Starting Flix Panel sync...");

// Step 1: Get backup info
$ch = curl_init();
curl_setopt_array($ch, [
    CURLOPT_URL => $panel_url . '/api/panel/last_backup_info/' . $api_key,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 30,
    CURLOPT_SSL_VERIFYPEER => false
]);
$response = curl_exec($ch);
$error = curl_error($ch);
curl_close($ch);

if($error) {
    logMsg("Failed to get backup info: $error");
    exit(1);
}

$backup_info = json_decode($response, true);
if(!$backup_info || !isset($backup_info['result']) || !$backup_info['result']) {
    logMsg("Invalid backup info response");
    exit(1);
}

$new_sha1 = $backup_info['sha1'];
$filename = $backup_info['filename'];

logMsg("Remote backup: $filename (SHA1: " . substr($new_sha1, 0, 8) . "...)");

// Step 2: Check if backup changed
$last_sha1 = file_exists($sha1_file) ? trim(file_get_contents($sha1_file)) : '';
if($new_sha1 === $last_sha1) {
    logMsg("Backup unchanged. No sync needed.");

    // Update last sync time
    $db->query("REPLACE INTO panel_settings (setting_key, setting_value) VALUES ('flix_sync_last', '" . time() . "')");
    exit(0);
}

logMsg("New backup detected. Downloading...");

// Step 3: Download backup
$ch = curl_init();
curl_setopt_array($ch, [
    CURLOPT_URL => $panel_url . '/api/panel/download_last_backup/' . $api_key,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 600, // 10 minutes for large files
    CURLOPT_SSL_VERIFYPEER => false
]);
$backup_content = curl_exec($ch);
$error = curl_error($ch);
curl_close($ch);

if($error || empty($backup_content)) {
    logMsg("Failed to download backup: $error");
    exit(1);
}

// Step 4: Save backup file
$backup_file = $backup_dir . '/' . $filename;
file_put_contents($backup_file, $backup_content);
file_put_contents($sha1_file, $new_sha1);

logMsg("Backup saved: $filename (" . round(strlen($backup_content) / 1024 / 1024, 2) . " MB)");

// Step 5: Extract backup
$extract_dir = $backup_dir . '/extract_' . time();
mkdir($extract_dir, 0755, true);

$sql_content = '';
try {
    if(substr($filename, -3) === '.gz' && substr($filename, -7) !== '.tar.gz') {
        // Plain .gz file - decompress directly
        $gz_content = gzdecode($backup_content);
        if($gz_content === false) {
            throw new Exception("Failed to decompress .gz file");
        }

        // Check if decompressed content is a tar archive
        if(strpos(substr($gz_content, 257, 5), "ustar") !== false) {
            // It's a tar archive inside gz
            $tar_file = $extract_dir . '/backup.tar';
            file_put_contents($tar_file, $gz_content);
            $phar = new PharData($tar_file);
            $phar->extractTo($extract_dir);
            unlink($tar_file);
        } else {
            // It's just SQL content compressed
            $sql_content = $gz_content;
        }
    } else {
        $phar = new PharData($backup_file);
        $phar->extractTo($extract_dir);
    }
    logMsg("Backup extracted to: $extract_dir");
} catch(Exception $e) {
    // Try alternative: use system commands
    exec("cd " . escapeshellarg($extract_dir) . " && gunzip -c " . escapeshellarg($backup_file) . " > backup.tar 2>&1 && tar -xf backup.tar 2>&1", $output, $retval);
    if($retval !== 0) {
        exec("gunzip -c " . escapeshellarg($backup_file) . " > " . escapeshellarg($extract_dir . '/backup.sql') . " 2>&1", $output2, $retval2);
    }
    logMsg("Extracted using system commands");
}

// Step 6: Find SQL content
if(empty($sql_content)) {
    $sql_files = glob($extract_dir . '/*.sql');
    if(empty($sql_files)) {
        $sql_files = glob($extract_dir . '/*/*.sql');
    }
    if(empty($sql_files)) {
        $all_files = glob($extract_dir . '/*');
        foreach($all_files as $file) {
            if(is_file($file) && filesize($file) > 1000) {
                $content = file_get_contents($file, false, null, 0, 1000);
                if(stripos($content, 'INSERT INTO') !== false) {
                    $sql_files[] = $file;
                    break;
                }
            }
        }
    }
    if(!empty($sql_files)) {
        $sql_content = file_get_contents($sql_files[0]);
        logMsg("SQL file loaded: " . basename($sql_files[0]));
    }
}

if(empty($sql_content)) {
    logMsg("No SQL content found in backup");
    cleanupDir($extract_dir);
    exit(1);
}

// Step 7: Parse users from SQL
$users = parseUsersFromSQL($sql_content);
logMsg("Found " . count($users) . " users in backup");

if(empty($users)) {
    logMsg("No users to sync");
    cleanupDir($extract_dir);
    exit(0);
}

// Step 8: Sync users
$imported = 0;
$updated = 0;
$skipped = 0;

foreach($users as $user) {
    // Skip users with empty username
    if(empty($user['username']) || trim($user['username']) === '') {
        $skipped++;
        continue;
    }

    $username_esc = $db->real_escape_string($user['username']);
    $password_esc = $db->real_escape_string($user['password']);
    $exp_date = intval($user['exp_date']);
    $max_conn = intval($user['max_connections']);
    $enabled = intval($user['enabled']);

    // Check if user exists
    $check = $db->query("SELECT id, password, exp_date, max_connections, enabled FROM users WHERE username='$username_esc'");

    if($check->num_rows == 0) {
        // Insert new user
        $db->query("INSERT INTO users (username, password, exp_date, enabled, admin_enabled, max_connections, created_at, member_id)
            VALUES ('$username_esc', '$password_esc', '$exp_date', '$enabled', 1, $max_conn, " . time() . ", 1)");
        $imported++;
    } else {
        // Check if update needed
        $existing = $check->fetch_assoc();
        $updates = [];

        if($existing['password'] != $user['password']) {
            $updates[] = "password='$password_esc'";
        }
        if($existing['exp_date'] != $exp_date) {
            $updates[] = "exp_date='$exp_date'";
        }
        if($existing['max_connections'] != $max_conn) {
            $updates[] = "max_connections='$max_conn'";
        }
        if($existing['enabled'] != $enabled) {
            $updates[] = "enabled='$enabled'";
        }

        if(!empty($updates)) {
            $db->query("UPDATE users SET " . implode(', ', $updates) . " WHERE username='$username_esc'");
            $updated++;
        } else {
            $skipped++;
        }
    }
}

// Cleanup
cleanupDir($extract_dir);

// Update last sync time
$db->query("REPLACE INTO panel_settings (setting_key, setting_value) VALUES ('flix_sync_last', '" . time() . "')");

logMsg("Sync complete! Imported: $imported, Updated: $updated, Skipped: $skipped");

$db->close();

// Helper functions
function parseUsersFromSQL($sql_content) {
    $users = [];

    // Look for INSERT statements for lines table
    if(preg_match_all('/INSERT INTO [`\']?lines[`\']?\s+.*?VALUES\s*(.+?);/is', $sql_content, $matches)) {
        foreach($matches[1] as $values_block) {
            preg_match_all('/\(([^)]+)\)/s', $values_block, $rows);
            foreach($rows[1] as $row) {
                $fields = str_getcsv($row, ',', "'");

                if(count($fields) >= 5) {
                    $users[] = [
                        'username' => trim($fields[1] ?? '', "'\" "),
                        'password' => trim($fields[2] ?? '', "'\" "),
                        'exp_date' => intval(trim($fields[3] ?? '0', "'\" ")),
                        'max_connections' => intval(trim($fields[4] ?? '1', "'\" ")),
                        'enabled' => 1 - intval(trim($fields[5] ?? '0', "'\" ")) // disabled field is inverted
                    ];
                }
            }
        }
    }

    return $users;
}

function cleanupDir($dir) {
    if(!is_dir($dir)) return;
    $files = array_diff(scandir($dir), ['.', '..']);
    foreach($files as $file) {
        $path = $dir . '/' . $file;
        is_dir($path) ? cleanupDir($path) : unlink($path);
    }
    rmdir($dir);
}
