<?php 
class IMDB
{
    public static $sNotFound = 'n/A';
    public $iId = null;
    public $isReady = false;
    public $sSeparator = ' / ';
    public $sUrl = null;
    public $bArrayOutput = false;
    private $iCache = 1440;
    private $sRoot = null;
    private $sSource = null;
    private $main_storyline = '';
    private $main_plotoutline = '';
    private $sSearchFor = 'all';
    const IMDB_DEBUG = false;
    const IMDB_LANG = 'en-US,en;q=0.9';
    const IMDB_TIMEOUT = 15;
    const IMDB_AKA = '~<td[^>]*>\s*Also\s*Known\s*As\s*</td>\s*<td>(.+)</td>~Uis';
    const IMDB_ASPECT_RATIO = '~<td[^>]*>Aspect\s*Ratio</td>\s*<td>(.+)</td>~Uis';
    const IMDB_AWARDS = '~<div\s*class="titlereference-overview-section">\s*Awards:(.+)</div>~Uis';
    const IMDB_BUDGET = '~<td[^>]*>Budget<\/td>\s*<td>\s*(.*)(?:\(estimated\))\s*<\/td>~Ui';
    const IMDB_CAST = '~<td[^>]*itemprop="actor"[^>]*>\s*<a\s*href="/name/([^/]*)/\?[^"]*"[^>]*>\s*<span.+>(.+)</span~Ui';
    const IMDB_CAST_IMAGE = '~(loadlate="(.*)"[^>]*><\/a>\s+<\/td>\s+)?<td[^>]*itemprop="actor"[^>]*>\s*<a\s*href="\/name\/([^/]*)\/\?[^"]*"[^>]*>\s*<span.+>(.+)<\/span+~Uis';
    const IMDB_CERTIFICATION = '~<td[^>]*>\s*Certification\s*</td>\s*<td>(.+)</td>~Ui';
    const IMDB_CHAR = '~<td class="character">(?:\s+)<div>(.*)(?:\s+)(?: /| \(.*\)|<\/div>)~Ui';
    const IMDB_COLOR = '~<a href="\/search\/title\?colors=(?:.*)">(.*)<\/a>~Ui';
    const IMDB_COMPANY = '~href="[^"]*update=[t0-9]+:production_companies[^"]*">Edit</a>\s*</header>\s*<ul\s*class="simpleList">.+<a href="\/company\/(.*)\/">(.*)</a>~Ui';
    const IMDB_COUNTRY = '~<a href="/country/(\w+)">(.*)</a>~Ui';
    const IMDB_CREATOR = '~<div[^>]*>\s*(?:Creator|Creators)\s*:\s*<ul[^>]*>(.+)</ul>~Uxsi';
    const IMDB_DIRECTOR = '~<div[^>]*>\s*(?:Director|Directors)\s*:\s*<ul[^>]*>(.+)</ul>~Uxsi';
    const IMDB_GENRE = '~href="/genre/([a-zA-Z_-]*)/?">([a-zA-Z_ -]*)</a>~Ui';
    const IMDB_GROSS = '~pl-zebra-list__label">Cumulative Worldwide Gross<\/td>\s+<td>\s+(.*)\s+<~Uxsi';
    const IMDB_ID = '~((?:tt\d{6,})|(?:itle\?\d{6,}))~';
    const IMDB_LANGUAGE = '~<a href="\/language\/(\w+)">(.*)<\/a>~Ui';
    const IMDB_LOCATION = '~href="\/search\/title\?locations=(.*)">(.*)<\/a>~Ui';
    const IMDB_MPAA = '~<li class="ipl-inline-list__item">(?:\s+)(TV-Y|TV-Y7|TV-G|TV-PG|TV-14|TV-MA|G|PG|PG-13|R|NC-17|NR|UR)(?:\s+)<\/li>~Ui';
    const IMDB_NAME = '~href="/name/(.+)/?(?:\?[^"]*)?"[^>]*>(.+)</a>~Ui';
    const IMDB_DESCRIPTION = '~<section class="titlereference-section-overview">\s+<div>(.*)</div>\s+<hr>~Ui';
    const IMDB_NOT_FOUND = '~<h1 class="findHeader">No results found for ~Ui';
    const IMDB_PLOT = '~<td[^>]*>\s*Plot\s*Summary\s*</td>\s*<td>\s*<p>(.+)</p>~Ui';
    const IMDB_PLOT_KEYWORDS = '~<td[^>]*>Plot\s*Keywords</td>\s*<td>(.+)(?:<a\s*href="/title/[^>]*>[^<]*</a>\s*</li>\s*</ul>\s*)?</td>~Ui';
    const IMDB_POSTER = '~<link\s*rel=\'image_src\'\s*href="(.*)">~Ui';
    const IMDB_RATING = '~class="ipl-rating-star__rating">(.*)<~Ui';
    const IMDB_RELEASE_DATE = '~href="/title/[t0-9]*/releaseinfo">(.*)<~Ui';
    const IMDB_RUNTIME = '~<td[^>]*>\s*Runtime\s*</td>\s*<td>(.+)</td>~Ui';
    const IMDB_SEARCH = '~<td class="result_text"> <a href="\/title\/(tt\d{6,})\/(?:.*)"(?:\s*)>(?:.*)<\/a>~Ui';
    const IMDB_SEASONS = '~episodes\?season=(?:\d+)">(\d+)<~Ui';
    const IMDB_SOUND_MIX = '~<td[^>]*>\s*Sound\s*Mix\s*</td>\s*<td>(.+)</td>~Ui';
    const IMDB_TAGLINE = '~<td[^>]*>\s*Taglines\s*</td>\s*<td>(.+)</td>~Ui';
    const IMDB_TITLE = '~itemprop="name">(.*)(<\/h3>|<span)~Ui';
    const IMDB_TITLE_ORIG = '~</h3>(?:\s+)(.*)(?:\s+)<span class=\"titlereference-original-title-label~Ui';
    const IMDB_TRAILER = '~href="videoplayer/(vi[0-9]*)"~Ui';
    const IMDB_URL = '~https?://(?:.*\.|.*)imdb.com/(?:t|T)itle(?:\?|/)(..\d+)~i';
    const IMDB_USER_REVIEW = '~href="/title/[t0-9]*/reviews"[^>]*>([^<]*)\s*User~Ui';
    const IMDB_VOTES = '~"ipl-rating-star__total-votes">\s*\((.*)\)\s*<~Ui';
    const IMDB_WRITER = '~<div[^>]*>\s*(?:Writer|Writers)\s*:\s*<ul[^>]*>(.+)</ul>~Ui';
    const IMDB_YEAR = '~og:title\' content="(?:.*)\((?:.*)(\d{4})(?:.*)\)~Ui';
    public function __construct($sSearch, $iCache = null, $sSearchFor = 'all')
    {
        $this->sRoot = dirname(__FILE__);
        if( !function_exists('curl_init') ) 
        {
            throw new Exception('You need to enable the PHP cURL extension.');
        }
        if( in_array($sSearchFor, [
            'movie', 
            'tv', 
            'episode', 
            'game', 
            'all'
        ]) ) 
        {
            $this->sSearchFor = $sSearchFor;
        }
        if( false ) 
        {
            ini_set('display_errors', 1);
            ini_set('display_startup_errors', 1);
            error_reporting(-1);
            echo '<pre><b>Running:</b> fetchUrl("' . $sSearch . '")</pre>';
        }
        if( $iCache !== null && (int)$iCache > 0 ) 
        {
            $this->iCache = (int)$iCache;
        }
        $this->fetchUrl($sSearch);
    }
    private function fetchUrl($sSearch)
    {
        $sSearch = trim($sSearch);
        $sId = IMDBHelper::matchRegex($sSearch, '~((?:tt\d{6,})|(?:itle\?\d{6,}))~', 1);
        if( $sId !== false ) 
        {
            $this->iId = preg_replace('~[\D]~', '', $sId);
            $this->sUrl = 'https://www.imdb.com/title/tt' . $this->iId . '/reference';
            $bSearch = false;
        }
        else
        {
            switch( strtolower($this->sSearchFor) ) 
            {
                case 'movie':
                    $sParameters = '&s=tt&ttype=ft';
                    break;
                case 'tv':
                    $sParameters = '&s=tt&ttype=tv';
                    break;
                case 'episode':
                    $sParameters = '&s=tt&ttype=ep';
                    break;
                case 'game':
                    $sParameters = '&s=tt&ttype=vg';
                    break;
                default:
                    $sParameters = '&s=tt';
            }
            $this->sUrl = 'https://www.imdb.com/find?q=' . str_replace(' ', '+', $sSearch) . $sParameters;
            $bSearch = true;
            $sRedirectFile = $this->sRoot . '/cache/' . sha1($this->sUrl) . '.redir';
            if( is_readable($sRedirectFile) ) 
            {
                if( false ) 
                {
                    echo '<pre><b>Using redirect:</b> ' . basename($sRedirectFile) . '</pre>';
                }
                $sRedirect = file_get_contents($sRedirectFile);
                $this->sUrl = trim($sRedirect);
                $this->iId = preg_replace('~[\D]~', '', IMDBHelper::matchRegex($sRedirect, '~((?:tt\d{6,})|(?:itle\?\d{6,}))~', 1));
                $bSearch = false;
            }
        }
        $sCacheFile = $this->sRoot . '/cache/' . sha1($this->iId) . '.cache';
        if( is_readable($sCacheFile) ) 
        {
            $iDiff = round(abs(time() - filemtime($sCacheFile)) / 60);
            if( $iDiff < $this->iCache ) 
            {
                if( false ) 
                {
                    echo '<pre><b>Using cache:</b> ' . basename($sCacheFile) . '</pre>';
                }
                $this->sSource = file_get_contents($sCacheFile);
                $this->isReady = true;
                return true;
            }
        }
        if( false ) 
        {
            echo '<pre><b>Running cURL:</b> ' . $this->sUrl . '</pre>';
        }
        $aCurlInfo = IMDBHelper::runCurl($this->sUrl);
        $sSource = $aCurlInfo['contents'];
        if( false === $sSource ) 
        {
            if( false ) 
            {
                echo '<pre><b>cURL error:</b> ' . var_dump($aCurlInfo) . '</pre>';
            }
            return false;
        }
        $sMatch = IMDBHelper::matchRegex($sSource, '~<td class="result_text"> <a href="\/title\/(tt\d{6,})\/(?:.*)"(?:\s*)>(?:.*)<\/a>~Ui', 1);
        if( $sMatch !== false ) 
        {
            $sUrl = 'https://www.imdb.com/title/' . $sMatch . '/reference';
            if( false ) 
            {
                echo '<pre><b>New redirect saved:</b> ' . basename($sRedirectFile) . ' => ' . $sUrl . '</pre>';
            }
            file_put_contents($sRedirectFile, $sUrl);
            $this->sSource = null;
            self::fetchUrl($sUrl);
            return true;
        }
        $sMatch = IMDBHelper::matchRegex($sSource, '~<h1 class="findHeader">No results found for ~Ui', 0);
        if( $sMatch !== false ) 
        {
            if( false ) 
            {
                echo '<pre><b>Movie not found:</b> ' . $sSearch . '</pre>';
            }
            return false;
        }
        $this->sSource = str_replace([
            "\n", 
            "\r\n", 
            "\r"
        ], '', $sSource);
        $this->isReady = true;
        if( false === $bSearch ) 
        {
            if( false ) 
            {
                echo '<pre><b>Cache created:</b> ' . basename($sCacheFile) . '</pre>';
            }
            file_put_contents($sCacheFile, $this->sSource);
        }
        return true;
    }
    public function getAll()
    {
        $aData = [];
        foreach( get_class_methods('IMDB') as $method ) 
        {
            if( substr($method, 0, 3) === 'get' && $method !== 'getAll' && $method !== 'getCastImages' ) 
            {
                $aData[$method] = [
                    'name' => ltrim($method, 'get'), 
                    'value' => $this->$method()
                ];
            }
        }
        array_multisort($aData);
        return $aData;
    }
    public function getAka()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<td[^>]*>\s*Also\s*Known\s*As\s*</td>\s*<td>(.+)</td>~Uis', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getAkas()
    {
        if( true === $this->isReady ) 
        {
            $sCacheFile = $this->sRoot . '/cache/' . sha1($this->iId) . '_akas.cache';
            $bUseCache = false;
            if( is_readable($sCacheFile) ) 
            {
                $iDiff = round(abs(time() - filemtime($sCacheFile)) / 60);
                if( $iDiff < $this->iCache || false ) 
                {
                    $bUseCache = true;
                }
            }
            if( $bUseCache ) 
            {
                $aRawReturn = file_get_contents($sCacheFile);
                $aReturn = unserialize($aRawReturn);
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn);
            }
            else
            {
                $fullAkas = sprintf('https://www.imdb.com/title/tt%s/releaseinfo', $this->iId);
                $aCurlInfo = IMDBHelper::runCurl($fullAkas);
                $sSource = $aCurlInfo['contents'];
                if( false === $sSource ) 
                {
                    if( false ) 
                    {
                        echo '<pre><b>cURL error:</b> ' . var_dump($aCurlInfo) . '</pre>';
                    }
                    return false;
                }
                $aReturned = IMDBHelper::matchRegex($sSource, '~<td>(.*?)<\/td>\s+<td>(.*?)<\/td>~');
                if( $aReturned ) 
                {
                    $aReturn = [];
                    foreach( $aReturned[1] as $i => $strName ) 
                    {
                        if( strpos($strName, '(') === false ) 
                        {
                            $aReturn[] = [
                                'title' => IMDBHelper::cleanString($aReturned[2][$i]), 
                                'country' => IMDBHelper::cleanString($strName)
                            ];
                        }
                    }
                    file_put_contents($sCacheFile, serialize($aReturn));
                    return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn);
                }
            }
        }
        return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound);
    }
    public function getAspectRatio()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<td[^>]*>Aspect\s*Ratio</td>\s*<td>(.+)</td>~Uis', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getAwards()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<div\s*class="titlereference-overview-section">\s*Awards:(.+)</div>~Uis', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getCastAsUrl($iLimit = 0, $bMore = true, $sTarget = '')
    {
        if( true === $this->isReady ) 
        {
            $aMatch = IMDBHelper::matchRegex($this->sSource, '~<td[^>]*itemprop="actor"[^>]*>\s*<a\s*href="/name/([^/]*)/\?[^"]*"[^>]*>\s*<span.+>(.+)</span~Ui');
            $aReturn = [];
            if( count($aMatch[2]) ) 
            {
                foreach( $aMatch[2] as $i => $sName ) 
                {
                    if( $iLimit !== 0 && $iLimit <= $i ) 
                    {
                        break;
                    }
                    else
                    {
                        $aReturn[] = '<a href="https://www.imdb.com/name/' . IMDBHelper::cleanString($aMatch[1][$i]) . '/"' . (($sTarget ? ' target="' . $sTarget . '"' : '')) . '>' . IMDBHelper::cleanString($sName) . '</a>';
                    }
                }
                $bHaveMore = $bMore && $iLimit < count($aMatch[2]);
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn, $bHaveMore);
            }
        }
        return self::$sNotFound;
    }
    public function getCast($iLimit = 0, $bMore = true)
    {
        if( true === $this->isReady ) 
        {
            $aMatch = IMDBHelper::matchRegex($this->sSource, '~<td[^>]*itemprop="actor"[^>]*>\s*<a\s*href="/name/([^/]*)/\?[^"]*"[^>]*>\s*<span.+>(.+)</span~Ui');
            $aReturn = [];
            if( count($aMatch[2]) ) 
            {
                foreach( $aMatch[2] as $i => $sName ) 
                {
                    if( $iLimit !== 0 && $iLimit <= $i ) 
                    {
                        break;
                    }
                    else
                    {
                        $aReturn[] = IMDBHelper::cleanString($sName);
                    }
                }
                $bMore = ($iLimit !== 0 && $bMore && $iLimit < count($aMatch[2]) ? 'â€¦' : '');
                $bHaveMore = $bMore && $iLimit < count($aMatch[2]);
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn, $bHaveMore);
            }
        }
        return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound);
    }
    public function getCastImages($iLimit = 0, $bMore = true, $sSize = 'small', $bDownload = false)
    {
        if( true === $this->isReady ) 
        {
            $aMatch = IMDBHelper::matchRegex($this->sSource, '~(loadlate="(.*)"[^>]*><\/a>\s+<\/td>\s+)?<td[^>]*itemprop="actor"[^>]*>\s*<a\s*href="\/name\/([^/]*)\/\?[^"]*"[^>]*>\s*<span.+>(.+)<\/span+~Uis');
            $aReturn = [];
            if( count($aMatch[4]) ) 
            {
                foreach( $aMatch[4] as $i => $sName ) 
                {
                    if( $iLimit !== 0 && $iLimit <= $i ) 
                    {
                        break;
                    }
                    else
                    {
                        $sMatch = $aMatch[2][$i];
                        if( 'big' === strtolower($sSize) && strstr($aMatch[2][$i], '@._') !== false ) 
                        {
                            $sMatch = substr($aMatch[2][$i], 0, strpos($aMatch[2][$i], '@._')) . '@.jpg';
                        }
                        else if( 'mid' === strtolower($sSize) && strstr($aMatch[2][$i], '@._') !== false ) 
                        {
                            $sMatch = substr($aMatch[2][$i], 0, strpos($aMatch[2][$i], '@._')) . '@._V1_UX214_AL_.jpg';
                        }
                        if( false === $bDownload ) 
                        {
                            $sMatch = IMDBHelper::cleanString($sMatch);
                        }
                        else
                        {
                            $sLocal = IMDBHelper::saveImageCast($sMatch, $aMatch[3][$i]);
                            if( file_exists(dirname(__FILE__) . '/' . $sLocal) ) 
                            {
                                $sMatch = $sLocal;
                            }
                            else
                            {
                                if( 'big' === strtolower($sSize) && strstr($aMatch[2][$i], '@._') !== false ) 
                                {
                                    $sMatch = substr($aMatch[2][$i], 0, strpos($aMatch[2][$i], '@._')) . '@._V1_UX214_AL_.jpg';
                                }
                                else
                                {
                                    $sMatch = substr($aMatch[2][$i], 0, strpos($aMatch[2][$i], '@._')) . '@.jpg';
                                }
                                $sLocal = IMDBHelper::saveImageCast($sMatch, $aMatch[3][$i]);
                                if( file_exists(dirname(__FILE__) . '/' . $sLocal) ) 
                                {
                                    $sMatch = $sLocal;
                                }
                                else
                                {
                                    $sMatch = IMDBHelper::cleanString($aMatch[2][$i]);
                                }
                            }
                        }
                        $aReturn[IMDBHelper::cleanString($aMatch[4][$i])] = $sMatch;
                    }
                }
                $bMore = ($iLimit !== 0 && $bMore && $iLimit < count($aMatch[4]) ? 'â€¦' : '');
                $bHaveMore = $bMore && $iLimit < count($aMatch[4]);
                $aReturn = array_replace($aReturn, array_fill_keys(array_keys($aReturn, self::$sNotFound), 'cast/not-found.jpg'));
                return $aReturn;
            }
        }
        return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound);
    }
    public function getCastAndCharacterAsUrl($iLimit = 0, $bMore = true, $sTarget = '')
    {
        if( true === $this->isReady ) 
        {
            $aMatch = IMDBHelper::matchRegex($this->sSource, '~<td[^>]*itemprop="actor"[^>]*>\s*<a\s*href="/name/([^/]*)/\?[^"]*"[^>]*>\s*<span.+>(.+)</span~Ui');
            $aMatchChar = IMDBHelper::matchRegex($this->sSource, '~<td class="character">(?:\s+)<div>(.*)(?:\s+)(?: /| \(.*\)|<\/div>)~Ui');
            $aReturn = [];
            if( count($aMatch[2]) ) 
            {
                foreach( $aMatch[2] as $i => $sName ) 
                {
                    if( $iLimit !== 0 && $iLimit <= $i ) 
                    {
                        break;
                    }
                    else
                    {
                        $aReturn[] = '<a href="https://www.imdb.com/name/' . IMDBHelper::cleanString($aMatch[1][$i]) . '/"' . (($sTarget ? ' target="' . $sTarget . '"' : '')) . '>' . IMDBHelper::cleanString($sName) . '</a> as ' . IMDBHelper::cleanString($aMatchChar[1][$i]);
                    }
                }
                $bHaveMore = $bMore && $iLimit < count($aMatch[2]);
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn, $bHaveMore);
            }
        }
        return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound);
    }
    public function getCastAndCharacter($iLimit = 0, $bMore = true)
    {
        if( true === $this->isReady ) 
        {
            $aMatch = IMDBHelper::matchRegex($this->sSource, '~<td[^>]*itemprop="actor"[^>]*>\s*<a\s*href="/name/([^/]*)/\?[^"]*"[^>]*>\s*<span.+>(.+)</span~Ui');
            $aMatchChar = IMDBHelper::matchRegex($this->sSource, '~<td class="character">(?:\s+)<div>(.*)(?:\s+)(?: /| \(.*\)|<\/div>)~Ui');
            $aReturn = [];
            if( count($aMatch[2]) ) 
            {
                foreach( $aMatch[2] as $i => $sName ) 
                {
                    if( $iLimit !== 0 && $iLimit <= $i ) 
                    {
                        break;
                    }
                    else
                    {
                        $aReturn[] = IMDBHelper::cleanString($sName) . ' as ' . IMDBHelper::cleanString($aMatchChar[1][$i]);
                    }
                }
                $bHaveMore = $bMore && $iLimit < count($aMatch[2]);
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn, $bHaveMore);
            }
        }
        return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound);
    }
    public function getCertification()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<td[^>]*>\s*Certification\s*</td>\s*<td>(.+)</td>~Ui', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getColor()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<a href="\/search\/title\?colors=(?:.*)">(.*)<\/a>~Ui', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getCompany()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = $this->getCompanyAsUrl();
            if( self::$sNotFound !== $sMatch ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getCompanyAsUrl($sTarget = '')
    {
        if( true === $this->isReady ) 
        {
            $aMatch = IMDBHelper::matchRegex($this->sSource, '~href="[^"]*update=[t0-9]+:production_companies[^"]*">Edit</a>\s*</header>\s*<ul\s*class="simpleList">.+<a href="\/company\/(.*)\/">(.*)</a>~Ui');
            if( isset($aMatch[2][0]) ) 
            {
                return '<a href="https://www.imdb.com/company/' . IMDBHelper::cleanString($aMatch[1][0]) . '/"' . (($sTarget ? ' target="' . $sTarget . '"' : '')) . '>' . IMDBHelper::cleanString($aMatch[2][0]) . '</a>';
            }
        }
        return self::$sNotFound;
    }
    public function getCountry()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = $this->getCountryAsUrl();
            if( self::$sNotFound !== $sMatch ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getCountryAsUrl($sTarget = '')
    {
        if( true === $this->isReady ) 
        {
            $aMatch = IMDBHelper::matchRegex($this->sSource, '~<a href="/country/(\w+)">(.*)</a>~Ui');
            $aReturn = [];
            if( count($aMatch[2]) ) 
            {
                foreach( $aMatch[2] as $i => $sName ) 
                {
                    $aReturn[] = '<a href="https://www.imdb.com/country/' . trim($aMatch[1][$i]) . '/"' . (($sTarget ? ' target="' . $sTarget . '"' : '')) . '>' . IMDBHelper::cleanString($sName) . '</a>';
                }
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn);
            }
        }
        return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound);
    }
    public function getCreator()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = $this->getCreatorAsUrl();
            if( self::$sNotFound !== $sMatch ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getCreatorAsUrl($sTarget = '')
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<div[^>]*>\s*(?:Creator|Creators)\s*:\s*<ul[^>]*>(.+)</ul>~Uxsi', 1);
            $aMatch = IMDBHelper::matchRegex($sMatch, '~href="/name/(.+)/?(?:\?[^"]*)?"[^>]*>(.+)</a>~Ui');
            $aReturn = [];
            if( count($aMatch[2]) ) 
            {
                foreach( $aMatch[2] as $i => $sName ) 
                {
                    $aReturn[] = '<a href="https://www.imdb.com/name/' . IMDBHelper::cleanString($aMatch[1][$i]) . '/"' . (($sTarget ? ' target="' . $sTarget . '"' : '')) . '>' . IMDBHelper::cleanString($sName) . '</a>';
                }
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn);
            }
        }
        return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound);
    }
    public function getDescription()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<section class="titlereference-section-overview">\s+<div>(.*)</div>\s+<hr>~Ui', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getDirector()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = $this->getDirectorAsUrl();
            if( self::$sNotFound !== $sMatch ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getDirectorAsUrl($sTarget = '')
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<div[^>]*>\s*(?:Director|Directors)\s*:\s*<ul[^>]*>(.+)</ul>~Uxsi', 1);
            $aMatch = IMDBHelper::matchRegex($sMatch, '~href="/name/(.+)/?(?:\?[^"]*)?"[^>]*>(.+)</a>~Ui');
            $aReturn = [];
            if( count($aMatch[2]) ) 
            {
                foreach( $aMatch[2] as $i => $sName ) 
                {
                    $aReturn[] = '<a href="https://www.imdb.com/name/' . IMDBHelper::cleanString($aMatch[1][$i]) . '/"' . (($sTarget ? ' target="' . $sTarget . '"' : '')) . '>' . IMDBHelper::cleanString($sName) . '</a>';
                }
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn);
            }
        }
        return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound);
    }
    public function getGenre()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = $this->getGenreAsUrl();
            if( self::$sNotFound !== $sMatch ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getGenreAsUrl($sTarget = '')
    {
        if( true === $this->isReady ) 
        {
            $aMatch = IMDBHelper::matchRegex($this->sSource, '~href="/genre/([a-zA-Z_-]*)/?">([a-zA-Z_ -]*)</a>~Ui');
            $aReturn = [];
            if( count($aMatch[2]) ) 
            {
                foreach( array_unique($aMatch[2]) as $i => $sName ) 
                {
                    $aReturn[] = '<a href="https://www.imdb.com/search/title?genres=' . IMDBHelper::cleanString($aMatch[1][$i]) . '"' . (($sTarget ? ' target="' . $sTarget . '"' : '')) . '>' . IMDBHelper::cleanString($sName) . '</a>';
                }
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn);
            }
        }
        return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound);
    }
    public function getGross()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~pl-zebra-list__label">Cumulative Worldwide Gross<\/td>\s+<td>\s+(.*)\s+<~Uxsi', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getLanguage()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = $this->getLanguageAsUrl();
            if( self::$sNotFound !== $sMatch ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getLanguageAsUrl($sTarget = '')
    {
        if( true === $this->isReady ) 
        {
            $aMatch = IMDBHelper::matchRegex($this->sSource, '~<a href="\/language\/(\w+)">(.*)<\/a>~Ui');
            $aReturn = [];
            if( count($aMatch[2]) ) 
            {
                foreach( $aMatch[2] as $i => $sName ) 
                {
                    $aReturn[] = '<a href="https://www.imdb.com/language/' . IMDBHelper::cleanString($aMatch[1][$i]) . '"' . (($sTarget ? ' target="' . $sTarget . '"' : '')) . '>' . IMDBHelper::cleanString($sName) . '</a>';
                }
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn);
            }
        }
        return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound);
    }
    public function getLocation()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = $this->getLocationAsUrl();
            if( self::$sNotFound !== $sMatch ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getLocationAsUrl($sTarget = '')
    {
        if( true === $this->isReady ) 
        {
            $aMatch = IMDBHelper::matchRegex($this->sSource, '~href="\/search\/title\?locations=(.*)">(.*)<\/a>~Ui');
            $aReturn = [];
            if( count($aMatch[2]) ) 
            {
                foreach( $aMatch[2] as $i => $sName ) 
                {
                    $aReturn[] = '<a href="https://www.imdb.com/search/title?locations=' . IMDBHelper::cleanString($aMatch[1][$i]) . '"' . (($sTarget ? ' target="' . $sTarget . '"' : '')) . '>' . IMDBHelper::cleanString($sName) . '</a>';
                }
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn);
            }
        }
        return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound);
    }
    public function getMpaa()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<li class="ipl-inline-list__item">(?:\s+)(TV-Y|TV-Y7|TV-G|TV-PG|TV-14|TV-MA|G|PG|PG-13|R|NC-17|NR|UR)(?:\s+)<\/li>~Ui', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getPlotKeywords()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<td[^>]*>Plot\s*Keywords</td>\s*<td>(.+)(?:<a\s*href="/title/[^>]*>[^<]*</a>\s*</li>\s*</ul>\s*)?</td>~Ui', 1);
            if( $sMatch !== false ) 
            {
                $aReturn = explode('|', IMDBHelper::cleanString($sMatch));
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn);
            }
        }
        return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound);
    }
    public function getPlot($iLimit = 0)
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<td[^>]*>\s*Plot\s*Summary\s*</td>\s*<td>\s*<p>(.+)</p>~Ui', 1);
            if( $sMatch !== false ) 
            {
                if( $iLimit !== 0 ) 
                {
                    return IMDBHelper::getShortText(IMDBHelper::cleanString($sMatch), $iLimit);
                }
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function XXXXXXXXEdit_maaking_getPlot($iLimit = 0)
    {
        if( true === $this->isReady ) 
        {
            $fallback = '';
            if( $this->main_plotoutline == '' ) 
            {
                $page = $this->sSource;
                if( preg_match('!class="summary_text">\s*(.*?)\s*</div>!ims', $page, $match) ) 
                {
                    $this->main_plotoutline = trim($match[1]);
                }
                else if( $fallback ) 
                {
                    $this->main_plotoutline = $this->storyline();
                }
            }
            return IMDBHelper::cleanString('xxxxxxxxx');
            $this->main_plotoutline = preg_replace('!\s*<a href="/title/tt\d{7,8}/(plotsummary|synopsis)[^>]*>See full (summary|synopsis).*$!i', '', $this->main_plotoutline);
            $this->main_plotoutline = preg_replace('#<a href="[^"]+"\s+>Add a Plot</a>&nbsp;&raquo;#', '', $this->main_plotoutline);
            return $this->main_plotoutline;
        }
        return self::$sNotFound;
    }
    public function storyline()
    {
        if( $this->main_storyline == '' ) 
        {
            $page = $this->sSource;
            if( @preg_match('~Storyline</h2>.*?<div.*?<p>.*?<span>(.*?)</span>.*?</p>~ims', $page, $match) ) 
            {
                $this->main_storyline = trim($match[1]);
            }
        }
        return $this->main_storyline;
    }
    public function getPoster($sSize = 'small', $bDownload = false)
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<link\s*rel=\'image_src\'\s*href="(.*)">~Ui', 1);
            if( $sMatch !== false ) 
            {
                if( 'big' === strtolower($sSize) && strstr($sMatch, '@._') !== false ) 
                {
                    $sMatch = substr($sMatch, 0, strpos($sMatch, '@._')) . '@.jpg';
                }
                if( false === $bDownload ) 
                {
                    return IMDBHelper::cleanString($sMatch);
                }
                else
                {
                    $sLocal = IMDBHelper::saveImage($sMatch, $this->iId);
                    if( file_exists(dirname(__FILE__) . '/' . $sLocal) ) 
                    {
                        return $sLocal;
                    }
                    else
                    {
                        return $sMatch;
                    }
                }
            }
        }
        return self::$sNotFound;
    }
    public function getRating()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~class="ipl-rating-star__rating">(.*)<~Ui', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function isReleased()
    {
        $strReturn = $this->getReleaseDate();
        if( $strReturn == self::$sNotFound || $strReturn == 'Not yet released' ) 
        {
            return false;
        }
        return true;
    }
    public function getReleaseDate()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~href="/title/[t0-9]*/releaseinfo">(.*)<~Ui', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getReleaseDates()
    {
        if( true === $this->isReady ) 
        {
            $sCacheFile = $this->sRoot . '/cache/' . sha1($this->iId) . '_akas.cache';
            $bUseCache = false;
            if( is_readable($sCacheFile) ) 
            {
                $iDiff = round(abs(time() - filemtime($sCacheFile)) / 60);
                if( $iDiff < $this->iCache || false ) 
                {
                    $bUseCache = true;
                }
            }
            if( $bUseCache ) 
            {
                $aRawReturn = file_get_contents($sCacheFile);
                $aReturn = unserialize($aRawReturn);
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn);
            }
            else
            {
                $fullAkas = sprintf('https://www.imdb.com/title/tt%s/releaseinfo', $this->iId);
                $aCurlInfo = IMDBHelper::runCurl($fullAkas);
                $sSource = $aCurlInfo['contents'];
                if( false === $sSource ) 
                {
                    if( false ) 
                    {
                        echo '<pre><b>cURL error:</b> ' . var_dump($aCurlInfo) . '</pre>';
                    }
                    return false;
                }
                $aReturned = IMDBHelper::matchRegex($sSource, '~>(.*)<\/a><\/td>\s+<td class="release_date">(.*)<\/td>~');
                if( $aReturned ) 
                {
                    $aReturn = [];
                    foreach( $aReturned[1] as $i => $strName ) 
                    {
                        if( strpos($strName, '(') === false ) 
                        {
                            $aReturn[] = [
                                'country' => IMDBHelper::cleanString($strName), 
                                'releasedate' => IMDBHelper::cleanString($aReturned[2][$i])
                            ];
                        }
                    }
                    file_put_contents($sCacheFile, serialize($aReturn));
                    return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn);
                }
            }
        }
        return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound);
    }
    public function getRuntime()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<td[^>]*>\s*Runtime\s*</td>\s*<td>(.+)</td>~Ui', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getSeasons()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = $this->getSeasonsAsUrl();
            if( self::$sNotFound !== $sMatch ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getSeasonsAsUrl($sTarget = '')
    {
        if( true === $this->isReady ) 
        {
            $aMatch = IMDBHelper::matchRegex($this->sSource, '~episodes\?season=(?:\d+)">(\d+)<~Ui');
            $aReturn = [];
            if( count($aMatch[1]) ) 
            {
                foreach( range(1, max($aMatch[1])) as $i => $sName ) 
                {
                    $aReturn[] = '<a href="https://www.imdb.com/title/tt' . $this->iId . '/episodes?season=' . $sName . '"' . (($sTarget ? ' target="' . $sTarget . '"' : '')) . '>' . $sName . '</a>';
                }
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn);
            }
        }
        return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound);
    }
    public function getSoundMix()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<td[^>]*>\s*Sound\s*Mix\s*</td>\s*<td>(.+)</td>~Ui', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getTagline()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<td[^>]*>\s*Taglines\s*</td>\s*<td>(.+)</td>~Ui', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getTitle($bForceLocal = false)
    {
        if( true === $this->isReady ) 
        {
            if( true === $bForceLocal ) 
            {
                $sMatch = IMDBHelper::matchRegex($this->sSource, '~</h3>(?:\s+)(.*)(?:\s+)<span class=\"titlereference-original-title-label~Ui', 1);
                if( $sMatch !== false && $sMatch !== '' ) 
                {
                    return IMDBHelper::cleanString($sMatch);
                }
            }
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~itemprop="name">(.*)(<\/h3>|<span)~Ui', 1);
            $sMatch = preg_replace('~\(\d{4}\)$~Ui', '', $sMatch);
            if( $sMatch !== false && $sMatch !== '' ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getTrailerAsUrl($bEmbed = false)
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~href="videoplayer/(vi[0-9]*)"~Ui', 1);
            if( $sMatch !== false ) 
            {
                $sUrl = 'https://www.imdb.com/video/imdb/' . $sMatch . '/' . (($bEmbed ? 'player' : ''));
                return IMDBHelper::cleanString($sUrl);
            }
        }
        return self::$sNotFound;
    }
    public function getUrl()
    {
        if( true === $this->isReady ) 
        {
            return IMDBHelper::cleanString(str_replace('reference', '', $this->sUrl));
        }
        return self::$sNotFound;
    }
    public function getUserReview()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~href="/title/[t0-9]*/reviews"[^>]*>([^<]*)\s*User~Ui', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getVotes()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~"ipl-rating-star__total-votes">\s*\((.*)\)\s*<~Ui', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getWriter()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = $this->getWriterAsUrl();
            if( self::$sNotFound !== $sMatch ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getWriterAsUrl($sTarget = '')
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<div[^>]*>\s*(?:Writer|Writers)\s*:\s*<ul[^>]*>(.+)</ul>~Ui', 1);
            $aMatch = IMDBHelper::matchRegex($sMatch, '~href="/name/(.+)/?(?:\?[^"]*)?"[^>]*>(.+)</a>~Ui');
            $aReturn = [];
            if( count($aMatch[2]) ) 
            {
                foreach( $aMatch[2] as $i => $sName ) 
                {
                    $aReturn[] = '<a href="https://www.imdb.com/name/' . IMDBHelper::cleanString($aMatch[1][$i]) . '/"' . (($sTarget ? ' target="' . $sTarget . '"' : '')) . '>' . IMDBHelper::cleanString($sName) . '</a>';
                }
                return IMDBHelper::arrayOutput($this->bArrayOutput, $this->sSeparator, self::$sNotFound, $aReturn);
            }
        }
        return self::$sNotFound;
    }
    public function getYear()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~og:title\' content="(?:.*)\((?:.*)(\d{4})(?:.*)\)~Ui', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
    public function getBudget()
    {
        if( true === $this->isReady ) 
        {
            $sMatch = IMDBHelper::matchRegex($this->sSource, '~<td[^>]*>Budget<\/td>\s*<td>\s*(.*)(?:\(estimated\))\s*<\/td>~Ui', 1);
            if( $sMatch !== false ) 
            {
                return IMDBHelper::cleanString($sMatch);
            }
        }
        return self::$sNotFound;
    }
}
class IMDBHelper extends IMDB
{
    public static function matchRegex($sContent, $sPattern, $iIndex = null)
    {
        preg_match_all($sPattern, $sContent, $aMatches);
        if( $aMatches === false ) 
        {
            return false;
        }
        if( $iIndex !== null && is_int($iIndex) ) 
        {
            if( isset($aMatches[$iIndex][0]) ) 
            {
                return $aMatches[$iIndex][0];
            }
            return false;
        }
        return $aMatches;
    }
    public static function arrayOutput($bArrayOutput, $sSeparator, $sNotFound, $aReturn = null, $bHaveMore = false)
    {
        if( $bArrayOutput ) 
        {
            if( $aReturn == null || !is_array($aReturn) ) 
            {
                return [];
            }
            if( $bHaveMore ) 
            {
                $aReturn[] = 'â€¦';
            }
            return $aReturn;
        }
        else
        {
            if( $aReturn == null || !is_array($aReturn) ) 
            {
                return $sNotFound;
            }
            foreach( $aReturn as $i => $value ) 
            {
                if( is_array($value) ) 
                {
                    $aReturn[$i] = implode($sSeparator, $value);
                }
            }
            return implode($sSeparator, $aReturn) . (($bHaveMore ? 'â€¦' : ''));
        }
    }
    public static function cleanString($sInput)
    {
        $aSearch = [
            'Full summary &raquo;', 
            'Full synopsis &raquo;', 
            'Add summary &raquo;', 
            'Add synopsis &raquo;', 
            'See more &raquo;', 
            'See why on IMDbPro.', 
            "\n", 
            "\r"
        ];
        $aReplace = [
            '', 
            '', 
            '', 
            '', 
            '', 
            '', 
            '', 
            ''
        ];
        $sInput = str_replace('</li>', ' | ', $sInput);
        $sInput = strip_tags($sInput);
        $sInput = str_replace('&nbsp;', ' ', $sInput);
        $sInput = str_replace($aSearch, $aReplace, $sInput);
        $sInput = html_entity_decode($sInput, ENT_QUOTES | ENT_HTML5);
        $sInput = preg_replace('/\s+/', ' ', $sInput);
        $sInput = trim($sInput);
        $sInput = rtrim($sInput, ' |');
        return ($sInput ? trim($sInput) : self::$sNotFound);
    }
    public static function getShortText($sText, $iLength = 100)
    {
        if( mb_strlen($sText) <= $iLength ) 
        {
            return $sText;
        }
        list($sShort) = explode("\n", wordwrap($sText, $iLength - 1));
        if( substr($sShort, -1) !== '.' ) 
        {
            return $sShort . 'â€¦';
        }
        return $sShort;
    }
    public static function saveImage($sUrl, $iId)
    {
        if( preg_match('~title_addposter.jpg|imdb-share-logo.png~', $sUrl) ) 
        {
            return 'posters/not-found.jpg';
        }
        $sFilename = dirname(__FILE__) . '/posters/' . $iId . '.jpg';
        if( file_exists($sFilename) ) 
        {
            return 'posters/' . $iId . '.jpg';
        }
        $aCurlInfo = self::runCurl($sUrl, true);
        $sData = $aCurlInfo['contents'];
        if( false === $sData ) 
        {
            return 'posters/not-found.jpg';
        }
        $oFile = fopen($sFilename, 'x');
        fwrite($oFile, $sData);
        fclose($oFile);
        return 'posters/' . $iId . '.jpg';
    }
    public static function runCurl($sUrl, $bDownload = false)
    {
        $oCurl = curl_init($sUrl);
        curl_setopt_array($oCurl, [
            CURLOPT_BINARYTRANSFER => ($bDownload ? true : false), 
            CURLOPT_CONNECTTIMEOUT => self::IMDB_TIMEOUT, 
            CURLOPT_ENCODING => '', 
            CURLOPT_FOLLOWLOCATION => 0, 
            CURLOPT_FRESH_CONNECT => 0, 
            CURLOPT_HEADER => ($bDownload ? false : true), 
            CURLOPT_HTTPHEADER => [
                'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8', 
                'Accept-Charset: utf-8, iso-8859-1;q=0.5', 
                'Accept-Language: ' . self::IMDB_LANG
            ], 
            CURLOPT_REFERER => 'https://www.imdb.com', 
            CURLOPT_RETURNTRANSFER => 1, 
            CURLOPT_SSL_VERIFYHOST => 0, 
            CURLOPT_SSL_VERIFYPEER => 0, 
            CURLOPT_TIMEOUT => self::IMDB_TIMEOUT, 
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:59.0) Gecko/20100101 Firefox/59.0', 
            CURLOPT_VERBOSE => 0
        ]);
        $sOutput = curl_exec($oCurl);
        $aCurlInfo = curl_getinfo($oCurl);
        curl_close($oCurl);
        $aCurlInfo['contents'] = $sOutput;
        if( $aCurlInfo['http_code'] !== 200 && $aCurlInfo['http_code'] !== 302 ) 
        {
            if( true === self::IMDB_DEBUG ) 
            {
                echo '<pre><b>cURL returned wrong HTTP code ' . $aCurlInfo['http_code'] . 'â€‌, aborting.</b></pre>';
            }
            return false;
        }
        return $aCurlInfo;
    }
    public static function saveImageCast($sUrl, $cId)
    {
        if( !preg_match('~http~', $sUrl) ) 
        {
            return 'cast/not-found.jpg';
        }
        $sFilename = dirname(__FILE__) . '/cast/' . $cId . '.jpg';
        if( file_exists($sFilename) ) 
        {
            return 'cast/' . $cId . '.jpg';
        }
        $aCurlInfo = self::runCurl($sUrl, true);
        $sData = $aCurlInfo['contents'];
        if( false === $sData ) 
        {
            return 'cast/not-found.jpg';
        }
        $oFile = fopen($sFilename, 'x');
        fwrite($oFile, $sData);
        fclose($oFile);
        return 'cast/' . $cId . '.jpg';
    }
}
