<?php
/**
 * Xtream Codes Player API with Local EPG Support
 *
 * EPG Actions:
 * - get_short_epg: Get short EPG for a stream (stream_id required, limit optional)
 * - get_simple_date_table: Get full EPG for a stream (stream_id required)
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Get request parameters
$username = isset($_GET['username']) ? $_GET['username'] : '';
$password = isset($_GET['password']) ? $_GET['password'] : '';
$action = isset($_GET['action']) ? $_GET['action'] : '';
$stream_id = isset($_GET['stream_id']) ? $_GET['stream_id'] : null;
$category_id = isset($_GET['category_id']) ? $_GET['category_id'] : null;
$vod_id = isset($_GET['vod_id']) ? $_GET['vod_id'] : null;
$series_id = isset($_GET['series_id']) ? $_GET['series_id'] : null;
$limit = isset($_GET['limit']) ? intval($_GET['limit']) : 4;

// Load config
require_once __DIR__ . '/includes/Config.php';

try {
    $pdo = new PDO(
        "mysql:host={$config['db']['hostname']};dbname={$config['db']['database']};charset=utf8",
        $config['db']['username'],
        $config['db']['password']
    );
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Get panel settings
    $stmt = $pdo->query("SELECT setting_key, setting_value FROM panel_settings");
    $settings = [];
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }

    $panel_url = $settings['panel_url'] ?? '';
    $source_username = $settings['panel_username'] ?? '';
    $source_password = $settings['panel_password'] ?? '';

    $parsed = parse_url($panel_url);
    $host = $parsed['host'];
    $base_url = 'http://' . $host . ':80';

} catch (Exception $e) {
    echo json_encode(['error' => 'Database connection failed']);
    exit;
}

/**
 * Get EPG from local cache
 */
function getLocalEpg($stream_id, $limit = 10, $full = false) {
    $cache_dir = __DIR__ . '/cache';
    $mapping_file = $cache_dir . '/epg_mapping.json';
    $epg_file = $cache_dir . '/epg.xml';

    // Get tvg-id from mapping
    if (!file_exists($mapping_file)) {
        return ['epg_listings' => []];
    }

    $mapping = json_decode(file_get_contents($mapping_file), true) ?: [];
    $tvg_id = $mapping[$stream_id] ?? '';

    if (empty($tvg_id) || !file_exists($epg_file)) {
        return ['epg_listings' => []];
    }

    $xml_content = file_get_contents($epg_file);
    if (empty($xml_content)) {
        return ['epg_listings' => []];
    }

    // Parse EPG
    $epg_listings = [];
    $now = time();

    // Find programmes for this tvg-id
    preg_match_all('/<programme[^>]*channel="' . preg_quote($tvg_id, '/') . '"[^>]*>.*?<\/programme>/s', $xml_content, $programmes);

    foreach ($programmes[0] as $prog) {
        preg_match('/start="(\d{14})/', $prog, $start_match);
        preg_match('/stop="(\d{14})/', $prog, $stop_match);

        if (!$start_match || !$stop_match) continue;

        $start_str = $start_match[1];
        $stop_str = $stop_match[1];

        $start_time = mktime(
            (int)substr($start_str, 8, 2),
            (int)substr($start_str, 10, 2),
            (int)substr($start_str, 12, 2),
            (int)substr($start_str, 4, 2),
            (int)substr($start_str, 6, 2),
            (int)substr($start_str, 0, 4)
        );

        $stop_time = mktime(
            (int)substr($stop_str, 8, 2),
            (int)substr($stop_str, 10, 2),
            (int)substr($stop_str, 12, 2),
            (int)substr($stop_str, 4, 2),
            (int)substr($stop_str, 6, 2),
            (int)substr($stop_str, 0, 4)
        );

        // For short EPG: current and future only
        // For full EPG: include past 24h too
        if ($full) {
            if ($stop_time < ($now - 86400) || $start_time > ($now + 86400 * 7)) continue;
        } else {
            if ($stop_time < $now || $start_time > ($now + 86400)) continue;
        }

        preg_match('/<title[^>]*>([^<]*)<\/title>/', $prog, $title_match);
        preg_match('/<desc[^>]*>([^<]*)<\/desc>/', $prog, $desc_match);

        $title = $title_match[1] ?? '';
        $desc = $desc_match[1] ?? '';

        $epg_listings[] = [
            'id' => md5($tvg_id . $start_time),
            'epg_id' => $stream_id,
            'title' => base64_encode($title),
            'lang' => 'en',
            'start' => date('Y-m-d H:i:s', $start_time),
            'end' => date('Y-m-d H:i:s', $stop_time),
            'description' => base64_encode($desc),
            'channel_id' => $tvg_id,
            'start_timestamp' => (string)$start_time,
            'stop_timestamp' => (string)$stop_time,
            'now_playing' => ($now >= $start_time && $now < $stop_time) ? 1 : 0,
            'has_archive' => 0
        ];

        if (!$full && count($epg_listings) >= $limit) break;
    }

    return ['epg_listings' => $epg_listings];
}

/**
 * Make API request to source server
 */
function makeApiRequest($base_url, $source_username, $source_password, $params = []) {
    $url = $base_url . '/player_api.php?username=' . urlencode($source_username) . '&password=' . urlencode($source_password);

    foreach ($params as $key => $value) {
        if ($value !== null) {
            $url .= '&' . $key . '=' . urlencode($value);
        }
    }

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_USERAGENT => 'IPTV Panel API'
    ]);

    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($http_code == 200 && $response) {
        return json_decode($response, true);
    }
    return null;
}

function getServerInfo($panel_url) {
    $parsed = parse_url($panel_url);
    return [
        'url' => $parsed['host'],
        'port' => $parsed['port'] ?? '80',
        'https_port' => '443',
        'server_protocol' => $parsed['scheme'],
        'rtmp_port' => '1935',
        'timezone' => 'UTC',
        'timestamp_now' => time(),
        'time_now' => date('Y-m-d H:i:s')
    ];
}

// Handle actions
$response = null;

switch ($action) {
    case 'get_short_epg':
        if ($stream_id) {
            $response = getLocalEpg($stream_id, $limit, false);
        } else {
            $response = ['error' => 'stream_id required'];
        }
        break;

    case 'get_simple_data_table':
    case 'get_simple_date_table':
        if ($stream_id) {
            $response = getLocalEpg($stream_id, 100, true);
        } else {
            $response = ['error' => 'stream_id required'];
        }
        break;

    case 'get_live_categories':
        $response = makeApiRequest($base_url, $source_username, $source_password, [
            'action' => 'get_live_categories'
        ]);
        break;

    case 'get_live_streams':
        $response = makeApiRequest($base_url, $source_username, $source_password, [
            'action' => 'get_live_streams',
            'category_id' => $category_id
        ]);
        break;

    case 'get_vod_categories':
        $response = makeApiRequest($base_url, $source_username, $source_password, [
            'action' => 'get_vod_categories'
        ]);
        break;

    case 'get_vod_streams':
        $response = makeApiRequest($base_url, $source_username, $source_password, [
            'action' => 'get_vod_streams',
            'category_id' => $category_id
        ]);
        break;

    case 'get_vod_info':
        $response = makeApiRequest($base_url, $source_username, $source_password, [
            'action' => 'get_vod_info',
            'vod_id' => $vod_id
        ]);
        break;

    case 'get_series_categories':
        $response = makeApiRequest($base_url, $source_username, $source_password, [
            'action' => 'get_series_categories'
        ]);
        break;

    case 'get_series':
        $response = makeApiRequest($base_url, $source_username, $source_password, [
            'action' => 'get_series',
            'category_id' => $category_id
        ]);
        break;

    case 'get_series_info':
        $response = makeApiRequest($base_url, $source_username, $source_password, [
            'action' => 'get_series_info',
            'series_id' => $series_id
        ]);
        break;

    default:
        // Authentication
        $source_result = makeApiRequest($base_url, $source_username, $source_password, []);

        if ($source_result && isset($source_result['user_info'])) {
            $response = [
                'user_info' => [
                    'username' => $username,
                    'password' => $password,
                    'message' => '',
                    'auth' => 1,
                    'status' => 'Active',
                    'exp_date' => $source_result['user_info']['exp_date'] ?? null,
                    'is_trial' => '0',
                    'active_cons' => '0',
                    'created_at' => $source_result['user_info']['created_at'] ?? time(),
                    'max_connections' => $source_result['user_info']['max_connections'] ?? '1',
                    'allowed_output_formats' => ['m3u8', 'ts', 'rtmp']
                ],
                'server_info' => getServerInfo($panel_url)
            ];
        } else {
            $response = [
                'user_info' => [
                    'auth' => 0,
                    'status' => 'Disabled',
                    'message' => 'Authentication failed'
                ]
            ];
        }
        break;
}

echo json_encode($response ?? ['error' => 'No data'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
?>
