<?php
/**
 * Bulk Streams Manager - Standalone Page
 * Provides a simple interface for bulk stream operations
 * Created to avoid conflicts with existing code
 */

define('_CHECK_X_ME', true);
require_once(dirname(__FILE__) . '/load2.php');
require_once(dirname(__FILE__) . '/includes/AdminStreams.php');

// Check authentication
if (!isset($sess_admin['adminid'])) {
    header('Location: ' . admin_path . 'login.php');
    exit;
}

// Handle AJAX bulk actions
if (isset($_POST['bulk_action']) && isset($_POST['stream_ids'])) {
    $action = $_POST['bulk_action'];
    $stream_ids = json_decode($_POST['stream_ids'], true);

    if (!is_array($stream_ids) || count($stream_ids) == 0) {
        echo json_encode(['success' => false, 'message' => 'No streams selected']);
        exit;
    }

    // Initialize AdminStreams
    $adminStreams = new AdminStreams();
    $adminStreams->appname = 'streams';
    $adminStreams->base = admin_path . 'index.php/streams';
    $adminStreams->type = 1;

    // Set POST data for bulk functions
    $_POST['ids'] = $stream_ids;

    ob_start();

    switch($action) {
        case 'delete':
            $adminStreams->BulkDelete();
            break;
        case 'restart':
            $adminStreams->BulkRestart();
            break;
        case 'stop':
            $adminStreams->BulkStop();
            break;
        case 'disable':
            $adminStreams->BulkDisable();
            break;
        case 'toggle_status':
            $adminStreams->BulkToggleStatus();
            break;
        default:
            echo '<span class="text-danger">Invalid action</span>';
    }

    $output = ob_get_clean();
    echo json_encode(['success' => true, 'message' => strip_tags($output), 'html' => $output]);
    exit;
}

// Get all streams for display
$intro->db->query("SELECT s.id, s.stream_display_name, s.direct_source, s.status,
                         ss.on_demand, ss.stream_status, ss.pid
                  FROM streams s
                  LEFT JOIN streams_sys ss ON s.id = ss.stream_id
                  WHERE s.type = 1
                  ORDER BY s.id ASC");

$streams = [];
while ($row = $intro->db->fetch_assoc()) {
    $streams[] = $row;
}
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Bulk Streams Manager</title>
    <link rel="stylesheet" href="<?php echo admin_path; ?>style/css/bootstrap.min.css">
    <link rel="stylesheet" href="<?php echo admin_path; ?>style/css/font-awesome.min.css">
    <style>
        body {
            padding: 20px;
            background: #f5f5f5;
        }
        .bulk-manager {
            background: white;
            border-radius: 8px;
            padding: 30px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .bulk-header {
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 3px solid #3498db;
        }
        .bulk-actions-bar {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 25px;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
        }
        .bulk-btn {
            margin: 5px;
            min-width: 140px;
            font-weight: bold;
            border: none;
            padding: 12px 24px;
            border-radius: 6px;
            transition: all 0.3s ease;
        }
        .bulk-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.2);
        }
        .bulk-btn-delete { background: #e74c3c; color: white; }
        .bulk-btn-delete:hover { background: #c0392b; }
        .bulk-btn-restart { background: #3498db; color: white; }
        .bulk-btn-restart:hover { background: #2980b9; }
        .bulk-btn-stop { background: #f39c12; color: white; }
        .bulk-btn-stop:hover { background: #d68910; }
        .bulk-btn-disable { background: #95a5a6; color: white; }
        .bulk-btn-disable:hover { background: #7f8c8d; }
        .bulk-btn-toggle { background: #9b59b6; color: white; }
        .bulk-btn-toggle:hover { background: #8e44ad; }
        .stream-table {
            margin-top: 20px;
        }
        .stream-table th {
            background: #34495e;
            color: white;
            padding: 12px;
        }
        .stream-row:hover {
            background: #ecf0f1;
        }
        .status-badge {
            padding: 5px 12px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: bold;
        }
        .status-live { background: #2ecc71; color: white; }
        .status-ondemand { background: #3498db; color: white; }
        .status-running { background: #27ae60; color: white; }
        .status-stopped { background: #e74c3c; color: white; }
        .select-all-box {
            background: #fff3cd;
            padding: 15px;
            border-radius: 6px;
            margin-bottom: 20px;
            border-left: 4px solid #ffc107;
        }
        #result-message {
            margin-top: 20px;
            padding: 15px;
            border-radius: 6px;
            display: none;
        }
        .success-msg { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error-msg { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="bulk-manager">
            <div class="bulk-header">
                <h2><i class="fa fa-tasks"></i> Bulk Streams Manager</h2>
                <p class="text-muted">Select streams and apply bulk actions</p>
                <a href="<?php echo admin_path; ?>index.php/streams/index" class="btn btn-default">
                    <i class="fa fa-arrow-left"></i> Back to Streams
                </a>
            </div>

            <div class="select-all-box">
                <label style="font-size: 16px; margin: 0;">
                    <input type="checkbox" id="select-all" style="width: 18px; height: 18px; vertical-align: middle;">
                    <strong> Select All Streams</strong>
                </label>
                <span id="selected-count" style="margin-left: 20px; color: #856404;">
                    (<span id="count">0</span> selected)
                </span>
            </div>

            <div class="bulk-actions-bar">
                <h4 style="color: white; margin-bottom: 15px;">
                    <i class="fa fa-flash"></i> Bulk Actions
                </h4>
                <button class="bulk-btn bulk-btn-restart" onclick="performBulkAction('restart')">
                    <i class="fa fa-refresh"></i> Restart Selected
                </button>
                <button class="bulk-btn bulk-btn-stop" onclick="performBulkAction('stop')">
                    <i class="fa fa-stop"></i> Stop Selected
                </button>
                <button class="bulk-btn bulk-btn-disable" onclick="performBulkAction('disable')">
                    <i class="fa fa-ban"></i> Disable Selected
                </button>
                <button class="bulk-btn bulk-btn-toggle" onclick="performBulkAction('toggle_status')">
                    <i class="fa fa-toggle-on"></i> Toggle Enable/Disable
                </button>
                <button class="bulk-btn bulk-btn-delete" onclick="performBulkAction('delete')">
                    <i class="fa fa-trash"></i> Delete Selected
                </button>
            </div>

            <div id="result-message"></div>

            <table class="table table-bordered table-striped stream-table">
                <thead>
                    <tr>
                        <th width="50">
                            <i class="fa fa-check-square-o"></i>
                        </th>
                        <th width="80">ID</th>
                        <th>Stream Name</th>
                        <th width="120">Mode</th>
                        <th width="120">Running</th>
                        <th width="100">Enabled</th>
                        <th width="100">PID</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($streams as $stream): ?>
                    <tr class="stream-row">
                        <td align="center">
                            <input type="checkbox" class="stream-checkbox" value="<?php echo $stream['id']; ?>"
                                   style="width: 18px; height: 18px;">
                        </td>
                        <td><strong><?php echo $stream['id']; ?></strong></td>
                        <td><?php echo htmlspecialchars($stream['stream_display_name']); ?></td>
                        <td>
                            <?php if ($stream['on_demand'] == 1 || $stream['direct_source'] == 0): ?>
                                <span class="status-badge status-ondemand">OnDemand</span>
                            <?php else: ?>
                                <span class="status-badge status-live">Live</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if ($stream['stream_status'] == 0 && !empty($stream['pid'])): ?>
                                <span class="status-badge status-running">Running</span>
                            <?php else: ?>
                                <span class="status-badge status-stopped">Stopped</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if ($stream['status'] == 1): ?>
                                <span class="status-badge" style="background: #27ae60; color: white;">Enabled</span>
                            <?php else: ?>
                                <span class="status-badge" style="background: #c0392b; color: white;">Disabled</span>
                            <?php endif; ?>
                        </td>
                        <td><?php echo $stream['pid'] ?? '-'; ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>

    <script src="<?php echo admin_path; ?>style/js/jquery.min.js"></script>
    <script>
        // Select all checkbox
        $('#select-all').on('change', function() {
            $('.stream-checkbox').prop('checked', $(this).is(':checked'));
            updateCount();
        });

        // Individual checkboxes
        $('.stream-checkbox').on('change', function() {
            updateCount();
            // Update select-all checkbox
            var total = $('.stream-checkbox').length;
            var checked = $('.stream-checkbox:checked').length;
            $('#select-all').prop('checked', total === checked);
        });

        // Update selected count
        function updateCount() {
            var count = $('.stream-checkbox:checked').length;
            $('#count').text(count);
        }

        // Perform bulk action
        function performBulkAction(action) {
            var selectedIds = [];
            $('.stream-checkbox:checked').each(function() {
                selectedIds.push(parseInt($(this).val()));
            });

            if (selectedIds.length === 0) {
                showMessage('Please select at least one stream', 'error');
                return;
            }

            var actionNames = {
                'delete': 'Delete',
                'restart': 'Restart',
                'stop': 'Stop',
                'disable': 'Disable',
                'toggle_status': 'Toggle Enable/Disable'
            };

            var confirmMsg = 'Are you sure you want to ' + actionNames[action].toLowerCase() + ' ' +
                           selectedIds.length + ' stream(s)?';

            if (!confirm(confirmMsg)) {
                return;
            }

            // Show loading
            showMessage('Processing ' + actionNames[action] + '...', 'info');

            $.ajax({
                url: 'streams_bulk_manager.php',
                method: 'POST',
                data: {
                    bulk_action: action,
                    stream_ids: JSON.stringify(selectedIds)
                },
                dataType: 'json',
                success: function(response) {
                    if (response.success) {
                        showMessage(response.html, 'success');

                        // If delete, reload page after 2 seconds
                        if (action === 'delete') {
                            setTimeout(function() {
                                location.reload();
                            }, 2000);
                        } else {
                            // Reload after 3 seconds for other actions
                            setTimeout(function() {
                                location.reload();
                            }, 3000);
                        }
                    } else {
                        showMessage(response.message || 'Action failed', 'error');
                    }
                },
                error: function() {
                    showMessage('Connection error. Please try again.', 'error');
                }
            });
        }

        // Show message
        function showMessage(message, type) {
            var $msg = $('#result-message');
            $msg.removeClass('success-msg error-msg info-msg')
                .addClass(type + '-msg')
                .html(message)
                .fadeIn();
        }
    </script>
</body>
</html>
