<?php
/**
 * VOD Import System API
 * Main API Router
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../classes/ServerManager.php';
require_once __DIR__ . '/../classes/MovieImporter.php';
require_once __DIR__ . '/../classes/FolderWatch.php';
require_once __DIR__ . '/../classes/Logger.php';
require_once __DIR__ . '/../classes/Auth.php';
require_once __DIR__ . '/../classes/RemoteDownloader.php';

// Parse request
$method = $_SERVER['REQUEST_METHOD'];
$uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$uri = str_replace('/VOD/api', '', $uri);
$uri = trim($uri, '/');
$segments = $uri ? explode('/', $uri) : [];

// Get JSON body for POST/PUT
$input = json_decode(file_get_contents('php://input'), true) ?? [];

// Simple router
$endpoint = $segments[0] ?? 'status';
$action = $segments[1] ?? null;
$id = isset($segments[2]) ? (int) $segments[2] : null;

// Handle case where second segment is numeric (it's an ID, not an action)
// URL format: /endpoint/{id}/action
if ($action !== null && is_numeric($action)) {
    $id = (int) $action;
    $action = $segments[2] ?? null;  // Get action from third segment if present
}

// Query parameters
$params = $_GET;

try {
    // PHP 7.4 compatible switch statement (instead of match)
    switch ($endpoint) {
        case 'status':
            $response = handleStatus();
            break;
        case 'stats':
            $response = handleStats();
            break;
        case 'servers':
            $response = handleServers($method, $action, $id, $input);
            break;
        case 'movies':
            $response = handleMovies($method, $action, $id, $input, $params);
            break;
        case 'series':
            $response = handleSeries($method, $action, $id, $input, $params);
            break;
        case 'episodes':
            $response = handleEpisodes($method, $action, $id, $input, $params);
            break;
        case 'import':
            $response = handleImport($method, $action, $input);
            break;
        case 'categories':
            $response = handleCategories($method, $action, $id, $params);
            break;
        case 'folderwatch':
            $response = handleFolderWatch($method, $action, $input);
            break;
        case 'logs':
            $response = handleLogs($method, $action, $params);
            break;
        case 'settings':
            $response = handleSettings($method, $input);
            break;
        case 'remote-servers':
            $response = handleRemoteServers($method, $action, $id, $input);
            break;
        case 'download-jobs':
            $response = handleDownloadJobs($method, $action, $id, $input, $params);
            break;
        case 'users':
            $response = handleUsers($method, $action, $id, $input);
            break;
        case 'auto-download':
            $response = handleAutoDownload($method, $action, $input);
            break;
        default:
            $response = ['error' => 'Unknown endpoint', 'available' => [
                'status', 'stats', 'servers', 'movies', 'series', 'episodes', 'import', 'categories', 'folderwatch', 'logs', 'settings',
                'remote-servers', 'download-jobs', 'users', 'auto-download'
            ]];
            break;
    }

    echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'error' => $e->getMessage(),
        'trace' => DEBUG_MODE ? $e->getTraceAsString() : null
    ]);
}

/**
 * Status endpoint
 */
function handleStatus(): array
{
    return [
        'status' => 'ok',
        'version' => API_VERSION,
        'timestamp' => date('c'),
        'php_version' => PHP_VERSION
    ];
}

/**
 * Statistics endpoint
 */
function handleStats(): array
{
    $sm = new ServerManager();
    return $sm->getStats();
}

/**
 * Servers endpoint
 */
function handleServers(string $method, ?string $action, ?int $id, array $input): array
{
    $sm = new ServerManager();

    if ($method === 'GET') {
        if ($action === 'check' && $id) {
            return $sm->checkServer($id);
        }
        if ($action === 'check-all') {
            return $sm->checkAllServers();
        }
        if ($id) {
            $server = $sm->getServer($id);
            return $server ?: ['error' => 'Server not found'];
        }
        return $sm->getAllServers();
    }

    if ($method === 'POST') {
        if ($action === 'add') {
            $serverId = $sm->addServer($input);
            if ($serverId) {
                return ['success' => true, 'server_id' => $serverId];
            }
            return ['error' => 'Failed to add server'];
        }
        if ($action === 'sync-categories' && $id) {
            return $sm->syncCategories($id);
        }
        if ($action === 'toggle' && $id) {
            return ['success' => $sm->toggleServer($id)];
        }
    }

    if ($method === 'PUT' && $id) {
        return ['success' => $sm->updateServer($id, $input)];
    }

    if ($method === 'DELETE' && $id) {
        return ['success' => $sm->deleteServer($id)];
    }

    return ['error' => 'Invalid request'];
}

/**
 * Movies endpoint
 */
function handleMovies(string $method, ?string $action, ?int $id, array $input, array $params): array
{
    $db = getDB();

    if ($method === 'GET') {
        // Get years for filter
        if ($action === 'years') {
            $stmt = $db->query("SELECT DISTINCT year FROM movies WHERE year IS NOT NULL AND year != '' ORDER BY year DESC");
            return $stmt->fetchAll(PDO::FETCH_COLUMN);
        }

        // Get single movie with links
        if ($id) {
            $stmt = $db->prepare("SELECT * FROM movies WHERE id = ?");
            $stmt->execute([$id]);
            $movie = $stmt->fetch();

            if ($movie) {
                $stmt = $db->prepare("SELECT * FROM movie_links WHERE movie_id = ?");
                $stmt->execute([$id]);
                $movie['links'] = $stmt->fetchAll();
            }

            return $movie ?: ['error' => 'Movie not found'];
        }

        // Search
        if ($action === 'search' || !empty($params['search'])) {
            $query = $params['q'] ?? $params['search'] ?? '';
            $stmt = $db->prepare("
                SELECT m.*, COUNT(ml.id) as link_count
                FROM movies m
                LEFT JOIN movie_links ml ON m.id = ml.movie_id
                WHERE m.title LIKE ?
                GROUP BY m.id
                ORDER BY m.year DESC, m.title ASC
                LIMIT 100
            ");
            $stmt->execute(["%{$query}%"]);
            return $stmt->fetchAll();
        }

        // Newest
        if ($action === 'newest') {
            $limit = min((int) ($params['limit'] ?? 10), 50);
            $stmt = $db->prepare("
                SELECT m.*, COUNT(ml.id) as link_count
                FROM movies m
                INNER JOIN movie_links ml ON m.id = ml.movie_id
                GROUP BY m.id
                HAVING link_count > 0
                ORDER BY m.created_at DESC
                LIMIT ?
            ");
            $stmt->execute([$limit]);
            return $stmt->fetchAll();
        }

        // List with filters
        $limit = min((int) ($params['limit'] ?? 24), 100);
        $offset = (int) ($params['offset'] ?? 0);
        $categoryId = $params['category_id'] ?? null;
        $year = $params['year'] ?? null;
        $search = $params['search'] ?? null;

        $where = [];
        $bindings = [];

        if ($categoryId) {
            $where[] = "m.id IN (SELECT movie_id FROM movie_links WHERE category_id = ?)";
            $bindings[] = $categoryId;
        }

        if ($year) {
            $where[] = "m.year = ?";
            $bindings[] = $year;
        }

        if ($search) {
            $where[] = "m.title LIKE ?";
            $bindings[] = "%{$search}%";
        }

        $whereClause = $where ? 'WHERE ' . implode(' AND ', $where) : '';

        $sql = "
            SELECT m.*, COUNT(ml.id) as link_count
            FROM movies m
            LEFT JOIN movie_links ml ON m.id = ml.movie_id
            {$whereClause}
            GROUP BY m.id
            ORDER BY m.created_at DESC
            LIMIT ? OFFSET ?
        ";

        $bindings[] = $limit;
        $bindings[] = $offset;

        $stmt = $db->prepare($sql);
        $stmt->execute($bindings);
        return $stmt->fetchAll();
    }

    return ['error' => 'Invalid request'];
}

function handleMoviesOld(string $method, ?string $action, ?int $id, array $input, array $params): array
{
    $importer = new MovieImporter();
    $db = getDB();

    if ($method === 'GET') {
        if ($action === 'newest') {
            $limit = (int) ($params['limit'] ?? 50);
            return $importer->getNewestMovies($limit);
        }

        if ($action === 'search') {
            $query = $params['q'] ?? '';
            if (empty($query)) {
                return ['error' => 'Search query required'];
            }
            return $importer->searchMovies($query);
        }

        if ($action === 'category' && $id) {
            $limit = (int) ($params['limit'] ?? 100);
            $offset = (int) ($params['offset'] ?? 0);
            return $importer->getMoviesByCategory($id, $limit, $offset);
        }

        if ($action === 'links' && $id) {
            return $importer->checkMovieLinks($id);
        }

        if ($action === 'best-link' && $id) {
            $link = $importer->selectBestLink($id);
            return $link ?: ['error' => 'No working link found'];
        }

        if ($action === 'stats') {
            return $importer->getImportStats();
        }

        if ($id) {
            $stmt = $db->prepare("
                SELECT m.*, COUNT(ml.id) as link_count
                FROM movies m
                LEFT JOIN movie_links ml ON m.id = ml.movie_id
                WHERE m.id = ?
                GROUP BY m.id
            ");
            $stmt->execute([$id]);
            $movie = $stmt->fetch();
            return $movie ?: ['error' => 'Movie not found'];
        }

        // List all movies with pagination
        $limit = min((int) ($params['limit'] ?? 50), 500);
        $offset = (int) ($params['offset'] ?? 0);
        $year = $params['year'] ?? null;

        $sql = "SELECT m.*, COUNT(ml.id) as link_count FROM movies m LEFT JOIN movie_links ml ON m.id = ml.movie_id";
        $where = [];
        $bindings = [];

        if ($year) {
            $where[] = "m.year = ?";
            $bindings[] = $year;
        }

        if ($where) {
            $sql .= " WHERE " . implode(' AND ', $where);
        }

        $sql .= " GROUP BY m.id ORDER BY m.created_at DESC LIMIT ? OFFSET ?";
        $bindings[] = $limit;
        $bindings[] = $offset;

        $stmt = $db->prepare($sql);
        $stmt->execute($bindings);

        return $stmt->fetchAll();
    }

    if ($method === 'DELETE' && $id) {
        $stmt = $db->prepare("DELETE FROM movies WHERE id = ?");
        return ['success' => $stmt->execute([$id])];
    }

    return ['error' => 'Invalid request'];
}

/**
 * Import endpoint
 */
function handleImport(string $method, ?string $action, array $input): array
{
    if ($method !== 'POST') {
        return ['error' => 'Method not allowed'];
    }

    $importer = new MovieImporter();

    switch ($action) {
        case 'server':
            $serverId = (int) ($input['server_id'] ?? 0);
            $categoryId = isset($input['category_id']) ? (int) $input['category_id'] : null;
            $force = (bool) ($input['force'] ?? false);

            if (!$serverId) {
                return ['error' => 'server_id required'];
            }

            return $importer->importFromServer($serverId, $categoryId, $force);

        case 'single':
            $serverId = (int) ($input['server_id'] ?? 0);
            $streamId = (int) ($input['stream_id'] ?? 0);

            if (!$serverId || !$streamId) {
                return ['error' => 'server_id and stream_id required'];
            }

            return $importer->importSingleMovie($serverId, $streamId);

        case 'smart-sync':
            $serverId = (int) ($input['server_id'] ?? 0);

            if (!$serverId) {
                return ['error' => 'server_id required'];
            }

            return $importer->smartSync($serverId);

        case 'bulk':
            $serverId = (int) ($input['server_id'] ?? 0);
            $streamIds = $input['stream_ids'] ?? [];

            if (!$serverId || empty($streamIds)) {
                return ['error' => 'server_id and stream_ids required'];
            }

            $results = [];
            foreach ($streamIds as $streamId) {
                $results[] = $importer->importSingleMovie($serverId, (int) $streamId);
            }

            return [
                'success' => true,
                'results' => $results,
                'imported' => count(array_filter($results, fn($r) => $r['success'] ?? false))
            ];

        case 'check-new':
            // Check for new movies across all servers
            $sm = new ServerManager();
            $servers = $sm->getAllServers(true);
            $newMovies = [];

            foreach ($servers as $server) {
                $result = $importer->smartSync($server['id']);
                if ($result['new_found'] > 0) {
                    $newMovies[$server['name']] = $result['new_found'];
                }
            }

            return [
                'success' => true,
                'new_movies_by_server' => $newMovies,
                'total_new' => array_sum($newMovies)
            ];

        case 'series':
            // Import series from server
            $serverId = (int) ($input['server_id'] ?? 0);
            $categoryId = isset($input['category_id']) ? (int) $input['category_id'] : null;
            $force = (bool) ($input['force'] ?? false);

            if (!$serverId) {
                return ['error' => 'server_id required'];
            }

            return importSeriesFromServer($serverId, $categoryId, $force);

        case 'series-sync':
            // Smart sync series
            $serverId = (int) ($input['server_id'] ?? 0);

            if (!$serverId) {
                return ['error' => 'server_id required'];
            }

            return smartSyncSeries($serverId);

        case 'check-new-series':
            // Check for new series across all servers
            $sm = new ServerManager();
            $servers = $sm->getAllServers(true);
            $newSeries = [];

            foreach ($servers as $server) {
                $result = smartSyncSeries($server['id']);
                if (($result['new_found'] ?? 0) > 0) {
                    $newSeries[$server['name']] = $result['new_found'];
                }
            }

            return [
                'success' => true,
                'new_series_by_server' => $newSeries,
                'total_new' => array_sum($newSeries)
            ];

        case 'url-categories':
            // Get categories from URL
            $server = $input['server'] ?? '';
            $username = $input['username'] ?? '';
            $password = $input['password'] ?? '';
            $type = $input['type'] ?? 'series';

            if (!$server || !$username || !$password) {
                return ['error' => 'server, username, and password required'];
            }

            return fetchUrlCategories($server, $username, $password, $type);

        case 'url-preview':
            // Preview content from URL
            $server = $input['server'] ?? '';
            $username = $input['username'] ?? '';
            $password = $input['password'] ?? '';
            $type = $input['type'] ?? 'series';
            $categoryId = $input['category_id'] ?? null;
            $limit = (int) ($input['limit'] ?? 10);

            if (!$server || !$username || !$password) {
                return ['error' => 'server, username, and password required'];
            }

            return previewUrlContent($server, $username, $password, $type, $categoryId, $limit);

        case 'url-series':
            // Import series directly from URL
            $server = $input['server'] ?? '';
            $username = $input['username'] ?? '';
            $password = $input['password'] ?? '';
            $serverName = $input['server_name'] ?? 'URL Import';
            $categoryId = $input['category_id'] ?? null;

            if (!$server || !$username || !$password) {
                return ['error' => 'server, username, and password required'];
            }

            return importSeriesFromUrl($server, $username, $password, $serverName, $categoryId);

        default:
            return ['error' => 'Unknown action', 'available' => [
                'server', 'single', 'smart-sync', 'bulk', 'check-new', 'series', 'series-sync', 'check-new-series',
                'url-categories', 'url-preview', 'url-series'
            ]];
    }
}

/**
 * Fetch categories from external URL
 */
function fetchUrlCategories(string $server, string $username, string $password, string $type): array
{
    // PHP 7.4 compatible
    switch($type) {
        case 'movie':
            $action = 'get_vod_categories';
            break;
        case 'series':
            $action = 'get_series_categories';
            break;
        case 'live':
            $action = 'get_live_categories';
            break;
        default:
            $action = 'get_series_categories';
            break;
    }

    $apiUrl = rtrim($server, '/') . "/player_api.php?username={$username}&password={$password}&action={$action}";

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_USERAGENT => 'TiviMate/5.1.6 (Android 7.1.2)'
    ]);
    $response = curl_exec($ch);
    $error = curl_error($ch);
    curl_close($ch);

    if (!$response) {
        return ['error' => 'Failed to connect: ' . $error];
    }

    $categories = json_decode($response, true);
    if (!is_array($categories)) {
        return ['error' => 'Invalid response from server'];
    }

    return ['categories' => $categories];
}

/**
 * Preview content from external URL
 */
function previewUrlContent(string $server, string $username, string $password, string $type, ?string $categoryId, int $limit): array
{
    // PHP 7.4 compatible
    switch($type) {
        case 'movie':
            $action = 'get_vod_streams';
            break;
        case 'series':
            $action = 'get_series';
            break;
        case 'live':
            $action = 'get_live_streams';
            break;
        default:
            $action = 'get_series';
            break;
    }

    $apiUrl = rtrim($server, '/') . "/player_api.php?username={$username}&password={$password}&action={$action}";
    if ($categoryId) {
        $apiUrl .= "&category_id={$categoryId}";
    }

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 60,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_USERAGENT => 'TiviMate/5.1.6 (Android 7.1.2)'
    ]);
    $response = curl_exec($ch);
    curl_close($ch);

    if (!$response) {
        return ['error' => 'Failed to fetch content'];
    }

    $items = json_decode($response, true);
    if (!is_array($items)) {
        return ['error' => 'Invalid response'];
    }

    return [
        'total' => count($items),
        'items' => array_slice($items, 0, $limit)
    ];
}

/**
 * Import series directly from URL (without saving server)
 */
function importSeriesFromUrl(string $server, string $username, string $password, string $serverName, ?string $categoryId): array
{
    $db = getDB();
    $logger = new Logger('url_import');

    $logger->info("Starting URL series import", ['server' => $server]);

    // First, add or find the server
    $stmt = $db->prepare("SELECT id FROM servers WHERE url = ? AND username = ?");
    $stmt->execute([rtrim($server, '/'), $username]);
    $existingServer = $stmt->fetch();

    if ($existingServer) {
        $serverId = $existingServer['id'];
    } else {
        // Add new server
        $stmt = $db->prepare("
            INSERT INTO servers (name, url, username, password, type, is_active, status)
            VALUES (?, ?, ?, ?, 'xtream', 1, 'online')
        ");
        $stmt->execute([$serverName, rtrim($server, '/'), $username, $password]);
        $serverId = (int) $db->lastInsertId();
    }

    // Build server array for import function
    $serverData = [
        'url' => rtrim($server, '/'),
        'username' => $username,
        'password' => $password
    ];

    // Fetch series
    $apiUrl = rtrim($server, '/') . "/player_api.php?username={$username}&password={$password}&action=get_series";
    if ($categoryId) {
        $apiUrl .= "&category_id={$categoryId}";
    }

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 120,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_USERAGENT => 'TiviMate/5.1.6 (Android 7.1.2)'
    ]);
    $response = curl_exec($ch);
    curl_close($ch);

    if (!$response) {
        return ['error' => 'Failed to fetch series'];
    }

    $seriesList = json_decode($response, true);
    if (!is_array($seriesList)) {
        return ['error' => 'Invalid response'];
    }

    $imported = 0;
    $totalEpisodes = 0;

    foreach ($seriesList as $s) {
        try {
            $externalId = (int) $s['series_id'];
            $title = $s['name'] ?? 'Unknown';
            $cleanName = preg_replace('/[^a-zA-Z0-9\s]/', '', strtolower($title));
            $cleanName = preg_replace('/\s+/', ' ', trim($cleanName));

            // Check if exists
            $stmt = $db->prepare("SELECT id FROM series WHERE tmdb_id = ?");
            $stmt->execute([$externalId]);
            if ($stmt->fetch()) continue;

            $year = isset($s['year']) && is_numeric($s['year']) ? (int) $s['year'] : null;
            $seriesCategoryId = isset($s['category_id']) ? (int) $s['category_id'] : null;

            $stmt = $db->prepare("
                INSERT INTO series (tmdb_id, title, clean_name, year, plot, genre, cast, rating, poster, backdrop, category_id)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $externalId,
                $title,
                $cleanName,
                $year,
                $s['plot'] ?? null,
                $s['genre'] ?? null,
                $s['cast'] ?? null,
                isset($s['rating']) ? (float) $s['rating'] : null,
                $s['cover'] ?? null,
                $s['backdrop_path'] ?? null,
                $seriesCategoryId
            ]);
            $localSeriesId = (int) $db->lastInsertId();
            $imported++;

            // Import episodes
            $epResult = importSeriesEpisodes($serverId, $externalId, $localSeriesId, $serverData);
            $totalEpisodes += $epResult['episodes_imported'] ?? 0;

        } catch (Exception $e) {
            $logger->error("Error importing series from URL", ['error' => $e->getMessage()]);
        }
    }

    return [
        'success' => true,
        'imported' => $imported,
        'episodes' => $totalEpisodes,
        'total' => count($seriesList)
    ];
}

/**
 * Import series from Xtream server
 */
function importSeriesFromServer(int $serverId, ?int $categoryId = null, bool $force = false): array
{
    $db = getDB();
    $sm = new ServerManager();
    $logger = new Logger('series_import');

    $server = $sm->getServer($serverId);
    if (!$server) {
        return ['error' => 'Server not found'];
    }

    $logger->info("Starting series import", ['server_id' => $serverId, 'category_id' => $categoryId]);

    // Build API URL with port
    $baseUrl = rtrim($server['url'], '/');
    // Check if port needs to be added to URL
    $port = $server['port'] ?? 80;
    if ($port && $port != 80 && !preg_match('/:\d+$/', $baseUrl)) {
        $baseUrl = preg_replace('/^(https?:\/\/[^\/]+)/', '$1:' . $port, $baseUrl);
    }
    $apiUrl = "{$baseUrl}/player_api.php?username={$server['username']}&password={$server['password']}&action=get_series";
    if ($categoryId) {
        $apiUrl .= "&category_id={$categoryId}";
    }

    // Fetch series list
    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 60,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_USERAGENT => 'TiviMate/5.1.6 (Android 7.1.2)'
    ]);
    $response = curl_exec($ch);
    $error = curl_error($ch);
    curl_close($ch);

    if (!$response) {
        $logger->error("Failed to fetch series", ['error' => $error]);
        return ['error' => 'Failed to fetch series from server: ' . $error];
    }

    $seriesList = json_decode($response, true);
    if (!is_array($seriesList)) {
        return ['error' => 'Invalid response from server'];
    }

    $imported = 0;
    $updated = 0;
    $errors = 0;

    foreach ($seriesList as $s) {
        try {
            $seriesId = (int) $s['series_id'];
            $title = $s['name'] ?? 'Unknown';
            $cleanName = preg_replace('/[^a-zA-Z0-9\s]/', '', strtolower($title));
            $cleanName = preg_replace('/\s+/', ' ', trim($cleanName));

            // Check if series exists
            $stmt = $db->prepare("SELECT id FROM series WHERE tmdb_id = ? OR (title = ? AND year = ?)");
            $year = isset($s['year']) && is_numeric($s['year']) ? (int) $s['year'] : null;
            $stmt->execute([$seriesId, $title, $year]);
            $existing = $stmt->fetch();

            if ($existing && !$force) {
                continue;
            }

            $seriesCategoryId = isset($s['category_id']) ? (int) $s['category_id'] : null;

            $data = [
                'tmdb_id' => $seriesId,
                'title' => $title,
                'clean_name' => $cleanName,
                'year' => $year,
                'plot' => $s['plot'] ?? null,
                'genre' => $s['genre'] ?? null,
                'cast' => $s['cast'] ?? null,
                'rating' => isset($s['rating']) ? (float) $s['rating'] : null,
                'poster' => $s['cover'] ?? null,
                'backdrop' => $s['backdrop_path'] ?? null
            ];

            if ($existing) {
                // Update
                $stmt = $db->prepare("
                    UPDATE series SET title = :title, clean_name = :clean_name, year = :year,
                    plot = :plot, genre = :genre, cast = :cast, rating = :rating,
                    poster = :poster, backdrop = :backdrop, updated_at = NOW()
                    WHERE id = :id
                ");
                $data['id'] = $existing['id'];
                $stmt->execute($data);
                $localSeriesId = $existing['id'];
                $updated++;
            } else {
                // Insert
                $stmt = $db->prepare("
                    INSERT INTO series (tmdb_id, title, clean_name, year, plot, genre, cast, rating, poster, backdrop)
                    VALUES (:tmdb_id, :title, :clean_name, :year, :plot, :genre, :cast, :rating, :poster, :backdrop)
                ");
                $stmt->execute($data);
                $localSeriesId = (int) $db->lastInsertId();
                $imported++;
            }

            // Import episodes for this series
            $episodesResult = importSeriesEpisodes($serverId, $seriesId, $localSeriesId, $server);

        } catch (Exception $e) {
            $logger->error("Error importing series", ['title' => $title, 'error' => $e->getMessage()]);
            $errors++;
        }
    }

    $logger->info("Series import completed", ['imported' => $imported, 'updated' => $updated, 'errors' => $errors]);

    return [
        'success' => true,
        'imported' => $imported,
        'updated' => $updated,
        'errors' => $errors,
        'total' => count($seriesList)
    ];
}

/**
 * Import episodes for a series
 */
function importSeriesEpisodes(int $serverId, int $externalSeriesId, int $localSeriesId, array $server): array
{
    $db = getDB();
    $logger = new Logger('series_import');

    // Fetch series info with episodes
    $baseUrl = rtrim($server['url'], '/');
    // Check if port needs to be added to URL
    $port = $server['port'] ?? 80;
    if ($port && $port != 80 && !preg_match('/:\d+$/', $baseUrl)) {
        $baseUrl = preg_replace('/^(https?:\/\/[^\/]+)/', '$1:' . $port, $baseUrl);
    }
    $apiUrl = "{$baseUrl}/player_api.php?username={$server['username']}&password={$server['password']}&action=get_series_info&series_id={$externalSeriesId}";

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_USERAGENT => 'TiviMate/5.1.6 (Android 7.1.2)'
    ]);
    $response = curl_exec($ch);
    curl_close($ch);

    if (!$response) {
        return ['error' => 'Failed to fetch episodes'];
    }

    $seriesInfo = json_decode($response, true);
    if (!is_array($seriesInfo) || !isset($seriesInfo['episodes'])) {
        return ['error' => 'No episodes found'];
    }

    $imported = 0;
    $seasonsCount = 0;

    foreach ($seriesInfo['episodes'] as $seasonNum => $episodes) {
        $seasonsCount++;
        foreach ($episodes as $ep) {
            try {
                $streamId = (int) ($ep['id'] ?? 0);
                $episodeNum = (int) ($ep['episode_num'] ?? 0);

                // Check if episode exists
                $stmt = $db->prepare("SELECT id FROM episodes WHERE series_id = ? AND stream_id = ?");
                $stmt->execute([$localSeriesId, $streamId]);

                if ($stmt->fetch()) {
                    continue;
                }

                $title = $ep['title'] ?? "Episode {$episodeNum}";
                $cleanName = preg_replace('/[^a-zA-Z0-9\s]/', '', strtolower($title));

                // Build stream URL
                $extension = $ep['container_extension'] ?? 'mp4';
                $streamUrl = "{$baseUrl}/series/{$server['username']}/{$server['password']}/{$streamId}.{$extension}";

                $stmt = $db->prepare("
                    INSERT INTO episodes (series_id, server_id, stream_id, season_number, episode_number,
                        title, clean_name, description, duration, stream_url, container_extension, poster, air_date)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");

                $stmt->execute([
                    $localSeriesId,
                    $serverId,
                    $streamId,
                    (int) $seasonNum,
                    $episodeNum,
                    $title,
                    $cleanName,
                    $ep['info']['plot'] ?? null,
                    isset($ep['info']['duration']) ? (int) $ep['info']['duration'] : null,
                    $streamUrl,
                    $extension,
                    $ep['info']['movie_image'] ?? null,
                    $ep['info']['air_date'] ?? null
                ]);

                $imported++;

            } catch (Exception $e) {
                $logger->error("Error importing episode", ['error' => $e->getMessage()]);
            }
        }
    }

    // Update series episode count
    $stmt = $db->prepare("UPDATE series SET seasons_count = ?, episodes_count = ? WHERE id = ?");
    $stmt->execute([$seasonsCount, $imported, $localSeriesId]);

    return ['success' => true, 'episodes_imported' => $imported, 'seasons' => $seasonsCount];
}

/**
 * Smart sync series - check for new series only
 */
function smartSyncSeries(int $serverId): array
{
    $db = getDB();
    $sm = new ServerManager();

    $server = $sm->getServer($serverId);
    if (!$server) {
        return ['error' => 'Server not found'];
    }

    // Get existing series IDs
    $stmt = $db->query("SELECT tmdb_id FROM series WHERE tmdb_id IS NOT NULL");
    $existingIds = $stmt->fetchAll(PDO::FETCH_COLUMN);

    // Fetch series from server
    $baseUrl = rtrim($server['url'], '/');
    // Check if port needs to be added to URL
    $port = $server['port'] ?? 80;
    if ($port && $port != 80 && !preg_match('/:\d+$/', $baseUrl)) {
        $baseUrl = preg_replace('/^(https?:\/\/[^\/]+)/', '$1:' . $port, $baseUrl);
    }
    $apiUrl = "{$baseUrl}/player_api.php?username={$server['username']}&password={$server['password']}&action=get_series";

    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 60,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_USERAGENT => 'TiviMate/5.1.6 (Android 7.1.2)'
    ]);
    $response = curl_exec($ch);
    curl_close($ch);

    if (!$response) {
        return ['error' => 'Failed to fetch series'];
    }

    $seriesList = json_decode($response, true);
    if (!is_array($seriesList)) {
        return ['error' => 'Invalid response'];
    }

    $newSeries = [];
    foreach ($seriesList as $s) {
        $seriesId = (int) ($s['series_id'] ?? 0);
        if (!in_array($seriesId, $existingIds)) {
            $newSeries[] = [
                'series_id' => $seriesId,
                'name' => $s['name'] ?? 'Unknown',
                'cover' => $s['cover'] ?? null
            ];
        }
    }

    return [
        'success' => true,
        'new_found' => count($newSeries),
        'new_series' => array_slice($newSeries, 0, 50),
        'total_on_server' => count($seriesList)
    ];
}

/**
 * Series endpoint
 */
function handleSeries(string $method, ?string $action, ?int $id, array $input, array $params): array
{
    $db = getDB();

    if ($method === 'GET') {
        // Get years for filter
        if ($action === 'years') {
            $stmt = $db->query("SELECT DISTINCT year FROM series WHERE year IS NOT NULL ORDER BY year DESC");
            return $stmt->fetchAll(PDO::FETCH_COLUMN);
        }

        // Get single series with episodes
        if ($id) {
            $stmt = $db->prepare("SELECT * FROM series WHERE id = ?");
            $stmt->execute([$id]);
            $series = $stmt->fetch();

            if ($series) {
                // Get episodes grouped by season
                $stmt = $db->prepare("
                    SELECT * FROM episodes
                    WHERE series_id = ?
                    ORDER BY season_number ASC, episode_number ASC
                ");
                $stmt->execute([$id]);
                $episodes = $stmt->fetchAll();

                $series['episodes'] = $episodes;
                $series['seasons'] = [];

                foreach ($episodes as $ep) {
                    $season = $ep['season_number'];
                    if (!isset($series['seasons'][$season])) {
                        $series['seasons'][$season] = [];
                    }
                    $series['seasons'][$season][] = $ep;
                }
            }

            return $series ?: ['error' => 'Series not found'];
        }

        // Search
        if ($action === 'search' || !empty($params['search'])) {
            $query = $params['q'] ?? $params['search'] ?? '';
            $stmt = $db->prepare("
                SELECT s.*, COUNT(e.id) as episode_count
                FROM series s
                LEFT JOIN episodes e ON s.id = e.series_id
                WHERE s.title LIKE ?
                GROUP BY s.id
                ORDER BY s.year DESC, s.title ASC
                LIMIT 100
            ");
            $stmt->execute(["%{$query}%"]);
            return $stmt->fetchAll();
        }

        // Newest
        if ($action === 'newest') {
            $limit = min((int) ($params['limit'] ?? 10), 50);
            $stmt = $db->prepare("
                SELECT s.*, COUNT(e.id) as episode_count
                FROM series s
                LEFT JOIN episodes e ON s.id = e.series_id AND e.stream_url IS NOT NULL AND e.stream_url != ''
                GROUP BY s.id
                ORDER BY s.created_at DESC
                LIMIT ?
            ");
            $stmt->execute([$limit]);
            return $stmt->fetchAll();
        }

        // List with filters
        $limit = min((int) ($params['limit'] ?? 24), 100);
        $offset = (int) ($params['offset'] ?? 0);
        $categoryId = $params['category_id'] ?? null;
        $year = $params['year'] ?? null;
        $search = $params['search'] ?? null;

        $where = [];
        $bindings = [];

        if ($categoryId) {
            $where[] = "s.category_id = ?";
            $bindings[] = $categoryId;
        }

        if ($year) {
            $where[] = "s.year = ?";
            $bindings[] = $year;
        }

        if ($search) {
            $where[] = "s.title LIKE ?";
            $bindings[] = "%{$search}%";
        }

        $whereClause = $where ? 'WHERE ' . implode(' AND ', $where) : '';

        // Show series with episodes first, then by creation date
        $sql = "
            SELECT s.*, COUNT(e.id) as episode_count
            FROM series s
            LEFT JOIN episodes e ON s.id = e.series_id
            {$whereClause}
            GROUP BY s.id
            ORDER BY s.episodes_count DESC, s.created_at DESC
            LIMIT ? OFFSET ?
        ";

        $bindings[] = $limit;
        $bindings[] = $offset;

        $stmt = $db->prepare($sql);
        $stmt->execute($bindings);
        return $stmt->fetchAll();
    }

    return ['error' => 'Invalid request'];
}

/**
 * Episodes endpoint
 */
function handleEpisodes(string $method, ?string $action, ?int $id, array $input, array $params): array
{
    $db = getDB();

    if ($method === 'GET') {
        // Get episodes for a series
        $seriesId = $params['series_id'] ?? null;
        $season = $params['season'] ?? null;

        if ($seriesId) {
            $sql = "SELECT * FROM episodes WHERE series_id = ?";
            $bindings = [$seriesId];

            if ($season) {
                $sql .= " AND season_number = ?";
                $bindings[] = $season;
            }

            $sql .= " ORDER BY season_number ASC, episode_number ASC";

            $stmt = $db->prepare($sql);
            $stmt->execute($bindings);
            return $stmt->fetchAll();
        }

        // Get single episode
        if ($id) {
            $stmt = $db->prepare("SELECT * FROM episodes WHERE id = ?");
            $stmt->execute([$id]);
            return $stmt->fetch() ?: ['error' => 'Episode not found'];
        }

        return ['error' => 'series_id parameter required'];
    }

    return ['error' => 'Invalid request'];
}

/**
 * Categories endpoint
 */
function handleCategories(string $method, ?string $action, ?int $id, array $params): array
{
    $db = getDB();
    $sm = new ServerManager();

    if ($method === 'GET') {
        $serverId = (int) ($params['server_id'] ?? 0);
        $type = $params['type'] ?? null;

        if ($serverId) {
            return $sm->getCategories($serverId, $type);
        }

        $sql = "SELECT c.*, s.name as server_name FROM categories c JOIN servers s ON c.server_id = s.id";
        if ($type) {
            $sql .= " WHERE c.type = ?";
            $stmt = $db->prepare($sql . " ORDER BY c.name");
            $stmt->execute([$type]);
        } else {
            $stmt = $db->query($sql . " ORDER BY c.type, c.name");
        }

        return $stmt->fetchAll();
    }

    return ['error' => 'Invalid request'];
}

/**
 * FolderWatch endpoint
 */
function handleFolderWatch(string $method, ?string $action, array $input): array
{
    $fw = new FolderWatch();

    switch ($action) {
        case 'scan':
            return $fw->scanWatchFolder();

        case 'process':
            return $fw->processWatchFolder();

        case 'validate':
            $filename = $input['filename'] ?? '';
            if (empty($filename)) {
                return ['error' => 'filename required'];
            }
            return $fw->validateFilename($filename);

        case 'suggest':
            $filename = $input['filename'] ?? '';
            if (empty($filename)) {
                return ['error' => 'filename required'];
            }
            return [
                'original' => $filename,
                'suggestion' => $fw->suggestFilename($filename)
            ];

        case 'orphans':
            if ($method === 'GET') {
                return $fw->findOrphanEpisodes();
            }
            if ($method === 'DELETE') {
                $episodeIds = $input['episode_ids'] ?? null;
                return ['deleted' => $fw->deleteOrphanEpisodes($episodeIds)];
            }
            break;

        case 'link-orphan':
            $episodeId = (int) ($input['episode_id'] ?? 0);
            $seriesId = (int) ($input['series_id'] ?? 0);

            if (!$episodeId || !$seriesId) {
                return ['error' => 'episode_id and series_id required'];
            }

            return ['success' => $fw->linkOrphanToSeries($episodeId, $seriesId)];

        case 'structure':
            return $fw->getFolderStructure();

        case 'rename-script':
            $files = $input['files'] ?? [];
            return ['script' => $fw->generateRenameScript($files)];

        default:
            return ['error' => 'Unknown action', 'available' => [
                'scan', 'process', 'validate', 'suggest', 'orphans', 'link-orphan', 'structure', 'rename-script'
            ]];
    }

    return ['error' => 'Invalid request'];
}

/**
 * Logs endpoint
 */
function handleLogs(string $method, ?string $action, array $params): array
{
    $db = getDB();
    $logger = new Logger('api', $db);

    // Clear all logs
    if ($method === 'DELETE' && $action === 'clear') {
        try {
            // Delete from import_logs (the actual log table used by Logger)
            $db->query("DELETE FROM import_logs");
            $db->query("ALTER TABLE import_logs AUTO_INCREMENT = 1");
            return ['success' => true, 'message' => 'All logs cleared'];
        } catch (PDOException $e) {
            return ['error' => 'Failed to clear logs: ' . $e->getMessage()];
        }
    }

    // Clear logs by status
    if ($method === 'DELETE' && in_array($action, ['success', 'warning', 'error', 'info'])) {
        $stmt = $db->prepare("DELETE FROM import_logs WHERE status = ?");
        $stmt->execute([$action]);
        return ['success' => true, 'message' => "All {$action} logs cleared", 'deleted' => $stmt->rowCount()];
    }

    // Clear old logs (older than X days)
    if ($method === 'DELETE' && $action === 'old') {
        $days = (int)($params['days'] ?? 7);
        $stmt = $db->prepare("DELETE FROM import_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)");
        $stmt->execute([$days]);
        return ['success' => true, 'message' => "Logs older than {$days} days cleared", 'deleted' => $stmt->rowCount()];
    }

    // Get log statistics
    if ($action === 'stats') {
        $stats = [];
        $stmt = $db->query("SELECT status, COUNT(*) as count FROM import_logs GROUP BY status");
        while ($row = $stmt->fetch()) {
            $stats[$row['status']] = (int)$row['count'];
        }
        $stmt = $db->query("SELECT COUNT(*) as total FROM import_logs");
        $stats['total'] = (int)$stmt->fetch()['total'];
        return $stats;
    }

    $filters = [
        'status' => $params['status'] ?? null,
        'type' => $params['type'] ?? null,
        'action' => $params['action'] ?? null,
        'server_id' => $params['server_id'] ?? null,
        'date_from' => $params['date_from'] ?? null,
        'date_to' => $params['date_to'] ?? null,
    ];

    $limit = min((int) ($params['limit'] ?? 100), 1000);
    $offset = (int) ($params['offset'] ?? 0);

    return $logger->getLogsFromDatabase(array_filter($filters), $limit, $offset);
}

/**
 * Settings endpoint
 */
function handleSettings(string $method, array $input): array
{
    if ($method === 'GET') {
        $db = getDB();
        $stmt = $db->query("SELECT * FROM settings ORDER BY `key`");
        $settings = [];

        while ($row = $stmt->fetch()) {
            $value = $row['value'];
            switch ($row['type']) {
                case 'int':
                    $value = (int) $value;
                    break;
                case 'bool':
                    $value = (bool) $value;
                    break;
                case 'json':
                    $value = json_decode($value, true);
                    break;
            }
            $settings[$row['key']] = [
                'value' => $value,
                'type' => $row['type'],
                'description' => $row['description']
            ];
        }

        return $settings;
    }

    if ($method === 'POST' || $method === 'PUT') {
        $updated = 0;

        foreach ($input as $key => $data) {
            $value = is_array($data) ? ($data['value'] ?? $data) : $data;
            $type = is_array($data) ? ($data['type'] ?? 'string') : 'string';

            if (setSetting($key, $value, $type)) {
                $updated++;
            }
        }

        return ['success' => true, 'updated' => $updated];
    }

    return ['error' => 'Invalid request'];
}

/**
 * Remote Servers endpoint
 */
function handleRemoteServers(string $method, ?string $action, ?int $id, array $input): array
{
    $rd = new RemoteDownloader();

    if ($method === 'GET') {
        if ($action === 'test' && $id) {
            return $rd->testConnection($id);
        }
        if ($action === 'storage' && $id) {
            return $rd->getServerStorage($id);
        }
        if ($action === 'stats' && $id) {
            return $rd->getServerStats($id);
        }
        if ($id) {
            $server = $rd->getServer($id);
            return $server ?: ['error' => 'Server not found'];
        }
        return $rd->getAllServers();
    }

    if ($method === 'POST') {
        if ($action === 'add') {
            $serverId = $rd->addServer($input);
            if ($serverId) {
                return ['success' => true, 'server_id' => $serverId];
            }
            return ['error' => 'Failed to add remote server'];
        }
        if ($action === 'test' && $id) {
            return $rd->testConnection($id);
        }
        if ($action === 'download') {
            $remoteServerId = (int) ($input['remote_server_id'] ?? 0);
            $sourceUrl = $input['source_url'] ?? '';
            $filename = $input['filename'] ?? '';
            $movieId = isset($input['movie_id']) ? (int) $input['movie_id'] : null;
            $episodeId = isset($input['episode_id']) ? (int) $input['episode_id'] : null;

            if (!$remoteServerId || !$sourceUrl || !$filename) {
                return ['error' => 'remote_server_id, source_url, and filename are required'];
            }

            return $rd->downloadToRemote($remoteServerId, $sourceUrl, $filename, $movieId, $episodeId);
        }
        if ($action === 'bulk-download') {
            $remoteServerId = (int) ($input['remote_server_id'] ?? 0);
            $items = $input['items'] ?? [];

            if (!$remoteServerId || empty($items)) {
                return ['error' => 'remote_server_id and items are required'];
            }

            return $rd->bulkDownload($remoteServerId, $items);
        }

        if ($action === 'browse-folders' && $id) {
            $path = $input['path'] ?? '/';
            // Always start from root if path is empty
            if ($path === '') {
                $path = '/';
            }
            return $rd->listRemoteFolders($id, $path);
        }

        if ($action === 'create-folder' && $id) {
            $path = $input['path'] ?? '';
            if (empty($path)) {
                return ['error' => 'path is required'];
            }
            return $rd->createRemoteFolder($id, $path);
        }
    }

    if ($method === 'PUT' && $id) {
        return ['success' => $rd->updateServer($id, $input)];
    }

    if ($method === 'DELETE' && $id) {
        return ['success' => $rd->deleteServer($id)];
    }

    return ['error' => 'Invalid remote server request'];
}

/**
 * Download Jobs endpoint
 */
function handleDownloadJobs(string $method, ?string $action, ?int $id, array $input, array $params): array
{
    $rd = new RemoteDownloader();
    $db = getDB();

    if ($method === 'GET') {
        if ($action === 'stats') {
            $stmt = $db->query("
                SELECT status, COUNT(*) as count
                FROM download_jobs
                GROUP BY status
            ");
            $stats = ['pending' => 0, 'downloading' => 0, 'completed' => 0, 'failed' => 0];
            while ($row = $stmt->fetch()) {
                $stats[$row['status']] = (int) $row['count'];
            }
            return $stats;
        }

        $serverId = isset($params['server_id']) ? (int) $params['server_id'] : null;
        $status = $params['status'] ?? null;
        $limit = min((int) ($params['limit'] ?? 50), 500);

        return $rd->getJobs($serverId, $status, $limit);
    }

    if ($method === 'POST') {
        if ($action === 'clear-completed') {
            $stmt = $db->prepare("DELETE FROM download_jobs WHERE status = 'completed'");
            $stmt->execute();
            return ['success' => true, 'deleted' => $stmt->rowCount()];
        }

        if ($action === 'clear-failed') {
            $stmt = $db->prepare("DELETE FROM download_jobs WHERE status = 'failed'");
            $stmt->execute();
            return ['success' => true, 'deleted' => $stmt->rowCount()];
        }

        if ($action === 'clear-all') {
            $stmt = $db->prepare("DELETE FROM download_jobs");
            $stmt->execute();
            return ['success' => true, 'deleted' => $stmt->rowCount()];
        }

        if ($action === 'retry' && $id) {
            $stmt = $db->prepare("UPDATE download_jobs SET status = 'pending', error_message = NULL, started_at = NULL, completed_at = NULL WHERE id = ?");
            $stmt->execute([$id]);
            return ['success' => true];
        }

        if ($action === 'cancel' && $id) {
            $stmt = $db->prepare("UPDATE download_jobs SET status = 'cancelled' WHERE id = ? AND status IN ('pending', 'downloading')");
            $stmt->execute([$id]);
            return ['success' => true];
        }
    }

    if ($method === 'DELETE' && $id) {
        $stmt = $db->prepare("DELETE FROM download_jobs WHERE id = ?");
        $stmt->execute([$id]);
        return ['success' => true];
    }

    return ['error' => 'Invalid download jobs request'];
}

/**
 * Users endpoint
 */
function handleUsers(string $method, ?string $action, ?int $id, array $input): array
{
    $auth = new Auth();
    $db = getDB();

    if ($method === 'GET') {
        if ($id) {
            $stmt = $db->prepare("SELECT id, username, email, role, is_active, last_login, login_ip, created_at FROM users WHERE id = ?");
            $stmt->execute([$id]);
            $user = $stmt->fetch();
            return $user ?: ['error' => 'User not found'];
        }
        return $auth->getAllUsers();
    }

    if ($method === 'POST') {
        if ($action === 'add') {
            $username = $input['username'] ?? '';
            $password = $input['password'] ?? '';
            $email = $input['email'] ?? '';
            $role = $input['role'] ?? 'user';

            if (empty($username) || empty($password)) {
                return ['error' => 'username and password are required'];
            }

            $userId = $auth->createUser($username, $password, $role, $email);
            if ($userId) {
                return ['success' => true, 'user_id' => $userId];
            }
            return ['error' => 'Failed to create user (username may already exist)'];
        }
        if ($action === 'toggle' && $id) {
            return ['success' => $auth->toggleUser($id)];
        }
        if ($action === 'change-password' && $id) {
            $newPassword = $input['password'] ?? '';
            if (empty($newPassword)) {
                return ['error' => 'password is required'];
            }
            return ['success' => $auth->updatePassword($id, $newPassword)];
        }
        // Generate API token
        if ($action === 'token' && $id) {
            $token = $auth->generateApiToken($id);
            if ($token) {
                return ['success' => true, 'token' => $token];
            }
            return ['error' => 'Failed to generate API token'];
        }
    }

    // Update user (PUT)
    if ($method === 'PUT' && $id) {
        // Reset password
        if ($action === 'password') {
            $newPassword = $input['password'] ?? '';
            if (empty($newPassword) || strlen($newPassword) < 6) {
                return ['error' => 'Password must be at least 6 characters'];
            }
            return ['success' => $auth->updatePassword($id, $newPassword)];
        }

        // Update user details
        $username = $input['username'] ?? null;
        $email = $input['email'] ?? null;
        $role = $input['role'] ?? null;

        $updates = [];
        $params = [];

        if ($username !== null) {
            $updates[] = "username = ?";
            $params[] = $username;
        }
        if ($email !== null) {
            $updates[] = "email = ?";
            $params[] = $email;
        }
        if ($role !== null) {
            // Check if trying to demote last admin
            if ($role !== 'admin') {
                $stmt = $db->prepare("SELECT role FROM users WHERE id = ?");
                $stmt->execute([$id]);
                $currentUser = $stmt->fetch();
                if ($currentUser && $currentUser['role'] === 'admin') {
                    $stmt = $db->query("SELECT COUNT(*) FROM users WHERE role = 'admin' AND is_active = 1");
                    $adminCount = $stmt->fetchColumn();
                    if ($adminCount <= 1) {
                        return ['error' => 'Cannot demote the last admin user'];
                    }
                }
            }
            $updates[] = "role = ?";
            $params[] = $role;
        }

        if (empty($updates)) {
            return ['error' => 'No fields to update'];
        }

        $updates[] = "updated_at = NOW()";
        $params[] = $id;

        try {
            $stmt = $db->prepare("UPDATE users SET " . implode(', ', $updates) . " WHERE id = ?");
            $stmt->execute($params);
            return ['success' => true];
        } catch (PDOException $e) {
            return ['error' => 'Failed to update user: ' . $e->getMessage()];
        }
    }

    if ($method === 'DELETE' && $id) {
        $result = $auth->deleteUser($id);
        if (!$result) {
            return ['error' => 'Cannot delete the last admin user'];
        }
        return ['success' => true];
    }

    return ['error' => 'Invalid users request'];
}

/**
 * Auto Download endpoint
 */
function handleAutoDownload(string $method, ?string $action, array $input): array
{
    if ($method !== 'POST') {
        return ['error' => 'Method not allowed'];
    }

    if ($action === 'run') {
        $db = getDB();
        $rd = new RemoteDownloader();
        
        $serverId = (int) ($input['server_id'] ?? 0);
        if (!$serverId) {
            return ['error' => 'Server ID required'];
        }

        $moviesQueued = 0;
        $episodesQueued = 0;

        // Get movies that haven't been downloaded yet
        $stmt = $db->query("
            SELECT m.id, m.title, ml.url, ml.container_extension
            FROM movies m
            INNER JOIN movie_links ml ON m.id = ml.movie_id
            WHERE m.is_downloaded = 0
            ORDER BY m.created_at DESC
            LIMIT 50
        ");
        $movies = $stmt->fetchAll();

        foreach ($movies as $movie) {
            $ext = $movie['container_extension'] ?: 'mp4';
            $filename = preg_replace('/[^a-zA-Z0-9\-_\.\s]/', '', $movie['title']) . '.' . $ext;
            
            $result = $rd->downloadToRemote($serverId, $movie['url'], $filename, $movie['id'], null);
            if ($result['success'] ?? false) {
                $moviesQueued++;
            }
        }

        // Get episodes that haven't been downloaded yet
        $stmt = $db->query("
            SELECT e.id, e.title, s.title as series_title, e.season_num, e.episode_num, el.url, el.container_extension
            FROM episodes e
            INNER JOIN series s ON e.series_id = s.id
            INNER JOIN episode_links el ON e.id = el.episode_id
            WHERE e.is_downloaded = 0
            ORDER BY e.created_at DESC
            LIMIT 100
        ");
        $episodes = $stmt->fetchAll();

        foreach ($episodes as $ep) {
            $ext = $ep['container_extension'] ?: 'mp4';
            $seriesName = preg_replace('/[^a-zA-Z0-9\-_\.\s]/', '', $ep['series_title']);
            $filename = sprintf('%s/S%02dE%02d.%s', $seriesName, $ep['season_num'], $ep['episode_num'], $ext);
            
            $result = $rd->downloadToRemote($serverId, $ep['url'], $filename, null, $ep['id']);
            if ($result['success'] ?? false) {
                $episodesQueued++;
            }
        }

        return [
            'success' => true,
            'movies_queued' => $moviesQueued,
            'episodes_queued' => $episodesQueued
        ];
    }

    return ['error' => 'Invalid action'];
}
