<?php
/**
 * Xtream Codes API Handler
 * Full XC API integration with caching and error handling
 */

class XtreamAPI
{
    private string $baseUrl;
    private string $username;
    private string $password;
    private int $port;
    private bool $useHttps;
    private int $timeout;
    private ?array $userInfo = null;
    private ?array $serverInfo = null;
    private array $errors = [];

    // API Actions
    private const ACTIONS = [
        'auth' => '',
        'live_categories' => 'get_live_categories',
        'vod_categories' => 'get_vod_categories',
        'series_categories' => 'get_series_categories',
        'live_streams' => 'get_live_streams',
        'vod_streams' => 'get_vod_streams',
        'series' => 'get_series',
        'series_info' => 'get_series_info',
        'vod_info' => 'get_vod_info',
        'epg' => 'get_simple_data_table',
        'short_epg' => 'get_short_epg',
    ];

    public function __construct(
        string $url,
        string $username,
        string $password,
        int $port = 80,
        bool $useHttps = false,
        int $timeout = 30
    ) {
        // Clean URL
        $this->baseUrl = rtrim(preg_replace('/^https?:\/\//', '', $url), '/');
        $this->username = $username;
        $this->password = $password;
        $this->port = $port;
        $this->useHttps = $useHttps;
        $this->timeout = $timeout;
    }

    /**
     * Build full API URL
     */
    private function buildUrl(string $endpoint = 'player_api.php', array $params = []): string
    {
        $protocol = $this->useHttps ? 'https' : 'http';
        $url = "{$protocol}://{$this->baseUrl}:{$this->port}/{$endpoint}";

        $params['username'] = $this->username;
        $params['password'] = $this->password;

        return $url . '?' . http_build_query($params);
    }

    /**
     * Make HTTP request
     */
    private function request(string $url): ?array
    {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => $this->timeout,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_USERAGENT => 'TiviMate/5.1.6 (Android 7.1.2)',
            CURLOPT_HTTPHEADER => [
                'Accept: application/json',
                'Connection: keep-alive'
            ]
        ]);

        $response = curl_exec($ch);
        $error = curl_error($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $responseTime = curl_getinfo($ch, CURLINFO_TOTAL_TIME);
        curl_close($ch);

        if ($response === false) {
            $this->errors[] = [
                'url' => $url,
                'error' => $error,
                'http_code' => $httpCode
            ];
            return null;
        }

        if ($httpCode !== 200) {
            $this->errors[] = [
                'url' => $url,
                'error' => "HTTP Error: $httpCode",
                'http_code' => $httpCode
            ];
            return null;
        }

        $data = json_decode($response, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->errors[] = [
                'url' => $url,
                'error' => 'Invalid JSON response',
                'response' => substr($response, 0, 500)
            ];
            return null;
        }

        // Add response metadata
        $data['_meta'] = [
            'response_time' => $responseTime,
            'http_code' => $httpCode
        ];

        return $data;
    }

    /**
     * Authenticate and get user/server info
     */
    public function authenticate(): bool
    {
        $url = $this->buildUrl('player_api.php');
        $data = $this->request($url);

        if ($data === null) {
            return false;
        }

        if (!isset($data['user_info']) || ($data['user_info']['auth'] ?? 0) != 1) {
            $this->errors[] = [
                'error' => 'Authentication failed',
                'message' => $data['user_info']['message'] ?? 'Unknown error'
            ];
            return false;
        }

        $this->userInfo = $data['user_info'];
        $this->serverInfo = $data['server_info'] ?? [];

        return true;
    }

    /**
     * Check if authenticated
     */
    public function isAuthenticated(): bool
    {
        if ($this->userInfo === null) {
            return $this->authenticate();
        }
        return true;
    }

    /**
     * Get user info
     */
    public function getUserInfo(): ?array
    {
        $this->isAuthenticated();
        return $this->userInfo;
    }

    /**
     * Get server info
     */
    public function getServerInfo(): ?array
    {
        $this->isAuthenticated();
        return $this->serverInfo;
    }

    /**
     * Get live categories
     */
    public function getLiveCategories(): array
    {
        $url = $this->buildUrl('player_api.php', ['action' => self::ACTIONS['live_categories']]);
        return $this->request($url) ?? [];
    }

    /**
     * Get VOD categories
     */
    public function getVODCategories(): array
    {
        $url = $this->buildUrl('player_api.php', ['action' => self::ACTIONS['vod_categories']]);
        return $this->request($url) ?? [];
    }

    /**
     * Get series categories
     */
    public function getSeriesCategories(): array
    {
        $url = $this->buildUrl('player_api.php', ['action' => self::ACTIONS['series_categories']]);
        return $this->request($url) ?? [];
    }

    /**
     * Get live streams (optionally by category)
     */
    public function getLiveStreams(?int $categoryId = null): array
    {
        $params = ['action' => self::ACTIONS['live_streams']];
        if ($categoryId !== null) {
            $params['category_id'] = $categoryId;
        }

        $url = $this->buildUrl('player_api.php', $params);
        return $this->request($url) ?? [];
    }

    /**
     * Get VOD streams (movies)
     */
    public function getVODStreams(?int $categoryId = null): array
    {
        $params = ['action' => self::ACTIONS['vod_streams']];
        if ($categoryId !== null) {
            $params['category_id'] = $categoryId;
        }

        $url = $this->buildUrl('player_api.php', $params);
        $result = $this->request($url) ?? [];

        // Remove metadata from results
        unset($result['_meta']);

        return $result;
    }

    /**
     * Get all VOD streams with pagination support
     */
    public function getAllVODStreams(callable $progressCallback = null): array
    {
        $categories = $this->getVODCategories();
        unset($categories['_meta']);

        $allMovies = [];
        $total = count($categories);
        $current = 0;

        foreach ($categories as $category) {
            $current++;
            $categoryId = $category['category_id'] ?? null;

            if ($categoryId === null) continue;

            $movies = $this->getVODStreams((int) $categoryId);

            foreach ($movies as $movie) {
                if (is_array($movie) && isset($movie['stream_id'])) {
                    $movie['category_name'] = $category['category_name'] ?? '';
                    $allMovies[$movie['stream_id']] = $movie;
                }
            }

            if ($progressCallback) {
                $progressCallback($current, $total, $category['category_name'] ?? '');
            }

            // Small delay to avoid rate limiting
            usleep(100000); // 100ms
        }

        return array_values($allMovies);
    }

    /**
     * Get series list
     */
    public function getSeries(?int $categoryId = null): array
    {
        $params = ['action' => self::ACTIONS['series']];
        if ($categoryId !== null) {
            $params['category_id'] = $categoryId;
        }

        $url = $this->buildUrl('player_api.php', $params);
        return $this->request($url) ?? [];
    }

    /**
     * Get series info (seasons, episodes)
     */
    public function getSeriesInfo(int $seriesId): ?array
    {
        $params = [
            'action' => self::ACTIONS['series_info'],
            'series_id' => $seriesId
        ];

        $url = $this->buildUrl('player_api.php', $params);
        return $this->request($url);
    }

    /**
     * Get VOD info (movie details)
     */
    public function getVODInfo(int $vodId): ?array
    {
        $params = [
            'action' => self::ACTIONS['vod_info'],
            'vod_id' => $vodId
        ];

        $url = $this->buildUrl('player_api.php', $params);
        return $this->request($url);
    }

    /**
     * Build stream URL for live
     */
    public function getLiveStreamUrl(int $streamId, string $extension = 'm3u8'): string
    {
        $protocol = $this->useHttps ? 'https' : 'http';
        return "{$protocol}://{$this->baseUrl}:{$this->port}/live/{$this->username}/{$this->password}/{$streamId}.{$extension}";
    }

    /**
     * Build stream URL for VOD/movie
     */
    public function getVODStreamUrl(int $streamId, string $extension = 'mp4'): string
    {
        $protocol = $this->useHttps ? 'https' : 'http';
        return "{$protocol}://{$this->baseUrl}:{$this->port}/movie/{$this->username}/{$this->password}/{$streamId}.{$extension}";
    }

    /**
     * Build stream URL for series episode
     */
    public function getSeriesStreamUrl(int $streamId, string $extension = 'mp4'): string
    {
        $protocol = $this->useHttps ? 'https' : 'http';
        return "{$protocol}://{$this->baseUrl}:{$this->port}/series/{$this->username}/{$this->password}/{$streamId}.{$extension}";
    }

    /**
     * Check if stream URL is working
     */
    public function checkStreamUrl(string $url, int $timeout = 10): array
    {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_NOBODY => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => $timeout,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
        ]);

        curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $contentLength = curl_getinfo($ch, CURLINFO_CONTENT_LENGTH_DOWNLOAD);
        $contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
        $responseTime = curl_getinfo($ch, CURLINFO_TOTAL_TIME);
        $error = curl_error($ch);
        curl_close($ch);

        return [
            'is_working' => $httpCode >= 200 && $httpCode < 400,
            'http_code' => $httpCode,
            'content_length' => $contentLength > 0 ? $contentLength : null,
            'content_type' => $contentType,
            'response_time' => round($responseTime * 1000), // ms
            'error' => $error ?: null
        ];
    }

    /**
     * Get M3U playlist URL
     */
    public function getM3UUrl(string $type = 'm3u_plus', string $output = 'ts'): string
    {
        $protocol = $this->useHttps ? 'https' : 'http';
        return "{$protocol}://{$this->baseUrl}:{$this->port}/get.php?" . http_build_query([
                'username' => $this->username,
                'password' => $this->password,
                'type' => $type,
                'output' => $output
            ]);
    }

    /**
     * Get XMLTV EPG URL
     */
    public function getXMLTVUrl(): string
    {
        $protocol = $this->useHttps ? 'https' : 'http';
        return "{$protocol}://{$this->baseUrl}:{$this->port}/xmltv.php?" . http_build_query([
                'username' => $this->username,
                'password' => $this->password
            ]);
    }

    /**
     * Transform VOD data to standard format
     */
    public function transformVODToStandard(array $vod): array
    {
        $name = $vod['name'] ?? '';
        $year = $vod['year'] ?? null;

        // Parse year from name if not provided
        if (!$year && preg_match('/\((\d{4})\)/', $name, $matches)) {
            $year = $matches[1];
            $name = trim(preg_replace('/\s*\(\d{4}\)\s*/', '', $name));
        }

        // Generate clean filename
        $cleanName = preg_replace('/[^\w\s\-]/u', '', $name);
        $cleanName = preg_replace('/\s+/', '.', trim($cleanName));
        if ($year) {
            $cleanName .= " ($year)";
        }
        $extension = $vod['container_extension'] ?? 'mp4';
        $cleanName .= '.' . $extension;

        return [
            'stream_id' => $vod['stream_id'] ?? null,
            'name' => $name,
            'original_name' => $vod['name'] ?? '',
            'year' => $year,
            'clean_name' => $cleanName,
            'category_id' => $vod['category_id'] ?? null,
            'category_ids' => $vod['category_ids'] ?? [],
            'container_extension' => $extension,
            'rating' => $vod['rating'] ?? null,
            'rating_5based' => $vod['rating_5based'] ?? null,
            'poster' => $vod['stream_icon'] ?? null,
            'added_timestamp' => $vod['added'] ?? null,
            'is_adult' => (bool) ($vod['is_adult'] ?? false),
        ];
    }

    /**
     * Get errors
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * Clear errors
     */
    public function clearErrors(): void
    {
        $this->errors = [];
    }

    /**
     * Get connection info for debugging
     */
    public function getConnectionInfo(): array
    {
        return [
            'base_url' => $this->baseUrl,
            'port' => $this->port,
            'username' => $this->username,
            'use_https' => $this->useHttps,
            'timeout' => $this->timeout,
            'is_authenticated' => $this->userInfo !== null,
            'user_status' => $this->userInfo['status'] ?? 'unknown',
            'server_url' => $this->serverInfo['url'] ?? null,
        ];
    }
}
