#############################################
#  VOD Import System - Windows Installation
#  Run as Administrator in PowerShell
#  Usage: .\install_windows.ps1
#############################################

# Check Admin
if (-NOT ([Security.Principal.WindowsPrincipal][Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
    Write-Host "Please run as Administrator!" -ForegroundColor Red
    Exit 1
}

Clear-Host
Write-Host "============================================" -ForegroundColor Cyan
Write-Host "   VOD Import System - Windows Installer" -ForegroundColor Cyan
Write-Host "============================================" -ForegroundColor Cyan
Write-Host ""

# Configuration
$INSTALL_DIR = "C:\xampp\htdocs\VOD"
$DB_NAME = "vod_import"
$DB_USER = "root"
$DB_PASS = ""
$XAMPP_PATH = "C:\xampp"
$PHP_PATH = "$XAMPP_PATH\php"
$MYSQL_PATH = "$XAMPP_PATH\mysql\bin"

#############################################
# Step 1: Check XAMPP
#############################################
Write-Host "[1/6] Checking XAMPP..." -ForegroundColor Green

if (-NOT (Test-Path $XAMPP_PATH)) {
    Write-Host ""
    Write-Host "XAMPP not found at $XAMPP_PATH" -ForegroundColor Yellow
    Write-Host ""
    Write-Host "Please install XAMPP first:" -ForegroundColor Yellow
    Write-Host "  1. Download from: https://www.apachefriends.org/download.html" -ForegroundColor White
    Write-Host "  2. Install to C:\xampp" -ForegroundColor White
    Write-Host "  3. Start Apache and MySQL from XAMPP Control Panel" -ForegroundColor White
    Write-Host "  4. Run this script again" -ForegroundColor White
    Write-Host ""

    $download = Read-Host "Do you want to download XAMPP now? (y/n)"
    if ($download -eq "y") {
        Start-Process "https://www.apachefriends.org/download.html"
    }
    Exit 1
}

Write-Host "  XAMPP found at $XAMPP_PATH" -ForegroundColor Gray

#############################################
# Step 2: Create Directory Structure
#############################################
Write-Host "[2/6] Creating directories..." -ForegroundColor Green

$directories = @(
    "$INSTALL_DIR",
    "$INSTALL_DIR\api",
    "$INSTALL_DIR\assets\css",
    "$INSTALL_DIR\assets\js",
    "$INSTALL_DIR\assets\img",
    "$INSTALL_DIR\classes",
    "$INSTALL_DIR\config",
    "$INSTALL_DIR\cache",
    "$INSTALL_DIR\logs",
    "$INSTALL_DIR\media",
    "$INSTALL_DIR\watch",
    "$INSTALL_DIR\temp",
    "$INSTALL_DIR\scripts"
)

foreach ($dir in $directories) {
    if (-NOT (Test-Path $dir)) {
        New-Item -ItemType Directory -Path $dir -Force | Out-Null
    }
}

Write-Host "  Directories created" -ForegroundColor Gray

#############################################
# Step 3: Create Database
#############################################
Write-Host "[3/6] Creating database..." -ForegroundColor Green

$SQL_SCHEMA = @"
CREATE DATABASE IF NOT EXISTS $DB_NAME CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE $DB_NAME;

-- Users
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100),
    role ENUM('admin', 'user', 'viewer') DEFAULT 'user',
    is_active TINYINT(1) DEFAULT 1,
    api_token VARCHAR(64),
    last_login DATETIME,
    login_ip VARCHAR(45),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Source Servers
CREATE TABLE IF NOT EXISTS servers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    url VARCHAR(255) NOT NULL,
    port INT DEFAULT 80,
    username VARCHAR(100),
    password VARCHAR(100),
    type ENUM('xtream', 'm3u', 'stalker') DEFAULT 'xtream',
    is_active TINYINT(1) DEFAULT 1,
    status ENUM('online', 'offline', 'unknown') DEFAULT 'unknown',
    last_check DATETIME,
    movies_count INT DEFAULT 0,
    series_count INT DEFAULT 0,
    live_count INT DEFAULT 0,
    priority INT DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Remote Servers
CREATE TABLE IF NOT EXISTS remote_servers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    type ENUM('ssh', 'ftp', 'sftp', 'local') DEFAULT 'ssh',
    host VARCHAR(255) NOT NULL,
    port INT DEFAULT 22,
    username VARCHAR(100),
    password VARCHAR(255),
    private_key TEXT,
    remote_path VARCHAR(255) DEFAULT '/media',
    is_active TINYINT(1) DEFAULT 1,
    status ENUM('online', 'offline', 'unknown') DEFAULT 'unknown',
    last_check DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Categories
CREATE TABLE IF NOT EXISTS categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    server_id INT,
    external_id INT,
    name VARCHAR(100) NOT NULL,
    type ENUM('movie', 'series', 'live') DEFAULT 'movie',
    parent_id INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (server_id) REFERENCES servers(id) ON DELETE CASCADE,
    UNIQUE KEY unique_cat (server_id, external_id, type)
) ENGINE=InnoDB;

-- Movies
CREATE TABLE IF NOT EXISTS movies (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tmdb_id INT,
    title VARCHAR(255) NOT NULL,
    clean_name VARCHAR(255),
    original_title VARCHAR(255),
    year INT,
    plot TEXT,
    genre VARCHAR(255),
    director VARCHAR(255),
    cast TEXT,
    rating DECIMAL(3,1),
    duration INT,
    poster VARCHAR(500),
    backdrop VARCHAR(500),
    trailer_url VARCHAR(500),
    is_downloaded TINYINT(1) DEFAULT 0,
    local_path VARCHAR(500),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_title (title),
    INDEX idx_year (year)
) ENGINE=InnoDB;

-- Movie Links
CREATE TABLE IF NOT EXISTS movie_links (
    id INT AUTO_INCREMENT PRIMARY KEY,
    movie_id INT NOT NULL,
    server_id INT NOT NULL,
    stream_id INT,
    category_id INT,
    url VARCHAR(500) NOT NULL,
    container_extension VARCHAR(10) DEFAULT 'mp4',
    quality VARCHAR(20),
    is_working TINYINT(1) DEFAULT 1,
    last_check DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (movie_id) REFERENCES movies(id) ON DELETE CASCADE,
    FOREIGN KEY (server_id) REFERENCES servers(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Series
CREATE TABLE IF NOT EXISTS series (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tmdb_id INT,
    imdb_id VARCHAR(20),
    title VARCHAR(255) NOT NULL,
    original_title VARCHAR(255),
    year INT,
    clean_name VARCHAR(255),
    description TEXT,
    plot TEXT,
    genre VARCHAR(255),
    cast TEXT,
    rating DECIMAL(3,1),
    poster VARCHAR(500),
    backdrop VARCHAR(500),
    seasons_count INT DEFAULT 0,
    episodes_count INT DEFAULT 0,
    category_id INT,
    status VARCHAR(20) DEFAULT 'ongoing',
    is_adult TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_title (title)
) ENGINE=InnoDB;

-- Episodes
CREATE TABLE IF NOT EXISTS episodes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    series_id INT NOT NULL,
    server_id INT,
    stream_id INT,
    season_number INT NOT NULL,
    episode_number INT NOT NULL,
    title VARCHAR(255),
    clean_name VARCHAR(255),
    description TEXT,
    duration INT DEFAULT 0,
    stream_url VARCHAR(500),
    container_extension VARCHAR(10) DEFAULT 'mp4',
    poster VARCHAR(500),
    air_date DATE,
    rating DECIMAL(3,1),
    is_downloaded TINYINT(1) DEFAULT 0,
    download_path VARCHAR(500),
    is_orphan TINYINT(1) DEFAULT 0,
    status VARCHAR(20) DEFAULT 'available',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (series_id) REFERENCES series(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Download Jobs
CREATE TABLE IF NOT EXISTS download_jobs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    remote_server_id INT NOT NULL,
    movie_id INT,
    episode_id INT,
    source_url VARCHAR(500) NOT NULL,
    filename VARCHAR(255) NOT NULL,
    destination_path VARCHAR(500),
    status ENUM('pending', 'downloading', 'completed', 'failed', 'cancelled') DEFAULT 'pending',
    progress INT DEFAULT 0,
    file_size BIGINT,
    downloaded_size BIGINT DEFAULT 0,
    error_message TEXT,
    started_at DATETIME,
    completed_at DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (remote_server_id) REFERENCES remote_servers(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Logs
CREATE TABLE IF NOT EXISTS import_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    server_id INT,
    action VARCHAR(50),
    type VARCHAR(50),
    target_id INT,
    target_name VARCHAR(255),
    status ENUM('success', 'error', 'warning', 'info') DEFAULT 'info',
    message TEXT,
    details JSON,
    ip_address VARCHAR(45),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Settings
CREATE TABLE IF NOT EXISTS settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    `key` VARCHAR(50) NOT NULL UNIQUE,
    `value` TEXT,
    `type` ENUM('string', 'int', 'bool', 'json') DEFAULT 'string',
    description VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Default Admin (password: admin123)
INSERT INTO users (username, password, email, role) VALUES
('admin', '\$2y\$12\$LQv3c1yqBWVHxkd0LHAkCOYz6TtxMQJqhN8/X4x5gL1GpLqEKRnwq', 'admin@localhost', 'admin')
ON DUPLICATE KEY UPDATE username=username;

-- Default Settings
INSERT INTO settings (`key`, `value`, `type`) VALUES
('site_name', 'VOD Import System', 'string'),
('items_per_page', '24', 'int'),
('auto_download_movies', '0', 'bool'),
('auto_download_series', '0', 'bool')
ON DUPLICATE KEY UPDATE `key`=`key`;
"@

$sqlFile = "$env:TEMP\vod_schema.sql"
$SQL_SCHEMA | Out-File -FilePath $sqlFile -Encoding UTF8

try {
    & "$MYSQL_PATH\mysql.exe" -u $DB_USER -e "source $sqlFile" 2>$null
    Write-Host "  Database created successfully" -ForegroundColor Gray
} catch {
    Write-Host "  Warning: Could not create database. Make sure MySQL is running." -ForegroundColor Yellow
}

Remove-Item $sqlFile -Force -ErrorAction SilentlyContinue

#############################################
# Step 4: Create Config File
#############################################
Write-Host "[4/6] Creating configuration..." -ForegroundColor Green

$CONFIG_PHP = @"
<?php
/**
 * VOD Import System Configuration
 * Windows Installation
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/../logs/php_errors.log');

date_default_timezone_set('UTC');

// Database
define('DB_HOST', 'localhost');
define('DB_NAME', '$DB_NAME');
define('DB_USER', '$DB_USER');
define('DB_PASS', '$DB_PASS');
define('DB_CHARSET', 'utf8mb4');

// Paths
define('BASE_PATH', dirname(__DIR__));
define('CONFIG_PATH', BASE_PATH . '/config');
define('CACHE_PATH', BASE_PATH . '/cache');
define('LOGS_PATH', BASE_PATH . '/logs');
define('MEDIA_PATH', BASE_PATH . '/media');
define('WATCH_PATH', BASE_PATH . '/watch');

// API
define('API_VERSION', '2.0');
define('DEBUG_MODE', true);

function getDB(): PDO {
    static `$pdo = null;
    if (`$pdo === null) {
        `$dsn = sprintf('mysql:host=%s;dbname=%s;charset=%s', DB_HOST, DB_NAME, DB_CHARSET);
        `$pdo = new PDO(`$dsn, DB_USER, DB_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false
        ]);
    }
    return `$pdo;
}

function getSetting(string `$key, `$default = null) {
    static `$settings = null;
    if (`$settings === null) {
        try {
            `$stmt = getDB()->query("SELECT ``key``, ``value``, ``type`` FROM settings");
            `$settings = [];
            while (`$row = `$stmt->fetch()) {
                `$value = `$row['value'];
                switch (`$row['type']) {
                    case 'int': `$value = (int) `$value; break;
                    case 'bool': `$value = (bool) `$value; break;
                    case 'json': `$value = json_decode(`$value, true); break;
                }
                `$settings[`$row['key']] = `$value;
            }
        } catch (Exception `$e) { return `$default; }
    }
    return `$settings[`$key] ?? `$default;
}

function setSetting(string `$key, `$value, string `$type = 'string'): bool {
    if (`$type === 'json') `$value = json_encode(`$value);
    elseif (`$type === 'bool') `$value = `$value ? '1' : '0';
    `$stmt = getDB()->prepare("INSERT INTO settings (``key``, ``value``, ``type``) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE ``value`` = VALUES(``value``)");
    return `$stmt->execute([`$key, (string) `$value, `$type]);
}

spl_autoload_register(function (`$class) {
    `$file = BASE_PATH . '/classes/' . `$class . '.php';
    if (file_exists(`$file)) require_once `$file;
});

foreach ([CACHE_PATH, LOGS_PATH, MEDIA_PATH, WATCH_PATH] as `$dir) {
    if (!is_dir(`$dir)) @mkdir(`$dir, 0755, true);
}
"@

$CONFIG_PHP | Out-File -FilePath "$INSTALL_DIR\config\config.php" -Encoding UTF8

Write-Host "  Configuration created" -ForegroundColor Gray

#############################################
# Step 5: Download Files
#############################################
Write-Host "[5/6] Downloading application files..." -ForegroundColor Green

$DOWNLOAD_URL = Read-Host "Enter the ZIP download URL (or press Enter to skip)"

if ($DOWNLOAD_URL) {
    $zipFile = "$env:TEMP\vod_system.zip"

    try {
        Write-Host "  Downloading..." -ForegroundColor Gray
        Invoke-WebRequest -Uri $DOWNLOAD_URL -OutFile $zipFile -UseBasicParsing

        Write-Host "  Extracting..." -ForegroundColor Gray
        Expand-Archive -Path $zipFile -DestinationPath "$env:TEMP\vod_extract" -Force

        # Copy files
        if (Test-Path "$env:TEMP\vod_extract\VOD") {
            Copy-Item -Path "$env:TEMP\vod_extract\VOD\*" -Destination $INSTALL_DIR -Recurse -Force
        }

        Remove-Item $zipFile -Force -ErrorAction SilentlyContinue
        Remove-Item "$env:TEMP\vod_extract" -Recurse -Force -ErrorAction SilentlyContinue

        Write-Host "  Files downloaded and extracted" -ForegroundColor Gray
    } catch {
        Write-Host "  Download failed. Please copy files manually." -ForegroundColor Yellow
    }
} else {
    Write-Host "  Skipped. Copy files manually to: $INSTALL_DIR" -ForegroundColor Yellow
}

#############################################
# Step 6: Create Shortcuts
#############################################
Write-Host "[6/6] Creating shortcuts..." -ForegroundColor Green

# Create batch file to open in browser
$batchContent = @"
@echo off
start http://localhost/VOD/
"@
$batchContent | Out-File -FilePath "$INSTALL_DIR\Open VOD Panel.bat" -Encoding ASCII

# Create desktop shortcut
$WshShell = New-Object -ComObject WScript.Shell
$Shortcut = $WshShell.CreateShortcut("$env:USERPROFILE\Desktop\VOD Panel.lnk")
$Shortcut.TargetPath = "http://localhost/VOD/"
$Shortcut.Save()

Write-Host "  Shortcuts created" -ForegroundColor Gray

#############################################
# Complete
#############################################
Write-Host ""
Write-Host "============================================" -ForegroundColor Green
Write-Host "       Installation Complete!" -ForegroundColor Green
Write-Host "============================================" -ForegroundColor Green
Write-Host ""
Write-Host "Access URL: " -NoNewline
Write-Host "http://localhost/VOD/" -ForegroundColor Cyan
Write-Host ""
Write-Host "Admin Login:" -ForegroundColor Yellow
Write-Host "  Username: admin"
Write-Host "  Password: admin123"
Write-Host ""
Write-Host "Install Directory: $INSTALL_DIR" -ForegroundColor Gray
Write-Host ""
Write-Host "IMPORTANT:" -ForegroundColor Red
Write-Host "  1. Make sure Apache and MySQL are running in XAMPP"
Write-Host "  2. Change admin password after first login!"
Write-Host ""

# Save credentials
$credentials = @"
============================================
VOD Import System - Windows Installation
============================================
Date: $(Get-Date)

URL: http://localhost/VOD/

Admin Login:
  Username: admin
  Password: admin123

Database:
  Name: $DB_NAME
  User: $DB_USER

Install Path: $INSTALL_DIR
============================================
"@
$credentials | Out-File -FilePath "$INSTALL_DIR\credentials.txt" -Encoding UTF8

Write-Host "Credentials saved to: $INSTALL_DIR\credentials.txt" -ForegroundColor Gray
Write-Host ""

# Open browser
$openBrowser = Read-Host "Open VOD Panel in browser now? (y/n)"
if ($openBrowser -eq "y") {
    Start-Process "http://localhost/VOD/"
}
