<?php
/**
 * API principale - FULL Xtream Codes Integration
 * Works as a proxy between the app and Xtream Panel
 */

header('Content-Type: application/json; charset=UTF-8');
require_once 'db_helper.php';
require_once 'xtream_helper.php';

$methode = isset($_GET['methode']) ? $_GET['methode'] : '';
$rawData = file_get_contents('php://input');
$postData = json_decode($rawData, true);

// Fix escaped exclamation marks
if ($postData === null && !empty($rawData)) {
    $fixedData = str_replace('\\!', '!', $rawData);
    $postData = json_decode($fixedData, true);
}

// Debug logging (after fixing JSON)
$logFile = __DIR__ . '/debug_log.txt';
$debugInfo = array(
    'timestamp' => date('Y-m-d H:i:s'),
    'method' => $_SERVER['REQUEST_METHOD'],
    'url' => $_SERVER['REQUEST_URI'],
    'get_methode' => $methode,
    'raw_input' => $rawData,
    'parsed_json' => $postData,
    'note' => 'Use format: username!!!password (3 exclamation marks)'
);
file_put_contents($logFile, json_encode($debugInfo, JSON_PRETTY_PRINT) . "\n---\n", FILE_APPEND);

if ($postData === null) {
    $postData = array();
}

function getCurrentDate() {
    return date('y-m-d H:i:s');
}

// Extract Xtream credentials from POST data
function getXtreamCredentials($postData) {
    $username = null;
    $password = null;

    // From xtream_username/xtream_password fields
    if (isset($postData['xtream_username'])) $username = $postData['xtream_username'];
    if (isset($postData['xtream_password'])) $password = $postData['xtream_password'];

    // From username/password fields
    if (!$username && isset($postData['username'])) $username = $postData['username'];
    if (!$password && isset($postData['password'])) $password = $postData['password'];

    // From code field (username!!!password)
    if (!$username && isset($postData['code'])) {
        $code = $postData['code'];

        if (strpos($code, '!!!') !== false) {
            // Format: username!!!password
            list($username, $password) = explode('!!!', $code, 2);
        } else if (strlen($code) > 0 && strlen($code) % 2 == 0) {
            // Check if it's username repeated twice (e.g., "12312311231231" -> "1231231" + "1231231")
            $half = strlen($code) / 2;
            $first_half = substr($code, 0, $half);
            $second_half = substr($code, $half);

            if ($first_half === $second_half) {
                // Code is username repeated twice
                $username = $first_half;
                $password = $first_half;
            } else {
                // Use the whole code as both username and password
                $username = $code;
                $password = $code;
            }
        } else {
            // Use the whole code as both username and password
            $username = $code;
            $password = $code;
        }
    }

    // Default credentials if not provided
    if (!$username) $username = '1231231';
    if (!$password) $password = '1231231';

    return array($username, $password);
}

// Note: التطبيق يرسل android_id في كل الطلبات، لذلك لا نستخدمه للتفريق بين login و data requests
// نعتمد على methode فقط

// Router
switch ($methode) {
    case 'login':
    case 'restore':
    case '':
        // Login/Restore endpoint
        if (isset($postData['code']) && isset($postData['android_id'])) {
            $code = $postData['code'];
            $macW = isset($postData['macW']) ? $postData['macW'] : '';
            $macE = isset($postData['macE']) ? $postData['macE'] : '';
            $android_id = $postData['android_id'];

            // Parse code - same logic as getXtreamCredentials
            $username = '';
            $password = '';

            if (strpos($code, '!!!') !== false) {
                // Format: username!!!password
                list($username, $password) = explode('!!!', $code, 2);
            } else if (strlen($code) > 0 && strlen($code) % 2 == 0) {
                // Check if it's username repeated twice (e.g., "12312311231231" -> "1231231" + "1231231")
                $half = strlen($code) / 2;
                $first_half = substr($code, 0, $half);
                $second_half = substr($code, $half);

                if ($first_half === $second_half) {
                    // Code is username repeated twice
                    $username = $first_half;
                    $password = $first_half;
                } else {
                    // Use the whole code as both username and password
                    $username = $code;
                    $password = $code;
                }
            } else {
                // Use the whole code as both username and password
                $username = $code;
                $password = $code;
            }

            if (empty($username)) $username = '1231231';
            if (empty($password)) $password = '1231231';

            // Authenticate
            $config = DB::getConfig();
            $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];
            $xtream = new XtreamAPI($xtream_url, $username, $password);
            $auth = $xtream->authenticate();

            if ($auth && isset($auth['username'])) {
                $user = array(
                    "id" => "1",
                    "username" => $username,
                    "password" => $password,
                    "code" => $code,
                    "macW" => $macW,
                    "macE" => $macE,
                    "android_id" => $android_id,
                    "account_expire_date" => isset($auth['account_expire_date']) ? $auth['account_expire_date'] : 'Unlimited',
                    "account_expire_timestamp" => isset($auth['account_expire_timestamp']) ? $auth['account_expire_timestamp'] : null,
                    "active" => true,
                    "status" => "active",
                    "created_at" => date('Y-m-d H:i:s'),
                    "server_info" => $auth
                );
                echo json_encode($user);
                exit;
            } else {
                http_response_code(401);
                echo json_encode(array("error" => "Authentication failed"));
                exit;
            }
        }
        break;

    case '1':
        // Method 1: Return ALL data (LIVE from Xtream)
        list($username, $password) = getXtreamCredentials($postData);
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        $xtream = new XtreamAPI($xtream_url, $username, $password);

        // Get all categories
        $all_categories = $xtream->getAllCategories();
        $formatted_categories = $xtream->transformCategoriesToAppFormat($all_categories);

        // Debug logging
        error_log("Method 1: Got " . count($all_categories) . " categories from Xtream");
        error_log("Method 1: Formatted to " . count($formatted_categories) . " categories");

        // If Xtream returns empty, fallback to db.json
        if (empty($formatted_categories)) {
            error_log("Method 1: Using fallback from db.json");
            $formatted_categories = DB::getBouquets();
        }

        $response = array(
            "rad" => DB::getRadios(),
            "bou" => $formatted_categories,
            "intro" => DB::getIntro(),
            "main" => DB::getMain(),
            "cat" => $formatted_categories,
            "date" => getCurrentDate()
        );
        echo json_encode($response);
        break;

    case '5':
        // Method 5: Return configuration
        $config = DB::getConfig();
        echo json_encode($config);
        break;

    case '2':
        // Method 2: Get bouquets (LIVE from Xtream)
        list($username, $password) = getXtreamCredentials($postData);
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        $xtream = new XtreamAPI($xtream_url, $username, $password);
        $all_categories = $xtream->getAllCategories();
        $formatted_categories = $xtream->transformCategoriesToAppFormat($all_categories);

        if (empty($formatted_categories)) {
            $formatted_categories = DB::getBouquets();
        }

        $response = array(
            "bou" => $formatted_categories,
            "date" => getCurrentDate()
        );
        echo json_encode($response);
        break;

    case '3':
        // Method 3: Get categories (LIVE from Xtream)
        list($username, $password) = getXtreamCredentials($postData);
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        $xtream = new XtreamAPI($xtream_url, $username, $password);
        $all_categories = $xtream->getAllCategories();
        $formatted_categories = $xtream->transformCategoriesToAppFormat($all_categories);

        if (empty($formatted_categories)) {
            $formatted_categories = DB::getCategories();
        }

        $response = array(
            "cat" => $formatted_categories,
            "date" => getCurrentDate()
        );
        echo json_encode($response);
        break;

    case '4':
        // Method 4: Get radios
        $response = array(
            "rad" => DB::getRadios(),
            "date" => getCurrentDate()
        );
        echo json_encode($response);
        break;

    case '6':
        // Method 6: Get main menu
        $response = array(
            "main" => DB::getMain(),
            "date" => getCurrentDate()
        );
        echo json_encode($response);
        break;

    case '7':
        // Method 7: Get intro
        $response = array(
            "intro" => DB::getIntro(),
            "date" => getCurrentDate()
        );
        echo json_encode($response);
        break;

    case '8':
        // Method 8: App info
        $appInfo = DB::getAppInfo();
        echo json_encode($appInfo);
        break;

    case '10':
        // Method 10: Get live channels by category
        list($username, $password) = getXtreamCredentials($postData);
        $category_id = isset($postData['category_id']) ? $postData['category_id'] : null;
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        $xtream = new XtreamAPI($xtream_url, $username, $password);
        $streams = $xtream->getLiveStreams($category_id);

        if (is_array($streams)) {
            $formatted_streams = $xtream->transformStreamsToAppFormat($streams, 'live');
            $response = array(
                "channels" => $formatted_streams,
                "date" => getCurrentDate()
            );
        } else {
            $response = array(
                "channels" => array(),
                "date" => getCurrentDate()
            );
        }
        echo json_encode($response);
        break;

    case '11':
        // Method 11: Get VOD by category
        list($username, $password) = getXtreamCredentials($postData);
        $category_id = isset($postData['category_id']) ? $postData['category_id'] : null;
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        $xtream = new XtreamAPI($xtream_url, $username, $password);
        $streams = $xtream->getVODStreams($category_id);

        if (is_array($streams)) {
            $formatted_streams = $xtream->transformStreamsToAppFormat($streams, 'vod');
            $response = array(
                "movies" => $formatted_streams,
                "date" => getCurrentDate()
            );
        } else {
            $response = array(
                "movies" => array(),
                "date" => getCurrentDate()
            );
        }
        echo json_encode($response);
        break;

    case '12':
        // Method 12: Get series by category
        list($username, $password) = getXtreamCredentials($postData);
        $category_id = isset($postData['category_id']) ? $postData['category_id'] : null;
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        $xtream = new XtreamAPI($xtream_url, $username, $password);
        $series = $xtream->getSeries($category_id);

        if (is_array($series)) {
            $response = array(
                "series" => $series,
                "date" => getCurrentDate()
            );
        } else {
            $response = array(
                "series" => array(),
                "date" => getCurrentDate()
            );
        }
        echo json_encode($response);
        break;

    case '13':
        // Method 13: Get series info (seasons and episodes)
        list($username, $password) = getXtreamCredentials($postData);
        $series_id = isset($postData['series_id']) ? $postData['series_id'] : null;
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        $xtream = new XtreamAPI($xtream_url, $username, $password);
        $series_info = $xtream->getSeriesInfo($series_id);

        if ($series_info) {
            $response = array(
                "series_info" => $series_info,
                "date" => getCurrentDate()
            );
        } else {
            $response = array(
                "series_info" => null,
                "date" => getCurrentDate()
            );
        }
        echo json_encode($response);
        break;

    case '14':
        // Method 14: Authentication check
        list($username, $password) = getXtreamCredentials($postData);
        $config = DB::getConfig();
        $xtream_url = isset($config['xtream_url']) ? $config['xtream_url'] : $config['str4'];

        $xtream = new XtreamAPI($xtream_url, $username, $password);
        $auth = $xtream->authenticate();

        if ($auth) {
            $response = array(
                "status" => "success",
                "user_info" => $auth,
                "date" => getCurrentDate()
            );
        } else {
            $response = array(
                "status" => "failed",
                "message" => "Invalid credentials",
                "date" => getCurrentDate()
            );
        }
        echo json_encode($response);
        break;

    default:
        echo json_encode(array(
            "error" => "Unknown method: $methode",
            "available_methods" => array(
                "1" => "Get all data",
                "2" => "Get bouquets",
                "3" => "Get categories",
                "4" => "Get radios",
                "5" => "Get config",
                "6" => "Get main menu",
                "7" => "Get intro",
                "8" => "Get app info",
                "10" => "Get live channels",
                "11" => "Get VOD movies",
                "12" => "Get series",
                "13" => "Get series info",
                "14" => "Authenticate"
            )
        ));
        break;
}
?>
